// $RCSfile: JavaTalkServer.java,v $
// $Id: JavaTalkServer.java,v 1.17 1996/06/27 23:43:21 fms Exp fms $
// by Frank Stajano, http://www.cam-orl.co.uk/~fms
// Development started on 1996 05 09
// (c) Olivetti Research Limited

/*
    Copyright (c) 1996 Olivetti Research Limited

    Permission is hereby granted, without written agreement and
    without license or royalty fees, to use, copy, modify, and
    distribute this software and its documentation for any purpose,
    provided that the above copyright notice and the following three
    paragraphs appear in all copies of this software, its documentation
    and any derivative work.

    In no event shall Olivetti Research Limited be liable to any party
    for direct, indirect, special, incidental, or consequential damages
    arising out of the use of this software and its documentation, even
    if Olivetti Research Limited has been advised of the possibility of
    such damage.

    Olivetti Research Limited specifically disclaims any warranties,
    including, but not limited to, the implied warranties of
    merchantability and fitness for a particular purpose. The
    software provided hereunder is on an "as is" basis, and Olivetti
    Research Limited has no obligation to provide maintenance,
    support, updates, enhancements, or modifications.

    Derived or altered versions must be plainly marked as such, and
    must not be misrepresented as being the original software.
*/


// TODO:
// make various logging activities be methods of the applet, not the threads
// make broadcast be a method of the applet
// remove debugging printouts
// check if access to participants shouldn't be synchronized
// tell others whether the one who left did so by choice or was kicked out

// ADVANCED TODO
// remember names of participants based on address (1st guess at least)
// implement timestamp echo protocol
// synchronise clocks and compute half-paths
// non-positional handling of command-line arguments


import java.io.*;
import java.net.*;
import java.util.*;

public class JavaTalkServer extends Thread implements Trigger {
    // These are automatically substituted
    public static final String rcsVersion = "$Revision: 1.17 $";
    public static final String rcsDate = "$Date: 1996/06/27 23:43:21 $";
    public static final String rcsProgram = "$RCSfile: JavaTalkServer.java,v $";

    // These extract the useful parts from the automatically substituted ones.
    public static final String version = rcsVersion.substring(11, rcsVersion.length()-2);
    public static final String date = rcsDate.substring(7, 17);
    public static final String program = rcsProgram.substring(10, rcsProgram.length()-9);
    public static final String about = "\n  " + program + " v. " + version
            + " of " + date + "\n  by Frank Stajano, (C) Olivetti Research Limited\n";

    public final static int DEFAULT_PORT = 6764;
    public final static int WARNING_IF_INACTIVE_FOR = 5*60;
    public final static int TIME_TO_LIVE_AFTER_WARNING = 60;
        // seconds before a connection is closed
        // (later transformed into milliseconds)
    public final static int REAPER_PERIOD = 10*1000;
        // milliseconds between checks of who is inactive

    protected int warningIfInactiveFor = WARNING_IF_INACTIVE_FOR;
    protected int timeToLiveAfterWarning = TIME_TO_LIVE_AFTER_WARNING;
    protected int port = DEFAULT_PORT;
    protected String logFileName = "javatalk." + Time.ymdhms(false) + ".log";

    protected int maxInactivity;
    protected PrintStream log;
    protected ServerSocket listen_socket;
    protected Hashtable participants;
    protected Ticker ticker;

    public static void main(String[] args) {
        //System.err.println("main() called");
        System.out.println(about);

        new JavaTalkServer(args);
    }

    public static void usage() {
        System.out.println ("USAGE (all arguments optional and positional):\n"
                + "  java " + program
                + " warningIfInactiveFor timeToLiveAfterWarning port logFileName\n"
                + "    warningIfInactiveFor     After this many seconds of inactivity: warning\n"
                + "    timeToLiveAfterWarning   After this many extra seconds: kicked out\n"
                + "    port                     Port number on which to listen for connections\n"
                + "    logFileName              A transcript of the talk goes here\n"
        );
        System.exit(1);
    }


    public JavaTalkServer(String[] args) {

        // parse arguments
        switch (args.length) {
            case 4:
                logFileName = args[3];
                // fallthrough
            case 3:
                try {
                    port = Integer.parseInt(args[2]);
                }
                catch (NumberFormatException e) {
                    usage();
                }
                // fallthrough
            case 2:
                try {
                    timeToLiveAfterWarning = Integer.parseInt(args[1]);
                }
                catch (NumberFormatException e) {
                    usage();
                }
                // fallthrough
            case 1:
                try {
                    warningIfInactiveFor = Integer.parseInt(args[0]);
                }
                catch (NumberFormatException e) {
                    usage();
                }
                // fallthrough
            case 0:
                break;

            default:
                usage();
        }


        // Print out the actual parameters for this invocation
        System.out.println (
                  "\nwarningIfInactiveFor     " + warningIfInactiveFor
                + "\ntimeToLiveAfterWarning   " + timeToLiveAfterWarning
                + "\nport                     " + port
                + "\nlogFileName              " + logFileName
                + "\n\nServer running and waiting for connections..."
        );

        // transform seconds into milliseconds, because that's what
        // the Java library routines eat.
        warningIfInactiveFor *= 1000;
        timeToLiveAfterWarning *= 1000;

        maxInactivity = warningIfInactiveFor + timeToLiveAfterWarning;

        try {
            log = new PrintStream(new FileOutputStream(logFileName));
        }
        catch (IOException e) {
            fail(e, "Can't open log file '" + logFileName + "' for writing.");
        }
        log.println(about);
        Date d = new Date();
        log.println("Server started on " + d.toGMTString() + " ("
                + Time.ymdhms(d, true) + " server's local time)\n" );
        log.flush();

        try {
            listen_socket = new ServerSocket(port);
        }
        catch (IOException e) {
            fail(e, "Exception creating server socket");
        }

        participants = new Hashtable();

        ticker = new Ticker(REAPER_PERIOD, this);

        this.start();
    }

    public void run() {
        try {
            while(true) {
                Socket client_socket = listen_socket.accept();
                Connection c = new Connection(client_socket, this);
            }
        }
        catch (IOException e) {
            fail(e, "Exception while listening for connections");
        }
    }


    public static void fail(Exception e, String msg) {
        System.err.println(msg + ": " +  e);
        System.exit(1);
    }


    public void tick() {
        // All the connections that have received no data for longer
        // than the timeout shall be killed.
        Vector toBeKilled = new Vector();
        Date d = new Date();
        long now = d.getTime();
        Enumeration e = participants.keys();
        while (e.hasMoreElements()) {
            String key = (String) e.nextElement();
            Connection thisClient = (Connection) participants.get(key);
            long inactivity = now - thisClient.timeOfLastActivity;
            if (inactivity > maxInactivity) {
                toBeKilled.addElement(thisClient);
            } else if (inactivity > warningIfInactiveFor) {
                thisClient.out.println("INACTIVITY WARNING: you'll be kicked off in "
                        + ((maxInactivity - inactivity)/1000) + " seconds");
                thisClient.out.flush();
            }
        }
        // now we've finished with participants, we can safely
        // do things that remove stuff from it.


        for (int i= 0; i < toBeKilled.size(); i++) {
            Connection thisClient = (Connection) toBeKilled.elementAt(i);
            // TODO: broadcast that we're throwing him out for inactivity
            //System.err.println ("Killing " + thisClient.clientName);
            try {
                thisClient.client.close();
                //System.err.println ("Client killed ok");
            }
            catch (IOException e2) {
                fail (e2, "Couldn't kill client");
            }
        }
    }

}





// For every chat client that logs in, there is a new connection thread.
class Connection extends Thread {
    protected Socket client;
    protected DataInputStream in;
    protected PrintStream out;
    protected InetAddress clientAddress;
    protected int clientPort;
    protected String clientName;
    protected long timeOfLastActivity;

    protected Hashtable participants;
    protected PrintStream log;

    public static void fail(Exception e, String msg) {
        System.err.println(msg + ": " +  e);
        System.exit(1);
    }

    // Initialize the streams and start the thread
    public Connection(Socket client_socket, JavaTalkServer server) {
        logActivity("Accepted incoming connection, spawned thread.");
        client = client_socket;
        timeOfLastActivity = (new Date()).getTime();
        participants = server.participants;
        log = server.log;

        try {
            in = new DataInputStream(client.getInputStream());
            logActivity("Successfully created input stream for new socket");
            out = new PrintStream(client.getOutputStream());
            logActivity("Successfully created output stream for new socket");
        }
        catch (IOException e) {
            try {
                client.close();
                logActivity("Successfully closed client socket");
            }
            catch (IOException e2) {
            }
            logException(e, "Exception while getting socket streams");
            //??? how to you kill this thread before having started it?
            this.stop();
            return;
        }


        clientAddress = client.getInetAddress();
        clientPort = client.getPort();
        clientName = clientAddress + ":" + clientPort;
        participants.put(clientName, this);
        logActivity("Got a connection from " + clientName);
        out.println(server.about);
        broadcast ("User " + clientName + " joined the conference.");
        logPrint(whoString());
        this.start();
    }


    public String whoString() {
        String result ="\n  CURRENTLY CONNECTED:\n";
        Enumeration e = participants.keys();
        while (e.hasMoreElements()) {
            String key = (String) e.nextElement();
            Connection thisClient = (Connection) participants.get(key);
            result += "  " + thisClient.clientName
                    + " (from " + key + ")\n";
        }
        return result;
    }

    public void run() {
        String line;
        int len;
        try {
            for(;;) {
                // read in a line
                line = in.readLine();
                logActivity("Received line: " + line);
                timeOfLastActivity = (new Date()).getTime();
                if (line == null) {
                    logActivity ("Received empty line: closing down");
                    break;
                }

                // command MYNAME changes the user's name
                if (line.startsWith("MYNAME ")) {
                    String msg;
                    msg = "RENAME: " + clientName + " --> "
                        + line.substring(7) + " ("
                        + clientAddress + ":" + clientPort + ")";
                    clientName = line.substring(7);
                    broadcast(msg);
                    logPrint(whoString());
                    continue;
                }

                // command WHO lists who is connected
                if (line.startsWith("WHO")) {
                    out.println (whoString());
                    out.flush();
                    continue;
                }

                // no command, just text.
                broadcast (clientName + "> " + line);
            }
        }
        catch (IOException e) {
            logException (e, "Exception in main loop talking to "
                + clientName + "(" + clientAddress
                + ":" + clientPort + ")");
        }
        finally {
            logActivity ("Connection to "
                + clientAddress + ":" + clientPort
                + " not working, I'll close it");
            try {
                client.close();
                logActivity ("Successfully closed broken socket for "
                    + clientAddress + ":" + clientPort);
            }
            catch (IOException e2) {
                logException (e2, "Exception while closing socket for "
                    + clientAddress + ":" + clientPort);
            }
            finally {
                client = null;
                logActivity("Broken socket forgotten");
                participants.remove(clientAddress + ":" + clientPort);
                broadcast ("User " + clientName + " ("
                    + clientAddress + ":" + clientPort
                    + ") left the conference.");
                logPrint(whoString());
            }
        }
    }

    public void logException (Exception e, String message) {
        System.err.println(message + ": " +  e);
    }

    public void logActivity (String message) {
        System.out.println(message);
    }

    public void logPrint (String message) {
        Date d = new Date();
        log.print("["+  Time.ymdhms(true) + "] " + message);
        log.flush();
    }

    public void broadcast (String s) {
        Enumeration e = participants.keys();
        while (e.hasMoreElements()) {
            String key = (String) e.nextElement();
            Connection thisClient = (Connection) participants.get(key);
            thisClient.out.println(s);
            thisClient.out.flush();
        }
        logActivity ("Broadcasting line: " + s);
        logPrint(s + "\n");
    }
}


class Ticker extends Thread {
    public int period; // period between ticks in milliseconds
    protected Trigger trigger;

    public Ticker (int period, Trigger trigger) {
        this.period = period;
        this.trigger = trigger;
        this.start();
    }

    public void run() {
        for (;;) {
            if (trigger != null) {
                trigger.tick();
            }
            try {
                sleep(period);
            } catch (InterruptedException e) {
                System.err.println("Ticker thread interrupted: " + e);
            }
        }
    }
}


interface Trigger {
    // Something that can be called by a ticker to make things happen.
    public abstract void tick();
}



class Time {
    /*
        NOTE: the year-month-day representation is a Good Thing (tm)
        because it puts the most significant digits first.
        So you can sort ymd strings alphabetically and they will
        be sorted chronologically for free.
    */

    /**
        Returns a string with a ymdhms representation of the supplied
        date, with or without lots of separators between the fields.
        The separators make the string more readable but get in the
        way if you are composing a file name.
    */
    public static String ymdhms(Date d, boolean separators) {
        int y = d.getYear();
        int month = d.getMonth()+1;
        int date = d.getDate();
        int h = d.getHours();
        int m = d.getMinutes();
        int s = d.getSeconds();
        String result = "";
        if (y < 10) {
            result += "0";
        }
        result += y;
        if (month < 10) {
            result += 0;
        }
        result += month;
        if (date < 10) {
            result += "0";
        }
        result += date + "-";
        if (h < 10) {
            result += "0";
        }
        result += h;
        if (separators) {
            result += ":";
        }
        if (m < 10) {
            result += 0;
        }
        result += m;
        if (separators) {
            result += ":";
        }
        if (s < 10) {
            result += "0";
        }
        result += s;
        return result;
    }


    /**
        Returns a string with a ymdhms representation of "now".
    */
    public static String ymdhms(boolean separators) {
        return ymdhms(new Date(), separators);
    }


    public static String hoursMins(Date d) {
        int h = d.getHours();
        int m = d.getMinutes();
        String result = "";
        if (h < 10) {
            result += "0";
        }
        result += h + ":";
        if (m < 10) {
            result += 0;
        }
        result += m;
        return result;
    }

    public static String hoursMins() {
        return hoursMins(new Date());
    }

}


