/*----------------------------------------------------------------------*
 * Bounds Checking for GCC.						*
 * Copyright (C) 1995 Richard W.M. Jones <rwmj@doc.ic.ac.uk>.		*
 *----------------------------------------------------------------------*
 * This program is free software; you can redistribute it and/or modify	*
 * it under the terms of the GNU General Public License as published by	*
 * the Free Software Foundation; either version 2 of the License, or	*
 * (at your option) any later version.					*
 *									*
 * This program is distributed in the hope that it will be useful,	*
 * but WITHOUT ANY WARRANTY; without even the implied warranty of	*
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the	*
 * GNU General Public License for more details.				*
 *									*
 * You should have received a copy of the GNU General Public License	*
 * along with this program; if not, write to the Free Software		*
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.		*
 *----------------------------------------------------------------------*
 * File:
 *	lib/string.c
 * Summary:
 *	Specially checked string and memory functions, replacing those
 *	found in your normal C library.
 * Other notes:
 *	Some of these functions are built into GCC, so you may need to
 *	use the '-fno-builtin' flag to get these checked versions. The
 *	rest of this library uses the '__bounds_*' versions which are
 *	unchecked.
 * Author      	Date		Notes
 * RWMJ		27/2/95		Initial implementation.
 * RWMJ		4/4/95		After profiling, some optimizations.
 *----------------------------------------------------------------------*/

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <ctype.h>

#include "bounds-lib.h"
#include "check.h"

/* Undefine the definitions made in "bounds-lib.h" above. See that header file
 * for comments.
 */
#undef memcpy
#undef memmove
#undef memccpy
#undef memset
#undef memcmp
#undef memchr
#undef strcpy
#undef strncpy
#undef strlen
#undef strcmp
#undef strncmp
#undef strcat
#undef strncat
#undef strpbrk
#undef strrchr
#undef strspn
#undef strcspn
#undef strstr
#undef strtok
#undef strchr
#undef strcoll
#undef strxfrm
#undef strcasecmp
#undef strncasecmp

#if defined(__BOUNDS_CHECKING_ON)
#error "This file must not be compiled with bounds checking enabled."
#endif

/* On `memchr' and `memccpy', it is probably valid to give a `n' (max. len.)
 * argument longer than the object, provided you know the `c' (character)
 * will match always before the end of the object. If you want strict
 * checking of `n', set these to `1'. If you do this, certain valid (?)
 * software fails, in particular GNU's `sprintf'.
 */
#define STRICT_MEMCHR		0
#define STRICT_MEMCCPY		0

#define CASECMP 		0 /* strcasecmp, strncasecmp functions    */

/* Since `bcopy' varies from machine to machine in the way it handles
 * overlapping arguments, you may set its behaviour here. Set
 * `BCOPY_OVERLAPPING_ARGUMENTS' to 1 to get memmove-type behaviour, and
 * to 0 to get memcpy behaviour. Set `BCOPY_WARN_OVERLAPPING_ARGUMENTS'
 * if you want a warning when `bcopy' is called with overlapping arguments
 * (since relying on this is not necessarily portable).
 *
 * - RWMJ 16/1/96
 * - Thanks to Mark W. Snitily (mark@sgcs.com) for clarification here.
 */
#define BCOPY_OVERLAPPING_ARGUMENTS		1
#define BCOPY_WARN_OVERLAPPING_ARGUMENTS	1


static inline void *
check_ptr(void *pointer,char *function,char *ptr_name)
{
  if (pointer == NULL || pointer == ILLEGAL)
    {
      __bounds_errorf (NULL, 0, pointer, NULL,
		       "NULL or ILLEGAL %s used in %s", ptr_name, function);
      return NULL;
    }
  return pointer;
}

/*----------------------------------------------------------------------
 *	Inline function that does the donkey work checking a single
 *	pointer is valid over a range of n bytes. If the pointer is
 *	invalid, the function exits with an appropriate message.
 *	  If size == 0, the pointer points to a string. We work out the
 *	size ourselves here, and check the extent will be OK.
 *----------------------------------------------------------------------*/

static inline size_t
check (void *pointer, size_t size, char *function, char *ptr_name)
{
  object *obj;

  if (pointer == NULL || pointer == ILLEGAL)
    {
      __bounds_errorf (NULL, 0, pointer, NULL,
		       "NULL or ILLEGAL %s used in %s", ptr_name, function);
      return 0;
    }
  if ((obj = __bounds_find_object (pointer)) == NULL) {
    if (maybe_is_unchecked_static (pointer, NULL, 0, function)
	|| maybe_is_unchecked_stack (pointer, NULL, 0, function)) {
      /* Unchecked pointer is OK. We have already delivered a warning at
       * this point. If this is a string, return its length, else return
       * anything.
       */
      if (size)		/* not a string */
	return 0;
      else {		/* unchecked operation => may fail */
	char *p = (char *) pointer;

        while (*p++);
        return p - (char *)pointer;
      }
    }
    __bounds_errorf (NULL, 0, pointer, NULL,
		     "invalid %s used in %s", ptr_name, function);
    return 0;
  }

  if (size) {
    /* The pointer itself is valid, and points to a checked object. Now make
     * sure that we won't overrun the object by using this pointer and size.
     */
    if (pointer + size > obj->extent)
      __bounds_errorf (NULL, 0, pointer, obj,
		       "%s with this %s would overrun the end of the object's allocated memory",
		       function, ptr_name);
    return 0;
  } else {
    /* This is a string.
     * Work out the size ourselves, and whether the pointer will be valid.
     * Return the length of the string + 1.
     */
    char *p = (char *) pointer;

    while (*p++);
    if (p > (char *)obj->extent)
      __bounds_errorf (NULL, 0, pointer, obj,
		       "in %s, %s is a string overrunning the end of the object's allocated memory",
		       function, ptr_name);
    return p - (char *)pointer;
  }
}

/*----------------------------------------------------------------------
 *	Check for overlapping objects in functions (such as memcpy, strcpy)
 *	where this is not permitted.
 *----------------------------------------------------------------------*/

static inline void
check_overlap (void *p1, size_t n1,
	       void *p2, size_t n2, char *function_name)
{
  void *p1e = (void *) ((char *) p1 + n1 - 1);
  void *p2e = (void *) ((char *) p2 + n2 - 1);
  if ((p1 <= p2 && p1e >= p2) ||	/* p1----p2====p1e----p2e */
      (p2 <= p1 && p2e >= p1) ||	/* p2----p1====p2e----p1e */
      (p1 >= p2 && p1e <= p2e) ||	/* p2----p1====p1e----p2e */
      (p2 >= p1 && p2e <= p1e))		/* p1----p2====p2e----p1e */
    __bounds_errorf (NULL, 0, p1, NULL,
		     "in %s, source and destination objects overlap",
		     function_name);
}

/*----------------------------------------------------------------------
 *	These are the checked functions themselves. The '__bounds_*'
 *	versions are internal unchecked functions used by this library
 *	itself (to avoid reentrancy problems).
 *   1.	The memory functions.
 *----------------------------------------------------------------------*/

inline void *
__bounds_memcpy (void *dest, void *src, size_t n)
{
  char *d = (char *) dest;
  char *s = (char *) src;

  for (;n;n--)
    *d++ = *s++;
  return dest;
}

void *
memcpy (void *dest, void *src, size_t n)
{
  if (n == 0 && (dest == NULL || src == NULL))
    {
      if (__bounds_warn_misc_strings)
	__bounds_warning (NULL, 0, "memcpy",
			  "NULL source or destination for a zero-sized memcpy");
      return dest;
    }
  if (n == 0) return dest;

  check (dest, n, "memcpy", "destination pointer");
  check (src,  n, "memcpy", "source pointer");
  check_overlap (dest, n, src, n, "memcpy");
  return __bounds_memcpy (dest, src, n);
}

inline void *
__bounds_memmove (void *dest, void *src, size_t n)
{
  char *d = (char *) dest;
  char *s = (char *) src;

  if (dest < src)
    for (;n;n--)
      *d++ = *s++;
  else if (dest > src) {
    d += n;
    s += n;
    for (;n;n--)
      *--d = *--s;
  }
  return dest;
}

void *
memmove (void *dest, void *src, size_t n)
{
  if (n == 0 && (dest == NULL || src == NULL))
    {
      if (__bounds_warn_misc_strings)
	__bounds_warning (NULL, 0, "memmove",
			  "NULL source or destination for a zero-sized memmove");
      return dest;
    }
  if (n == 0) return dest;

  check (dest, n, "memmove", "destination pointer");
  check (src,  n, "memmove", "source pointer");
  return __bounds_memmove (dest, src, n);
}

/* See comment on `bcopy' at the top of this file.
 */
#ifdef bcopy
#undef bcopy
#endif

void
bcopy (void *src, void *dest, int n)
{
#if BCOPY_OVERLAPPING_ARGUMENTS
#if BCOPY_WARN_OVERLAPPING_ARGUMENTS
  check_overlap (dest, n, src, n, "bcopy");
#endif
  memmove (dest, src, n);
#else /* !BCOPY_OVERLAPPING_ARGUMENTS */
  memcpy (dest, src, n);
#endif /* !BCOPY_OVERLAPPING_ARGUMENTS */
}

inline void *
__bounds_memset (void *dest, int c, size_t n)
{
  /* `memset' is called very frequently, and therefore has to be fast. In a
   * future release of GCC, we will call '__builtin_memset' in this library,
   * but that isn't implemented yet. In the mean time, we spot the common
   * case when the memory is aligned, and write integers.
   */
  if (PTR_TO_UNSIGNED (dest) & (sizeof (int) - 1)) /* not aligned */
    {
      char *d = (char *) dest;

      for (;n;n--)
	*d++ = c;
    }
  else						/* aligned */
    {
      int *di = (int *) dest;
      size_t n_longs = n / sizeof (int);
      int wr = (unsigned char) c << 24 |
	       (unsigned char) c << 16 |
	       (unsigned char) c << 8  |
	       (unsigned char) c;
      char *dc;

      n &= sizeof (int) - 1;
      for (;n_longs;n_longs--)
	*di++ = wr;
      dc = (char *) di;
      for (;n;n--)
	*dc++ = c;
    }

  return dest;
}

void *
memset (void *dest, int c, size_t n)
{
  if (n == 0 && dest == NULL)
    {
      if (__bounds_warn_misc_strings)
	__bounds_warning (NULL, 0, "memset",
			  "NULL destination for a zero-sized memset");
      return dest;
    }
  if (n == 0) return dest;

  check (dest, n, "memset", "destination pointer");
  return __bounds_memset (dest, c, n);
}

#ifdef bzero
#undef bzero
#endif

void
bzero (void *dest, int n)
{
  memset (dest, 0, n);
}

inline int
__bounds_memcmp (void *s1, void *s2, size_t n)
{
  char *cs1 = (char *) s1;
  char *cs2 = (char *) s2;
  for (;n;n--) {
    if (*cs1 - *cs2 != 0)
      return *cs1 - *cs2;
    cs1++, cs2++;
  }
  return 0;
}

int
memcmp (void *s1, void *s2, size_t n)
{
  if (n == 0 && (s1 == NULL || s2 == NULL))
    {
      if (__bounds_warn_misc_strings)
	__bounds_warning (NULL, 0, "memcmp",
			  "NULL source or destination for a zero-sized memcmp");
      return 1;
    }
  if (n == 0) return 0;

  check (s1, n, "memcmp", "first pointer argument");
  check (s2, n, "memcmp", "second pointer argument");
  return __bounds_memcmp (s1, s2, n);
}

/* I'm assuming here that bcmp is the pre-ANSI memcmp function. Since I've
 * never actually used the b* functions, I'm not sure if this is totally
 * right.
 */
#ifdef bcmp
#undef bcmp
#endif

int
bcmp (void *s1, void *s2, int n)
{
  return memcmp (s1, s2, n);
}

/*----------------------------------------------------------------------
 *   2. The string functions.
 *----------------------------------------------------------------------*/

inline char *
__bounds_strcpy (char *dest, char *src)
{
  char *d = dest;

  while ((*dest++ = *src++) != 0);
  return d;
}

char *
strcpy (char *dest, char *src)
{
  size_t n = check (src, 0, "strcpy", "source string");
  check (dest, n, "strcpy", "destination string");
  check_overlap (dest, n, src, n, "strcpy");
  return __bounds_strcpy (dest, src);
}

static inline char *
__bounds_strncpy (char *dest, char *src, size_t n)
{
  char *d = dest;

  if (n) {
    for (;;) {
      if (--n == -1) return d;
      else if ((*dest++ = *src++) == 0) break;
    }
    while (n--) *dest++ = 0;
  }
  return d;
}

char *
strncpy (char *dest, char *src, size_t n)
{
  char *t1;
  size_t n1;

  if (n == 0 && (dest == NULL || src == NULL))
    {
      if (__bounds_warn_misc_strings)
	__bounds_warning (NULL, 0, "strncpy",
			  "NULL source or destination for a zero-sized strncpy");
      return dest;
    }
  if (n == 0) return dest;
  t1 = check_ptr(src, "strncpy", "source argument");
  if (t1 == NULL) return NULL;
  n1 = n;
  while (n1 && *t1) { n1--; t1++; }
  n1 = (((t1 - src) + 1) < n) ? (t1 - src) + 1 : n;
  check (src, n1, "strncpy", "source string");
  check (dest, n, "strncpy", "destination string");
  check_overlap (dest, n, src, n1, "strncpy");
  return __bounds_strncpy (dest, src, n);
}

inline int
__bounds_strlen (char *s)
{
  char *r = s;

  while (*s++);
  return (s - r) - 1;
}

int
strlen (char *s)
{
  return check (s, 0, "strlen", "string argument") - 1;
}

inline int
__bounds_strcmp (char *s1, char *s2)
{ 
  while (*s1 && *s1 == *s2) { s1++; s2++; }
  return(*s1 - *s2);
}

int
strcmp(char *s1, char *s2)
{
  char *t1 = check_ptr(s1, "strcmp", "string1 argument");
  char *t2 = check_ptr(s2, "strcmp", "string2 argument");
  size_t n;

  if (t1 == NULL || t2 == NULL) return -1;

  while (*t1 && *t2) { t1++; t2++; }
  n = (t1 - s1) + 1;
  check (s1, n, "strcmp", "string1 argument");
  check (s2, n, "strcmp", "string2 argument");
  return(__bounds_strcmp (s1, s2));
}

inline int
__bounds_strncmp (char *s1, char *s2, size_t n)
{ 
  do {
    if (--n == -1) break;
    else if (*s1 != *s2) return *s1 - *s2;
    s2++;
  } while (*s1++);
  return 0;
}

int
strncmp(char *s1, char *s2, size_t n)
{
  char *t1;
  char *t2;
  size_t n1;

  if (n == 0 && (s1 == NULL || s2 == NULL))
    {
      if (__bounds_warn_misc_strings)
	__bounds_warning (NULL, 0, "strncmp",
			  "NULL source or destination for a zero-sized strncmp");
      return 0;
    }
  if (n == 0) return 0;
  t1 = check_ptr(s1, "strncmp", "string1 argument");
  t2 = check_ptr(s2, "strncmp", "string2 argument");
  if (t1 == NULL || t2 == NULL) return -1;
  n1 = n;
  while (n1 && *t1 && *t2) { n1--; t1++; t2++; }
  if (((t1 - s1) + 1) < n) n = (t1 - s1) + 1;
  check (s1, n, "strncmp", "string1 argument");
  check (s2, n, "strncmp", "string2 argument");
  return(__bounds_strncmp (s1, s2,n));
}

static inline char *
__bounds_strcat(char *dest, char *src)
{
  char *r = dest;

  while (*dest++);
  dest--;
  while ((*dest++ = *src++) != 0);
  return r;
}

char *
strcat(char *dest, char *src)
{
  char *r = check_ptr(dest, "strcat", "destination argument");
  size_t n = check(src, 0, "strcat", "source argument");

  if (r == NULL) return NULL;

  while (*r++);
  check(dest, n + (r - dest) - 1, "strcat", "destination argument");
  return(__bounds_strcat(dest,src));
}

static inline char *
__bounds_strncat(char *dest, char *src, size_t n)
{
  char *r = dest;

  while (*dest++);
  dest--;
  for (;;) {
	if (n-- == 0) break;
	else if ((*dest++ = *src++) == 0) return(r);
  }
  *dest = 0;
  return(r);
}


char *
strncat(char *dest, char *src, size_t n)
{
  char *r;
  char *p;
  size_t n1;

  if (n == 0 && (dest == NULL || src == NULL))
    {
      if (__bounds_warn_misc_strings)
	__bounds_warning (NULL, 0, "strncat",
			  "NULL source or destination for a zero-sized strncat");
      return dest;
    }
  if (n == 0) return dest;
  r = check_ptr(dest, "strncat", "destination argument");
  p = check_ptr(src, "strncat", "source argument");
  if (r == NULL || p == NULL) return NULL;
  n1 = n;
  while (n1 && *p) { n1--; p++; }
  if (((p - src) + 1) < n) n = (p - src) + 1;
  check(src, n, "strncat", "source argument");
  while (*r++);
  check(dest, n + (r - dest) - 1, "strncat", "destination argument");
  return(__bounds_strncat(dest,src,n));
}

static inline char *
__bounds_strpbrk(char *string, char *chs)
{
  char ch;

  while ((ch = *string++) != 0) {
    char *chk = chs;
    char tch;

    while ((tch = *chk++) != 0)
      if (ch == tch) return string - 1;
  }
  return(NULL);
}

char *
strpbrk(char *string, char *chs)
{
  check(string, 0, "strpbrk", "string argument");
  check(chs, 0, "strpbrk", "check argument");
  return __bounds_strpbrk(string, chs);
}

static inline char *
__bounds_strrchr(char *string, int c)
{
  unsigned char ch = c;
  unsigned char tch;
  unsigned char *result = NULL;

  for (;;) {
    if (ch == (tch = (unsigned char) (*string++))) result = string - 1;
    if (tch == 0) break;
  }
  return result;
}

char *
strrchr(char *string, int c)
{
  check(string, 0, "strrchr", "string argument");
  return __bounds_strrchr(string, c);
}

#ifdef rindex
#undef rindex
#endif

char *
rindex (char *string, int c)
{
  return strrchr (string, c);
}

static inline size_t
__bounds_strspn(char *string, char *chs)
{
  char *r = string;
  char ch;

  while ((ch = *r++) != 0) {
    char *chk = chs;
    char tch;

    do {
      if ((tch = *chk++) == 0) return (r - string) - 1;
    } while (tch != ch);
  }
  return (r - string) - 1;
}

size_t
strspn(char *string, char *chs)
{
  check(string, 0, "strspn", "string argument");
  check(chs, 0, "strspn", "check argument");
  return __bounds_strspn(string, chs);
}

static inline size_t
__bounds_strcspn(char *string, char *chs)
{
  char *r = string;
  char ch;

  while ((ch = *r++) != 0) {
    char *chk = chs;
    char tch;

    while ((tch = *chk++) != 0) {
      if (ch == tch) return (r - string) - 1;
    }
  }
  return (r - string) - 1;
}

size_t
strcspn(char *string, char *chs)
{
  check(string, 0, "strcspn", "string argument");
  check(chs, 0, "strcspn", "check argument");
  return __bounds_strcspn(string, chs);
}

static inline char *
__bounds_strstr(char *s1, char *s2)
{
  char cp1, cp2;
  char *pos = s1;

  while (*s1) {
    char *cmp = s2;

    while (((cp1 = *s1++) == (cp2 = *cmp++)) && cp1);
    if (cp2 == 0) return (char *)pos;
    s1 = ++pos;
  }
  return NULL;
}

char *
strstr(char *s1, char *s2)
{
  check(s1, 0, "strstr", "string1 argument");
  check(s2, 0, "strstr", "string2 argument");
  return __bounds_strstr(s1, s2);
}

static inline char *
__bounds_strtok(char *str1, char *str2)
{
  /* NOTE: For multi-threaded libraries, make last_end a thread-local
   * variable.
   */
  static char *last_end = NULL;
  char        *start;

  if (str1) last_end = str1;

  if (!last_end) return NULL;

  last_end += __bounds_strspn(last_end, str2);
  if (*last_end == '\0') return NULL;

  start = last_end; 
  last_end += __bounds_strcspn(last_end, str2);

  if (*last_end != '\0') *last_end++ = '\0';

  return start;
}

char *
strtok(char *str1, char *str2)
{
  if (str1 != NULL)
    check(str1, 0, "strtok", "first argument");
  check(str2, 0, "strtok", "second argument");
  return __bounds_strtok(str1, str2);
}

static inline char *
__bounds_strchr(char *string, int ch)
{
  int   tch;

  for (;;)
    if      ( (tch = (unsigned char) (*string++))
	      == (unsigned char) ch ) return string - 1;
    else if ( tch == 0 ) return NULL;
}

char *
strchr(char *string, int ch)
{
  check(string, 0, "strchr", "string argument");
  return __bounds_strchr(string, ch);
}

#ifdef index
#undef index
#endif

char *
index (char *string, int ch)
{
  return strchr (string, ch);
}

static inline int
__bounds_strcoll(const char *string1, const char *string2)
{
   while ( *string1 && (*string1 == *string2)) {
		string1++; string2++;
   }
   return *string1 - *string2;
}

int
strcoll(char *string1, char *string2)
{
  check(string1, 0, "strcoll", "first argument");
  check(string2, 0, "strcoll", "second argument");
  return __bounds_strcoll(string1, string2);
}

static inline size_t
__bounds_strxfrm(char *to, char *from, size_t n)
{
  int count = 0; 
  while (*from++) count++;
  from -= count;
  from--;

  if (n != 0) while((*to++ = *from++) && (--n != 0));

  return ((size_t) count);
}

size_t
strxfrm(char *to, char *from, size_t n)
{
  check(from, 0, "strxfrm", "src argument");
  if (n) check(to, n, "strxfrm", "dest argument");
  return __bounds_strxfrm(to, from, n);
}

#if CASECMP
#define	LOW(c)	(islower(c) ? c : _tolower(c))

static inline int
__bounds_strcasecmp(char *s1, char *s2)
{
  while (*s1 && LOW(*s1) == LOW(*s2)) { s1++; s2++; }
  return(LOW(*s1) - LOW(*s2));
}

int
strcasecmp(char *s1, char *s2)
{
  char *t1 = check_ptr(s1, "strcasecmp", "first argument");
  char *t2 = check_ptr(s2, "strcasecmp", "second argument");
  size_t n;

  if (t1 == NULL || t2 == NULL) return NULL;

  while (*t1 && *t2) { t1++; t2++; }
  n = (t1 - s1) + 1;
  check (s1, n, "strcasecmp", "string1 argument");
  check (s2, n, "strcasecmp", "string2 argument");
  return __bounds_strcasecmp(s1,s2);
}

static inline int
__bounds_strncasecmp(char *s1, char *s2, size_t n)
{
  do {
    if (--n == -1) break;
    else if (LOW(*s1) != LOW(*s2)) return LOW(*s1) - LOW(*s2);
    s2++;
  } while (*s1++);
  return 0;
}

int
strncasecmp(char *s1, char *s2, size_t n)
{
  char *t1;
  char *t2;
  size_t n1;

  if (n == 0 && (dest == NULL || src == NULL))
    {
      if (__bounds_warn_misc_strings)
	__bounds_warning (NULL, 0, "strncasecmp",
			  "NULL source or destination for a zero-sized strncasecmp");
      return 0;
    }
  if (n == 0) return 0;
  t1 = check_ptr(s1, "strncasecmp", "string1 argument");
  t2 = check_ptr(s2, "strncasecmp", "string2 argument");
  if (t1 == NULL || t2 == NULL) return NULL;
  n1 = n;
  while (n1 && *t1 && *t2) { n1--; t1++; t2++; }
  if (((t1 - s1) + 1) < n) n = (t1 - s1) + 1;
  check (s1, n, "strncasecmp", "string1 argument");
  check (s2, n, "strncasecmp", "string2 argument");
  return __bounds_strncasecmp(s1,s2,n);
}
#endif

/*----------------------------------------------------------------------
 *	The odd functions `memccpy' and `memchr' are treated separately
 *	here. With these functions, it is permissible to have a `size'
 *	argument that is too large, so long as we will always meet
 *	the `c' terminating character before the end of the object.
 *----------------------------------------------------------------------
 */

static inline void
check2 (void *pointer, size_t size, int c, char *function, char *ptr_name)
{
  object *obj;

#if 0 /* This check is redundant. All callers should cast the argument to
       * unsigned char before calling.
       * - Eberhard Mattes <mattes@azu.informatik.uni-stuttgart.de>
       * & RWMJ.
       */
  if (c < 0 || c > 255)
    __bounds_errorf (NULL, 0, pointer, NULL,
		     "%s called with terminating character < 0 or > 255",
		     function);
#endif

  if (pointer == NULL || pointer == ILLEGAL)
    {
      __bounds_errorf (NULL, 0, pointer, NULL,
		       "NULL or ILLEGAL %s used in %s", ptr_name, function);
      return;
    }
  if ((obj = __bounds_find_object (pointer)) == NULL) {
    if (maybe_is_unchecked_static (pointer, NULL, 0, function)
	|| maybe_is_unchecked_stack (pointer, NULL, 0, function)) {
      /* Unchecked pointer is OK. We have already delivered a warning at
       * this point.
       */
      return;
    }
    __bounds_errorf (NULL, 0, pointer, NULL,
		     "invalid %s used in %s", ptr_name, function);
    return;
  }

  /* The pointer itself is valid, and points to a checked object. Now make
   * sure that we won't overrun the object by using this pointer and size.
   * However, if size does overrun, give it a second chance by looking for
   * the terminating character.
   */
  if (pointer + size > obj->extent)
    {
      char *p = (char *) pointer;
      size_t n = obj->extent - pointer;

      for (; n && *p != c; n--, p++)
	;
      if (n == 0)
	{
	  __bounds_errorf (NULL, 0, pointer, obj,
			   "%s with this %s would overrun the end of the object's allocated memory",
			   function, ptr_name);
	  return;
	}
      /* If n > 0, then OK: The area is still acceptable. */
      }
  return;
}

inline void *
__bounds_memchr (void *s, int c, size_t n)
{
  unsigned char *cs = (unsigned char *) s;
  for (;n;n--) {
    if (*cs == (unsigned char) c)
      return cs;
    cs++;
  }
  return NULL;
}

void *
memchr (void *s, int c, size_t n)
{
#if STRICT_MEMCHR
  check (s, n, "memchr", "pointer argument");
#else
  check2 (s, n, (unsigned char) c, "memchr", "pointer argument");
#endif
  return __bounds_memchr (s, c, n);
}

inline void *
__bounds_memccpy (void *dest, void *src, int c, size_t n)
{
  int c2;

  for (;n;n--) {
    c2 = *(char *)dest++ = *(char *)src++;
    if (c == c2)
      return dest;
  }
  return NULL;
}

void *
memccpy (void *dest, void *src, int c, size_t n)
{
#if STRICT_MEMCHR
  check (src, n, "memccpy", "source pointer");
#else
  check2 (src, n, (unsigned char) c, "memccpy", "source pointer");
#endif
  check (dest, n, "memccpy", "destination pointer");
  check_overlap (dest, n, src, n, "memccpy");
  return __bounds_memccpy (dest, src, c, n);
}
