unit Programmer;
interface
uses Employee, EmployeeCaller;
type
	(* TProgrammerSuperClasses is a set containing an enumeration
	of the superclasses. this works like a multi-dimensional
	boolean (the flag for the dimension is set if its corresponding
	value in the enumeration is in the set) *)
	TProgrammerSuperClasses	=	set of ( EmployeeSuperClass );

	(* the TProgrammer class is a class that semanticly inherits
	from the TEmployee class, but by using aggregation *)
	TProgrammer	=	class
		public
			constructor Create;
			destructor Destroy; override;
			(* all of the virtual methods from the superclass must be
			redefined here in case any derived classes want to
			override them later. they must of course be declared
			as virtual here the same way they were in the base class.
			notice that they are declared virtual and not override
			since they're syntacticly not declared in this family
			before but rather forwarded by a member class *)
			procedure Work( Hours : integer ); virtual;
			function Wage : real; virtual;
			(* since we're not formally inheriting, this object
			cannot be casted into one of the superclasses. therefore,
			we provide functions that return the superclasses of the
			object. since all the virtual methods will be forwarded,
			we are in effect returning a pointer to the superclass
			part of ourself - a kind of casting *)
			function AsEmployee : TEmployee; virtual;
			(* the new functions *)
			function LinesWritten : integer; virtual;
		private
			(* instead of deriving from the TEmployee class the
			usual way, we make the superclass a member inside the
			subclass. (has-a instead of is-a relationship) *)
			m_SuperClass		:	TEmployeeCaller;
			(* this member is a multi-dimensional flag that tells us
			which of the superclasses that have already been
			destructed *)
			m_Destructed		:	TProgrammerSuperClasses;
			(* regular members of the class *)
			m_LinesWritten		:	integer;
	end;

implementation
const
	g_ProgrammerSalary	=	15;
	g_LinesPerHour			=	5; (* industry average... *)

type
	(* the purpose of the TProgrammerEmployee class is to have
	a proxy that can forward virtual methods from the TEmployee
	superclass to the TProgrammer subclass *)
	TProgrammerEmployee	=	class( TEmployeeCaller )
			(* since the memory of the superclass and the subclass
			is located apart, we need to maintain a reference to
			the subclass so we can forward virtual methods *)
			m_SubClass		:	TProgrammer;
			(* we need a constructor for this class since we have
			to set up the link to the subclass. it takes a
			reference to the subclass followed by all the
			parameters that the superclass needs to be created *)
			constructor Create(
				BackPtr	:	TProgrammer;
				Salary	:	real );
			(* we must override the destructor so that we can forward
			the call to the destructor in the subclass *)
			destructor Destroy; override;
			(* all the virtual methods must be overriden here so
			that they can be forwarded to the subclass *)
			procedure Work( Hours : integer ); override;
			function Wage : real; override;
	end;

(* this constructor initialize all the members specific for this
class (i.e. the subclass reference) and construct the superclass
with the parameters given *)
constructor TProgrammerEmployee.Create;
begin
	m_SubClass	:= BackPtr;
	(* construct the superclass with the rest of the parameters
	given. we don't modify the parameters sent to the superclass
	at all - if any other behavior is wanted, this should be
	taken care of in the subclass, not here. the only purpose
	of this class is to provide a bridge between the superclass
	and the subclass *)
	inherited Create( Salary );
end;

(* this destructor is invoked when someone destructs the
TProgrammer object through its TEmployee superclass (which is
a TProgrammerEmployee)
it shall NOT take down the superclass, since the superclasses
must be destructed in opposite order of creation, something only
the subclass can do since it knows about the other superclasses
*)
destructor TProgrammerEmployee.Destroy;
begin
	(* we have to tell the subclass that the destructor of this
	superclass already is called by putting the enumeration
	representing this class into the set of destructed superclasses
	*)
	m_SubClass.m_Destructed	:=
		m_SubClass.m_Destructed + [ EmployeeSuperClass ];
	(* the subclass shall coordinate the destruction of the
	entire object. hence the call to a destructor must be forwarded
	*)
	m_SubClass.Destroy;
end;

(* the virtual methods only forward the call to the corresponding
method in the subclass. this is also a good place to do name
resolution (in case methods in different superclasses have the
same name) *)
procedure TProgrammerEmployee.Work;
begin
	m_SubClass.Work( Hours );
end;

function TProgrammerEmployee.Wage;
begin
	Result	:=	m_SubClass.Wage;
end;

constructor TProgrammer.Create;
begin
	(* we initialize the base class BEFORE we initialize ourself
	so that all our member functions can rely on the superclass
	to exist.
	here goes all the code that customizes the superclass for use
	in the subclass, in this case every programmer get a fixed
	salary.
	we also provide a reference to ourself, so that the superclass
	can link to us and forward its messages *)
	m_SuperClass := TProgrammerEmployee.Create(
		Self, g_ProgrammerSalary );
	(* the set is initialized empty so that if none of the
	destructors in the superclasses are invoked, they will be
	called *)
	m_Destructed := [ ];
	(* now that we've created the superclass, we can initialize
	the members that is defined in this class *)
	m_LinesWritten	:=	0;
	(* allow us to follow the order of creation *)
	writeln( 'DEBUG : Programmer created.' );
end;

(* this destructor is capable of taking the entire object down.
all the superclasses will forward its destructors here. it can
safely be overridden( so that all destructors called through
superclasses will be calling the destructor of the new subclass)
as long as the new destructor calls this one to destruct the
superclasses properly *)
destructor TProgrammer.Destroy;
begin
	(* the order of destruction is the opposite of the order of
	creation *)
	writeln( 'DEBUG : Programmer destroyed.' );
	(* the superclasses need to be destroyed in opposite order of
	creation. we don't know if the object is destroyed through
	any of its superclasses and hence we don't know if any
	destructors are already called. therefore, we take down the
	superclasses through a regular method so that they can be
	destructed in proper order, and thereafter release the memory
	for the superclasses which it is needed *)
	m_SuperClass.InheritedDestroy;
	(* we need to release the memory of those superclasses that has
	not yet have their destructor called (those superclasses which
	has not marked themselves as destructed) *)
	if not ( EmployeeSuperClass in m_Destructed ) then
		m_SuperClass.FreeInstance;
end;

(* if any of the virtual methods from the base class wasn't
intended to be overridden, then the default implmentation can
just call the implementation of the superclass, and it will seem
as it was never overridden (until some derived class does it) *)
function TProgrammer.Wage;
begin
	(* we call the inherited version so that we get the
	implementation of the superclass. if we call the virtual
	version (i.e. Wage()) then it will be forwarded to this point
	again! *)
	Result	:=	m_SuperClass.InheritedWage;
end;

procedure TProgrammer.Work;
begin
	(* first let the superclass have a shot at it *)
	m_SuperClass.InheritedWork( Hours );
	(* we want to do some extra work in this method *)
	inc( m_LinesWritten, Hours * g_LinesPerHour );
end;

(* return the actual superclass part of this object, so that we
can pass this object when a reference to one of the superclasses
is required. this way, we still have polymorphism *)
function TProgrammer.AsEmployee;
begin
	Result	:=	m_SuperClass;
end;

(* perform some work on the members of this class *)
function TProgrammer.LinesWritten;
begin
	Result	:=	m_LinesWritten;
end;

end.
