#include <string.h>
#include <sys\stat.h>
#include <sys\types.h>
#include <fcntl.h>
#include <io.h>
#include <math.h>

#include "defines.hpp"

#include "vesa.hpp"
#include "3d.hpp"
#include "dpmi.hpp"
#include "vesa3d.hpp"

void *VESA3D_ZBuffer;
DWORD *VESA3D_ZBuffer_MULTable;
WORD VESA3D_ZBuffer_Selector;

struct VESA3D_BMPHeader {
   char Check [2];
   DWORD FileSize;
   DWORD Reserved;
   DWORD BitmapStart;
   DWORD ImageSize;
   DWORD Columns;
   DWORD Rows;
   WORD Planes;
   WORD BitsPerPix;
   DWORD Compression;
   DWORD CompressionSize;
   DWORD XScale;
   DWORD YScale;
   DWORD Colors;
   DWORD VIPColors;
};

WORD VESA3D_LoadTexture( char *File, VESA3D_Texture &TEX ) {

    int FileHandle;
    static struct VESA3D_BMPHeader Header;

    FileHandle = open( File, O_RDONLY | O_BINARY );
    if ( FileHandle == -1 ) return 1;

    read( FileHandle, &Header, sizeof( VESA3D_BMPHeader ) );
    TEX.TDX = Header.Columns;
    TEX.TDY = Header.Rows;
    if ( ( Header.Columns & 3 ) != 0 )
        Header.Columns = ( Header.Columns >> 2 << 2 ) + 4;

    TEX.PAL = new char [1024];
    read( FileHandle, TEX.PAL, 1024 );

    TEX.TEXPtr = DWORD( new char [ (Header.Rows) * Header.Columns ] );
    TEX.TEXLength = Header.Rows * Header.Columns;
    lseek( FileHandle, Header.BitmapStart, SEEK_SET );
    read( FileHandle, (void*)(TEX.TEXPtr), Header.Rows * Header.Columns );
    close( FileHandle );
    
    TEX.MULTable = new DWORD [Header.Rows];
    for ( int j=0; j < Header.Rows; j++ ){
        TEX.MULTable[j] = TEX.TEXPtr + j*Header.Columns;
    };
    
    return 0;
};

void VESA3D_SetTexturePalette( VESA3D_Texture &TEX, WORD N ) {
    for( WORD i = 1; i <= N; i++ ){
        VESA_SetColor8B( i, TEX.PAL[4*i+2], TEX.PAL[4*i+1], TEX.PAL[4*i] );
    };
};

void VESA3D_ConvertTexture( VESA3D_Texture TEX ) {
    char MOVEPAL [256] = {0};
    char *SAVEPAL = new char [1024];
    char *TEXBytePtr = (char*)(TEX.TEXPtr);
    char PALCount = 1;

    for ( int i=0; i <= 1023; i++ )
        SAVEPAL[i] = TEX.PAL[i];
   
    for ( i=0; i < TEX.TEXLength; i++ ) {
        if ( MOVEPAL[*TEXBytePtr] == 0 ) {
            MOVEPAL[*TEXBytePtr] = PALCount;
            ((DWORD*)(TEX.PAL))[PALCount] = ((DWORD*)(SAVEPAL))[*TEXBytePtr];
            PALCount++;
        };
        *TEXBytePtr = MOVEPAL[*TEXBytePtr];
        TEXBytePtr++;
    };
};

WORD VESA3D_PageWrite = 0;
WORD VESA3D_PageShow  = 0;
WORD VESA3D_WriteSelector;

void VESA3D_FlipPageWrite() {
    VESA3D_PageWrite ^= 1;
    VESA3D_WriteSelector = ( VESA3D_PageWrite == 0 ? VESA_Selector : VESA_SecondScreen_Sel );
};

void VESA3D_FlipPageShow() {
    VESA3D_PageShow ^= 1;
    if ( VESA3D_PageShow == 0 ) {
        VESA_SetDisplayStart( 0, 0 );
    }
    else {
        VESA_SetDisplayStart( 0, WORD( VESA_SecondScreen_StartY ) );
        // VESA_SetDisplayStart( WORD(VESA_XResolution), 0 );
    };
};

void VESA3D_GouraudTriangle( long X1, long Y1, DWORD C1, long X2, long Y2, DWORD C2,
                      long X3, long Y3, DWORD C3 ) {
    VESA_FilledTriangleG8B( X1, Y1, X2, Y2, X3, Y3, C1, C2, C3, VESA3D_WriteSelector );
};

void VESA3D_FlatTriangle( long X1, long Y1, long X2, long Y2, long X3, long Y3, DWORD C ) {
    VESA_FilledTriangle8B( X1, Y1, X2, Y2, X3, Y3, C, VESA3D_WriteSelector );
};

void VESA3D_TexturedTriangle( long X1, long Y1, long TX1, long TY1,
                        long X2, long Y2, long TX2, long TY2,
                        long X3, long Y3, long TX3, long TY3, void *TEX ) {
    VESA_FilledTriangleTex8BC( X1, Y1, X2, Y2, X3, Y3, TX1, TY1, TX2, TY2, TX3, TY3,
                                DWORD((*((VESA3D_Texture*)TEX)).MULTable),
                                VESA3D_WriteSelector );
};

void VESA3D_TexturedTriangleZBuf( long X1, long Y1, long TX1, long TY1, double Z1,
                                   long X2, long Y2, long TX2, long TY2, double Z2,
                                   long X3, long Y3, long TX3, long TY3, double Z3,
                                   void *TEX ) {    
    VESA_FilledTriangleTexZ8BC( X1, Y1, X2, Y2, X3, Y3, TX1, TY1, TX2, TY2, TX3, TY3,
                                DWORD( Z1 * 256 * 65536 ),
                                DWORD( Z2 * 256 * 65536 ),
                                DWORD( Z3 * 256 * 65536 ),
                                DWORD((*((VESA3D_Texture*)TEX)).MULTable),
                                VESA3D_ZBuffer_Selector,
                                VESA3D_WriteSelector );
};

void VESA3D_FlatTriangleZBuf( long X1, long Y1, double Z1, long X2, long Y2, double Z2,
                               long X3, long Y3, double Z3, DWORD C ) {
    VESA_FilledTriangleZBuf8BC( X1, Y1, X2, Y2, X3, Y3,
                                DWORD( Z1 * 256 * 65536 ),
                                DWORD( Z2 * 256 * 65536 ),
                                DWORD( Z3 * 256 * 65536 ),
                                C, VESA3D_ZBuffer_Selector,
                                VESA3D_WriteSelector );
};

void VESA3D_GouraudTriangleZBuf( long X1, long Y1, double Z1, DWORD C1,
                               long X2, long Y2, double Z2, DWORD C2,
                               long X3, long Y3, double Z3, DWORD C3 ) {
    VESA_FilledTriangleGZBuf8BC( X1, Y1, X2, Y2, X3, Y3,
                                 DWORD( Z1 * 256 * 65536 ),
                                 DWORD( Z2 * 256 * 65536 ),
                                 DWORD( Z3 * 256 * 65536 ),
                                 C1, C2, C3,
                                 VESA3D_ZBuffer_Selector,
                                 VESA3D_WriteSelector );
};

double VESA3D_ShadeMul, VESA3D_ShadeSub = 0;

DWORD VESA3D_DistColor( double D, DWORD C ) {
    return C + DWORD( ( D - VESA3D_ShadeSub ) * VESA3D_ShadeMul );
};

void VESA3D_New3DTexQuad( long PA, long PB, long PC, long PD, VESA3D_Texture &TEX,
                          _3D_Triangle_SideTyp S, _3D_Triangle_Typ T ) {
    long AX = 0;
    long AY = TEX.TDY-1;
    long BX = TEX.TDX-1;
    long BY = TEX.TDY-1;
    long CX = TEX.TDX-1;
    long CY = 0;
    long DX = 0;
    long DY = 0;
    long MX = TEX.TDX/2;
    long MY = TEX.TDY/2;
    long PM = _3D_NewPoint( (_3D_Point_GetX(PA)+_3D_Point_GetX(PB)+
                             _3D_Point_GetX(PC)+_3D_Point_GetX(PD))/4,
                            (_3D_Point_GetY(PA)+_3D_Point_GetY(PB)+
                             _3D_Point_GetY(PC)+_3D_Point_GetY(PD))/4,
                            (_3D_Point_GetZ(PA)+_3D_Point_GetZ(PB)+
                             _3D_Point_GetZ(PC)+_3D_Point_GetZ(PD))/4 );
    _3D_NewObject( new _3D_Triangle( PA,AX,AY,PB,BX,BY,PM,MX,MY,&TEX,S,RectFourTri,T ) );
    _3D_NewObject( new _3D_Triangle( PB,BX,BY,PC,CX,CY,PM,MX,MY,&TEX,S,RectFourTri,T ) );
    _3D_NewObject( new _3D_Triangle( PC,CX,CY,PD,DX,DY,PM,MX,MY,&TEX,S,RectFourTri,T ) );
    _3D_NewObject( new _3D_Triangle( PD,DX,DY,PA,AX,AY,PM,MX,MY,&TEX,S,RectFourTri,T ) );
   
};

void VESA3D_New3DGourQuad( long PA, long PB, long PC, long PD, DWORD C,
                           _3D_Triangle_SideTyp S) {
    _3D_NewObject( new _3D_Triangle( PA, PB, PC, C, S, Rectangle ) );
    _3D_NewObject( new _3D_Triangle( PC, PD, PA, C, S, Rectangle ) );
};

void VESA3D_New3DZBufQuad( long PA, long PB, long PC, long PD, DWORD C,
                            _3D_Triangle_SideTyp S ) {
    _3D_NewObject( new _3D_Triangle( PA, PB, PC, C, S, Rectangle, FlatShadedZBuf ) );
    _3D_NewObject( new _3D_Triangle( PC, PD, PA, C, S, Rectangle, FlatShadedZBuf ) );
};

void VESA3D_New3DGourZBufQuad( long PA, long PB, long PC, long PD, DWORD C,
                                _3D_Triangle_SideTyp S ) {
    _3D_NewObject( new _3D_Triangle( PA, PB, PC, C, S, Rectangle, GouraudShadedZBuf ) );
    _3D_NewObject( new _3D_Triangle( PC, PD, PA, C, S, Rectangle, GouraudShadedZBuf ) );
};

void VESA3D_New3DQuad( long PA, long PB, long PC, long PD, DWORD C,
                                _3D_Triangle_SideTyp S, _3D_Triangle_Typ T ) {
    _3D_NewObject( new _3D_Triangle( PA, PB, PC, C, S, Rectangle, T ) );
    _3D_NewObject( new _3D_Triangle( PC, PD, PA, C, S, Rectangle, T ) );
};

void VESA3D_New3DTri( long PA, long PB, long PC, DWORD C, _3D_Triangle_SideTyp S,
                      _3D_Triangle_MidPointTyp M, _3D_Triangle_Typ T, long Accurance = 0 ) {
    if ( Accurance == 0 )
        _3D_NewObject( new _3D_Triangle( PA, PB, PC, C, S, M, T ) );
    else {
        double AX = _3D_Point_GetX(PA);
        double AY = _3D_Point_GetY(PA);
        double AZ = _3D_Point_GetZ(PA);
        double BX = _3D_Point_GetX(PB);
        double BY = _3D_Point_GetY(PB);
        double BZ = _3D_Point_GetZ(PB);
        double CX = _3D_Point_GetX(PC);
        double CY = _3D_Point_GetY(PC);
        double CZ = _3D_Point_GetZ(PC);
        double LAC = sqrt( (AX-CX)*(AX-CX) + (AY-CY)*(AY-CY) + (AZ-CZ)*(AZ-CZ) );
        double LAB = sqrt( (AX-BX)*(AX-BX) + (AY-BY)*(AY-BY) + (AZ-BZ)*(AZ-BZ) );
        double LBC = sqrt( (BX-CX)*(BX-CX) + (BY-CY)*(BY-CY) + (BZ-CZ)*(BZ-CZ) );
        long PM;
        double LX,LY,LZ;
        if ( LAC > LAB && LAC > LBC ) {
            LX = (AX+CX)/2;
            LY = (AY+CY)/2;
            LZ = (AZ+CZ)/2;
            PM = _3D_NewPoint( LX,LY,LZ );
            VESA3D_New3DTri( PA, PB, PM, C, S, M, T, Accurance-1 );
            VESA3D_New3DTri( PB, PC, PM, C, S, M, T, Accurance-1 );
        }
        else if ( LAB > LBC ) {
            LX = (AX+BX)/2;
            LY = (AY+BY)/2;
            LZ = (AZ+BZ)/2;
            PM = _3D_NewPoint( LX,LY,LZ );
            VESA3D_New3DTri( PC, PA, PM, C, S, M, T, Accurance-1 );
            VESA3D_New3DTri( PB, PC, PM, C, S, M, T, Accurance-1 );
        }
        else {
            LX = (BX+CX)/2;
            LY = (BY+CY)/2;
            LZ = (BZ+CZ)/2;
            PM = _3D_NewPoint( LX,LY,LZ );
            VESA3D_New3DTri( PA, PB, PM, C, S, M, T, Accurance-1 );
            VESA3D_New3DTri( PC, PA, PM, C, S, M, T, Accurance-1 );
        };
    };
}

void VESA3D_New3DQuadx( long PA, long PB, long PC, long PD, DWORD C,
                        _3D_Triangle_SideTyp S, _3D_Triangle_Typ T, long Accurance ) {
    VESA3D_New3DTri( PA, PB, PC, C, S, Rectangle, T, Accurance );
    VESA3D_New3DTri( PC, PD, PA, C, S, Rectangle, T, Accurance );
};

void VESA3D_MakeZBuffer() {
    VESA3D_ZBuffer = new DWORD [ VESA_YResolution * VESA_BpL ];
    /* VESA3D_ZBuffer_MULTable = new DWORD [ VESA_YResolution ];
    for ( int i = 0; i < VESA_YResolution; i++ )
        VESA3D_ZBuffer_MULTable[i] = i*VESA_XResolution*4; */
    VESA3D_ZBuffer_Selector = DPMI_GetSelector( VESA3D_ZBuffer );
};

void VESA3D_ClearZBuffer() {
    VESA_ClearZBuffer( DWORD(VESA3D_ZBuffer), VESA_YResolution * VESA_BpL );
};

void VESA3D_Init() {
    _3D_FlatTriangle = VESA3D_FlatTriangle;
    _3D_GouraudTriangle = VESA3D_GouraudTriangle;
    _3D_TexturedTriangle = VESA3D_TexturedTriangle;
    _3D_FlatTriangleZBuf = VESA3D_FlatTriangleZBuf;
    _3D_GouraudTriangleZBuf = VESA3D_GouraudTriangleZBuf;
    _3D_TexturedTriangleZBuf = VESA3D_TexturedTriangleZBuf;
    _3D_DistColor = VESA3D_DistColor;
    _3D_SetVParams( VESA_MaxX/2, VESA_MaxY/2, VESA_MaxX, VESA_MaxY*4/3 );
    VESA3D_WriteSelector = VESA_Selector;
};

