#include <math.h>
#include <i86.h>

#include "defines.hpp"

#include "3d.hpp"

_3D_Point *PointArray = new _3D_Point [1000];
long PointArrayPtr = 0;

_3D_Object **Obj3DArray = new _3D_Object* [1000];
long Obj3DArrayPtr = 0;

double _3D_ScreenDistance = 1;

double _3D_EyeX, _3D_EyeY , _3D_EyeZ;

double _3D_Alpha, _3D_SINAlpha, _3D_COSAlpha;
double _3D_Phi, _3D_SINPhi, _3D_COSPhi, _3D_SINnPhi, _3D_COSnPhi;
double _3D_Gamma, _3D_SINGamma, _3D_COSGamma, _3D_SINnGamma, _3D_COSnGamma;
double _3D_Omega, _3D_SINOmega, _3D_COSOmega, _3D_SINnOmega, _3D_COSnOmega;

struct _3D_Matrix _3D_Projection_Matrix;

double _3D_MaxDistance = 1000;

long vMidX,vMidY,vMulX,vMulY;

extern void (*_3D_TexturedTriangle) ( long, long, long, long, long, long, long, long,
                                     long, long, long, long, void* ) = 0;
extern void (*_3D_TexturedTriangleZBuf) ( long, long, long, long, double,
                                           long, long, long, long, double,
                                           long, long, long, long, double, void* ) = 0;
extern void (*_3D_FlatTriangle) ( long, long ,long, long, long, long, DWORD ) = 0;                                  
extern void (*_3D_GouraudTriangle) ( long, long, DWORD, long, long, DWORD, long, long, DWORD ) = 0;
extern DWORD (*_3D_DistColor) ( double, DWORD ) = 0;
extern void (*_3D_FlatTriangleZBuf) ( long, long, double, long, long, double,
                                        long, long, double, DWORD ) = 0;
extern void (*_3D_GouraudTriangleZBuf) ( long, long, double, DWORD, long, long,
                                           double, DWORD, long, long, double, DWORD ) = 0;                                        

void _3D_InitMatrix_Rotate( _3D_Matrix &M,
                            double sinPhi, double cosPhi,
                            double sinGamma, double cosGamma,
                            double sinOmega, double cosOmega ) {
    M.MXX =  cosPhi*cosOmega+sinPhi*sinGamma*sinOmega;
    M.MXY =  sinPhi*cosOmega-cosPhi*sinGamma*sinOmega;
    M.MXZ =  cosGamma*sinOmega;
    M.MYX = -sinPhi*cosGamma;
    M.MYY =  cosPhi*cosGamma;
    M.MYZ =  sinGamma;
    M.MZX = -cosPhi*sinOmega+sinPhi*sinGamma*cosOmega;
    M.MZY = -sinPhi*sinOmega-cosPhi*sinGamma*cosOmega;
    M.MZZ =  cosGamma*cosOmega;
};

void _3D_Translate( double &X, double &Y, double &Z,
                    double TX, double TY, double TZ ) {
    X += TX;
    Y += TY;
    Z += TZ;
};

void _3D_Rotate_Forward( double &X, double &Y, double &Z, _3D_Matrix M ) {
    double NX,NY,NZ;
    NX = X;
    NY = Y;
    NZ = Z;
    X = M.MXX*NX + M.MXY*NY + M.MXZ*NZ;
    Y = M.MYX*NX + M.MYY*NY + M.MYZ*NZ;
    Z = M.MZX*NX + M.MZY*NY + M.MZZ*NZ;
};

void _3D_SetEye( double X, double Y, double Z ) {
    _3D_EyeX = X;
    _3D_EyeY = Y;
    _3D_EyeZ = Z;
};

void _3D_SetEye_Globe( double MX, double MY, double MZ, double Rad ) {
    _3D_SetEye( MX - Rad * _3D_SINPhi * _3D_COSGamma,
                MY - Rad * _3D_COSPhi * _3D_COSGamma,
                MZ + Rad * _3D_SINGamma );
};

void _3D_SetVParams( long MidX, long MidY, long MulX, long MulY ) {
    vMidX = MidX;
    vMidY = MidY;
    vMulX = MulX;
    vMulY = MulY;
};

void _3D_SetAlpha( double A ) {
    _3D_Alpha = A;
    _3D_SINAlpha = sin( _3D_Alpha );
    _3D_COSAlpha = cos( _3D_Alpha );
};

void _3D_SetPhi( double A ) {
    _3D_Phi = A;
    _3D_SINPhi  = sin(  _3D_Phi );
    _3D_COSPhi  = cos(  _3D_Phi );
    _3D_SINnPhi = sin( -_3D_Phi );
    _3D_COSnPhi = cos( -_3D_Phi );
};

void _3D_SetGamma( double A ) {
    _3D_Gamma = A;
    _3D_SINGamma  = sin(  _3D_Gamma );
    _3D_COSGamma  = cos(  _3D_Gamma );
    _3D_SINnGamma = sin( -_3D_Gamma );
    _3D_COSnGamma = cos( -_3D_Gamma );
};

void _3D_SetOmega( double A ) {
    _3D_Omega = A;
    _3D_SINOmega  = sin(  _3D_Omega );
    _3D_COSOmega  = cos(  _3D_Omega );
    _3D_SINnOmega = sin( -_3D_Omega );
    _3D_COSnOmega = cos( -_3D_Omega );
};

void _3D_InitRotation() {
    _3D_InitMatrix_Rotate( _3D_Projection_Matrix,
                           _3D_SINnPhi, _3D_COSnPhi,
                           _3D_SINnGamma, _3D_COSnGamma,
                           _3D_SINnOmega, _3D_COSnOmega );
};

void ProjectPoint( _3D_Point &P ) {
    double NX,NY,NZ;
    
    P.V = 1;

    NX = P.X - _3D_EyeX;
    NY = P.Y - _3D_EyeY;
    NZ = P.Z - _3D_EyeZ;
    
    P.D = sqrt( NX*NX + NY*NY + NZ*NZ );

    if ( P.D > _3D_MaxDistance ) return;

    if ( P.D == 0 ) return;

    _3D_Rotate_Forward( NX, NY, NZ, _3D_Projection_Matrix );
    
    if ( NY / P.D < _3D_COSAlpha ) return;

    P.V = 0;

    P.PX = vMidX + _3D_ScreenDistance * vMulX * NX / NY;
    P.PY = vMidY - _3D_ScreenDistance * vMulY * NZ / NY;
};

long _3D_NewPoint( double X, double Y, double Z ) {
    PointArray[ PointArrayPtr ].X = X;
    PointArray[ PointArrayPtr ].Y = Y;
    PointArray[ PointArrayPtr ].Z = Z;
    PointArrayPtr++;
    return (PointArrayPtr-1);
};

double _3D_Point_GetX( long P ) {
    return PointArray[P].X;
};

double _3D_Point_GetY( long P ) {
    return PointArray[P].Y;
};

double _3D_Point_GetZ( long P ) {
    return PointArray[P].Z;
};

void _3D_ProjectPoints( long Start, long End ) {
    if ( Start == -1 )
        Start = 0;
    if ( End == -1 )
        End = PointArrayPtr-1;
    for ( int i=Start ; i <= End ; i++ ) {
        ProjectPoint( PointArray[i] );
    };
};

void _3D_ForAllPointsDo( _3D_PointFunc P ) {
    for ( int i=0; i<PointArrayPtr; i++ ) {
        P(PointArray[i]);
    };
};

void _3D_RotatePointsForward( long Start, long End,
                              double MX, double MY, double MZ, _3D_Matrix M ) {
    for ( long i = Start; i <= End ; i++ ) {
        _3D_Translate( PointArray[i].X, PointArray[i].Y, PointArray[i].Z, -MX, -MY, -MZ );                           
        _3D_Rotate_Forward( PointArray[i].X,
                            PointArray[i].Y,
                            PointArray[i].Z, M );
        _3D_Translate( PointArray[i].X, PointArray[i].Y, PointArray[i].Z, MX, MY, MZ );                           
    };
};

void _3D_NewObject( _3D_Object *O ) {
    Obj3DArray[ Obj3DArrayPtr ] = O;
    Obj3DArrayPtr++;
};

void _3D_DrawObjects() {
    for ( int i=0; i < Obj3DArrayPtr; i++ ) {
        (*Obj3DArray[i]).Draw();
    };
};

void _3D_PrepareObjects() {
    for ( int i=0; i < Obj3DArrayPtr; i++ ) {
        (*Obj3DArray[i]).PrepareDraw();
    };
};

void _3D_SortObjects() {
    for( long a = 0; a < Obj3DArrayPtr-1; a++ ) {
        long MaxDObj = a;
        for( long b = a+1; b < Obj3DArrayPtr; b++ ) {
            if ( (*Obj3DArray[b]).D > (*Obj3DArray[MaxDObj]).D )
                MaxDObj = b;
        };
        _3D_Object* M = Obj3DArray[a];
        Obj3DArray[a] = Obj3DArray[MaxDObj];
        Obj3DArray[MaxDObj] = M;
    };
};

void _3D_SortObjectsZ() {
    for( long a = 0; a < Obj3DArrayPtr-1; a++ ) {
        long MinDObj = a;
        for( long b = a+1; b < Obj3DArrayPtr; b++ ) {
            if ( (*Obj3DArray[b]).D < (*Obj3DArray[MinDObj]).D )
                MinDObj = b;
        };
        _3D_Object* M = Obj3DArray[a];
        Obj3DArray[a] = Obj3DArray[MinDObj];
        Obj3DArray[MinDObj] = M;
    };
};

void _3D_Normal( double AX, double AY, double AZ, double BX, double BY, double BZ,
             double &NX, double &NY, double &NZ ) {
    NX = AY*BZ-AZ*BY;
    NY = AZ*BX-AX*BZ;
    NZ = AX*BY-AY*BX;
};

_3D_Triangle::_3D_Triangle( long P1, long P2, long P3, DWORD C, _3D_Triangle_SideTyp S,
                            _3D_Triangle_MidPointTyp M,_3D_Triangle_Typ T ) {
    PA = P1;
    PB = P2;
    PC = P3;
    Color = C;
    TriangleTyp = T;
    SideTyp = S;
    MidPointTyp = M;
    CalcNormal();
    CalcMidPoint();
    CommonInit();
};

_3D_Triangle::_3D_Triangle( long P1, long tx1, long ty1,
                            long P2, long tx2, long ty2,
                            long P3, long tx3, long ty3, void *T,
                            _3D_Triangle_SideTyp S, _3D_Triangle_MidPointTyp M,
                            _3D_Triangle_Typ TYP ){
    PA = P1;
    PB = P2;
    PC = P3;
    Texture = T;
    TriangleTyp = TYP;
    SideTyp = S;
    MidPointTyp = M;
    TXA = tx1; TYA = ty1;
    TXB = tx2; TYB = ty2;
    TXC = tx3; TYC = ty3;
    CalcNormal();
    CalcMidPoint();
    CommonInit();
};

void _3D_Triangle::CommonInit() {
    if ( TriangleTyp == FlatShadedZBuf )
        CalcDistance = False;
    else
        CalcDistance = True;
    CalcDistance = True;
};

void _3D_Triangle::CalcMidPoint() {
    switch ( MidPointTyp ) {
        case Average :
            MX = (PointArray[PA].X + PointArray[PB].X + PointArray[PC].X ) / 3;
            MY = (PointArray[PA].Y + PointArray[PB].Y + PointArray[PC].Y ) / 3;
            MZ = (PointArray[PA].Z + PointArray[PB].Z + PointArray[PC].Z ) / 3;
            break;
        case Rectangle:
            MX = ( PointArray[PA].X + PointArray[PC].X ) / 2;
            MY = ( PointArray[PA].Y + PointArray[PC].Y ) / 2;
            MZ = ( PointArray[PA].Z + PointArray[PC].Z ) / 2;
            break;
        case RectFourTri:
            MX = PointArray[PC].X;
            MY = PointArray[PC].Y;
            MZ = PointArray[PC].Z;
            break;
    };
};

void _3D_Triangle::CalcNormal() {
    _3D_Normal( PointArray[PB].X - PointArray[PA].X,
                PointArray[PB].Y - PointArray[PA].Y,
                PointArray[PB].Z - PointArray[PA].Z,
                PointArray[PC].X - PointArray[PA].X,
                PointArray[PC].Y - PointArray[PA].Y,
                PointArray[PC].Z - PointArray[PA].Z,
                NX,NY,NZ);
};

void _3D_Triangle::PrepareDraw() {
    double RX, RY, RZ;
    CalcNormal();
    RX = MX - _3D_EyeX;
    RY = MY - _3D_EyeY;
    RZ = MZ - _3D_EyeZ;
    V = 0;
    if ( CalcDistance == True )
        D = sqrt( RX*RX + RY*RY + RZ*RZ );
    if ( SideTyp == Singlesided )
        if ( RX*NX + RY*NY + RZ*NZ < 0 )
            V = 1;
    V += char( PointArray[PA].V + PointArray[PB].V + PointArray[PC].V );
};

void _3D_Triangle::Draw() {
    if ( V==0 ) {
        switch ( TriangleTyp ) {
            case FlatShaded:                    
                _3D_FlatTriangle( PointArray[PA].PX, PointArray[PA].PY,
                                  PointArray[PB].PX, PointArray[PB].PY,
                                  PointArray[PC].PX, PointArray[PC].PY,
                                  _3D_DistColor( D, Color ) );
                break;
            case GouraudShaded:                           
                _3D_GouraudTriangle( PointArray[PA].PX, PointArray[PA].PY,
                                     _3D_DistColor(PointArray[PA].D,Color),
                                     PointArray[PB].PX, PointArray[PB].PY,
                                     _3D_DistColor(PointArray[PB].D,Color),
                                     PointArray[PC].PX, PointArray[PC].PY,
                                     _3D_DistColor(PointArray[PC].D,Color) );
                break;
            case Textured:
                _3D_TexturedTriangle( PointArray[PA].PX, PointArray[PA].PY,
                                      TXA, TYA,
                                      PointArray[PB].PX, PointArray[PB].PY,
                                      TXB, TYB, 
                                      PointArray[PC].PX, PointArray[PC].PY,
                                      TXC, TYC, Texture );
                break;
            case FlatShadedZBuf:
                _3D_FlatTriangleZBuf( PointArray[PA].PX, PointArray[PA].PY,
                                      PointArray[PA].D,
                                      PointArray[PB].PX, PointArray[PB].PY,
                                      PointArray[PB].D,
                                      PointArray[PC].PX, PointArray[PC].PY,
                                      PointArray[PC].D, _3D_DistColor( D, Color ) );
                break;
            case GouraudShadedZBuf:
                _3D_GouraudTriangleZBuf( PointArray[PA].PX, PointArray[PA].PY,
                                         PointArray[PA].D,
                                         _3D_DistColor( PointArray[PA].D, Color ),
                                         PointArray[PB].PX, PointArray[PB].PY,
                                         PointArray[PB].D,
                                         _3D_DistColor( PointArray[PB].D, Color ),
                                         PointArray[PC].PX, PointArray[PC].PY,
                                         PointArray[PC].D, 
                                         _3D_DistColor( PointArray[PC].D, Color ) );
                break;
            case TexturedZBuf:
                _3D_TexturedTriangleZBuf( PointArray[PA].PX, PointArray[PA].PY,
                                          TXA, TYA,
                                          PointArray[PA].D,
                                          PointArray[PB].PX, PointArray[PB].PY,
                                          TXB, TYB, 
                                          PointArray[PB].D,
                                          PointArray[PC].PX, PointArray[PC].PY,
                                          TXC, TYC, 
                                          PointArray[PC].D, Texture );
                break;
        };
    };
};
