unit zldt;

interface

{ Written for BPW 7.0 }

{ This contains an object declartion (LDTmesser) that can be used to       }
{ access the LDT for various reasons.                                      }
{ It includes methods for modifying selector properties and creating       }
{ and deleting callgates.                                                  }

{ This object could be used for:                                           }
{ - Dumping out the LDT to see what memory has been allocated with what    }
{   properties, but note that there are cleaner ways to do this            }
{   (i.e. TOOLHELP).                                                       }
{ - Validating pointers (ditto on the dirt)                                }
{ - Changing selector properties (very useful, but very dirty, causes lots }
{   of system crashes).                                                    }
{ - Creating CallGates.                                                    }

{ The reference for this unit comes from:                                  }
{                                                                          }
{     MS Developer Network CD, Disk Seven, April 94                        }
{                                                                          }
{ It is probably on earlier verisons of the CD as well. The original       }
{ article comes from Microsoft System Journal, May 1993 and is entitled:   }
{                                                                          }
{    "Run Privileged Code from your Windows based prgram using Call Gates" }
{               by Matt Pietrek                                            }
{                                                                          }
{ If you want to understand how this code works you should read that       }
{ article.                                                                 }

{ Call Gates have numerous uses, I am interested in them for two reasons:  }
{ - Getting real (no prefixes) 32 bit code to execute under Windows.       }
{ - Getting ring 0 priveleged instructions execute under Windows.          }


{ CAUTION:                                                                 }
{ messer=knife in german, with this object you can "messup" your LDT very  }
{ badly, during testing I was often reduced to a cold boot to get my       }
{ system back, once I lost my setup information (!). So be careful when    }
{ you use this object.                                                     }

{ This code is hereby placed in the public domain.                         }
{ I am not responsible for any errors or damage that the program           }
{ may cause.                                                               }
{ Author: Mike Wise, 20 July 1994, Luebeck, Germany                        }
{         CompuServe 100014,170                                            }

uses wintypes;


{ TsegDescriptor is the structure of the processor selector, determined by the hardware }
{ See i486 Microprocessor Programers Reference Manual or something similar for further  }
{ information.                                                                          }

type TselDescriptor=object limit_0_15 :word;
                           base_0_15  :word;
                           base_16_23 :byte;
                           flags1     :byte;{ No bit fields, what a drag }
                           flags2     :byte;{ No bit fields, what a drag }
                           base_31_24 :byte;
                           procedure dump( var tx:text; lab:string );
                           function valid:wordbool;
                           function getbase:longint;
                           function getlimit:longint;
                           function getGranularity:word;{ returns 1 or 4096 }
                           function getCodeOperationSize:word;{ returns 16 or 32 }
                           function getDPL:word;{ DPL=Descriptor Privilege Level, returns 0-3 }
                           function setDPL(newDPL:word):word;{ returns old DPL }
                           function issystem:wordbool;
                           function ispresent:wordbool;
                           function iscodeseg:wordbool;
                           function isaccessed:wordbool;
                           end;
      PselDescriptor=^TselDescriptor;

{ TcallGateDescriptor is the structure of the call gate selector, determined by the hardware }
{ See i486 Microprocessor Programers Reference Manual or something similar for further       }
{ information.                                                                               }

type TcallGateDescriptor=object offset_0_15 :word;
                                selector    :word;
                                flags1      :byte;{ No bit fields, what a drag }
                                flags2      :byte;{ No bit fields, what a drag }
                                offset_16_31:word;
                                procedure dump( var tx:text; lab:string );
                                function getDPL:word;{ DPL=Descriptor Privilege Level, returns 0-3 }
                                function getoffset:longint;
                                end;
     PcallGateDescriptor=^TcallGateDescriptor;

type LDTmesser=object LDTalias:word;
                      D7:PselDescriptor;
                      lastDescriptor:integer;
                      constructor init;
                      procedure getLDT;
                      function getSelectorDesP(selector:word):PselDescriptor;
                      function getCallGateDesP(selector:word):PcallGateDescriptor;
                      procedure dumpall(var tx:text);
                      procedure dumpSelector(var tx:text; lab:string; selector:word );
                      procedure dumpCallGate(var tx:text; lab:string; selector:word );
                      function CreateCallGateFlags(funcAddress:pointer; ring:byte; flags:byte; params:word ):pointer;
                      function CreateCallGate16(funcAddress:pointer; ring:byte; nWordParams:word ):pointer;
                      function CreateCallGate32(funcAddress:pointer; ring:byte; nDwordParams:word ):pointer;
                      procedure FreeCallGate(p:pointer);
                      destructor done;
                      end;
     PLDTmesser=^LDTmesser;

procedure dumpLDT(var tx:text);

implementation

uses winprocs,hexunit;
{------------------------- TselDescriptor methods ----------------------------------------}

function TselDescriptor.valid:wordbool;
{ Returns true if this is a valid descriptor.  }
{ There is probably a better way to do this.   }
begin
     valid := (getbase<>0) or (getlimit<>0);
end;

procedure TselDescriptor.dump(var tx:text; lab:string);
{ Writes out a line of text summarizing what is in a TselDescriptor }
var limit:longint;
    typ:string;
    sel:word;
    iscode:wordbool;
begin
     limit := getlimit*getGranularity;
     if iscodeseg then begin
        typ :='C'
        end
     else begin
        typ := 'D';
        end;
     if issystem then begin
        typ := typ+'S'
        end;
     if not ispresent then begin
        typ := typ+'P' { P for paged }
        end;
     sel := loword(longint(@self));
     iscode := iscodeseg;
     write(tx,lab,' ');
     write(tx,hexad(@self),' S:',hex(sel));
     write(tx,' B:',hexl(getbase),' L:',limit:5,' T:',typ,' Ring:',getDPL);
     if iscode then write(tx,' ',getCodeOperationSize,' bit code');
     writeln(tx);
end;


function TselDescriptor.getDPL:word;
{ Get Descriptor Privelege Level (i.e. the socalled "rings") }
{ They are always between 0 and 3                            }
begin
     getDPL := ((flags1 and $60) shr 5);
end;

function TselDescriptor.setDPL(newDPL:word):word;
{ Changes the DPL to what the user specified }
{ returns the old DPL                        }
begin
     setDPL := (flags1 and $60) shr 5;
     flags1 := (flags1 and $9f) or ((newDPL and 3) shl 5);
end;

function TselDescriptor.getbase:longint;
begin
     getbase := longint(base_31_24) shl 24 + longint(base_16_23) shl 16 +
                longint(base_0_15);
end;

function TselDescriptor.getlimit:longint;
begin
     getlimit := ((flags2 and $f) shl 16) + longint(limit_0_15);
end;

function TselDescriptor.getGranularity:word;
{ returns 1 or 4096 }
begin
     if (flags2 and $80)>0 then getGranularity := $1000
                           else getGranularity := 1;
end;

function TselDescriptor.getCodeOperationSize:word;
{ returns 16 or 32 }
begin
     if (flags2 and $40)<>0 then getCodeOperationSize := 32
                            else getCodeOperationSize := 16;
end;

function TselDescriptor.issystem:wordbool;
begin
     if (flags1 and $10)=0 then issystem := true
                           else issystem := false;
end;

function TselDescriptor.ispresent:wordbool;
begin
     if (flags1 and $80)<>0 then ispresent := true
                            else ispresent := false;
end;

function TselDescriptor.iscodeseg:wordbool;
begin
     if (flags1 and $8)<>0 then iscodeseg := true
                           else iscodeseg := false;
end;
function TselDescriptor.isaccessed:wordbool;
begin
     if (flags1 and $1)<>0 then isaccessed := true
                           else isaccessed := false;
end;

{------------------------- TcallGateDescriptor methods -------------------------------}

procedure TcallGateDescriptor.dump(var tx:text; lab:string);
{ Writes out a line of text summarizing what is in a TcallGateDescriptor }
var limit:longint;
    typ:string;
    sel:word;
    iscode:wordbool;
begin
     sel := loword(longint(@self));
     write(tx,lab,' ');
     write(tx,hexad(@self),' S:',hex(sel));
     write(tx,' O:',hexl(getoffset),' Ring:',getDPL,' flags:',hex2(flags1),' ',hex2(flags2));
     {if iscode then write(tx,' ',getCodeOperationSize,' bit code');}
     writeln(tx);
end;

function TcallGateDescriptor.getoffset:longint;
begin
     getoffset := longint(offset_16_31) shl 16 + offset_0_15;
end;

function TcallGateDescriptor.getDPL:word;
{ Get Descriptor Privelege Level (i.e. the "rings") }
{ They are always between 0 and 3                            }
begin
     getDPL := ((flags2 and $60) shr 5);
end;



{------------------------------ LDTmesser methods ---------------------------------}


constructor LDTmesser.init;
begin
     LDTalias := 0;
     lastDescriptor := 0;
     D7 := nil;
end;

destructor LDTmesser.done;
begin
     LDTalias := 0;
     lastDescriptor := 0;
     D7 := nil;
end;


procedure LDTmesser.getLDT;
{ Get the LDT address from somehwere or other }
const MS_DOS_STR:Pchar='MS-DOS';
var DPMIproc:function:word;
label extensions_not_found;
begin
     LDTalias := 0;
     lastDescriptor := 0;
     D7 := nil;
     asm push ds
         lds si, MS_DOS_STR
         mov ax, $168a
         xor di,di
         mov es,di
         int $2f
         pop ds
         cmp al,$8a
         je  extensions_not_found
         mov word ptr [DPMIproc],   di
         mov word ptr [DPMIproc+2], es
         mov ax,$100
         end;
     LDTalias := DPMIproc;
     asm jc extensions_not_found;
         end;

     { Note - there must be a better way to get the limit than this! }
         D7 := getSelectorDesP( LDTalias  );
         lastDescriptor := D7^.limit_0_15 div 8;

     exit;

extensions_not_found:
     LDTalias := 0;
end;

function LDTmesser.getSelectorDesP(selector:word):PselDescriptor;
{ Extract a pointer to a selector descriptor gate out of the LDT }
begin
     if LDTalias=0 then begin
        getLDT;
        if LDTalias=0 then exit;
        end;

     getSelectorDesP := ptr(LDTalias,selector and $fff8);
end;

function LDTmesser.getCallGateDesP(selector:word):PcallGateDescriptor;
{ Extract a pointer to a call gate out of the LDT }
begin
     if LDTalias=0 then begin
        getLDT;
        if LDTalias=0 then exit;
        end;

     getCallGateDesP := ptr(LDTalias,selector and $fff8);
end;

procedure LDTmesser.dumpSelector(var tx:text; lab:string; selector:word );
{ This routine just prints out assorted information that goes with a selector }
var Dp:PselDescriptor;
begin
     if LDTalias=0 then begin
        getLDT;
        if LDTalias=0 then exit;
        end;

     Dp := getSelectorDesP(selector);
     Dp^.dump(tx,lab);
end;

procedure LDTmesser.dumpCallgate(var tx:text; lab:string; selector:word );
{ This routine just prints out assorted information that goes with a call gate }
var Gp:PcallGateDescriptor;
begin
     if LDTalias=0 then begin
        getLDT;
        if LDTalias=0 then exit;
        end;

     Gp := getCallGateDesP(selector);
     Gp^.dump(tx,lab);
end;

procedure LDTmesser.dumpAll(var tx:text);
{ This routine runs through all the selectors and prints out those that }
{ are valid.                                                            }
{ Note, this can UAE if the size of the LDT suddenly gets smaller       }
{       before the end of the list comes. This does seem to happen      }
{       sometimes. It could be avoided by reprogramming this with a     }
{       while statement and doing a getLDT before every selector.       }
var Dp:PselDescriptor;
    i:integer;
    stri:string15;
begin
     getLDT;

     writeln(tx,'LDT at:',hex(LDTalias),' lastDescriptor:',lastDescriptor);

     Dp := ptr(LDTalias,0);
     for i := 0 to lastDescriptor do begin
         if (Dp^.valid) then begin
            str( i:4, stri );
            Dp^.dump(tx,stri);
            end;
         if i<lastDescriptor then inc( Dp );
         end;
end;


function LDTmesser.CreateCallGate16( funcAddress:pointer; ring:byte; nWordParams:word ):pointer;
{ Given a far 16:16 function address, create a new function pointer for it which uses a callgate }
{ and executes code with a 16 bit code operation size                                            }
const flags  = $80  {present bit}
            or $60  {lowly ring 3 code can call me}
            or $04; {type is 286 code}
begin
     CreateCallGate16 := CreateCallGateFlags( funcAddress,ring,flags,nWordParams );
end;

function LDTmesser.CreateCallGate32( funcAddress:pointer; ring:byte; nDwordParams:word ):pointer;
{ Given a far 16:16 function address, create a new function pointer for it which uses a callgate }
{ and executes code with a 32 bit code operation size                                            }
const flags = $80  {present bit}
           or $60  {lowly ring 3 code can call me}
           or $0c; {type is 386 code}
begin
     CreateCallGate32 := CreateCallGateFlags( funcAddress,ring,flags,nDwordParams );
end;

function LDTmesser.CreateCallGateFlags( funcAddress:pointer; ring:byte; flags:byte; params:word ):pointer;
{ Given a far 16:16 function address, create a new function pointer for it which uses a callgate that  }
{ has the characteristics specfied by "flags"                                                          }
var ringAlias:word;
    callgate_selector:word;
    ringDP:PselDescriptor;
    callGP:PcallGateDescriptor;
begin
     CreateCallGateFlags := nil;
     if LDTalias=0 then begin
        getLDT;
        if LDTalias=0 then exit;
        end;

     { Grab a selector from Windows that has the exact same  }
     { attributes as the CS portion of funcAddress           }
         ringAlias := AllocSelector( hiword(longint(funcAddress)) );
         if ringAlias=0 then exit;


     { Change the new selector to run at ring "ring" }
        ringDP := getSelectorDesP( ringAlias );
        ringDP^.setDPL( ring );

     { Get a selector for the call gate to reside in }
        callgate_selector := AllocSelector( 0 );
        if callgate_selector=0 then begin
           FreeSelector( ringAlias );{ If that failed, then we don't need this anymore }
           exit;
           end;

     { Now setup the call gate to look the way we wanted it to look }
         callGP := getCallGateDesP( callgate_selector );

        callGP^.offset_0_15 := loword(longint(funcAddress));
        callGP^.selector := ringAlias;
        callGP^.flags1 := params;
        callGP^.flags2 := flags;
        callGP^.offset_16_31 := 0;

     CreateCallGateFlags := ptr( callgate_selector, 0 );
end;

procedure LDTmesser.FreeCallGate(p:pointer);
var callGP:PcallGateDescriptor;
    selp:word;
begin
     if LDTalias=0 then begin
        getLDT;
        if LDTalias=0 then exit;
        end;

     selp := hiword(longint(p));
     if selp=0 then exit;

     callGP := getCallGateDesP(selp);
     FreeSelector( callGP^.selector );
     FreeSelector( selp );
end;


procedure dumpLDT(var tx:text);
var LDT:LDTmesser;
begin
     LDT.init;
     LDT.getLDT;
     LDT.dumpSelector(tx,'CS',cseg);
     LDT.dumpSelector(tx,'DS',dseg);
     LDT.dumpall(tx);
     readln;
end;

end.