#include <stdlib.h>
#include <stdio.h>
#include "jtag.h"

/*
 * RTN ShiftOut: this subroutine will shift an n-bit number into the
 *               shift register we built to hold the inputs to the
 *               adder.
 */
#define SHIFT 2  /* port bit that enables the shifting of the input reg. */
#define IN    3  /* port bit through which input bits are passed */
void ShiftOut( int n, bitstream* s )
    {
    int i;

    /* The TMS port bit is used as the clock for the input shift
       register. We lower TMS before raising the enable control bit
       so we don't generate some spurious noise on the shift register
       clock.  If that happened, we would get all kinds of strange
       stuff in the shift register. */
    SetTMSValue(0);         /* first lower the TMS bit */
    SetPortBit( SHIFT, 1 ); /* now enable the input shift register */

    /* now shift bits into the register, starting with the MSB */
    for( i=n-1; i>=0; i-- )
        {
        SetTMSValue(0);                /* lower the clock */
        SetPortBit( IN, GetBit(s,i) ); /* set the input bit */
        SetTMSValue(1);           /* raise the clock to shift in the bit */
        }

    /* Now lower the TMS bit and disable the shifting of the register
       by lowering the shift control line.  We use this sequence for
       the same reason: eliminate spurious clock pulses. */
    SetTMSValue(0);         /* first lower the TMS bit */
    SetPortBit( SHIFT, 0 ); /* disable the input shift register */

    /* finally, force the TMS line high since this is its normal state */
    SetTMSValue(1);
    }

/*
 * RTN DoAdd: this subroutine shifts two 8-bit numbers into the shift
 *            register which feeds the 8-bit adder and then uses the
 *            SAMPLE/PRELOAD JTAG instruction to read out the result.
 */
bitstream DoAdd( bitstream a, bitstream b )
    {
    bitstream sum;
    bitstream *out = AllocBits(264);  /* allocate storage for the
                                         EPX780 BSDR storage */

    /* shift the operands into the input shift register */
    ShiftOut( 8, &b );
    ShiftOut( 8, &a );

    /* now use the SAMPLE/PRELOAD instruction to get the adder output */
    EPX780SamplePreload( NULL, out );

    /* now look at the bits in the BSDR storage that correspond to the
       output bits of the adder.  We have to pull these out of the
       ADD.RPT file by looking for the macrocells associated with
       sum0, sum1, ... , sum7.  Take each output and put it in the
       right place for the final result. */
    sum = 0;
    SetBit( &sum, 0, GetMacrocellOutput(out,29) );
    SetBit( &sum, 1, GetMacrocellOutput(out,30) );
    SetBit( &sum, 2, GetMacrocellOutput(out,35) );
    SetBit( &sum, 3, GetMacrocellOutput(out,40) );
    SetBit( &sum, 4, GetMacrocellOutput(out,20) );
    SetBit( &sum, 5, GetMacrocellOutput(out,22) );
    SetBit( &sum, 6, GetMacrocellOutput(out,24) );
    SetBit( &sum, 7, GetMacrocellOutput(out,45) );

    /* free the storage used to hold the BSDR bits */
    FreeBits( out );

    /* return the sum output by the adder */
    return sum;
    }


main()
    {
    bitstream a, b, sum;

    SelectPort(1);   /* we'll assume the TAP is on printer port #1 */
    JTAGPortInit();  /* initialize the port */
    HardTAPReset();  /* reset the TAP controller */

    /* now add all possible 8-bit operands and make sure the sum
       has the correct value */
    for( a=0; a<256; a++ )
        for( b=0; b<256; b++ )
            {
            sum = DoAdd(a,b);  /* do the sum in the FPGA and return
                                  the result */

            /* now check the result with the one we calculate using the PC.
               remember to use only the lower 8-bits of the sum from the
               PC since the FPGA adder is limited to 8 bits. */
            if( sum != ((a+b) & 0xff) )
                {
                printf( "ERROR: %ld + %ld != %ld\n", a, b, sum );
                exit(0);
                }
            else
                printf( "%ld + %ld = %ld\n", a, b, sum );
            }
    }
