/****************************************************************************\
*                                                                            *
*  char.c -- character display and management functions for use with the     *
*  menu code and other programs.                                             *
*                                                                            *
\****************************************************************************/

#include "defs.h"

/****************************************************************************\
*                                                                            *
*  put_btext -- put 5X5 bitmapped characters on the screen.                  *
*  for better fonts, use Fastgraph/Fonts!                                    *
*                                                                            *
\****************************************************************************/

char chars[43][5] = {
   0,   0,   0,   0,   0,
-120,  -8,-120,  80,  32,
 -16,-120, -16,-120, -16,
 120,-128,-128,-128, 120,
 -16,-120,-120,-120, -16,
  -8,-128, -16,-128,  -8,
-128,-128, -16,-128, -16,
 112,-120,-104,-128, 120,
-120,-120,  -8,-120,-120,
 112,  32,  32,  32, 112,
  96,-112,  16,  16,  56,
-120,-112, -32,-112,-120,
  -8,-128,-128,-128,-128,
-120, -88, -88, -40,-120,
-120,-104, -88, -56,-120,
 112,-120,-120,-120, 112,
-128,-128, -16,-120, -16,
 120, -88,-120,-120, 112,
-112, -96, -16,-120, -16,
 -16,   8, 112,-128, 120,
  32,  32,  32,  32,  -8,
 112,-120,-120,-120,-120,
  32,  80,-120,-120,-120,
-120, -40, -88, -88,-120,
-120,  80,  32,  80,-120,
  32,  32,  32,  80,-120,
  -8,  64,  32,  16,  -8,
 112,-120,-120,-120, 112,
 112,  32,  32,  96,  32,
 -16,  64,  32,-112,  96,
 -32,  16,  96,  16, -32,
  16,  16, -16,-112,-112,
 112,   8, -16,-128,  -8,
 112,-120, -16,-128, 112,
  64,  64,  32,  16,  -8,
 112,-120, 112,-120, 112,
  16,   8, 120,-120, 112,
 128,  64,   0,   0,   0,
  64,   0,   0,   0,   0,
   0,   0, 112,   0,   0,
  64,   0,   0,  64,   0,
  16,   0,  16,  72,  48,
   0, 112,   0, 112,   0
};

void put_bstring(char *string,int ix,int iy)
{
   register int i;
   int nchar;
   int blank;
   char ch;

   nchar = strlen(string);

   for (i = 0; i < nchar; i++)
   {
      blank = FALSE;
      ch = string[i];

      /* upper case */

      if (ch >= 65 && ch <= 90)
         ch -= 64;

      /* lower case */

      else if (ch >= 97 && ch <= 122)
         ch -= 96;

      /* numbers */

      else if (ch >= 48 && ch <= 57)
         ch -= 21;

      /* comma */

      else if (ch == 44)
         ch = 37;

      /* period */

      else if (ch == 46)
         ch = 38;

      /* minus */

      else if (ch == 45)
         ch = 39;

     /* colon */

      else if (ch == 58)
         ch = 40;

     /* question mark */

      else if (ch == 63)
         ch = 41;

     /* equals */

      else if (ch == 61)
         ch = 42;

      else
         blank = TRUE;

      fg_move(ix,iy);
      if (!blank) fg_drawmap(&chars[ch][0],1,5);
      ix += 6;
   }
}

/****************************************************************************\
*                                                                            *
*  center_bstring -- display the bitmapped font, centered                    *
*                                                                            *
\****************************************************************************/

void center_bstring(char *string,int x1,int x2,int y)
{
   int x;

   x = get_center(string,x1,x2);
   put_bstring(string,x,y);
}

/****************************************************************************\
*                                                                            *
*  erase_char -- erase a single 5x5 character                                *
*                                                                            *
\****************************************************************************/

void erase_char(int x,int y)
{
   register int color;

   color = fg_getcolor();
   fg_setcolor(white);
   fg_rect(x,x+5,y-5,y);
   fg_setcolor(color);
}

/****************************************************************************\
*                                                                            *
*  get_center -- determine the center of a string                            *
*                                                                            *
\****************************************************************************/

get_center(char *string,int x1,int x2)
{
   int nchar;

   nchar = strlen(string);
   return(((x1 + x2) / 2) - nchar*3);
}

/****************************************************************************\
*                                                                            *
*  put_char -- put a 1 character string                                      *
*                                                                            *
\****************************************************************************/

void put_char(unsigned char key,int x,int y)
{
   char string[2];
   string[1] = '\0';
   string[0] = key;
   put_bstring(string,x,y);
}

/****************************************************************************\
*                                                                            *
*  put_cursor -- a little line that goes under the character                 *
*                                                                            *
\****************************************************************************/

void put_cursor(int x,int y,int cursor_color)
{
   register int color;

   color = fg_getcolor();
   fg_setcolor(cursor_color);
   fg_rect(x,x+5,y,y);
   fg_setcolor(color);
}

/****************************************************************************\
*                                                                            *
*  file_help_screen -- pop help for load file screen                         *
*                                                                            *
\****************************************************************************/

void file_help_screen()
{
   register int i;
   int y;
   static char *string[] =
   {
      "    Popup Help",
      " ",
      "Arrows: Move between fields",
      "PGUP:   Next thing",
      "PGDN:   Prev thing",
      "INS:    Insert thing",
      "DEL:    Delete thing",
      "F2:     F2 thing",
      "F10:    F10 thing",
      "Esc:    Abort"
   };

   fg_setpage(0);
   fg_mousevis(0);
   fg_save(48,271,40,170);

   fg_setcolor(white);
   fg_rect(56,263,45,165);
   fg_setcolor(black);
   fg_box(56,263,45,165);

   y = 60;
   for (i = 0; i < 10; i++)
   {
      put_bstring(string[i], 80,y);
      y += 10;
   }

   fg_waitkey();
   fg_restore(48,271,40,170);
   fg_setpage(1);
   fg_setcolor(blue);
   fg_rect(48,271,40,170);
   fg_setpage(0);
}

/****************************************************************************\
*                                                                            *
*  get_field get character string, allow for up arrows and down arrows       *
*                                                                            *
\****************************************************************************/

get_field(char *string,int x,int y,int max_length,unsigned char key,unsigned char aux)
{
   register int i;
   int color;
   int cursor_timer;
   int foreground;
   int background;
   int xmax, ymin;
   int first;

   first = TRUE;

   foreground = fg_getcolor();
   background = white;

   xmax = x + 6*max_length;
   ymin = y - 6;

   i = 0;
   cursor_timer = 16;
   color = foreground;
   fg_setcolor(foreground);

   for (;;)
   {
      cursor_timer--;
      if (cursor_timer == 8)
         color = background;
      else if (cursor_timer == 0)
      {
         cursor_timer = 16;
         color = foreground;
      }
      if (i < max_length) put_cursor(x,y+1,color);
      if (key+aux > 0)
         if (i < max_length) put_cursor(x,y+1,background);

      if (i == 0 && islower(key)) key ^= 32;

      if ((isalnum(key) || key == SPACE || ispunct(key)) && i < max_length)
      {
         if (first)
         {
            string[i] = '\0';
            fg_setcolor(background);
            fg_rect(x-2,xmax+1,ymin,y+1);
            first = FALSE;
            fg_setcolor(foreground);
         }

         put_cursor(x,y+1,background);
         put_char(key,x,y);
         x += 6;
         string[i++] = key;
         string[i] = '\0';
      }

      else if (key == BS && i > 0)
      {
         if (i < max_length) put_cursor(x,y+1,background);
         x -= 6;
         erase_char(x,y);
         i--;
         string[i] = '\0';
      }

      else if (key == ESC)
         return(ESC);
      else if (aux == UP_ARROW)
         return(UP_ARROW);
      else if (aux == DOWN_ARROW)
         return(DOWN_ARROW);
      else if (aux == PGDN)
         return(PGDN);
      else if (aux == PGUP)
         return(PGUP);
      else if (aux == LEFT_ARROW)
         return(LEFT_ARROW);
      else if (aux == RIGHT_ARROW)
         return(RIGHT_ARROW);
      else if (key == ENTER)
         return(ENTER);
      else if (aux == INSERT)
         return(INSERT);
      else if (aux == DELETE)
         return(DELETE);
      else if (aux == F10)
         return(F10);
      else if (aux == F2)
         return(F2);
      else if (aux == F1)
         file_help_screen();

      fg_waitfor(1);
      fg_intkey(&key,&aux);
   }
}

/****************************************************************************\
*                                                                            *
*  get_string -- accept and display a character string                       *
*                                                                            *
\****************************************************************************/

get_string(char *string, int x, int y, int max_length)
{
   register int i;
   int color;
   int cursor_timer;
   int foreground;
   int background;
   int xmax, ymin;
   unsigned char key,aux;

   foreground = fg_getcolor();
   background = white;

   /* first keystroke begins text input */

   fg_getkey(&key,&aux);

   /* clear the input area */

   fg_setcolor(background);
   xmax = x + 6*max_length;
   ymin = y - 6;
   fg_rect(x-2,xmax+1,ymin,y+1);

   /* initialize the string and cursor */

   i = 0;
   string[0] = '\0';
   cursor_timer = 16;
   color = foreground;
   fg_setcolor(foreground);

   do
   {
      /* alternate the cursor color - make it blink */

      cursor_timer--;
      if (cursor_timer == 8)
         color = background;
      else if (cursor_timer == 0)
      {
         cursor_timer = 16;
         color = foreground;
      }
      if (i < max_length) put_cursor(x,y,color);

      if (i == 0 && islower(key)) key ^= 32;

      /* check if it is a printable character */

      if ((isalnum(key) || key == SPACE || ispunct(key)) && i < max_length)
      {
         put_cursor(x,y,background);
         put_char(key,x,y);
         x += 6;
         string[i++] = key;
      }

      /* if it is backspace, erase the character and back up one */

      else if (key == BS && i > 0)
      {
         if (i < max_length) put_cursor(x,y,background);
         x -= 6;
         erase_char(x,y);
         i--;
      }

      /* pressing escape bounces you right out of here */

      else if (key == ESC)
      {
         if (i < max_length) put_cursor(x,y,background);
            string[i] = '\0';
         return(ESC);
      }

      /* get another key */

      fg_waitfor(1);
      fg_intkey(&key,&aux);
   }
   while (key != ENTER);

   if (i < max_length) put_cursor(x,y,background);
   string[i] = '\0';
   return(0);
}
