/****************************************************************************\
*                                                                            *
*  FGFDEMO.C                                                                 *
*                                                                            *
*  This program demonstrates some features of Fastgraph/Fonts.               *
*                                                                            *
*  Fastgraph/Fonts lets you easily add bit-mapped font support to Fastgraph  *
*  or Fastgraph/Light applications.                                          *
*                                                                            *
*  Copyright (c) 1992-1993 Ted Gruber Software.  All Rights Reserved.        *
*                                                                            *
*                                                                            *
*  Ted Gruber Software would like to acknowledge the contributions made by   *
*  Randall Dryburgh of Micron Software Sciences in creating FGFDEMO.  Randy  *
*  developed the original versions of the functions relating to the palette  *
*  fades and the digital odometer.                                           *
*                                                                            *
\****************************************************************************/

#define NFONTS 11
#define NPALETTES 16
#define NSTEPS 32

#define FALSE 0
#define TRUE  1

#define LEFT   -1
#define CENTER  0
#define RIGHT   1
#define TOP     1
#define BOTTOM -1

#include <fastgraf.h>
#include <fgfonts.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>

/****************************************************************************\
*                                                                            *
*  function prototypes                                                       *
*                                                                            *
\****************************************************************************/

void main(void);

void average_palette(void);
void fade_in(int,int);
void fade_out(int,int);
void odometer(int,int,double,int,int);
int  widest_numeral(void);

/****************************************************************************\
*                                                                            *
*  font handles, font names, and other global variables                      *
*                                                                            *
\****************************************************************************/

int austin, broadway, cognac, crystal14, crystal26, fountain, modern, plaza;
int regal, royal, standard;

char *fontname[NFONTS] = {
   "Austin 36",
   "Broadway 18",
   "Cognac 19",
   "Crystal 14",
   "Crystal 26",
   "Fountain 27",
   "Modern 28",
   "Plaza 14",
   "Regal 24",
   "Royal 15",
   "Standard 8"};

unsigned int clockspeed;

char default_palette[NPALETTES*3];
char new_palette[NPALETTES*3];
char zeroes[NPALETTES*3];

float average[NPALETTES*3];

/****************************************************************************\
*                                                                            *
*  main program                                                              *
*                                                                            *
\****************************************************************************/

void main()
{
   register int i;
   int abort;
   int old_mode;
   int width;
   int x, y;
   double amount;
   unsigned char key, aux;
   char string[24];

   /* make sure we're running on a VGA system; exit if not */

   if (fg_testmode(18,0) == 0)
   {
      printf("This demo requires 640 x 480 16 color VGA graphics.\n");
      exit(1);
   }

   /* load the font files */

   austin    = fgf_load("AUSTIN36.FGF");
   broadway  = fgf_load("BRODWY18.FGF");
   cognac    = fgf_load("COGNAC19.FGF");
   crystal14 = fgf_load("CRYSTL14.FGF");
   crystal26 = fgf_load("CRYSTL26.FGF");
   fountain  = fgf_load("FOUNTN27.FGF");
   modern    = fgf_load("MODERN28.FGF");
   plaza     = fgf_load("PLAZA14.FGF");
   regal     = fgf_load("REGAL24.FGF");
   royal     = fgf_load("ROYAL15.FGF");
   standard  = fgf_load("STNDRD08.FGF");

   /* verify all fonts were loaded successfully; exit if not */

   abort = FALSE;
   if (austin == 0) abort = TRUE;
   if (broadway == 0) abort = TRUE;
   if (cognac == 0) abort = TRUE;
   if (crystal14 == 0) abort = TRUE;
   if (crystal26 == 0) abort = TRUE;
   if (fountain == 0) abort = TRUE;
   if (modern == 0) abort = TRUE;
   if (plaza == 0) abort = TRUE;
   if (regal == 0) abort = TRUE;
   if (royal == 0) abort = TRUE;
   if (standard == 0) abort = TRUE;
   if (abort)
   {
      printf("Failure loading one or more font files.\n");
      exit(1);
   }

   /* benchmark the system speed */

   clockspeed = fg_measure();

   /* initialize the array that zeroes the DAC values */

   for (i = 0; i < NPALETTES*3; i++)
      zeroes[i] = 0;

   /* initialize Fastgraph's video environment */

   old_mode = fg_getmode();
   fg_setmode(18);
   fg_getdacs(0,NPALETTES,default_palette);

   /* create palette increment averages */

   average_palette();

   /* draw the TGS logo on the hidden page */

   fg_setpage(1);
   fg_showpcx("TGS.PCX",0);

   /* fade in the TGS logo on the visual page */

   fg_setdacs(0,NPALETTES,zeroes);
   fg_transfer(0,185,0,89,227,285,1,0);
   fade_in(0,16);

   /* fade in "and" */

   fg_setpage(0);
   fg_setrgb(15,0,0,0);
   fgf_select(standard);
   fg_setcolor(15);
   fg_move(320,315);
   fgf_justify(CENTER,CENTER);
   fgf_print("and",3);
   fade_in(15,1);

   /* fade out the visual page */

   fg_waitfor(30);
   fade_out(0,16);

   /* erase both pages */

   fg_setpage(0);
   fg_erase();
   fg_setpage(1);
   fg_erase();

   /* draw the Micron logo on the hidden page */

   fg_move(0,0);
   fg_showpcx("MICRON.PCX",2);

   /* fade in the Micron logo on the visual page */

   fg_setdacs(0,NPALETTES,zeroes);
   fg_transfer(0,165,0,52,237,266,1,0);
   fade_in(0,16);

   /* fade in "present" */

   fg_setpage(0);
   fg_setrgb(14,0,0,0);
   fg_setcolor(14);
   fg_move(320,315);
   fgf_print("present",7);
   fade_in(14,1);

   /* fade to black */

   fg_waitfor(30);
   fade_out(0,16);

   /* erase both pages */

   fg_setpage(0);
   fg_erase();
   fg_setpage(1);
   fg_erase();

   /* display and then fade out the Fastgraph/Fonts logo */

   fg_setpage(0);
   fgf_select(austin);
   fg_setcolor(10);
   fg_move(320,240);
   fgf_justify(CENTER,BOTTOM);
   fgf_print("Fastgraph/Fonts",15);
   fgf_select(crystal14);
   fg_setcolor(15);
   fg_move(320,270);
   fgf_justify(CENTER,CENTER);
   fgf_print("Copyright (c) 1992-1993 Ted Gruber Software",43);
   fg_move(320,286);
   fgf_print("All Rights Reserved.",20);
   fade_in(10,1);
   fg_waitfor(18);
   fade_in(15,1);
   fg_waitfor(30);
   fade_out(0,16);

   /* display the info screen */

   fg_setpage(0);
   fg_erase();

   fgf_select(regal);
   fg_setcolor(10);
   fg_box(0,639,0,479);
   fg_move(0,32);
   fg_draw(639,32);
   fg_move(320,5);
   fgf_justify(CENTER,TOP);
   fgf_print("Fastgraph/Fonts",15);

   fgf_select(modern);
   fg_setcolor(12);
   fg_move(320,60);
   fgf_justify(CENTER,CENTER);
   fgf_print("Fastgraph/Fonts\177\011 lets you easily add bit-mapped",48);
   fg_move(320,90);
   fgf_print("character support to Fastgraph applications.  It",48);
   fg_move(320,120);
   fgf_print("includes a wide range of fonts in several point sizes.",54);
   fg_move(320,150);
   fgf_print("An application can load up to 32 fonts at once.",47);

   fgf_select(broadway);
   fg_setcolor(12);
   fg_move(320,190);
   fgf_print("Fastgraph/Fonts\177\017 includes functions for font loading",53);
   fg_move(320,215);
   fgf_print("and unloading, string display with horizontal and",49);
   fg_move(320,240);
   fgf_print("vertical justification, font selection, determining",51);
   fg_move(320,265);
   fgf_print("string height and width, and other useful features.",51);
   fg_move(320,290);
   fgf_print("Font files that come with Fastgraph/Fonts may be",48);
   fg_move(320,315);
   fgf_print("distributed freely as part of your applications.",48);

   fgf_select(fountain);
   fg_setcolor(9);
   fg_move(320,355);
   fgf_print("The \177\014Fastgraph/Fonts User's Guide\177\011 includes a",47);
   fg_move(320,385);
   fgf_print("description of the font file format, so you can",47);
   fg_move(320,415);
   fgf_print("create your own font files in case Fastgraph/Fonts",50);
   fg_move(320,445);
   fgf_print("doesn't include your favorite fonts.",36);

   fade_in(0,16);
   fg_waitkey();
   fade_out(0,16);
   fg_erase();

   /* display the features screen */

   fgf_select(cognac);
   fg_setcolor(14);
   fg_move(320,240);
   fgf_print("...and now to demonstrate some Fastgraph/Fonts features...",58);
   fade_in(14,1);
   fg_waitfor(30);
   fade_out(14,1);
   fg_setcolor(1);
   fg_setpage(1);
   fg_rect(0,639,0,319);
   fg_setpage(0);
   fg_rect(0,639,0,479);
   fg_setdacs(0,NPALETTES,default_palette);

   fgf_select(crystal26);
   fg_setcolor(15);
   fg_move(320,50);
   fgf_justify(CENTER,BOTTOM);
   fgf_print("Load up to 32 fonts at once!",28);

   fgf_select(crystal26);
   fg_setcolor(4);
   fg_move(240,90);
   fgf_print("Jackpot is $ ",13);
   x = fg_getxpos();
   y = fg_getypos();

   fgf_select(broadway);
   fg_setcolor(2);
   fg_move(320,120);
   fgf_print("Change \177\003colors \177\004anywhere \177\005in \177\006a \177\007string",44);

   fgf_select(royal);
   fg_setcolor(14);
   fg_move(320,160);
   fgf_print("Justify strings horizontally and vertically:",44);
   fg_setcolor(7);
   fg_move(0,180);
   fg_dash(639,180,0x1111);
   fg_setcolor(14);
   fg_move(0,180);
   fgf_justify(LEFT,BOTTOM);
   fgf_print("LEFT AND ABOVE",14);
   fg_move(320,180);
   fgf_justify(CENTER,CENTER);
   fgf_print("CENTERED BOTH DIRECTIONS",24);
   fg_move(639,180);
   fgf_justify(RIGHT,TOP);
   fgf_print("RIGHT AND BELOW",15);

   fgf_select(modern);
   fg_setcolor(12);
   fg_move(320,220);
   width = fgf_width(" ",1);
   fgf_space(width/2);
   fgf_justify(CENTER,CENTER);
   fgf_print("narrow spacing between words",28);
   fg_move(320,250);
   fgf_space(width);
   fgf_print("normal spacing between words",28);
   fg_move(320,280);
   fgf_space(width*2);
   fgf_print("wide spacing between words",26);

   do
      fg_intkey(&key,&aux);
   while (key+aux > 0);

   fgf_select(crystal26);
   fg_setcolor(4);
   amount = 1998.31;
   do
   {
      odometer(x,y,amount,4,1);
      amount += 0.01;
      fg_intkey(&key,&aux);
   } while (key+aux == 0 && amount < 10000.00);

   /* cast of characters screen */

   fg_setpage(0);
   fg_erase();
   fgf_select(regal);
   fg_setcolor(10);
   fg_move(320,0);
   fgf_justify(CENTER,TOP);
   fgf_print("*** Partial Cast of Characters ***",34);

   strcpy(string,"ABCDabcd1234.,?!+-&@#$");
   y = 80;

   for (i = 0; i < NFONTS; i++)
   {
      fgf_select(i+1);
      fg_setcolor(i+1);
      fg_move(20,y);
      fgf_justify(LEFT,BOTTOM);
      fgf_print(fontname[i],strlen(fontname[i]));
      fg_move(620,y);
      fgf_justify(RIGHT,BOTTOM);
      fgf_print(string,22);
      y += 36;
   }

   fg_setcolor(10);
   fg_move(320,y);
   fgf_justify(CENTER,BOTTOM);
   fgf_print("and many more!",14);

   fg_waitkey();
   fade_out(0,16);

   /* unload fonts and restore the original video state before exiting */

   fgf_unload(-1);
   fg_setmode(old_mode);
   fg_reset();

   /* display ordering information */

   printf("Fastgraph/Fonts (tm) is available for $49 from:\n\n");
   printf("Ted Gruber Software     orders/info (702) 735-1980\n");
   printf("PO Box 13408                    FAX (702) 735-4603\n");
   printf("Las Vegas, NV  89112            BBS (702) 796-7134\n\n");
   printf("Please add $3 shipping within the U.S. and Canada,\n");
   printf("or $6 to other countries.\n");
}

/****************************************************************************\
*                                                                            *
*  average_palette                                                           *
*                                                                            *
*  Compute the palette fade increments used by fade_in and fade_out.         *
*                                                                            *
\****************************************************************************/

void average_palette()
{
   register int i;

   for (i = 0; i < NPALETTES*3 ; i++)
      average[i] = (float)(default_palette[i]) / (float)(NSTEPS);
}

/****************************************************************************\
*                                                                            *
*  fade_in                                                                   *
*                                                                            *
*  Fade one or more DACs from black to their target colors.                  *
*                                                                            *
\****************************************************************************/

void fade_in(start,count)
int start, count;
{
   register int i, j;
   int k, n;
   int last;

   last = start + count;

   for (i = 0; i < NSTEPS; i++)
   {
      k = 0;
      n = start * 3;
      for (j = start; j < last; j++)
      {
         new_palette[k++] = (char)(average[n++] * i);
         new_palette[k++] = (char)(average[n++] * i);
         new_palette[k++] = (char)(average[n++] * i);
      }
      fg_setdacs(start,count,new_palette);
      fg_waitfor(1);
   }
}

/****************************************************************************\
*                                                                            *
*  fade_out                                                                  *
*                                                                            *
*  Fade one or more DACs from their current colors to black.                 *
*                                                                            *
\****************************************************************************/

void fade_out(start,count)
int start, count;
{
   register int i, j;
   int k, n;
   int last;

   last = start + count;

   for (i = 0; i < NSTEPS; i++)
   {
      k = 0;
      n = start * 3;
      for (j = start; j < last; j++)
      {
         new_palette[k++] = default_palette[n] - (char)(average[n] * i);
         n++;
         new_palette[k++] = default_palette[n] - (char)(average[n] * i);
         n++;
         new_palette[k++] = default_palette[n] - (char)(average[n] * i);
         n++;
      }
      fg_setdacs(start,count,new_palette);
      fg_waitfor(1);
   }
}

/****************************************************************************\
*                                                                            *
*  odometer                                                                  *
*                                                                            *
*  Increment the odometer amount by one cent and scroll its new amount on    *
*  the screen.                                                               *
*                                                                            *
\****************************************************************************/

void odometer(x,y,amount,foreground_color,background_color)
int x, y;
double amount;
int foreground_color, background_color;
{
   register int i, j;
   int char_width;
   int delay_diff, delay_same;
   int height, width;
   int length_old, length_new;
   int xpos;
   char amount_old[20], amount_new[20];

   /* create strings for old and new amounts */

   sprintf(amount_old,"%.2f",amount);
   sprintf(amount_new,"%.2f",amount+0.01);
   length_old = strlen(amount_old);
   length_new = strlen(amount_new);

   /* get height of the amount string */

   height = fgf_height(amount_old,length_old);

   /* get its width, rounded up to a byte boundary multiple */

   width = (widest_numeral() + 7) & 0xFFF8;

   /* create a box in the background color */

   fg_setpage(1);
   fg_setcolor(background_color);
   fg_rect(200,200+width*length_new,100-height,100+height);
   fg_setcolor(foreground_color);
   fgf_justify(LEFT,BOTTOM);

   /* put old amount on hidden video page */

   for (i = 0; i < length_old; i++)
   {
      char_width = (width - fgf_width(&amount_old[i],1)) / 2;
      fg_move(200+i*width+char_width,100);
      fgf_print(&amount_old[i],1);
   }

   /* put new amount on hidden video page */

   for (i = 0; i < length_new; i++)
   {
      char_width = (width - fgf_width(&amount_new[i],1)) / 2;
      fg_move(200+i*width+char_width,100+height);
      fgf_print(&amount_new[i],1);
   }

   /* copy old amount to visual video page */

   fg_transfer(200,200+width*length_old,100-height,100,x,y,1,0);

   /* roll the odometer */

   delay_diff = (clockspeed / 60) / length_new;
   delay_same = (clockspeed / 10) / length_new;

   for (j = 0; j < height; j++)
   {
      for (i = 0; i < length_new; i++)
      {
         if (amount_new[i] != amount_old[i] && amount_old[i] != '.')
         {
            fg_stall(delay_diff);
            xpos = 200 + i * width;
            fg_transfer(xpos,xpos+width,100+j-height,100+j,x+xpos-200,y,1,0);
         }
         else
            fg_stall(delay_same);
      }
   }
}

/****************************************************************************\
*                                                                            *
*  widest_numeral                                                            *
*                                                                            *
*  Compute the width of the widest digit in the current font.                *
*                                                                            *
\****************************************************************************/

int widest_numeral()
{
   int widest;
   int width;
   char c;

   widest = 0;
   for (c = '0'; c <= '9'; c++)
   {
      width = fgf_width(&c,1);
      if (width > widest) widest = width;
   }

   return(width);
}
