//////////////////////////////////////////////////////////////////////////
//
// rsh daemon for Windows NT/95
// (c) 1996 Silviu Marghescu - Cornerstone Technologies, Inc.
//
//
// This program is free software; you can redistribute it and/or modify
//	it.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//
//
// rshd - Remote Shell Daemon for Windows NT/95
// Author: Silviu C. Marghescu (http://www.cs.umd.edu/~silviu)
// Date:   May 16, 1996
//
//////////////////////////////////////////////////////////////////////////

#define VISUALCPP
//#define BORLANDCPP

#define COPYRIGHT "Copyright 1996  Silviu C. Marghescu, Cornerstone Technologies, Inc."
#define VERSION_MAJOR 1
#define VERSION_MINOR 0

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <winsock.h>

/* the .rhosts file path, relative to the "windir" directory */
#define RHOSTS "\\system32\\drivers\\etc\\.rhosts"

/* the rshd shutdown command */
#define RSH_SHUTDOWN "rshShutdown"

struct sockaddr_in anaddr; /* socket address structure */
u_short rshPort; /* the rshd port; basically, the 'cmd' port from services */
u_short rshProto; /* the rshd protocol ("tcp") */
SOCKET rshServer; /* the rshd server socket for incoming connections */
int client=0; /* number of clients so far (used for debugging purposes only) */
int runFlag=1; /* cleared when the rshd daemon is to be shutdown */
int securityFlag=1; /* set to loosen up the security when not all information is available on the client */
int noRHosts=0; /* set in order to disable .rhosts cheking */
int noStdout=0, noStderr=0; /* redirection flags */
int debugFlag=0;
int winntFlag=0; /* OS flag; set if we're running on NT.  The "OS" env. variable has to be set to "Windows_NT" */
int shell4dosFlag=0; /* 4DOS shell flag */

/* socket options variables */
int on=1;
struct linger linger;

/* the trusted host list; loaded from the .rhosts file */
struct _rhosts
{
    char* hostname;
    struct _rhosts* next;
}* rhostsList=NULL;

// debugging function
void
	debug (char* message)
{
	if(debugFlag)
		fprintf(stderr, "[%d] %s\n", client, message);
}

// displays the current Winsock error in text format
void winsockError ()
{
    fprintf(stderr, "[%d] Winsock error: ", client);
    int nErrCode=WSAGetLastError();
    switch(nErrCode)
    {
        case WSAENETDOWN:
            fprintf(stderr, "The network subsystem has failed.\n");
            break;
        case WSAEINTR:
            fprintf(stderr, "A blocking call was cancelled.  This can be caused by\n1) a short response time, or\n2) User interrupts the process.\n");
            break;
        case WSAEINPROGRESS:
            fprintf(stderr, "A blocking call is in progress.\n");
            break;
        case WSAENOBUFS:
            fprintf(stderr, "No buffer space is available.\n");
            break;
        case WSAENOTSOCK:
            fprintf(stderr, "Invalid socket descriptor.\n");
            break;
        case WSAEADDRINUSE:
            fprintf(stderr, "The specified address is already in use.\n");
            break;
        case WSAEADDRNOTAVAIL:
            fprintf(stderr, "The specified address is not available\nfrom the local machine.\n");
            break;
        case WSAECONNREFUSED:
            fprintf(stderr, "The connection attempt was refused.\n");
            break;
        case WSAEINVAL:
            fprintf(stderr, "The socket is not already bound to an address.\n");
            break;
        case WSAEISCONN:
            fprintf(stderr, "The socket is already connected.\n");
            break;
        case WSAEMFILE:
            fprintf(stderr, "The maximum number of sockets has exceeded.\n");
            break;
        case WSAENETUNREACH:
            fprintf(stderr, "Network cannot be reached from this host at this time.\n");
            break;
        case WSAETIMEDOUT:
            fprintf(stderr, "Attempt to connect timed out without establishing a connection.\n");
            break;
        case WSAENOTCONN:
            fprintf(stderr, "The socket is not connected.\n");
            break;
        case WSAESHUTDOWN:
            fprintf(stderr, "The socket has been shut down.\n");
            break;
        case WSAECONNABORTED:
            fprintf(stderr, "The virtual circuit was aborted due to timeout or other failure.\n");
            break;
        case WSAECONNRESET:
            fprintf(stderr, "The virtual circuit was reset by the remote side.\n");
            break;
        case WSAEACCES:
            fprintf(stderr, "The requested address is a broadcast address.\n");
            break;
        case WSAENETRESET:
            fprintf(stderr, "The connection must be reset.\n");
            break;
        case WSAHOST_NOT_FOUND:
            fprintf(stderr, "Authoritative Answer Host is not found.\n");
            break;
        default:
            fprintf(stderr, "Error number = %d.\n", nErrCode);
            break;
    }
}

// display an error message and possibly the last Winsock error
void
	error (char* message, int ex=1)
{
    fprintf(stderr, "*** [%d] ERROR: %s\n", client, message);
    winsockError();
    if(ex)
    {
        WSACleanup( );
        exit(1);
    }
}

// the windows version of rresvport
int
    rresvport (int* alport)
{
    struct sockaddr_in sin;
    int s;

    sin.sin_family=AF_INET;
    sin.sin_addr.s_addr=INADDR_ANY;
    s=socket(AF_INET, SOCK_STREAM, 0);
    if(s<0)
        return -1;
    for(;;)
    {
        sin.sin_port=htons((u_short)*alport);
        if(bind(s, (struct sockaddr*)&sin, sizeof(sin))>=0)
            return s;
        if(WSAGetLastError()!=WSAEADDRINUSE)
        {
            closesocket(s);
            return -1;
        }
        (*alport)--;
        if(*alport==IPPORT_RESERVED/2)
        {
            closesocket(s);
            return -1;
        }
    }
}

// receive a null terminated string from the given socket
int
    receive (SOCKET rshClient, char* buff, int blen)
{
    int bufflen;
    int totallen=0;
    debug("Receiving...");
    do
    {
        bufflen=recv(rshClient, buff, blen, 0);
        if(bufflen==SOCKET_ERROR)
            return bufflen;
        if(debugFlag)
            fprintf(stderr, "[%d] ...got %d chars.\n", client, bufflen);
        totallen+=bufflen;
    } while(bufflen && buff[bufflen-1]);
    if(!bufflen)
        buff[0]=0;
    buff[totallen+1]=0;
    return totallen;
}

// send back to the client whatever was redirected in a temporary file
void
    dumpFile (char* fileName, SOCKET s)
{
    char buff[4096];
    int bufflen;
    FILE* temp=fopen(fileName, "r");
    if(temp==NULL)
    {
        error("cannot open temporary file...", 0);
        return;
    }
    while(!feof(temp))
    {
        buff[0]=0;
        fgets(buff, 4096, temp);
        bufflen=strlen(buff);
        if(bufflen)
            if(send(s, buff, bufflen, 0) < bufflen)
            {
                error("error sending results.", 0);
                break;
            }
    }
    fclose(temp);
}

// execute the command given in "comm" and send back the results;
// unless disabled thru command line options, both the stdout and
// the stderr of the command are redirected into temporary files
// which are then sent back to the client
//
void
    runCommand (SOCKET rshClient, SOCKET rshClientErr, char* comm)
{
    char buff[1024];
    char tempOut[128];
    char tempErr[128];
    char* tempDir=getenv("TEMP");

    if(!strcmp(comm, RSH_SHUTDOWN))
    {
        /* the "rshShutdown" command is an internal one; it is used to
         *	gracefully stop the rshd daemon
         */
        strcpy(buff, "rshd shutdown!");
        int bufflen=strlen(buff);
        send(rshClient, buff, bufflen, 0);
        runFlag=0;
        closesocket(rshClient);
        if(rshClientErr!=INVALID_SOCKET)
            closesocket(rshClientErr);
        WSACleanup( );
        exit(0);
    }
    strcpy(buff, comm);
    if(!noStdout)
    {
        /* stdout redirection on */
        *tempOut=0;
        if(tempDir)
        {
            strcpy(tempOut, tempDir);
#ifndef VISUALCPP
            strcat(tempOut, "\\");
#endif
        }
        tmpnam(tempOut+strlen(tempOut));
        strcat(buff, " >");
        strcat(buff, tempOut);
    }
    if(!noStderr && (winntFlag || shell4dosFlag))
    {
        /* stderr redirection on */
        *tempErr=0;
        if(tempDir)
        {
            strcpy(tempErr, tempDir);
#ifndef VISUALCPP
            strcat(tempErr, "\\");
#endif
        }
        tmpnam(tempErr+strlen(tempErr));
        if(shell4dosFlag)
            strcat(buff, " >&>");
        else
            strcat(buff, " 2>");
        strcat(buff, tempErr);
    }
    if(debugFlag)
        fprintf(stderr, "[%d] Executing '%s'...\n", client, buff);
    /* run the command and wait for it to end */
    system(buff);

    /* send the results over... */
    debug("Sending results...");
    /* stdout goes to the main client port */
    if(!noStdout)
    {
        dumpFile(tempOut, rshClient);
        unlink(tempOut);
    }
    /* if an additional port was specified, use it for stderr */
    if(!noStderr && (winntFlag || shell4dosFlag))
    {
        if(rshClientErr != INVALID_SOCKET)
            dumpFile(tempErr, rshClientErr);
        else
            /* otherwise, send stderr to the same main client port */
    	    dumpFile(tempErr, rshClient);
    	unlink(tempErr);
    }
}

/* make sure we have the right winsock.dll version; 1.1 required */
void
    winsockCheck ()
{
    debug("Checking winsock.dll version...");
    WORD wVersionRequested;
    WSADATA wsaData;
    int err;
    wVersionRequested = MAKEWORD( 1, 1 );
    err = WSAStartup( wVersionRequested, &wsaData );
    if ( err != 0 )
        error("unsupported version of winsock.dll!\n");

    /* Confirm that the Windows Sockets DLL supports 1.1.*/
    /* Note that if the DLL supports versions greater */
    /* than 1.1 in addition to 1.1, it will still return */
    /* 1.1 in wVersion since that is the version we */
    /* requested.     */
    if ( LOBYTE( wsaData.wVersion ) != 1 || HIBYTE( wsaData.wVersion ) != 1 )
    {
        error("unsupported version of winsock.dll!\n");
    }
    /* The Windows Sockets DLL is acceptable. Proceed. */
}

/* standard socket initialization procedure */
void
    initSocket ()
{
    /* get port number for rshd */
    struct servent FAR* sp=getservbyname("cmd", "tcp");
    if(sp==NULL)
        error("cannot determine port number for the rshd daemon.");
    rshPort=htons(sp->s_port);

    /* get protocol number for tcp */
    LPPROTOENT lpProto=getprotobyname("tcp");
    if(!lpProto)
    {
        debug("cannot obtain the protocol number; using default...");
        rshProto=IPPROTO_TCP;
    }
    else
        rshProto=lpProto->p_proto;

    debug("Creating socket...");
    /* create socket */
    rshServer=socket(PF_INET, SOCK_STREAM, rshProto);
    if(rshServer==INVALID_SOCKET)
        error("cannot allocate socket for the rshd daemon.");

    /* bind our socket */
    anaddr.sin_port=htons(rshPort);
    anaddr.sin_addr.s_addr=INADDR_ANY;
    anaddr.sin_family=PF_INET;
    debug("Binding socket...");
    if(bind(rshServer, (struct sockaddr FAR*)&anaddr, sizeof(anaddr)))
    {
        closesocket(rshServer);
        error("cannot bind to the rshd daemon port.");
    }
}

/* if an additional port is received from the client, use it to create
 * a socket for stderr output
 */
int
    openErrSocket (SOCKET rshClient, SOCKET& rshClientErr, char* buff)
{
    /* read the stderr port number */
    rshClientErr=INVALID_SOCKET;
    u_short errPort=(u_short)atoi(buff);
    if(!errPort)
    {
        error("wrong stderr port number!", 0);
        return 1;
    }
    if(debugFlag)
        fprintf(stderr, "[%d] stderr port read: %d\n", client, errPort);

    /* make sure the client stderr port is within the reserved range */
    if(errPort<512 || errPort>1023)
    {
        error("client stderr port outside the 512-1023 range\n");
        return 0;
    }
    /* get the necessary info on the stderr socket */
    struct sockaddr_in cliaddr;
    int len=sizeof(cliaddr);
    if(getpeername(rshClient, (struct sockaddr FAR*)&cliaddr, &len))
    {
        error("cannot determine client's IP address!", 0);
        closesocket(rshClientErr);
        rshClientErr=INVALID_SOCKET;
        return 1;
    }

    /* create the new socket */
    int lport=IPPORT_RESERVED-1;
    rshClientErr=rresvport(&lport);
    if(rshClientErr==INVALID_SOCKET)
    {
        error("cannot create stderr socket!", 0);
        return 1;
    }

    debug("Setting options on the stderr socket...");
    if(setsockopt(rshClientErr, SOL_SOCKET, SO_KEEPALIVE, (char*)&on, sizeof(on))<0)
        error("cannot set SO_KEEPALIVE!", 0);
    linger.l_onoff=1;
    linger.l_linger=60;
    if(setsockopt(rshClientErr, SOL_SOCKET, SO_LINGER, (char*)&linger, sizeof(linger))<0)
        error("cannot set SO_LINGER!", 0);

    /* now, connect to the client stderr port */
    cliaddr.sin_family=PF_INET;
    cliaddr.sin_port=htons(errPort);
    if(connect(rshClientErr, (struct sockaddr FAR*)&cliaddr, sizeof(cliaddr)))
    {
        error("cannot connect to the client stderr port!", 0);
        closesocket(rshClientErr);
        rshClientErr=INVALID_SOCKET;
    }
    return 1;
}

/* performs a security check on the remote username;
 * normally, it should check the .rhosts file for allowed users
 */
int
    userNameCheck (char*)
{
    return 1; /* no check for now */
}

/* process the input from the client and runs the received command */
void
    command (SOCKET rshClient, SOCKET& rshClientErr)
{
    char buff[4096];
    int blen=sizeof(buff);
    
    /* receive data from the client */
    if(receive(rshClient, buff, sizeof(buff))==SOCKET_ERROR)
    {
        error("cannot receive client data.", 0);
        return;
    }

    int crt=0;
    if(buff[crt])
    {
        /* an additional socket will be open for stderr */
        if(debugFlag)
            fprintf(stderr, "[%d] stderr port: %s\n", client, buff);
        if(!openErrSocket(rshClient, rshClientErr, buff))
            return;
    }
    /* skip the stderr port */
    while(buff[crt++]);
    /* retrieve and then skip the remote user name */
    if(!buff[crt])
    {
        /* the remote user name hasn't been provided yet */
        if(receive(rshClient, buff, blen)==SOCKET_ERROR)
        {
            error("cannot receive local user name.", 0);
            return;
        }           
        crt=0;
    }
    if(debugFlag)
        fprintf(stderr, "[%d] Remote user name: %s\n", client, buff+crt);
    if(!userNameCheck(buff+crt))
        return;
    while(buff[crt++]);
    /* ignore the local user name */
    if(!buff[crt])
    {
        /* the local user name hasn't been provided yet */
        if(receive(rshClient, buff, blen)==SOCKET_ERROR)
        {
            error("cannot receive local user name.", 0);
            return;
        }
        crt=0;
    }
    if(debugFlag)
        fprintf(stderr, "[%d] Local user name: %s\n", client, buff+crt);
    while(buff[crt++]);
    /* the rest is the command to be executed */
    if(!buff[crt])
    {
        /* the command hasn't been provided yet */
        if(receive(rshClient, buff, blen)==SOCKET_ERROR)
        {
            error("cannot receive remote command.", 0);
            return;
        }
        crt=0;
    }
    if(debugFlag)
        fprintf(stderr, "[%d] Command: '%s'\n", client, buff+crt);
    debug("Sending null byte result...");
    buff[0]=0;
    if(send(rshClient, buff, 1, 0) < 1)
    {
        error("error sending result status.", 0);
        return;
    }

    /* run the command and send the results over*/
    runCommand(rshClient, rshClientErr, buff+crt);
}

/* performs a security check on the remote hostname;
 * normally, the host should be listed in the .rhosts file
 */
int
    hostNameCheck (char* hostname)
{
    debug("Checking host against the .rhosts file...");
    struct _rhosts* ptr=rhostsList;
    while(ptr)
    {
        if(!strcmpi(ptr->hostname, hostname))
            return 1;
        ptr=ptr->next;
    }
    fprintf(stderr, "[%d] Access denied to host %s...\n", client, hostname);
    return 0;
}

/* performs a security clearance on the remote client;
 * the following things should check:
 * - the foreign port should be in the 512-1023 range;
 * - the remote host should be listed in the .rhosts file;
 * - the remote client should be allowed to login from the remote host
 * the 'securityFlag' is used to resolve the cases when not enough
 * information is available (the default is to let people pass)
 */
int
    clientCheck (SOCKET rshClient)
{
    /* get the necessary info on the client socket */
    struct sockaddr_in cliaddr;
    int len=sizeof(cliaddr);
    if(getpeername(rshClient, (struct sockaddr FAR*)&cliaddr, &len))
    {
        error("cannot determine client's IP address!", 0);
        return securityFlag;
    }
    /* make sure the client port is within the reserved  range */
    cliaddr.sin_port=ntohs(cliaddr.sin_port);
    if(debugFlag)
        fprintf(stderr, "[%d] Client port: %d...\n", client, cliaddr.sin_port);
    if(cliaddr.sin_port<512 || cliaddr.sin_port>1023)
    {
        fprintf(stderr, "[%d] client port outside the 512-1023 range!\n",
            client);
        return 0;
    }

    /* now, check the remote host */
    if(noRHosts)
        return 1; /* .rhosts checking disabled */
    struct hostent* remoteHostPtr=gethostbyaddr((const char FAR*)&cliaddr.sin_addr,
        4, PF_INET);
    if(!remoteHostPtr)
    {
        error("cannot determine remote host credentials!\n");
        return securityFlag;
    }
    if(debugFlag)
        fprintf(stderr, "[%d] Client host: %s...\n", client,
            remoteHostPtr->h_name);
    return hostNameCheck(remoteHostPtr->h_name);
}

/* this is the client thread; it is started for each new connection */
long
    clientThread (SOCKET* rshClientPtr)
{
    debug("Thread started...");
    SOCKET rshClient=*rshClientPtr;
    SOCKET rshClientErr=INVALID_SOCKET;
    delete rshClientPtr;
    debug("Setting options on the main socket...");
    if(setsockopt(rshClient, SOL_SOCKET, SO_KEEPALIVE, (char*)&on, sizeof(on))<0)
        error("cannot set SO_KEEPALIVE!\n", 0);
    linger.l_onoff=1;
    linger.l_linger=60;
    if(setsockopt(rshClient, SOL_SOCKET, SO_LINGER, (char*)&linger, sizeof(linger))<0)
        error("cannot set SO_LINGER!\n", 0);
    debug("Checking client...");
    if(!clientCheck(rshClient))
    {
        closesocket(rshClient);
        return 0;
    }
    debug("Processing client data...");
    command(rshClient, rshClientErr);
    debug("Closing main socket...");
    shutdown(rshClient, 2);
    closesocket(rshClient);
    if(rshClientErr!=INVALID_SOCKET)
    {
        debug("Closing stderr socket...");
        shutdown(rshClientErr, 2);
        closesocket(rshClientErr);
    }
    debug("Client disconnected...");
    return 0;
}

/* loop for new connections */
void
    loop ()
{
    DWORD threadID;
    debug("Listening...");
    /* listen for connections */
    if(listen(rshServer, 5))
    {
        closesocket(rshServer);
        error("error while listening to the rshd daemon port.");
    }

    debug("Ready for connections...");
    while(1)
    {
        debug("Accepting connection...");
        /* ready to accept connections */
        int len=sizeof(anaddr);
        SOCKET rshClient=accept(rshServer, (struct sockaddr FAR*)&anaddr, &len);
        if(!runFlag)
            return;
        if(rshClient==INVALID_SOCKET)
        {
            error("error accepting connection from rsh client.", 0);
            continue;
        }
        client++;
        debug("Client connected!");
        /* got a new connection; start a separate thread */
        SOCKET* rshSocket=new SOCKET;
        if(!rshSocket)
        {
            error("heap overflow!", 0);
            continue;
        }
        *rshSocket=rshClient;
        debug("Starting client thread...");
        HANDLE threadHnd=CreateThread(NULL, 0, (LPTHREAD_START_ROUTINE)clientThread,
            (LPVOID)rshSocket, 0, (LPDWORD)&threadID);
        if(!threadHnd)
            error("cannot start client thread...");
    }
}

/* check for command line parameters and set various flags */
void
    parseCommandLine (int argc, char* argv[])
{
    for(int i=1; i<argc; i++)
    if(!strcmpi(argv[i], "-d"))
        debugFlag=1;
    else
    if(!strcmpi(argv[i], "-s"))
    {
        securityFlag=0;
        debug("Tight security enabled!");
    }
    else
    if(!strcmp(argv[i], "-1"))
    {
        noStdout=1;
        debug("No stdout redirection!");
    }
    else
    if(!strcmp(argv[i], "-2"))
    {
        noStderr=1;
        debug("No stderr redirection!");
    }
    else
    if(!strcmp(argv[i], "-4"))
    {
        shell4dosFlag=1;
        debug("Running in 4DOS!");
    }
    else
    if(!strcmpi(argv[i], "-r"))
    {
        noRHosts=1;
        debug(".rhosts checking disabled!");
    }
    else
    if(!strcmpi(argv[i], "-v"))
    {
        fprintf(stderr, "\nrhsd - remote shell daemon for Windows NT, version %d.%d\n%s\n",
            VERSION_MAJOR, VERSION_MINOR, COPYRIGHT);
        exit(0);
    }
    else
        fprintf(stderr, "Ignoring unknown option '%s'...\n", argv[i]);

    // since we're here, check some environment variables
    char* os=getenv("OS");
    if(os && !strcmp(os, "Windows_NT"))
        winntFlag=1;
    else
        winntFlag=0; // "OS" undefined; most probably Windows 95
}


void
    loadRHosts ()
{
    rhostsList=NULL;
    char* windir=getenv("windir");
    if(!windir)
        error("The WINDIR environment variable is not set!");
    char rhosts[256];
    strcpy(rhosts, windir);
    strcat(rhosts, RHOSTS);
    if(debugFlag)
        fprintf(stderr, "[%d] Loading %s...\n", client, rhosts);
    FILE* rhostsFile=fopen(rhosts, "r");
    if(!rhostsFile)
        error("Cannot open the .rhosts file.  Either create one or use the '-r' option...");
    char buff[1024];
    buff[1023]=0;
    while(!feof(rhostsFile))
    {
        fgets(buff, 1023, rhostsFile);
        if(feof(rhostsFile))
            break;
        int i=0;
        if(buff[i]=='#')
            continue; /* ignoring comment line */
        while(buff[i] && buff[i]!=' ' && buff[i]!='\t' && buff[i]!='\n')
            i++;
        if(!i)
            continue; /* empty line */
        char* hostname=(char*)calloc(sizeof(char), i+1);
        strncpy(hostname, buff, i);
        struct _rhosts* rhostCell=new struct _rhosts;
        if(!rhostCell)
            error("Heap overflow!");
        rhostCell->next=rhostsList;
        rhostCell->hostname=hostname;
        rhostsList=rhostCell;
        if(debugFlag)
            fprintf(stderr, "[%d] Trusting host %s...\n", client, hostname);
    }
    fclose(rhostsFile);
}


void
    main (int argc, char* argv[])
{
    parseCommandLine(argc, argv);
    winsockCheck();
    if(!noRHosts)
        loadRHosts();
    initSocket();
    loop();
    WSACleanup();
    exit(0);
}
