/*
   Termminal hooks for Windows NT(tm) port of GNU Emacs.
   Copyright (C) 1992 Free Software Foundation, Inc.

   This file is part of GNU Emacs.

   GNU Emacs is free software; you can redistribute it and/or modify it
   under the terms of the GNU General Public License as published by the
   Free Software Foundation; either version 1, or (at your option) any later
   version.

   GNU Emacs is distributed in the hope that it will be useful, but WITHOUT
   ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
   FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
   more details.

   You should have received a copy of the GNU General Public License along
   with GNU Emacs; see the file COPYING.  If not, write to the Free Software
   Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.

   Tim Fleehart (apollo@online.com)	17-Jan-92
*/


#include	<stdio.h>

#include	<windows.h>
#include	"termhooks.h"


extern int meta_key;


int	MoveCursor			(int row, int col);
int	ClearToEnd			(void);
int	ClearScreen			(void);
int	ClearEndOfLine			(int);
int	InsDelLines			(int vpos, int n);
int	ChangeLineHighlight		(int, int, int);
int	ReassertLineHighlight		(int, int);
int	InsertChars			(char *start, int len);
int	OutputChars			(char *string, int len);
int	DeleteChars			(int n);
int	RingBell			(void);
int	ResetTerminalModes		(void);
int	SetTerminalModes		(void);
int	CalculateCosts			(int, int *, int *);
int	initialize_win_nt_display	(void);
int	SetTerminalWindow		(void);
int	UpdateBegin			(void);
int	UpdateEnd			(void);


int (*move_cursor_hook)()		= MoveCursor;
int (*raw_move_cursor_hook)()		= MoveCursor;
int (*clear_to_end_hook)()		= ClearToEnd;
int (*clear_screen_hook)()		= ClearScreen;
int (*clear_end_of_line_hook)()		= ClearEndOfLine;
int (*ins_del_lines_hook)()		= InsDelLines;
int (*change_line_highlight_hook)()	= ChangeLineHighlight;
int (*reassert_line_highlight_hook)()	= ReassertLineHighlight;
int (*insert_chars_hook)()		= InsertChars;
int (*output_chars_hook)()		= OutputChars;
int (*delete_chars_hook)()		= DeleteChars;
int (*ring_bell_hook)()			= RingBell;
int (*reset_terminal_modes_hook)()	= ResetTerminalModes;
int (*set_terminal_modes_hook)()	= SetTerminalModes;
int (*set_terminal_window_hook)()	= SetTerminalWindow;
int (*read_socket_hook)()		= NULL;
int (*fix_screen_hook)()		= NULL;		/* FixScreen; */
int (*calculate_costs_hook)()		= CalculateCosts;
int (*update_begin_hook)()		= UpdateBegin;
int (*update_end_hook)()		= UpdateEnd;


/*
 * current cursor position
 */

BOOL	Updating = FALSE;
COORD	Cursor;
HANDLE	PrevScreen, CurScreen;
UCHAR	CAttr;

extern int	screen_height,
		screen_width;


/*
 * move the cursor to (row, col)
 */

int
MoveCursor(row, col)
int	row,
	col;
{
	Cursor.X = col;
	Cursor.Y = row;

	if (!Updating) {
		SetConsoleCursorPosition(CurScreen, Cursor);
	}

	return 1;
}


/*
 * clear from cursor to end of screen
 */

int
ClearToEnd()
{
	ClearEndOfLine(screen_width);

	InsDelLines(Cursor.Y, screen_height - Cursor.Y);

	return 1;
}


/*
 * clear the whole screen
 */

int
ClearScreen()
{
	SMALL_RECT	Clip, Scroll;
	COORD		Dest;
	CHAR_INFO	Fill;

	Scroll.Top = 0;
	Scroll.Bottom = screen_height;
	Scroll.Left = 0;
	Scroll.Right = screen_width;

	Dest.Y = screen_height;
	Dest.X = 0;

	Clip.Left = 0;
	Clip.Top = 0;
	Clip.Right = screen_width;
	Clip.Bottom = screen_height;

	Fill.Char.AsciiChar = 0x20;
	Fill.Attributes = CAttr;

	ScrollConsoleScreenBuffer(CurScreen, &Scroll, &Clip, Dest, &Fill);

	MoveCursor(0, 0);
}


/* clear from Cursor to end (what's "standout marker"?) */

int
ClearEndOfLine(end)
int	end;
{
	static	char	base[256];
	static	BOOL	beenhere = FALSE;

	if (!beenhere) {
		int	i;

		for (i = 0; i < 256; i++) {
			base[i] = ' ';		/* empty space	*/
		}

		beenhere = TRUE;
	}

	OutputChars(base, end - Cursor.X);	/* fencepost ?	*/
}


/*
 * insert n lines at vpos. if n is negative delete -n lines
 */

int
InsDelLines(vpos, n)
int	vpos,
	n;
{
	int		i, nb;
	SMALL_RECT	Clip, Scroll;
	COORD		Dest;
	CHAR_INFO	Fill;

	if (n < 0) {
		Scroll.Top = vpos - n;
		Scroll.Bottom = screen_height;
		Dest.Y = vpos;
	} else {
		Scroll.Top = vpos;
		Scroll.Bottom = screen_height - n;
		Dest.Y = vpos + n;
	}

	Scroll.Left = 0;
	Scroll.Right = screen_width;

	Dest.X = 0;

	/*
	 * I'm not really sure what to do with this... where's the spec? (I
	 * imagine I almost don't care if I calculate the scroll region
	 * correctly...)
	 */

	Clip.Left = 0;
	Clip.Top = 0;
	Clip.Right = screen_width;
	Clip.Bottom = screen_height;

	Fill.Char.AsciiChar = 0x20;
	Fill.Attributes = CAttr;

	ScrollConsoleScreenBuffer(CurScreen, &Scroll, &Clip, Dest, &Fill);

	/*
	 * here we have to deal with a win32 console flake: If the scroll
	 * region looks like abc and we scroll c to a and fill with d we get
	 * cbd... if we scroll block c one line at a time to a, we get cdd...
	 * Jove expects cdd consistently... So we have to deal with that
	 * here... (this also occurs scrolling the same way in the other
	 * direction.
	 */

	if (n > 0) {
		if (Scroll.Bottom < Dest.Y) {
			for (i = Scroll.Bottom; i < Dest.Y; i++) {
				MoveCursor(i, 0);
				ClearEndOfLine(screen_width);
			}
		}
	} else {
		nb = Dest.Y + (Scroll.Bottom - Scroll.Top) + 1;

		if (nb < Scroll.Top) { 
			for (i = nb; i < Scroll.Top; i++) {
				MoveCursor(i, 0);
				ClearEndOfLine(screen_width);
			}
		}
	}

	Cursor.X = 0;
	Cursor.Y = vpos;
}


/*
 * Call this when about to modify line at position VPOS and change whether it
 * is highlighted.
 */

int
ChangeLineHighlight(new_highlight, vpos, first_unused_hpos)
int	new_highlight;
int	vpos;
int	first_unused_hpos;
{
	MoveCursor(vpos, 0);
	ClearEndOfLine(first_unused_hpos);
}

/*
 * External interface to control of standout mode. Call this when about to
 * modify line at position VPOS and not change whether it is highlighted.
 */

int
ReassertLineHighlight(highlight, vpos)
int	highlight;
int	vpos;
{
	highlight;		/* pedantic compiler silencer */
	vpos;			/* pedantic compiler silencer */
}


#undef	LEFT
#undef	RIGHT
#define	LEFT	1
#define	RIGHT	0

ScrollLine(dist, direction)
int	dist,
	direction;
{
	/*
	 * The idea here is to implement a horizontal scroll in one line to
	 * implement delete and half of insert.
	 */

	int		i, nb;
	SMALL_RECT	Clip, Scroll;
	COORD		Dest;
	CHAR_INFO	Fill;

	Scroll.Top = Cursor.Y;
	Scroll.Bottom = Cursor.Y;

	Scroll.Left = Cursor.X - ((direction == LEFT) ? dist : 0);
	Scroll.Right = screen_width - ((direction == LEFT) ? 0 : dist);

	Dest.X = Cursor.X;
	Dest.Y = Cursor.Y;

	/*
	 * I'm not really sure what to do with this... where's the spec? (I
	 * imagine I almost don't care if I calculate the scroll region
	 * correctly...)
	 */

	Clip.Left = Cursor.X;
	Clip.Right = screen_width;
	Clip.Top = Cursor.Y;
	Clip.Bottom = Cursor.Y;

	Fill.Char.AsciiChar = 0x20;
	Fill.Attributes = CAttr;

	ScrollConsoleScreenBuffer(CurScreen, &Scroll, &Clip, Dest, &Fill);
}


/* if start is zero insert blanks instead of a string at start ? */

int
InsertChars(start, len)
char	*start;
int	len;
{
	ScrollLine(len, RIGHT);

	/*
	 * move len chars to the right starting at Cursor, fill with blanks
	 */

	if (start) {
		/*
		 * print the first len characters of start, Cursor.X adjusted
		 * by OutputChars.
		 */

		OutputChars(start, len);
	} else {
		Cursor.X += len;
		MoveCursor(Cursor.Y, Cursor.X);
	}
}


int
OutputChars(string, len)
char	*string;
int	len;
{
	static	SHORT	Attrs[256]; /* can't be more than this many chars/line */
	DWORD	i;

	/*
	 * Just dump the characters starting at Cursor, update cursor when
	 * done.
	 */

	for (i = 0; i < len; i++)
		Attrs[i] = CAttr;

	WriteConsoleOutputAttribute(CurScreen, (LPWORD)Attrs, len, Cursor, &i);

	WriteConsoleOutputCharacter(CurScreen, string, len, Cursor, &i);

	Cursor.X += len;
	MoveCursor(Cursor.Y, Cursor.X);
}


int
DeleteChars(n)
int	n;
{
	/*
	 * delete chars means scroll chars from Cursor.X + n to Cursor.X,
	 * anything beyond the edge of the screen should come out empty...
	 */

	ScrollLine(n, LEFT);
}


int
RingBell()
{
	Beep(666, 100);		/* wha'da-ya-want-fer-nothin'?	*/
}


int
ResetTerminalModes()
{
	UnsetKbd();

	SetConsoleActiveScreenBuffer(PrevScreen);

	CloseHandle(CurScreen);
}


int
SetTerminalModes()
{
	ResetKbd();

	CurScreen = CreateConsoleScreenBuffer(GENERIC_READ | GENERIC_WRITE,
		0,
		NULL,
		CONSOLE_TEXTMODE_BUFFER,
		NULL);

	if (CurScreen == INVALID_HANDLE_VALUE) {
		printf("CreateConsoleScreenBuffer failed in ResetTerm\n");
		printf("LastError = 0x%lx\n", GetLastError());

		printf("CreateConsoleScreenBuffer failed in ResetTerm\n");
		printf("LastError = 0x%lx\n", GetLastError());

		exit(0);
	}

	SetConsoleActiveScreenBuffer(CurScreen);
}


/*
 * hmmm... perhaps these let us bracket screen changes so that we can flush
 * clumps rather than one-character-at-a-time...
 *
 * we'll start with not moving the cursor while an update is in progress
 */

int
UpdateBegin()
{
	Updating = TRUE;
}


int
UpdateEnd()
{
	Updating = FALSE;

	SetConsoleCursorPosition(CurScreen, Cursor);
}


SetTerminalWindow()
{
}



FixScreen()
{
	/* This gets called a *lot*... why? */
}


int
initialize_win_nt_display()
{
	CONSOLE_SCREEN_BUFFER_INFO	Info;

	PrevScreen = GetStdHandle(STD_OUTPUT_HANDLE);

	SetTerminalModes();

	GetConsoleScreenBufferInfo(CurScreen, &Info);

	meta_key = 1;
	screen_height = Info.dwSize.Y;		/* lines per page */
	screen_width = Info.dwSize.X;		/* characters per line */
	CAttr = Info.wAttributes & 0xFF;

	MoveCursor(0, 0);

	ClearScreen();
}


/*
 * snagged from x11term.c
 */

int
CalculateCosts(extra, costvec, ncostvec)
int	extra;
int	*costvec,
	*ncostvec;
{
	CalcLID(0, screen_width / 2, 0, 0, costvec, ncostvec);
}

