/*
   Keyboard support for Windows NT(tm) port of GNU Emacs.
   Copyright (C) 1992 Free Software Foundation, Inc.

   This file is part of GNU Emacs.

   GNU Emacs is free software; you can redistribute it and/or modify it
   under the terms of the GNU General Public License as published by the
   Free Software Foundation; either version 1, or (at your option) any later
   version.

   GNU Emacs is distributed in the hope that it will be useful, but WITHOUT
   ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
   FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
   more details.

   You should have received a copy of the GNU General Public License along
   with GNU Emacs; see the file COPYING.  If not, write to the Free Software
   Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.

   Tim Fleehart (apollo@online.com)	11-Jan-92
*/


#include	<windows.h>


#define MAX_INPUT_RECORDS	42
#define META_KEY_BIT		0x80


/*
 * select_alarmed definition in sysdep.c depends on HAVE_SELECT not being
 * defined.
 */

extern	int	select_alarmed;


/*
 * CurScreen and CAttr allow us to make sure we have the current screen colour
 * when we get focus messages.
 */

extern	HANDLE	CurScreen;
extern	UCHAR	CAttr;


INPUT_RECORD	KbdEvents[MAX_INPUT_RECORDS];

HANDLE	Keyboard;
DWORD	KbdEventCount = 0,
	KbdEventIndex = 0;

static	char	last = 0;
int	specialkey;

void	ResetKbd			(void);

void	(*keyboard_init_hook)()		= ResetKbd;


/*
 * Reads the keyboard buffer, does some translation and shoves bytes into the
 * buffer.  It is assumed the buffer is empty at the start.  size is the
 * size of the buffer.
 *
 * Returns the number of entries placed in the buffer.
 */

int
read_nt_keyboard(buffer, size)
unsigned	char	*buffer;
int	size;
{
	int	i = 0;

	while (i < size) {
		int	a;

		a = getrawinchar();
		if (a == -1) {
			return i;
		} else {
			buffer[i++] = (char)a;
		}
	}

	return i;
}


/*
 * Deal with <alt> and <ctrl> keys.  <alt> sets meta-bit, and <ctrl> masks
 * some junk.  The two are independent.
 */

#define	HAC(x)	((x) & ((Key -> dwControlKeyState & (RIGHT_CTRL_PRESSED |    \
	LEFT_CTRL_PRESSED)) ? 0x1f : 0xff)) | ((Key -> dwControlKeyState &   \
	(RIGHT_ALT_PRESSED | LEFT_ALT_PRESSED)) ? 0x80 : 0)


/*
 * I think this was doing something strange for <atl>-keys for JOVE...
 * beware...
 */

int
getrawinchar()
{
	INPUT_RECORD	*InputEvent;
	KEY_EVENT_RECORD	*Key = NULL;
	unsigned scan;

	if (specialkey = last) {
		scan = last;
		last = 0;

		return scan;
	}

tryagain:

	while (!Key) {
		/*
		 * bail out like read would if an alarm occurred... well, kind
		 * of, since we're only doing this if the alarm is the one set
		 * up by select().
		 */
		if (select_alarmed) {
			return -1;
		}

		if (!KbdEventCount) {
			/*
			 * brain damage sucks up all the cpu without a short
			 * sleep here.
			 */
			Sleep(10);		// 10ms.
			GetKbdEvents();
		}

		while (KbdEventCount) {
			KbdEventCount--;
			InputEvent = &(KbdEvents[KbdEventIndex++]);

			switch (InputEvent -> EventType) {
			case WINDOW_BUFFER_SIZE_EVENT:
			case MENU_EVENT:
				
				break;				// switch
				
			case MOUSE_EVENT:
				/*
				 * Does X code support a mouse?  We should be
				 * able to if there's anything interesteing to
				 * do.
				 */

				break;				// switch
				
			case FOCUS_EVENT: {
				/*
				 * We get a focus event when we send away the
				 * System-menu dialogs (I'd expect a menu
				 * event?)  Use this as the signal to look
				 * for a possible change of screen colours.
				 */
					
				CONSOLE_SCREEN_BUFFER_INFO	Info;

				GetConsoleScreenBufferInfo(CurScreen, &Info);

				CAttr = Info.wAttributes & 0xFF;

				break;				// switch
			}

			case KEY_EVENT:
				if (InputEvent -> Event.KeyEvent.bKeyDown) {
					Key = &(InputEvent -> Event.KeyEvent);

					switch (Key -> wVirtualKeyCode) {
					case VK_MENU:
					case VK_CAPITAL:
					case VK_SHIFT:
					case VK_CONTROL:
						/*
						 * throw these away they
						 * aren't useful
						 */

						Key = NULL;

						break;		// switch

					default:
						break;		// switch
					}
				}

				break;				// switch

			default:
				break;				// switch
			}

			if (Key) {
				break;				// while
			}
		}
	}

	/*
	 * a key with 0 repeat count is garbage
	 */

	if (!(Key -> wRepeatCount))
		goto tryagain;

	if (--(Key -> wRepeatCount)) {
		/*
		 * have to use the same key more than once...
		 */
		++KbdEventCount;
		--KbdEventIndex;
	}

	Key -> dwControlKeyState &= ~ENHANCED_KEY;

	if ((Key -> dwControlKeyState & (RIGHT_CTRL_PRESSED |
	    LEFT_CTRL_PRESSED)) /* && (Key -> uChar.UnicodeChar & 0x0ff) */ ) {
		/*
		 * a 'normal' <ctrl> key.
		 */

		return HAC(Key -> uChar.UnicodeChar);
	} else if ((Key -> dwControlKeyState & (RIGHT_ALT_PRESSED |
	    LEFT_ALT_PRESSED | RIGHT_CTRL_PRESSED | LEFT_CTRL_PRESSED)) ||
	    !(Key -> uChar.UnicodeChar & 0x0ff)) {
		switch (Key -> wVirtualKeyCode) {
			case VK_DELETE:
				if (!(Key -> dwControlKeyState &
				    (RIGHT_ALT_PRESSED | LEFT_ALT_PRESSED |
				    RIGHT_CTRL_PRESSED | LEFT_CTRL_PRESSED))) {
					return 0x7f;
				} else {
					last = HAC(0x7f);
				}

				break;

			/*
			 * There's not really much rhyme or reason to these
			 * designations, they're just what's convienient
			 */

			case VK_INSERT:
				last = HAC('A');

				break;

			case VK_HOME:
				last = HAC('C');

				break;

			case VK_END:
				last = HAC('D');

				break;

			case VK_PRIOR:
				last = HAC('E');

				break;

			case VK_NEXT:
				last = HAC('F');

				break;

			case VK_UP:
				last = HAC('G');

				break;

			case VK_DOWN:
				last = HAC('H');

				break;

			case VK_LEFT:
				last = HAC('I');

				break;

			case VK_RIGHT:
				last = HAC('J');

				break;

			case VK_F1:
				last = HAC('K');

				break;

			case VK_F2:
				last = HAC('L');

				break;

			case VK_F3:
				last = HAC('M');

				break;

			case VK_F4:
				last = HAC('N');

				break;

			case VK_F5:
				last = HAC('O');

				break;

			case VK_F6:
				last = HAC('P');

				break;

			case VK_F7:
				last = HAC('Q');

				break;

			case VK_F8:
				last = HAC('R');

				break;

			case VK_F9:
				last = HAC('S');

				break;

			case VK_F10:
				last = HAC('T');

				break;

			case VK_F11:
				last = HAC('U');

				break;

			case VK_F12:
				last = HAC('V');
				
				break;

			default:
				return (Key -> uChar.UnicodeChar) ?
					HAC(Key -> uChar.UnicodeChar) :
					HAC(Key -> wVirtualKeyCode);
		}

		return 0xff;
	} else {
		/*
		 * as far as I know this case is all "interpreted
		 * properly"
		 */

		return Key -> uChar.UnicodeChar & 0x0FF;
	}
}

#undef	HAC


/*
 * Called when the KbdEvents buffer has no more entries, will put up to
 * MAX_INPUT_RECORDS into KbdEvent for getrawinchar to zip through.
 */

BOOL
GetKbdEvents()
{
	//index == 0, event count set by return, 

	BOOL	Success;
	DWORD	NumberOfEvents;

	KbdEventIndex = 0;

	Success = PeekConsoleInput(Keyboard,
		KbdEvents,
		MAX_INPUT_RECORDS,
		&NumberOfEvents);

	if (!Success || !NumberOfEvents) {
		/*
		 * nobody home
		 */

		return FALSE;
	}

	Success = ReadConsoleInput(Keyboard,
		KbdEvents,
		MAX_INPUT_RECORDS,
		&KbdEventCount);

	/*
	 * we would expect Success == TRUE, but if it isn't there aren't any
	 * keys to look at anyway...
	 */

	return Success;
}


void
UnsetKbd()
{
	SetConsoleMode(Keyboard, ENABLE_PROCESSED_INPUT | ENABLE_LINE_INPUT |
		ENABLE_ECHO_INPUT | ENABLE_MOUSE_INPUT);
}

void
ResetKbd()
{
	Keyboard = GetStdHandle(STD_INPUT_HANDLE);

	SetConsoleMode(Keyboard, ENABLE_MOUSE_INPUT);

	KbdEventCount = 0;
	KbdEventIndex = 0;
}
