/*
   Signal support for Windows NT (tm) port of GNU EMACS.
   Copyright (C) 1992 Free Software Foundation, Inc.

   This file is part of GNU Emacs.

   GNU Emacs is free software; you can redistribute it and/or modify it
   under the terms of the GNU General Public License as published by the
   Free Software Foundation; either version 1, or (at your option) any later
   version.

   GNU Emacs is distributed in the hope that it will be useful, but WITHOUT
   ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
   FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
   more details.

   You should have received a copy of the GNU General Public License along
   with GNU Emacs; see the file COPYING.  If not, write to the Free Software
   Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.

   Tim Fleehart (apollo@online.com)	2-May-92

   Microsoft C doesn't support much in the way of signals...
*/


#include	<stdio.h>
#include	<stdlib.h>
#include	<errno.h>

#ifdef	WIN32GNU	/* remainder of file */

#include	<windows.h>
#include	"sig.h"


#define	P_NOWAIT	1
#define	WAIT_FOREVER	0xFFFFFFFF
#define	MAX_WAIT	0xFFFFFFFE


static	CRITICAL_SECTION	cs;

static	DWORD	AlarmTime = WAIT_FOREVER,
		Pids[MAXIMUM_WAIT_OBJECTS],
		AlarmThreadId,
		TmpAlarmTime;

static	HANDLE	sshAlarmThread,
		sshCompleteEvent,
		sshPauseEvent,
		WaitHandles[MAXIMUM_WAIT_OBJECTS];

static	int	WaitCount = 1,	/*
				 * We'll always be waiting on at least
				 * WaitHandles[0].
				 */

		(*SigFn[NSIG])() = {SIG_DFL};
				/*
				 * Initialise the array of current signal
				 * handlers to SIG_DFL. (Since SIG_DFL is 0
				 * this works nicely.)
				 */


int
alarm(
	int	seconds
)
{
	DWORD	oldtime,
		WaitResult;

	/*
	 * Children cannot inherit our alarm time without doing some
	 * shared-memory stuff and using this signal package...  too much to
	 * expect so we don't do it.
	 */

	/*
	 * Save away old alarm time for return value.
	 */

	oldtime = AlarmTime;

	/*
	 * Change the alarm time and let the other thread know unless it
	 * already does.
	 */

	TmpAlarmTime = seconds;

	change_alarm();

	if (GetCurrentThreadId() != AlarmThreadId) {
		if (!SetEvent(WaitHandles[0])) {
			exit(-1);
		}

		/*
		 * Wait until alarm thread says that it is ready.
		 */

		WaitResult = WaitForSingleObject(sshCompleteEvent,
			WAIT_FOREVER);

		/*
		 * Event was triggered.
		 */

		if (WaitResult != 0) {
			exit(-1);
		}
	}

	/*
	 * supposed to return the ammount of time previously remaining before
	 * an alarm would have occurred... hahaha!
	 *
	 * More importantly WAIT_FOREVER is really an alarm time of 0.
	 */

	return (oldtime == WAIT_FOREVER) ? 0 : oldtime;
}


int
(*Signal(int sig, int (*func)()))()
{
	int	(*oldfunc)();
	DWORD	WaitResult;

	/*
	 * Save old signal handler pointer for return value.
	 */

	switch (sig) {
		case SIGALRM:
		case SIGCHLD:
			/*
			 * These ones we allow.
			 *
			 * Set new signal handler; protect by critical section
			 * since another thread depends on a consistent value.
			 */

			EnterCriticalSection(&cs);

			oldfunc = SigFn[sig];
			SigFn[sig] = func;

			LeaveCriticalSection(&cs);

			break;

		default:
			/*
			 * anything else we don't
			 */

			errno = EINVAL;
			return SIG_ERR;
	}

	/*
	 * Let the other thread know, if it doesn't already.
	 */

	if (GetCurrentThreadId() != AlarmThreadId) {
		if (!SetEvent(WaitHandles[0])) {
			exit(-1);
		}

		/*
		 * Wait until alarm thread says that it is ready.
		 */

		WaitResult = WaitForSingleObject(sshCompleteEvent,
			WAIT_FOREVER);

		/*
		 * Event was triggered.
		 */

		if (WaitResult != 0) {
			exit(-1);
		}
	}

	return oldfunc;
}


/*
 *	pause - works like pause on unix (almost).
 */

int
pause()
{
	DWORD WaitResult;

	/*
	 * Wait until alarm_thread indicates a signal.
	 */

	WaitResult = WaitForSingleObject(sshPauseEvent, WAIT_FOREVER);

	/*
	 * Event was triggered.
	 */

	if (WaitResult != 0) {
		exit(-1);
	}

	/*
	 * Always return EINTR and -1.
	 */

	errno = EINTR;
	return -1;
}


/*
 * 	kill - Works like kill on unix (almost).  except that we do nothing
 * 	with pid...
 */

int
kill(
	int	pid,
	int	sig
)
{
	int	(*Fn)();

	switch (sig) {
		case SIGALRM:
		case SIGCHLD:
			EnterCriticalSection(&cs);

			Fn = SigFn[sig];

			LeaveCriticalSection(&cs);

			if (Fn == SIG_DFL) {
				/*
				 * The program is supposed to die here!
				 */

				exit(-1);
			} else if (Fn == SIG_IGN) {
				/*
				 * Okay, ignore it.
				 */
			} else {
				(void)(Fn)(sig);

				/*
				 * We call change_alarm in case signal was
				 * called by the handler we just called and
				 * caused the alarm time to change.
				 */

				change_alarm();

				/*
				 * Let a paused thread know that it got a
				 * signal of some kind.
				 */

				if (!SetEvent(sshPauseEvent)) {
					exit(-1);
				}
			}

			break;

		default:
			errno = EINVAL;

			return -1;
	}

	return 0;

	pid;		// -W3 warning elim
}


void
InitializeSignalProcessing()
{
	DWORD	idThread;

	sshCompleteEvent = CreateEvent(NULL, FALSE, FALSE, NULL);
	sshPauseEvent = CreateEvent(NULL, FALSE, FALSE, NULL);
	WaitHandles[0] = CreateEvent(NULL, FALSE, FALSE, NULL);

	if ((WaitHandles[0] == NULL) || (sshCompleteEvent == NULL) ||
	    (sshPauseEvent == NULL)) {
		exit(-1);
	}

	InitializeCriticalSection(&cs);

	sshAlarmThread = CreateThread(NULL,
		1024L,				// Stack size.
		(LPTHREAD_START_ROUTINE)alarm_thread,
		NULL,
		0L,
		&idThread);

	if (sshAlarmThread == NULL) {
		exit(-1);
	}
}


int
alarm_thread(
	void	*param
)
{
	DWORD	WaitResult;

	AlarmThreadId = GetCurrentThreadId();

	while (1) {
		int	i;
		DWORD	WaitTime;

		EnterCriticalSection(&cs);

		WaitTime = AlarmTime;

		LeaveCriticalSection(&cs);

		WaitResult = WaitForMultipleObjects(WaitCount,
			WaitHandles,
			FALSE,
			AlarmTime);

		if (WaitResult == -1) {
			/*
			 * an error occurred, bail
			 */

			exit(-1);
		} else if (!WaitResult) {
			/*
			 * This means the alarm time changed or the handler
			 * changed.  Make sure we know the correct timeout and
			 * start another wait with the current (new) timeout.
			 */

			change_alarm();

			continue;
		} else {
			/*
			 * If the wait didn't time out, one of the child
			 * processes terminated; figure out which signal to
			 * raise.
			 */

			int	(*Fn)(int);
			int	Sig = SIGCHLD;

			if (WaitResult == WAIT_TIMEOUT) {
				Sig = SIGALRM;
			}

			EnterCriticalSection(&cs);

			Fn = SigFn[Sig];

			LeaveCriticalSection(&cs);

			if (Fn == SIG_DFL) {
				/*
				 * The program is supposed to die here!
				 */

				exit(-1);
			} else if (Fn == SIG_IGN) {
				/*
				 * Okay, ignore it!
				 */

				continue;
			} else {
				(void)Fn(Sig);

				/*
				 * if signal handler changed the alarm time,
				 * we need to make sure that we're aware of the
				 * correct value.
				 */

				change_alarm();

				/*
				 * Let a paused thread know.
				 */

				if (!SetEvent(sshPauseEvent)) {
					exit(-1);
				}
			}
		}
	}

	return 0;
}


void
change_alarm(
	void
)
{

	EnterCriticalSection(&cs);

	if (!TmpAlarmTime) {
		/*
		 * Wait forever.
		 */

		AlarmTime = WAIT_FOREVER;
	} else {
		if (TmpAlarmTime > MAX_SECONDS) {
			AlarmTime = MAX_WAIT;
		} else {
			AlarmTime = TmpAlarmTime * 1000L;
		}
	}

	LeaveCriticalSection(&cs);

	/*
	 * Let the other thread know.
	 */

	if (!SetEvent(sshCompleteEvent)) {
		exit(-1);
	}
}


/*
 * wait needs to wait for a child's termination and remove a signaled child
 * from wait handles, this will involve moving the last handle in the list to a
 * lower position.
 */

// #include <sys/wait.h>

int
Wait(
	int	*status
)
{
	DWORD	WaitResult;

	/*
	 * Note that the first handle will never be signaled while we're
	 * waiting here... (how about *should* never be signaled...)
	 * None-the-less exclude it from consideration so it doesn't screw us
	 * over accidently.
	 *
	 * Do we have to deal with alarms here?  if so WAIT_FOREVER
	 * changes and we have to add more stuff...
	 */

	if (WaitCount == 1) {
		/*
		 * No unwaited upon children exist.
		 */

		errno = ECHILD;
		return -1;
	}

	WaitResult = WaitForMultipleObjects(WaitCount - 1,
		&WaitHandles[1],
		FALSE,
		WAIT_FOREVER);

	if (WaitResult == -1) {
		exit(-1);
	} else {
		int	pid;
		DWORD   ExitCode;

		/*
		 * Adjust WaitResult since we ignored first handle.
		 */

		WaitResult++;

		/*
		 * Get the exit code of the process, while we're at it make
		 * sure the process is really dead.
		 */

		if (!GetExitCodeProcess(WaitHandles[WaitResult], &ExitCode)) {
			exit(-1);
		}

		if (ExitCode == STILL_ACTIVE) {
			exit(-1);
		}

//  Need to work on status.
		if (status) {
			*status = (ExitCode & 0xff) << 8;
		}

		/*
		 * WaitResult is the index of the handle that was signaled, we
		 * need to return the corresponding pid.  We can now also
		 * close that process' handle since we're done with it.
		 */

		pid = Pids[WaitResult];
		CloseHandle(WaitHandles[WaitResult]);

		/*
		 * Remove the signaled process from both lists by moving the
		 * last one into its place, then return the pid.
		 */

		Pids[WaitResult] = Pids[--WaitCount];
		WaitHandles[WaitResult] = WaitHandles[WaitCount];

		return pid;
	}
}


/*
 * spawnve needs to add children to wait handles as well as actually spawn
 * them.  it will always be adding a handle to the end of the list.
 */

int
Spawnve(
	int	mode,
	char	*cmdname,
	char	**argv,
	char	**envp
)
{
	char			*CommandLine, *Environment, *parg, **targ;
	int			arglen;
	PROCESS_INFORMATION	Pi = {0};
	STARTUPINFO		Si = {0};

	if (WaitCount == MAXIMUM_WAIT_OBJECTS) {
		errno = ENOEXEC;

		return -1;
	}

	/*
	 * support only P_NOWAIT
	 */

	if (mode != P_NOWAIT) {
		errno = EINVAL;

		return -1;
	}

	/*
	 * we have to do some conjuring here to put argv and envp into the
	 * form CreateProcess wants...  argv needs to be a space separated/null
	 * terminated list of parameters, and envp is a null
	 * separated/double-null terminated list of parameters.
	 *
	 * Since I have no idea how large argv and envp are likely to be
	 * we figure out list lengths on the fly and allocate them.
	 */

	/*
	 * do argv...
	 */

	arglen = 0;
	targ = argv;
	while (*targ) {
		arglen += strlen(*targ++) + 1;
	}

	CommandLine = malloc(arglen);

	if (!CommandLine) {
		errno = ENOMEM;

		return -1;
	}

	targ = argv;
	parg = CommandLine;
	while (*targ) {
		strcpy(parg, *targ);
		parg += strlen(*targ++);
		*parg++ = ' ';
	}

	*--parg = '\0';

	/*
	 * and envp...
	 */

	arglen = 1;
	targ = envp;
	while (*targ) {
		arglen += strlen(*targ++) + 1;
	}

	Environment = malloc(arglen);

	if (!Environment) {
		errno = ENOMEM;

		return -1;
	}

	targ = envp;
	parg = Environment;
	while (*targ) {
		strcpy(parg, *targ);
		parg += strlen(*targ++);
		*parg++ = '\0';
	}

	*parg = '\0';

	/*
	 * Now create the process.
	 */

	if (!CreateProcess(cmdname,
	    CommandLine,
	    NULL,				// lpsaProcess
	    NULL,				// lpsaThread
	    FALSE,				// fInheritHandles
	    DETACHED_PROCESS,			// fdwCreate
	    Environment,
	    NULL,				// lpszCurDir
	    &Si,				// lpsiStartupInfo
	    &Pi)) {				// lppiProcInfo
		errno = ENOEXEC;

		return -1;
	}

	Pids[WaitCount] = Pi.dwProcessId;

	WaitHandles[WaitCount++] = Pi.hProcess;

// we should trip the alarm_thread event so it will start waiting on this
// new child.

	return Pi.dwProcessId;
}

#endif /* WIN32GNU */
