/* makepath.c -- Ensure that a directory path exists.
   Copyright (C) 1990 Free Software Foundation, Inc.

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.  */

/* Written by David MacKenzie <djm@ai.mit.edu> and
   Jim Meyering <meyering@cs.utexas.edu>. */

#ifdef __GNUC__
#define alloca __builtin_alloca
#else
#ifdef sparc
#include <alloca.h>
#else
#ifdef _AIX
 #pragma alloca
#else
char *alloca ();
#endif
#endif
#endif


#include <stdio.h>
#include <sys/types.h>
#include <sys/stat.h>
#ifdef POSIX
#include <unistd.h>
#endif
#if !defined(S_ISDIR) && defined(S_IFDIR)
#define	S_ISDIR(m) (((m) & S_IFMT) == S_IFDIR)
#endif

#include "cpio.h"



#ifdef STDC_HEADERS
#include <errno.h>
#include <stdlib.h>
#else
extern int errno;
#endif

#if defined(USG) || defined(STDC_HEADERS)
#include <string.h>
#define index strchr
#else
#include <strings.h>
#endif

void error ();

/* Ensure that the directory CONST_DIRPATH exists
   (remove any trailing slashes from CONST_DIRPATH before calling).

   Make any leading directories that don't already exist, with
   permissions PARENT_MODE.
   If the last element of CONST_DIRPATH does not exist, create it as
   a new directory with permissions MODE.
   If OWNER and GROUP are non-negative, make them the UID and GID of
   created directories.
   If VERBOSE_FMT_STRING is nonzero, use it as a printf format
   string for printing a message after successfully making a directory,
   with the name of the directory that was just made as an argument.

   Return 0 if CONST_DIRPATH exists as a directory with the proper
   ownership and permissions when done, otherwise 1. */

int
make_path (const_dirpath, mode, parent_mode, owner, group, verbose_fmt_string)
     char *const_dirpath;
     int mode;
     int parent_mode;
     int owner;
     int group;
     char *verbose_fmt_string;
{
  char *dirpath;		/* A copy we can scribble nulls on. */
  struct stat stats;

  dirpath = alloca (strlen (const_dirpath) + 1);
  strcpy (dirpath, const_dirpath);

  if (stat (dirpath, &stats))
    {
      char *slash;
      int tmp_mode;
      int need_to_re_protect;	/* Should intermediate dirs be unwritable? */
      struct ptr_list
      {
	char *dirname_end;
	struct ptr_list *next;
      };
      struct ptr_list *p, *leading_dirs = NULL;

      if ((parent_mode & 0300) == 0300)
	{
	  /* We will have write and execute permission for created dirs. */
	  tmp_mode = parent_mode;
	  need_to_re_protect = 0;
	}
      else
	{
	  tmp_mode = 0700;
	  need_to_re_protect = 1;
	}

      slash = dirpath;
      while (*slash == '/')
	slash++;
      while (slash = index (slash, '/'))
	{
	  *slash = '\0';
	  if (stat (dirpath, &stats))
	    {
	      if (mkdir (dirpath, tmp_mode))
		{
		  error (0, errno, "cannot make directory `%s'", dirpath);
		  return 1;
		}
	      else
		{
		  if (verbose_fmt_string != NULL)
		    error (0, 0, verbose_fmt_string, dirpath);

		  if ((owner >= 0 && group >= 0)
		      && chown (dirpath, owner, group))
		    {
		      error (0, errno, "%s", dirpath);
		      return 1;
		    }
		  if (need_to_re_protect)
		    {
		      struct ptr_list *new = alloca (sizeof (struct ptr_list));
		      new->dirname_end = slash;
		      new->next = leading_dirs;
		      leading_dirs = new;
		    }
		}
	    }
	  else if (!S_ISDIR (stats.st_mode))
	    {
	      error (0, 0, "`%s' exists but is not a directory", dirpath);
	      return 1;
	    }
	  *slash++ = '/';

	  /* Avoid unnecessary calls to `stat' when given
	     pathnames containing multiple adjacent slashes.  */
	  while (*slash == '/')
	    slash++;
	}

      if (mkdir (dirpath, mode))
	{
	  error (0, errno, "cannot make directory `%s'", dirpath);
	  return 1;
	}
      if (verbose_fmt_string != NULL)
	error (0, 0, verbose_fmt_string, dirpath);

      if ((owner >= 0 && group >= 0) && chown (dirpath, owner, group))
	{
	  error (0, errno, "%s", dirpath);
	  return 1;
	}

      /* If the mode for leading directories didn't include owner "wx"
	 privileges, we have to reset their protections to the correct
	 value.  */
      for (p = leading_dirs; p != NULL; p = p->next)
	{
	  *(p->dirname_end) = '\0';
	  if (chmod (dirpath, parent_mode))
	    {
	      error (0, errno, "%s", dirpath);
	      return 1;
	    }
	}
    }

  /* We get here if the entire path already exists. */

  else if (!S_ISDIR (stats.st_mode))
    {
      error (0, 0, "`%s' exists but is not a directory", dirpath);
      return 1;
    }
  else if (chmod (dirpath, mode))
    {
      error (0, errno, "%s", dirpath);
      return 1;
    }
  else if ((owner >= 0 && group >= 0) && chown (dirpath, owner, group))
    {
      error (0, errno, "%s", dirpath);
      return 1;
    }
  return 0;
}
