/* util.c - Several utility routines for cpio.
   Copyright (C) 1988, 1989, 1990 Free Software Foundation, Inc.

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.  */

#include <stdio.h>
#include <sys/types.h>
#if  defined(__MSDOS__) || defined (WIN32GNU)
#include <stdlib.h>
#if  defined(__MSDOS__)
#include <io.h>
#endif
#else
#include <pwd.h>
#include <grp.h>
#ifndef POSIX
struct passwd *getpwuid ();
struct group *getgrgid ();
#endif
#endif
#include <errno.h>
#if  ! ( defined(__MSDOS__) || defined (WIN32GNU))
extern int errno;
#include <sys/ioctl.h>
#endif
#if defined(USG) || defined(STDC_HEADERS)
#include <string.h>
#else
#include <strings.h>
#endif
#ifndef NO_MTIO
#include <sys/mtio.h>
#endif
#include "cpio.h"
#include "extern.h"
#include "rmt.h"

char *dirname ();

/* Write `output_size' bytes of `output_buffer' to file
   descriptor OUT_DES and reset `output_size' and `out_buff'. */

void
empty_output_buffer (out_des)
     int out_des;
{
  int bytes_written;

  bytes_written = rmtwrite (out_des, output_buffer, output_size);
  if (bytes_written != output_size)
    {
      error (0, errno, "write error");
      if (bytes_written == 0 && output_is_special)
	{
	  get_next_reel (out_des);
	  bytes_written = rmtwrite (out_des, output_buffer, output_size);
	}
      if (bytes_written != output_size)
	exit (1);
    }
  output_bytes += output_size;
  out_buff = output_buffer;
  output_size = 0;
}

/* Read at most NUM_BYTES or `io_block_size' bytes, whichever is smaller,
   into the start of `input_buffer' from file descriptor IN_DES.
   Set `input_size' to the number of bytes read and reset `in_buff'.
   Exit with an error if end of file is reached.  */

void
fill_input_buffer (in_des, num_bytes)
     int in_des;
     int num_bytes;
{
  in_buff = input_buffer;
  num_bytes = (num_bytes < io_block_size) ? num_bytes : io_block_size;
  input_size = rmtread (in_des, input_buffer, num_bytes);

  if (input_size == 0)
    {
      error (0, 0, "premature end of file");
      if (input_is_special)
	{
	  get_next_reel (in_des);
	  /* Try again with a new tape. */
	  input_size = rmtread (in_des, input_buffer, num_bytes);
	}
    }
  if (input_size < 0)
    error (1, errno, "read error");
  if (input_size == 0)
    exit (1);
  input_bytes += input_size;
}

/* Copy NUM_BYTES of buffer IN_BUF to `out_buff', which may be partly full.
   When `out_buff' fills up, flush it to file descriptor OUT_DES.  */

void
copy_buf_out (in_buf, out_des, num_bytes)
     char *in_buf;
     int out_des;
     long num_bytes;
{
  register long bytes_left = num_bytes;	/* Bytes needing to be copied. */
  register long space_left;	/* Room left in output buffer. */

  while (bytes_left > 0)
    {
      space_left = io_block_size - output_size;
      if (space_left == 0)
	empty_output_buffer (out_des);
      else
	{
	  if (bytes_left < space_left)
	    space_left = bytes_left;
	  bcopy (in_buf, out_buff, space_left);
	  out_buff += space_left;
	  output_size += space_left;
	  in_buf += space_left;
	  bytes_left -= space_left;
	}
    }
}

/* Copy NUM_BYTES of buffer `in_buff' into IN_BUF.
   `in_buff' may be partly full.
   When `in_buff' is exhausted, refill it from file descriptor IN_DES.  */

void
copy_in_buf (in_buf, in_des, num_bytes)
     char *in_buf;
     int in_des;
     long num_bytes;
{
  register long bytes_left = num_bytes;	/* Bytes needing to be copied. */
  register long space_left;	/* Bytes to copy from input buffer. */

  while (bytes_left > 0)
    {
      if (input_size == 0)
	fill_input_buffer (in_des, io_block_size);
      if (bytes_left < input_size)
	space_left = bytes_left;
      else
	space_left = input_size;
      bcopy (in_buff, in_buf, space_left);
      in_buff += space_left;
      in_buf += space_left;
      input_size -= space_left;
      bytes_left -= space_left;
    }
}

/*  Skip the next NUM_BYTES bytes of file descriptor IN_DES.
    If possible, use lseek to keep from reading all the bytes;
    otherwise (as from tapes), read all bytes. */

void
toss_input (in_des, num_bytes)
     int in_des;
     long num_bytes;
{
  int blocks_to_toss;		/* Number of io_block_size blocks to toss.  */
  int i;			/* Index for loop.  */

  if (num_bytes > input_size)
    {
      /* We need to toss more than what is currently in the input buffer.
         Calculate the number of full blocks to toss and either seek past
	 them or read them from tape.  The last partial block is then read
	 and prepared for a toss of less than one full block.  */
      num_bytes -= input_size;
      blocks_to_toss = num_bytes / io_block_size;
      if (input_is_seekable)
	{
	  /* We can seek past the blocks, saving input disk reads. */
	  if (lseek (in_des, (long) blocks_to_toss * io_block_size, 1) < 0L)
	    error (1, errno, "cannot seek on input");
	  input_bytes += (long) blocks_to_toss * io_block_size;
	}
      else
	{
	  /* The input is something that must be read serially. */
	  for (i = 1; i <= blocks_to_toss; i++)
	    fill_input_buffer (in_des, io_block_size);
	}
      num_bytes -= (long) blocks_to_toss * io_block_size;
      /* Get next block containing last "few" bytes to toss.  */
      fill_input_buffer (in_des, io_block_size);
    }

  /* Adjust to correct place in the current buffer full.  */
  in_buff += num_bytes;
  input_size -= num_bytes;
}

/* Copy a file using the input and output buffers, which may start out
   partly full.  After the copy, the files are not closed nor the last
   block flushed to output, and the input buffer may still be partly
   full.
   IN_DES is the file descriptor for input;
   OUT_DES is the file descriptor for output;
   NUM_BYTES is the number of bytes to copy.  */

void
copy_files (in_des, out_des, num_bytes)
     int in_des;
     int out_des;
     long num_bytes;
{
  long size;

  while (num_bytes > 0)
    {
      if (input_size == 0)
	fill_input_buffer (in_des, io_block_size);
      size = (input_size < num_bytes) ? input_size : num_bytes;
      copy_buf_out (in_buff, out_des, size);
      num_bytes -= size;
      input_size -= size;
      in_buff += size;
    }
}

/* Create all directories up to but not including the last part of NAME.
   Do not destroy any nondirectories while creating directories.  */

void
create_all_directories (name)
     char *name;
{
  char *dir;

  dir = dirname (name);
  if (dir == NULL)
    error (2, 0, "virtual memory exhausted");

  if (dir[0] != '.' || dir[1] != '\0')
    make_path (dir, 0700, 0700, -1, -1, "created directory %s");

  free (dir);
}

/* Support for remembering inodes with multiple links.  Used in the
   "copy in" and "copy pass" modes for making links instead of copying
   the file.  */

struct inode_val
{
  unsigned inode;
  char *file_name;
};

/* Inode hash table.  Allocated by first call to add_inode. */
static struct inode_val **hash_table = NULL;

/* Size of current hash table.  Initial size is 47.  (47 = 2*22 + 3) */
static int hash_size = 22;

/* Number of elements in current hash table. */
static int hash_num;

/* Find the file name associated with NODE_NUM.  If there is no file
   associated with NODE_NUM, return NULL. */

char *
find_inode_file (node_num)
     unsigned node_num;
{
#if     ! ( defined( __MSDOS__ ) || defined( WIN32GNU))
  int start;			/* Initial hash location.  */
  int temp;			/* Rehash search variable.  */

  if (hash_table != NULL)
    {
      /* Hash function is node number modulo the table size. */
      start = node_num % hash_size;

      /* Initial look into the table.  */
      if (hash_table[start] == NULL)
	return NULL;
      if (hash_table[start]->inode == node_num)
	return hash_table[start]->file_name;

      /* The home position is full with a different inode record.
	 Do a linear search terminated by a NULL pointer.  */
      for (temp = start + 1; hash_table[temp] != NULL && temp != start;
	   temp = (temp + 1) % hash_size)
	{
	  if (hash_table[temp]->inode == node_num)
	    return hash_table[temp]->file_name;
	}
    }
#endif
  return NULL;
}

/* Associate FILE_NAME with the inode NODE_NUM.  (Insert into hash table.) */

void
add_inode (node_num, file_name)
     unsigned short node_num;
     char *file_name;
{
  struct inode_val *temp;

  /* Create new inode record.  */
  temp = (struct inode_val *) xmalloc (sizeof (struct inode_val));
  temp->inode = node_num;
  temp->file_name = xstrdup (file_name);

  /* Do we have to increase the size of (or initially allocate)
     the hash table?  */
  if (hash_num == hash_size || hash_table == NULL)
    {
      struct inode_val **old_table;	/* Pointer to old table. */
      int i;			/* Index for re-insert loop. */

      /* Save old table.  */
      old_table = hash_table;
      if (old_table == NULL)
	hash_num = 0;

      /* Calculate new size of table and allocate it.
         Sequence of table sizes is 47, 97, 197, 397, 797, 1597, 3197, 6397 ...
	 where 3197 and most of the sizes after 6397 are not prime.  The other
	 numbers listed are prime.  */
      hash_size = 2 * hash_size + 3;
      hash_table = (struct inode_val **)
	xmalloc (hash_size * sizeof (struct inode_val *));

      /* Insert the values from the old table into the new table.  */
      for (i = 0; i < hash_num; i++)
	hash_insert (old_table[i]);

      if (old_table != NULL)
	free (old_table);
    }

  /* Insert the new record and increment the count of elements in the
      hash table. */
  hash_insert (temp);
  hash_num++;
}

/* Do the hash insert.  Used in normal inserts and resizing the hash
   table.  It is guaranteed that there is room to insert the item.
   NEW_VALUE is the pointer to the previously allocated inode, file
   name association record.  */

void
hash_insert (new_value)
     struct inode_val *new_value;
{
  int start;			/* Home position for the value.  */
  int temp;			/* Used for rehashing. */

  /* Hash function is node number modulo the table size. */
  start = new_value->inode % hash_size;

  /* Do the initial look into the table.  */
  if (hash_table[start] == NULL)
    {
      hash_table[start] = new_value;
      return;
    }

  /* If we get to here, the home position is full with a different inode
     record.  Do a linear search for the first NULL pointer and insert
     the new item there.  */
  temp = (start + 1) % hash_size;
  while (hash_table[temp] != NULL)
    temp = (temp + 1) % hash_size;

  /* Insert at the NULL. */
  hash_table[temp] = new_value;
}

/* Attempt to rewind the tape drive on file descriptor TAPE_DES
   and take it offline.  */

void
tape_offline (tape_des)
     int tape_des;
{
#ifdef MTIOCTOP
  struct mtop control;

  control.mt_op = MTOFFL;
  control.mt_count = 1;
  rmtioctl (tape_des, MTIOCTOP, &control); /* Don't care if it fails.  */
#endif
}

/* The file on file descriptor TAPE_DESC is assumed to be magnetic tape
   and the end of the tape has been reached.
   Ask the user for to mount a new tape to continue the processing.  */

void
get_next_reel (tape_des)
     int tape_des;
{
  FILE *tty_in;			/* File for interacting with user.  */
  FILE *tty_out;		/* File for interacting with user.  */
  int c;

  /* Open files for interactive communication.  */
  tty_in = fopen (CONSOLE, "r");
  if (tty_in == NULL)
    error (2, errno, CONSOLE);
  tty_out = fopen (CONSOLE, "w");
  if (tty_out == NULL)
    error (2, errno, CONSOLE);

  /* Give message and wait for carrage return.  User should hit carrage return
     only after loading the next tape.  */
  fprintf (tty_out, "Found end of tape.  Load next tape and press RETURN. ");
  fflush (tty_out);

  tape_offline (tape_des);

  do
    c = getc (tty_in);
  while (c != EOF && c != '\n');

  fclose (tty_in);
  fclose (tty_out);
}

#if defined( __MSDOS__ ) || defined (WIN32GNU)
int
chown (path, owner, group)
     char *path;
     int owner, group;
{
  return 0;
}
#endif

#ifdef __TURBOC__
#include <time.h>
#include <fcntl.h>
#include <io.h>

struct utimbuf
{
  time_t  actime;		/* Access time. */
  time_t  modtime;		/* Modification time. */
};

int
utime (char *filename, struct utimbuf *utb)
{
  extern int errno;
  struct tm *tm;
  struct ftime filetime;
  time_t when;
  int fd;
  int status;

  if (utb == 0)
    when = time (0);
  else
    when = utb->modtime;

  fd = _open (filename, O_RDWR);
  if (fd == -1)
    return -1;

  tm = localtime (&when);
  if (tm->tm_year < 80)
    filetime.ft_year = 0;
  else
    filetime.ft_year = tm->tm_year - 80;
  filetime.ft_month = tm->tm_mon + 1;
  filetime.ft_day = tm->tm_mday;
  if (tm->tm_hour < 0)
    filetime.ft_hour = 0;
  else
    filetime.ft_hour = tm->tm_hour;
  filetime.ft_min = tm->tm_min;
  filetime.ft_tsec = tm->tm_sec / 2;

  status = setftime (fd, &filetime);
  _close (fd);
  return status;
}
#endif
