/****************************************************************************
  PROGRAM: Generic.c
  PURPOSE: Generic template for Windows applications
  FUNCTIONS:
  WinMain() - calls initialization function, processes message loop
  InitApplication() - initializes window data and registers window
  InitInstance() - saves instance handle and creates main window
  MainWndProc() - processes messages
  About() - processes messages for "About" dialog box
  COMMENTS:
      Windows can have several copies of your application running at the
      same time.  The variable hInst keeps track of which instance this
      application is so that processing will be to the correct window.
****************************************************************************/
#include <windows.h>            /* required for all Windows applications */
#include <ctl3d.h>                        /* 3d controls dll prototypes    */
#include "generic.h"                      /* specific to this program      */

#include "KeyCal.h"           // control message constants

HANDLE hInst;                             /* current instance              */

/*********************************************/
/* variables used by the calendar dialog box */
/* be careful how much stuff you put on the  */
/* stack with windows routines!              */
/*********************************************/
static long l, lDays[ 31 ];
static int  nDays[ 31 ];
static char sz[ 64 ];

/***********************************************************************/
/* This routine was added to the generic application template provided */
/* by Microsoft. The rest of the program is a very simple example of a */
/* Windows program.                                                    */
/*                                                                     */
/* This routine intercepts notification messages from the calendar and */
/* displays the julian date number in an edit box below the calendar.  */
/*                                                                     */
/* A test button is provided that runs a suite of tests against the    */
/* control to check it's operation. The code as written only works     */
/* for the day that it was written for. It was the only way to test    */
/* some of the functions without getting ridiculous.                   */
/*                                                                     */
/* A couple of selection buttons allow you to disallow selection of    */
/* certain days of the week.                                           */
/*                                                                     */
/* This code may be freely used as an example in your own program      */
/***********************************************************************/
BOOL FAR PASCAL Calendar( HWND hWnd, unsigned message, WORD wParam, LONG lParam )
  {
    switch ( message )
      {
        case WM_INITDIALOG:
          return ( TRUE );

        case WM_COMMAND:
          switch ( wParam )
            {
              /* user pressed the test button */
              case IDD_TEST:
                /* send reset and check that one date is selected */
                SendDlgItemMessage( hWnd, IDD_CAL, DM_RESET, 0, 0L );

                /* test all the functions */

                if ( SendDlgItemMessage( hWnd, IDD_CAL, DM_GET_MONTH, 0, 0L ) != 4 )
                  MessageBox( NULL, "DM_GET_MONTH does not return april", NULL, MB_ICONEXCLAMATION );

                if ( SendDlgItemMessage( hWnd, IDD_CAL, DM_GET_YEAR, 0, 0L ) != 1993 )
                  MessageBox( NULL, "DM_GET_YEAR does not return 1993", NULL, MB_ICONEXCLAMATION );

                if ( SendDlgItemMessage( hWnd, IDD_CAL, DM_GET_DAY, 0, 0L ) != 8 )
                  MessageBox( NULL, "DM_GET_DAY does not return 8th", NULL, MB_ICONEXCLAMATION );

                l = SendDlgItemMessage( hWnd, IDD_CAL, DM_GETSELITEMS, 31, (LONG)(LPINT)nDays );
                if ( l != 1 )
                  {
                    wsprintf( sz, "DM_GETSELITEMS returns %ld ( %d )", l, nDays[ 0 ] );
                    MessageBox( NULL, sz, NULL, MB_ICONEXCLAMATION );
                  }
                if ( nDays[ 0 ] != 8 )
                  {
                    wsprintf( sz, "DM_GETSELITEMS returns wrong selected item %d", nDays[ 0 ] );
                    MessageBox( NULL, sz, NULL, MB_ICONEXCLAMATION );
                  }

                l = SendDlgItemMessage( hWnd, IDD_CAL, DM_GETSELDATES, 31, (LONG)(LPLONG)lDays );
                if ( l != 1 )
                  {
                    wsprintf( sz, "DM_GETSELDATES returns %ld ( %ld )", l, lDays[ 0 ] );
                    MessageBox( NULL, sz, NULL, MB_ICONEXCLAMATION );
                  }

                if ( SendDlgItemMessage( hWnd, IDD_CAL, DM_GET_DAYS, 0, 0L ) != 0 )
                  MessageBox( NULL, "DM_GET_DAYS does not return 0", NULL, MB_ICONEXCLAMATION );


                /* set the date and try tests again */
                SendDlgItemMessage( hWnd, IDD_CAL, DM_SET_DATE, 0, 2444605L );

                if ( SendDlgItemMessage( hWnd, IDD_CAL, DM_GET_MONTH, 0, 0L ) != 1 )
                  MessageBox( NULL, "DM_GET_MONTH does not return January", NULL, MB_ICONEXCLAMATION );

                if ( SendDlgItemMessage( hWnd, IDD_CAL, DM_GET_YEAR, 0, 0L ) != 1981 )
                  MessageBox( NULL, "DM_GET_YEAR does not return 1981", NULL, MB_ICONEXCLAMATION );

                if ( SendDlgItemMessage( hWnd, IDD_CAL, DM_GET_DAY, 0, 0L ) != 1 )
                  MessageBox( NULL, "DM_GET_DAY does not return 1st", NULL, MB_ICONEXCLAMATION );

                l = SendDlgItemMessage( hWnd, IDD_CAL, DM_GETSELITEMS, 31, (LONG)(LPINT)nDays );
                if ( l != 1 )
                  {
                    wsprintf( sz, "DM_GETSELITEMS returns %ld ( %d )", l, nDays[ 0 ] );
                    MessageBox( NULL, sz, NULL, MB_ICONEXCLAMATION );
                  }
                if ( nDays[ 0 ] != 1 )
                  {
                    wsprintf( sz, "DM_GETSELITEMS returns wrong selected item %d", nDays[ 0 ] );
                    MessageBox( NULL, sz, NULL, MB_ICONEXCLAMATION );
                  }

                l = SendDlgItemMessage( hWnd, IDD_CAL, DM_GETSELDATES, 31, (LONG)(LPLONG)lDays );
                if ( l != 1 )
                  {
                    wsprintf( sz, "DM_GETSELDATES returns %ld ( %ld )", l, lDays[ 0 ] );
                    MessageBox( NULL, sz, NULL, MB_ICONEXCLAMATION );
                  }

                if ( SendDlgItemMessage( hWnd, IDD_CAL, DM_GET_DAYS, 0, 0L ) !=  ( 2449085L - 2444605L ) )
                  MessageBox( NULL, "DM_GET_DAYS does not return correct value", NULL, MB_ICONEXCLAMATION );

                MessageBox( NULL, "All function tests complete", NULL, MB_ICONEXCLAMATION );

                break;

              /* any day of the week is selectable */
              case IDD_ANY:
                SendDlgItemMessage( hWnd, IDD_CAL, DM_SELECT_ANY, 0, 0L );
                break;

              /* only weekends are selectable */
              case IDD_WEEKEND:
                SendDlgItemMessage( hWnd, IDD_CAL, DM_SELECT_DAY_MASK, 0, (LONG)0x81 );
                break;

              /* only weekdays are selectable */
              case IDD_WEEKDAY:
                SendDlgItemMessage( hWnd, IDD_CAL, DM_SELECT_DAY_MASK, 0, (LONG)0x3E );
                break;

              /* calendar notification messages */
              case IDD_CAL:
                /* high order word of lParam contains notification code */
                /* low order word of lParam contains window handle */
                switch ( HIWORD( lParam ) )
                  {
                    case DN_DATECHANGED:
                      // get new date from date control
                      l = SendMessage( LOWORD( lParam ), DM_GET_DATE, 0, 0L );

                      // put date into text control
                      wsprintf( sz, "%ld", l );
                      SetDlgItemText( hWnd, IDD_DATETEXT, (LPSTR) sz );
                      break;
                  }
                break;

              case IDD_HELP:
                WinHelp( hWnd, "keycal.hlp", HELP_INDEX, 0L );
                break;

              case IDOK:
                EndDialog( hWnd, TRUE );
                return ( TRUE );
                break;

              case IDCANCEL:
                EndDialog( hWnd, TRUE );
                return ( TRUE );
                break;
            }
          break;
      }
    return ( FALSE );
  }

/****************************************************************************
  FUNCTION: WinMain(HANDLE, HANDLE, LPSTR, int)
  PURPOSE: calls initialization function, processes message loop
  COMMENTS:
    Windows recognizes this function by name as the initial entry point 
    for the program.  This function calls the application initialization 
    routine, if no other instance of the program is running, and always 
    calls the instance initialization routine.  It then executes a message 
    retrieval and dispatch loop that is the top-level control structure 
    for the remainder of execution.  The loop is terminated when a WM_QUIT 
    message is received, at which time this function exits the application 
    instance by returning the value passed by PostQuitMessage(). 
    If this function must abort before entering the message loop, it 
    returns the conventional value NULL.  
****************************************************************************/
int PASCAL WinMain( HANDLE hInstance,       /* current instance         */
                    HANDLE hPrevInstance,   /* previous instance         */
                    LPSTR lpCmdLine,        /* command line             */
                    int nCmdShow            /* show-window type (open/icon) */
                  )
  {
  HANDLE   hLibrary;
  MSG msg;                     /* message                 */

    /****************************************************/
    /* The following two calls use the 3d look controls */
    /* provided by Microsoft in ctl3d.dll               */
    /****************************************************/
    Ctl3dRegister( hInstance );         /* use new 3d look controls        */
    Ctl3dAutoSubclass( hInstance );

    if ( ! hPrevInstance )             /* Other instances of app running? */
      if ( ! InitApplication( hInstance ) ) /* Initialize shared things */
        return ( FALSE );         /* Exits if unable to initialize     */

    /* Perform initializations that apply to a specific instance */
    if ( ! InitInstance( hInstance, nCmdShow ) )
      return (FALSE);

    /****************************************************************/
    /* This loads the calendar DLL so it can be used in the program */
    /****************************************************************/
    hLibrary = LoadLibrary( "KeyCal.DLL" );
    if ( hLibrary < 32 )
      {
        MessageBox( NULL, "Cannot Load KeyCal DLL", NULL, MB_ICONEXCLAMATION );
        return ( -1 );
      }

    /* Acquire and dispatch messages until a WM_QUIT message is received. */
    while (GetMessage(&msg,       /* message structure                 */
        NULL,           /* handle of window receiving the message */
        NULL,           /* lowest message to examine             */
        NULL))           /* highest message to examine         */
      {
        TranslateMessage(&msg);           /* Translates virtual key codes  */
        DispatchMessage(&msg);            /* Dispatches message to window  */
      }

    /**************************************************************/
    /* Don't forget to unload the calendar DLL when done using it */
    /* Otherwise Windows will keep it in memory                   */
    /**************************************************************/
    FreeLibrary( hLibrary );

    /***********************************/
    /* done using the 3d look controls */
    /* unregister them to free memory  */
    /***********************************/
    Ctl3dUnregister( hInst );             /* remove subclassing on controls*/

    return (msg.wParam);       /* Returns the value from PostQuitMessage */
  }


/****************************************************************************
  FUNCTION: InitApplication(HANDLE)
  PURPOSE: Initializes window data and registers window class
  COMMENTS:
    This function is called at initialization time only if no other 
    instances of the application are running.  This function performs 
    initialization tasks that can be done once for any number of running 
    instances.  
    In this case, we initialize a window class by filling out a data 
    structure of type WNDCLASS and calling the Windows RegisterClass() 
    function.  Since all instances of this application use the same window 
    class, we only need to do this when the first instance is initialized.  
****************************************************************************/
BOOL InitApplication( HANDLE hInstance )
  {
  WNDCLASS  wc;

    wc.style = NULL;                   /* Class style(s).                   */
    wc.lpfnWndProc = MainWndProc;      /* Function to retrieve messages for */
                                       /* windows of this class.            */
    wc.cbClsExtra = 0;                 /* No per-class extra data.          */
    wc.cbWndExtra = 0;                 /* No per-window extra data.         */
    wc.hInstance = hInstance;          /* Application that owns the class.  */
    wc.hIcon = LoadIcon(NULL, IDI_APPLICATION);
    wc.hCursor = LoadCursor(NULL, IDC_ARROW);
    wc.hbrBackground = GetStockObject(WHITE_BRUSH); 
    wc.lpszMenuName =  "GenericMenu";   /* Name of menu resource in .RC file. */
    wc.lpszClassName = "GenericWClass"; /* Name used in call to CreateWindow. */

    /* Register the window class and return success/failure code. */
    return ( RegisterClass( & wc ) );
  }


/****************************************************************************
  FUNCTION:  InitInstance(HANDLE, int)
  PURPOSE:  Saves instance handle and creates main window
  COMMENTS:
    This function is called at initialization time for every instance of 
    this application.  This function performs initialization tasks that 
    cannot be shared by multiple instances.  
    In this case, we save the instance handle in a static variable and 
    create and display the main program window.  
****************************************************************************/
BOOL InitInstance(hInstance, nCmdShow)
    HANDLE          hInstance;          /* Current instance identifier.       */
    int             nCmdShow;           /* Param for first ShowWindow() call. */
  {
  HWND            hWnd;               /* Main window handle.                */

    /* Save the instance handle in static variable, which will be used in  */
    /* many subsequence calls from this application to Windows.            */

    hInst = hInstance;

    /* Create a main window for this application instance.  */

    hWnd = CreateWindow(
        "GenericWClass",                /* See RegisterClass() call.          */
        "Generic Sample Application",   /* Text for window title bar.         */
        WS_OVERLAPPEDWINDOW,            /* Window style.                      */
        CW_USEDEFAULT,                  /* Default horizontal position.       */
        CW_USEDEFAULT,                  /* Default vertical position.         */
        CW_USEDEFAULT,                  /* Default width.                     */
        CW_USEDEFAULT,                  /* Default height.                    */
        NULL,                           /* Overlapped windows have no parent. */
        NULL,                           /* Use the window class menu.         */
        hInstance,                      /* This instance owns this window.    */
        NULL                            /* Pointer not needed.                */
    );

    /* If window could not be created, return "failure" */

    if (!hWnd)
        return (FALSE);

    /* Make the window visible; update its client area; and return "success" */

    /* Show the window */
    ShowWindow(hWnd, nCmdShow );
    UpdateWindow(hWnd);          /* Sends WM_PAINT message                 */
    return (TRUE);               /* Returns the value from PostQuitMessage */
  }

/****************************************************************************
  FUNCTION: MainWndProc(HWND, unsigned, WORD, LONG)
  PURPOSE:  Processes messages
  MESSAGES:
    WM_COMMAND    - application menu (About dialog box)
    WM_DESTROY    - destroy window
  COMMENTS:
    To process the IDM_ABOUT message, call MakeProcInstance() to get the
    current instance address of the About() function.  Then call Dialog
    box which will create the box according to the information in your
    generic.rc file and turn control over to the About() function.    When
    it returns, free the intance address.
****************************************************************************/
long FAR PASCAL MainWndProc( HWND hWnd,         /* window handle           */
                             unsigned message,  /* type of message         */
                             WORD wParam,       /* additional information  */
                             LONG lParam        /* additional information  */
                           )
  {
  FARPROC lpProcAbout;          /* pointer to the "About" function */

    switch (message)
      {
        case WM_COMMAND:       /* message: command from application menu */
          switch ( wParam )
            {
              /**********************************/
              /* exit for a windows application */
              /**********************************/
              case IDM_F_EXIT:
                PostQuitMessage( 0 );
                break;

              case IDM_CAL:
                lpProcAbout = MakeProcInstance( Calendar, hInst );
                DialogBox( hInst,              /* current instance         */
                           "CalendarTest",  /* resource to use          */
                           hWnd,               /* parent handle            */
                           lpProcAbout         /* About() instance address */
                         );         
                FreeProcInstance( lpProcAbout );
                break;

              case IDM_ABOUT:
                lpProcAbout = MakeProcInstance( About, hInst );
                DialogBox(hInst,                  /* current instance         */
                          "AboutBox",             /* resource to use         */
                          hWnd,                   /* parent handle         */
                          lpProcAbout);           /* About() instance address */
                FreeProcInstance( lpProcAbout );
                break;

              default:
                return (DefWindowProc(hWnd, message, wParam, lParam));
            }
          break;

        case WM_DESTROY:          /* message: window being destroyed */
          PostQuitMessage(0);
          break;

        default:              /* Passes it on if unproccessed    */
          return (DefWindowProc(hWnd, message, wParam, lParam));
      }
    return (NULL);
  }


/****************************************************************************
  FUNCTION: About(HWND, unsigned, WORD, LONG)
  PURPOSE:  Processes messages for "About" dialog box
  MESSAGES:
    WM_INITDIALOG - initialize dialog box
    WM_COMMAND    - Input received
  COMMENTS:
    No initialization is needed for this particular dialog box, but TRUE
    must be returned to Windows.
    Wait for user to click on "Ok" button, then close the dialog box.
****************************************************************************/
BOOL FAR PASCAL About(hDlg, message, wParam, lParam)
HWND hDlg;                                /* window handle of the dialog box */
unsigned message;                         /* type of message                 */
WORD wParam;                              /* message-specific information    */
LONG lParam;
  {
    switch (message) {
    case WM_INITDIALOG:           /* message: initialize dialog box */
        return (TRUE);

    case WM_COMMAND:              /* message: received a command */
        if (wParam == IDOK                /* "OK" box selected?         */
                || wParam == IDCANCEL) {      /* System menu close command? */
        EndDialog(hDlg, TRUE);          /* Exits the dialog box         */
        return (TRUE);
        }
        break;
    }
    return (FALSE);                  /* Didn't process a message    */
  }
