; Generic PUS() and POP() procedures.
;----------------------------------------------------------------------
; This proc pushes an item on a stack
; Parameters:        WHAT - the item to push, DYN - the dynarray
; Globals Assumed:   NONE
; Procs Called:      NONE
; Returns:           True
;----------------------------------------------------------------------
proc PUSH( WHAT, DYN )
   private X, Z

   ; move 'em all down one
   Z = dynarraysize(DYN)
   for X from Z to 1 step -1
      DYN[ X + 1 ] = DYN[ X ]
   endfor

   ; put what in the no. 1 slot
   DYN[1] = WHAT

   return true

endproc


;----------------------------------------------------------------------
; This proc is a pops an item off of a stack
; Parameters:        DYN - the dynarray with items
; Globals Assumed:   NONE
; Procs Called:      NONE
; Returns:           WHAT or false (false if DYN is empty)
;----------------------------------------------------------------------
proc POP( DYN )
   private WHAT, X, Z

   Z = dynarraysize(DYN)
   if Z = 0 then
      return false
   endif

   ; WHAT is the first item on the stack

   WHAT = DYN[1]

   ; move em all up one

   for X from 1 to Z - 1
      DYN[X] = DYN[ X + 1 ]
   endfor

   ; release the last stack holder (reduce the stack size by 1)

   release vars DYN[ Z ]
   return WHAT

endproc


; Hypertext, context-sensitive help code.
;----------------------------------------------------------------------
; This proc displays context sensitive help for each dialog box
; Parameters:        NONE
; Globals Assumed:   SYS_BAG["HELPTAG"] - a dialog box indicator
; Procs Called:      SHOWMSG(),
;                    HELP_DISPLAY(),
;                    CLEARMSG()
; Returns:           True or False
;----------------------------------------------------------------------
proc HELP_CONTROL()
   private RV, TAG, BACK, HMODE, DBAG, SEEALSO, HELPTEXT

   if not isassigned(SYS_BAG["HELPTAG"]) then
      SHOWMSG("No help is defined",true)
      return false
   endif

   ; HELPTEXT[] is defined as an array of 1 element, in case help
   ; does not exist for a specific screen or menu choice - this
   ; allows the error proc to recover with text in the "standard" array

   dynarray BACK[]
   dynarray DBAG[]
   dynarray SEEALSO[]
   array HELPTEXT[1]

   ; save the current help tag

   TAG = SYS_BAG["HELPTAG"]

   DBAG["INDEX"] = ""
   DBAG["SEL"] = "Ok"
   HELPTEXT[1] = ""

   ; load the helptext array and the seealso dynarray

   execproc "HELPTXT_"+TAG

   HELP_DISPLAY()

   if DBAG["SEL"] = "Ok" then
      return false
   else
      return true
   endif

endproc

;----------------------------------------------------------------------
; This proc displays the help text
; Parameters:           None
; Globals Assumed:      All in HELP_CONTROL()
; Procs Called:         HELP_PROC()
; Returns:              Nothing
;----------------------------------------------------------------------
proc HELP_DISPLAY()

  showdialog "Help"
     proc "HELP_PROC" trigger "UPDATE" key "Esc"

     @3,9 height 17 width 56

     ; the actual help text in a wide pickarray

     pickarray @1,1 height 11 width 51
        columns 1
        HELPTEXT
        tag "HELPTEXT"
        to DBAG["INDEX"]

     pushbutton @13,1 width 10
        "~O~K"
        ok default
        value "Ok"
        tag "Ok"
        to DBAG["SEL"]

     pushbutton @13,13 width 11
        "~S~eeAlso"
        value "SeeAlso"
        tag "SeeAlso"
        to DBAG["SEL"]

     pushbutton @13,26 width 10
        "~B~ack"
        value "Back"
        tag "Back"
        to DBAG["SEL"]

  enddialog

endproc

;----------------------------------------------------------------------
; This proc handles triggers from within the help dbox
; Parameters:        4 dialog box proc parameters
; Globals Assumed:   BACK[], SEEALSO[]
; Procs Called:      PUSH(), POP()
; Returns:           true or false
;----------------------------------------------------------------------
proc HELP_PROC( TRIGTYPE, TAGVALUE, EVENTVALUE, ELEMVALUE )
   private IDX, TEMP

   switch
      case TRIGTYPE = "EVENT" and EVENTVALUE["TYPE"] = "KEY" :
         if EVENTVALUE["KEYCODE"] = asc("Esc") then
            canceldialog
         endif
         return true

      case TRIGTYPE = "UPDATE" :
         switch
            case TAGVALUE = "SeeAlso" :
               if dynarraysize(SEEALSO) = 0 then
                  SHOWMSG("There are no see also links for this topic",false)
                  return true
               endif

               ; display the popup list of SeeAlso topics in a new dialog box
               ; return the selected INDEX, or False if user cancelled

               IDX = SHOW_POPUP( SEEALSO )
               if IDX <> false then

                  ; push the current help tag on the hypertext stack

                  PUSH( TAG, BACK )

                  ; make the see-also list the active help screen
                  TAG = IDX

                  ; execute the help proc which will load arrays
                  execproc "HELPTXT_"+TAG

                  ; reset the highlight bar
                  DBAG["INDEX"] = ""

                  ; redisplay the help text
                  refreshcontrol "HELPTEXT"

               endif
               return true

            case TAGVALUE = "Back" :

               ; pop the last subject off of the hypertext stack
               TEMP = POP( BACK )
               if TEMP = false then
                  SHOWMSG("You are at the first help screen",false)
                  return true
               endif

               ; make the popped item the current help topic
               TAG = TEMP

               ; execute the help proc which will load arrays
               execproc "HELPTXT_"+TAG

               ; reset the highlight bar
               DBAG["INDEX"] = ""

               ; redisplay the help text
               refreshcontrol "HELPTEXT"

               return true

         endswitch

   endswitch

   return true

endproc


;A sample help procedure created with a help code generator.
proc HELPTXT_SYS02()
   array HELPTEXT[29]
   dynarray SEEALSO[]

   HELPTEXT[1] = "Report Frequency"
   HELPTEXT[2] = ""
   HELPTEXT[3] = "This dialog box allows the user to set"
   HELPTEXT[4] = "frequencies associated with various reports"
   HELPTEXT[5] = "that are available in the system.  "
   HELPTEXT[6] = ""
   HELPTEXT[7] = "If a report is late in being generated on the"
   HELPTEXT[8] = "assigned frequency, the user will be notified"
   HELPTEXT[9] = "upon signon to the system.  The reminders will"
   HELPTEXT[10] = "continue each time the user logs on, until the"
   HELPTEXT[11] = "report is generated, or the frequency is"
   HELPTEXT[12] = "changed."
   HELPTEXT[13] = ""
   HELPTEXT[14] = "The report name list box shows a list of"
   HELPTEXT[15] = "available reports.  To set a frequency, the"
   HELPTEXT[16] = "user selects the desired report and toggles the"
   HELPTEXT[17] = "appropriate frequency radio button in the"
   HELPTEXT[18] = "Frequency box.  The user may remove a frequency"
   HELPTEXT[19] = "set for a report by toggling the None radio "
   HELPTEXT[20] = "button in the Frequency box."
   HELPTEXT[21] = ""
   HELPTEXT[22] = "Buttons"
   HELPTEXT[23] = ""
   HELPTEXT[24] = "Ok:  saves changes and returns to the main"
   HELPTEXT[25] = "menu."
   HELPTEXT[26] = ""
   HELPTEXT[27] = "Cancel:  cancels any changes that may have been"
   HELPTEXT[28] = "made and returns to the main menu."
   HELPTEXT[29] = ""

   SEEALSO["SYS15"] = "Late Reports"

endproc


;Many portions of the system required a generic pop-up utility which can display a dynarray or an array. ;The following code shows how this is handled.
;----------------------------------------------------------------------
; This procedure is the generic popup script
; Parameters:       DPOP     = dynarray containing select list elements
; Globals assumed:  TAGVALUE = the 2nd dbox parameter (for the dbox title)
; Procedure called: SHOW_PROC()
; Returns:          retval = the selected element, or False if canceled
;----------------------------------------------------------------------
proc SHOW_POPUP( DPOP )
   private DYIND, DLEN, DCOL, L
   DLEN = 0

   DYIND = ""

   ; code for a dynarray popup

   if type(DPOP) = "DY" then

         ; get the length of the largest element in the popup
      foreach X in DPOP
         L = len(DPOP[X])
         if L > DLEN then
            DLEN = L
         endif
      endforeach

      if DLEN > 72 then
         DLEN = 72
      else
         if DLEN < 32 then
            DLEN = 32
         else
            DLEN = DLEN + 4
         endif
      endif

      DCOL = int ((80 - DLEN ) / 2)


      ShowDialog
         TAGVALUE
         proc "SHOW_PROC" trigger "SELECT"
         @ 5,DCOL Height 11 Width DLEN

         pickdynarray
            @ 1,1 height 5 width DLEN - 5
            DPOP
            tag "List"
            to DYIND

         PushButton
            @ 7,1 Width 10
            "~O~K"
            OK
            default
            Value "OK"
            Tag "OK"
         To DBAG["SELBUTTON"]

         PushButton
            @ 7,13 Width 10
            "Cancel"
            Cancel
            Value "Cancel"
            Tag "Cancel"
         To DBAG["SELBUTTON"]


      EndDialog

   else

      ; code for an array popup

      ; get the length of the largest element in the popup
      for X from 1 to arraysize(DPOP)
         L = len(DPOP[X])
         if L > DLEN then
            DLEN = L
         endif
      endfor

      if DLEN > 72 then
         DLEN = 72
      else
         if DLEN < 32 then
            DLEN = 32
         else
            DLEN = DLEN + 4
         endif
      endif
      DCOL = int ((80 - DLEN ) / 2)


      ShowDialog
         TAGVALUE
         proc "SHOW_PROC" trigger "SELECT"
         @ 5,DCOL Height 11 Width DLEN

         pickarray
            @ 1,1 height 5 width DLEN - 5
            DPOP
            tag "List"
            to DYIND

         PushButton
            @ 7,1 Width 10
            "~O~K"
            OK
            default
            Value "OK"
            Tag "OK"
         To DBAG["SELBUTTON"]

         PushButton
            @ 7,13 Width 10
            "Cancel"
            Cancel
            Value "Cancel"
            Tag "Cancel"
         To DBAG["SELBUTTON"]


      EndDialog

   endif

   if retval = true then
      return DYIND
   else
      return false
   endif

endproc

;----------------------------------------------------------------------
; This procedure is the dialog proc for SHOW_POPUP
; Parameters:       TRIGTYPE = Trigger type
;                   TAGVALUE = Tag value of the active control element
;                   EV       = Event value
;                   ELEMENT  = Element value
; Globals assumed:  DYIND[]  = index of dynarray
;                   DPOP[]   = dynarray for pickdynarray
; Returns:          True to allow the event to occur
;                   False to deny the event
;----------------------------------------------------------------------
proc SHOW_PROC( TRIGTYPE, TAGVALUE, EV, ELEMENT )

   switch
      case TRIGTYPE = "SELECT" :
         acceptdialog

      otherwise :
         return true

   endswitch

endproc


;This procedure is used to convert a number to a number string of any base. It was used to compress ;numeric information so that semaphor locking could control the 1:M:M relationships for an all-dialog ;box interface.
;----------------------------------------------------------------------
; This proc takes a number and returns the number converted to any base
; the returned number is a character string.  Allowable bases are
; from 2 to 51
; Parameters:       NUM, BASE
; Globals Assumed:  NONE
; Procs Called:     NONE
; Returns:          a converted number as a string or false
;----------------------------------------------------------------------
proc BASEX( NUM, BASE )
   private TEMP, NEWNUM, C, X

   ; don't accept a base higher than 51 or less than 2

   if BASE > 51 or BASE < 2 then
      return false
   endif

   ; declare the array to hold symbols for bases > 10

   array C[52]

   ; initialize the first 10 places

   for X from 1 to 10
      C[X] = strval( X - 1 )
   endfor

   ; initialize A-Z

   for X from 11 to 37
      C[X] = chr(65 + X - 11)
   endfor

   ; initialize several misc. but valid DOS filename characters

   C[38] = "~"
   C[39] = "#"
   C[40] = "$"
   C[41] = "%"
   C[42] = "^"
   C[43] = "&"
   C[44] = "("
   C[45] = ")"
   C[46] = "{"
   C[47] = "}"
   C[48] = "-"
   C[49] = "_"
   C[50] = "@"
   C[51] = "`"
   C[52] = "'"

   ; NEWNUM will hold our converted "number", actually a string

   NEWNUM = ""

   while NUM > 0

      ; get the remainder of NUM/BASE

      TEMP = mod( NUM, BASE )

      ; get the character from the C array which represents
      ; the remainder digit code

      TEMP = C[ TEMP + 1 ]

      ; keep placing that code on the left side of NEWNUM
      ; this way we'll work from the least significant digit to
      ; the most significant digit

      NEWNUM = TEMP + NEWNUM

      ; get the next number by dividing the number by the base
      ; keep processing while NUM is greater than 0

      NUM = int( NUM / BASE )

   endwhile

   return NEWNUM

endproc


;Following are some generic table read/write procedures which handle saving and restoring dialog box ;variables.
;----------------------------------------------------------------------
; This proc will read a field(s) into a dynarray from a table
; Parameters:        OPT[] - an options dynarray (see below)
;                    TARGET[] - the dynarray target
;                    FVA[] - a dynarray with field values
;                    KFE[] - a dynarray listing key fields
; Globals Assumed:   NONE
; Procs Called:      SHOW_LOCK_ERROR()
;                    READ_FROM_TABLE_ALL()
;                    READ_FROM_TABLE_SOME()
;                    READ_FROM_TABLE_SOME_IDX()
; Returns:           True or False
;----------------------------------------------------------------------
;  "OPTIONS" are:    TABLE     FIELD     FIELDPROC      SELECTPROC
;                    SCOPE     INDEX     ONWORKSPACE    INDEXFIELD
;
;                    TABLE is the table name to read
;                    ONWORKSPACE = true will assume the table is
;                                  already on the workspace
;                    SCOPE = "All" or "Some" If "All" then the
;                            entire table will be read
;                    FIELD is the field name to copy contents to TARGET
;                    INDEX is the secondary index name to locate on
;                    INDEXFIELD is the 1st field in the secondary index
;                    FIELDPROC is a proc name which will place the
;                              contents in TARGET, overriding the default
;                    SELECTPROC is a proc name which will select records
;                              so that certain records are not put in TARGET
;----------------------------------------------------------------------
proc READ_FROM_TABLE( OPT, TARGET, FVA, KFE )
   private ORIG_TABLE

   ; define records to be read

   if not isassigned(OPT["SCOPE"]) then
      OPT["SCOPE"] = "All"
   endif

   ; save current position if necessary

   if nimages() > 0 then
      ORIG_TABLE = table()
   endif

   ; is the table on the workspace

   if not isassigned( OPT["ONWORKSPACE"] ) then
      OPT["ONWORKSPACE"] = false
   endif

   ; put the table on the workspace

   if OPT["ONWORKSPACE"] = true then
      moveto OPT["TABLE"]
   else
      lock OPT["TABLE"] pfl
      if retval = false then
         SHOW_LOCK_ERROR()
         return false
      endif
      view OPT["TABLE"]
   endif

   ; call the appropriate proc
   ;    read the whole table
   ;    read some of the table in secondary index order
   ;    read some of the table in primary index order

   if OPT["SCOPE"] = "All" then
      READ_FROM_TABLE_ALL()
   else
      if isassigned(OPT["INDEX"]) then
         READ_FROM_TABLE_SOME_IDX()
      else
         READ_FROM_TABLE_SOME()
      endif
   endif

   ; clean up

   if OPT["ONWORKSPACE"] = false then
      clearimage
      unlock OPT["TABLE"] pfl
   endif

   ; restore position

   if nimages() > 0 then
      moveto ORIG_TABLE
   endif

   return true

endproc


;----------------------------------------------------------------------
; This proc reads all records from a table
; Parameters:        NONE
; Globals Assumed:   all in READ_FROM_TABLE()
; Procs Called:      NONE
; Returns:           true
;----------------------------------------------------------------------
proc READ_FROM_TABLE_ALL()
   private Y

   moveto field OPT["FIELD"]
   Y = 1

   scan

      ; test the curernt value to determine if you want it
      ; using a custom procedure

      if isassigned(OPT["SELECTPROC"]) then
         execproc OPT["SELECTPROC"]
         if retval = false then    ; exclude the record
            loop
         endif
      endif

      if isassigned(OPT["FIELDPROC"]) then
         execproc OPT["FIELDPROC"]
      else
         TARGET[Y] = []
         Y = Y + 1
      endif

   endscan

   return true

endproc


;----------------------------------------------------------------------
; This proc reads portions of a table
; Parameters:       NONE
; Globals Assumed:  all in READ_FROM_TABLE()
; Procs Called:     NONE
; Returns:          true or false
;----------------------------------------------------------------------
proc READ_FROM_TABLE_SOME()
   private X, Y, FOUND

   Y = 1
   X = dynarraysize( KFE )

   ; go to the top of the table, first primary keyfield

   home
   ctrlhome right

   ; "generically" allow for up to 5 key fields in a table

   switch
      case X = 1 :
         locate indexorder FVA[ KFE[1] ]
      case X = 2 :
         locate indexorder FVA[ KFE[1] ],
                           FVA[ KFE[2] ]
      case X = 3 :
         locate indexorder FVA[ KFE[1] ],
                           FVA[ KFE[2] ],
                           FVA[ KFE[3] ]
      case X = 4 :
         locate indexorder FVA[ KFE[1] ],
                           FVA[ KFE[2] ],
                           FVA[ KFE[3] ],
                           FVA[ KFE[4] ]
      case X = 5 :
         locate indexorder FVA[ KFE[1] ],
                           FVA[ KFE[2] ],
                           FVA[ KFE[3] ],
                           FVA[ KFE[4] ],
                           FVA[ KFE[5] ]
      otherwise :
         return false
   endswitch

   while retval = true

      if isassigned(OPT["SELECTPROC"]) then
         execproc OPT["SELECTPROC"]
         if retval = false then    ; exclude the record
            loop
         endif
      endif

      if isassigned(OPT["FIELDPROC"]) then
         execproc OPT["FIELDPROC"]
      else
         moveto field OPT["FIELD"]
         TARGET[Y] = []
         Y = Y + 1
         ctrlhome right
      endif

      ; find the next matching record

      switch
         case X = 1 :
            locate indexorder next FVA[ KFE[1] ]
         case X = 2 :
            locate indexorder next FVA[ KFE[1] ],
                                   FVA[ KFE[2] ]
         case X = 3 :
            locate indexorder next FVA[ KFE[1] ],
                                   FVA[ KFE[2] ],
                                   FVA[ KFE[3] ]
         case X = 4 :
            locate indexorder next FVA[ KFE[1] ],
                                   FVA[ KFE[2] ],
                                   FVA[ KFE[3] ],
                                   FVA[ KFE[4] ]
         case X = 5 :
            locate indexorder next FVA[ KFE[1] ],
                                   FVA[ KFE[2] ],
                                   FVA[ KFE[3] ],
                                   FVA[ KFE[4] ],
                                   FVA[ KFE[5] ]
      endswitch
   endwhile
   return true

endproc


;----------------------------------------------------------------------
; This proc reads some records from a table based on an index order
; Parameters:        NONE
; Globals Assumed:   all in READ_FROM_TABLE
; Procs Called:      NONE
; Returns:           true or false
;----------------------------------------------------------------------
proc READ_FROM_TABLE_SOME_IDX()
   private X, Y, FOUND

   Y = 1
   X = dynarraysize( KFE )

   ; go to the first keyfield

   moveto field OPTIONS["INDEXFIELD"]

   ; "generically" allow for up to 5 fields in the secondary index

   switch
      case X = 1 :
         locate indexorder BY OPT["INDEX"] FVA[ KFE[1] ]
      case X = 2 :
         locate indexorder BY OPT["INDEX"] FVA[ KFE[1] ],
                                           FVA[ KFE[2] ]
      case X = 3 :
         locate indexorder BY OPT["INDEX"] FVA[ KFE[1] ],
                                           FVA[ KFE[2] ],
                                           FVA[ KFE[3] ]
      case X = 4 :
         locate indexorder BY OPT["INDEX"] FVA[ KFE[1] ],
                                           FVA[ KFE[2] ],
                                           FVA[ KFE[3] ],
                                           FVA[ KFE[4] ]
      case X = 5 :
         locate indexorder BY OPT["INDEX"] FVA[ KFE[1] ],
                                           FVA[ KFE[2] ],
                                           FVA[ KFE[3] ],
                                           FVA[ KFE[4] ],
                                           FVA[ KFE[5] ]
      otherwise :
         return false
   endswitch

   while retval = true

      if isassigned(OPT["SELECTPROC"]) then
         execproc OPT["SELECTPROC"]
         if retval = false then    ; exclude the record
            loop
         endif
      endif

      if isassigned(OPT["FIELDPROC"]) then
         execproc OPT["FIELDPROC"]
      else
         moveto field OPT["FIELD"]
         TARGET[Y] = []
         Y = Y + 1
         moveto field OPTIONS["INDEXFIELD"]
      endif

      switch
         case X = 1 :
            locate indexorder next BY OPT["INDEX"] FVA[ KFE[1] ]
         case X = 2 :
            locate indexorder next BY OPT["INDEX"] FVA[ KFE[1] ],
                                                   FVA[ KFE[2] ]
         case X = 3 :
            locate indexorder next BY OPT["INDEX"] FVA[ KFE[1] ],
                                                   FVA[ KFE[2] ],
                                                   FVA[ KFE[3] ]
         case X = 4 :
            locate indexorder next BY OPT["INDEX"] FVA[ KFE[1] ],
                                                   FVA[ KFE[2] ],
                                                   FVA[ KFE[3] ],
                                                   FVA[ KFE[4] ]
         case X = 5 :
            locate indexorder next BY OPT["INDEX"] FVA[ KFE[1] ],
                                                   FVA[ KFE[2] ],
                                                   FVA[ KFE[3] ],
                                                   FVA[ KFE[4] ],
                                                   FVA[ KFE[5] ]
      endswitch
   endwhile
   return true

endproc


;This procedure converts a binary string to a number. Is is used to store a series of check box values in ;one field, saving valuable database space at a very small cost.
;----------------------------------------------------------------------
; This proc converts a binary string to a number
; Parameters:        BINSTR - a binary string
; Globals Assumed:   None
; Procs Called:      None
; Returns:           NUM - a number
;----------------------------------------------------------------------
proc BIN_TO_INT( BINSTR )
   private X, Y, Z, NUM
   NUM = 0

   Z = len(BINSTR)

   ; start from the least significant digit to the most
   ; in other words, process the string from right to left
   ; NUM = sum of Y * 2^X where Y is the value of the
   ;       current digit (1 or 0) and X is the place the digit
   ;       occupies, from 0 to Z-1
   ;       e.g., "1101" = 1*2^0 + 0*2^1 + 1*2^2 + 1*2^3
   ;                    = 1     + 0      + 4    + 8
   ;                    = 13

   for X from Z to 1 step -1
      Y = numval( substr( BINSTR, X, 1 ) )
      NUM = NUM + ( Y * pow( 2, (Z-X) ) )
   endfor

   return NUM

endproc


;This is a sample dialog proc which handles a simulation of a pickrecord object which lets a dialog box
; mimic a multi-record form in read-only mode.
;----------------------------------------------------------------------
; This is a Dialog Procedure to manage events
;----------------------------------------------------------------------
proc TEST_PROC(TRIGTYPE, TAGVALUE, EVENTVALUE, ELEMVALUE)
   private RV
   RV = true

   switch
      case TRIGTYPE = "OPEN" :
         DOPTIONS["COMMAND"] = "Redisplay"
         DBOX_TABLE( DOPTIONS, DWINDOW, DBAG )
         RV = true

      case TRIGTYPE = "CANCEL" :
         return true

   endswitch

   DBAG["ROWCOUNT"] = ALLTRIM(format("ec,w30",recno())) + 
                      " of "+ALLTRIM(format("ec,w30",DBAG["MAX"]))
   repaintdialog

   return RV

endproc


;----------------------------------------------------------------------
; This proc handles key events
;----------------------------------------------------------------------
proc TEST_PROC_EVENT()
   private RV
   RV = true

   switch

      case TAGVALUE = "DWindow" :    ; pickarray "Window"
         switch
            case EVENTVALUE["KEYCODE"] = asc("Tab") :
               ; prevent the user from tabbing to the slider
               selectcontrol "Description"
               return false

            case EVENTVALUE["KEYCODE"] = asc("Home") :
               DOPTIONS["COMMAND"] = "Home"

            case EVENTVALUE["KEYCODE"] = asc("End") :
               DOPTIONS["COMMAND"] = "End"

            case EVENTVALUE["KEYCODE"] = asc("Down") :
               DOPTIONS["COMMAND"] = "Down"

            case EVENTVALUE["KEYCODE"] = asc("Up") :
               DOPTIONS["COMMAND"] = "Up"

            case EVENTVALUE["KEYCODE"] = asc("PgDn") :
               DOPTIONS["COMMAND"] = "PgDn"

            case EVENTVALUE["KEYCODE"] = asc("PgUp") :
               DOPTIONS["COMMAND"] = "PgUp"

            case EVENTVALUE["KEYCODE"] = asc("CtrlPgDn") :
               DOPTIONS["COMMAND"] = "CtrlPgDn"

            case EVENTVALUE["KEYCODE"] = asc("CtrlPgUp") :
               DOPTIONS["COMMAND"] = "CtrlPgUp"

            case EVENTVALUE["KEYCODE"] = asc("CtrlLeft") :
               DOPTIONS["COMMAND"] = "PageStepUp"

            case EVENTVALUE["KEYCODE"] = asc("CtrlRight") :
               DOPTIONS["COMMAND"] = "PageStepDn"

            case EVENTVALUE["KEYCODE"] = asc("Left") :
               DOPTIONS["COMMAND"] = "PgUp"

            case EVENTVALUE["KEYCODE"] = asc("Right") :
               DOPTIONS["COMMAND"] = "PgDn"

         endswitch
         RV = DBOX_TABLE( DOPTIONS, DWINDOW, DBAG )

      case TAGVALUE = "Description" :

         switch
            case EVENTVALUE["KEYCODE"] = asc("ReverseTab") :
               ; prevent the user from tabbing to the slider
               selectcontrol "DWindow"
               return false
         endswitch

   endswitch

   return RV

endproc


;----------------------------------------------------------------------
; This proc handles update events
;----------------------------------------------------------------------
proc TEST_PROC_UPDATE()
   private RV

   switch
      case TAGVALUE = "WindowSlider" :
         DOPTIONS["COMMAND"] = "SliderGoto"
         RV = DBOX_TABLE( DOPTIONS, DWINDOW, DBAG )
         selectcontrol DOPTIONS["TAG"]      ; pickarray "Window"
         return RV

      case TAGVALUE = "DWindow" :
         if controlvalue("DWindow") = DBAG["DWindow"] then
            ; the user selected the currently highlighted item
            return true
         endif

         ; otherwise the user selected one of the other items in the window

         DOPTIONS["COMMAND"] = "Select"
         RV = DBOX_TABLE( DOPTIONS, DWINDOW, DBAG )
         return RV

   endswitch
   return true

endproc


;----------------------------------------------------------------------
; This proc handles synchronizing a table "window" and a table
; simulating a pick record from within a dialog box
; Parameters:         OPTIONS[] - an options dynarray
;                       "COMMAND" - required
;                       "PROC"    - required, fills the window
;                     DARRAY[] - the "window" array
;                     DBAG[] - window and slider control value variables
; Globals Assumed:    NONE
; Procs Called:       DBOX_TABLE_REDISPLAY()
;                     SHOWMSG()
; Returns:            true or false (for dbox proc handling)
;----------------------------------------------------------------------
proc DBOX_TABLE( OPTIONS, DARRAY, DBAG )
   private X, Z, SK, TEMP

   Z = arraysize(DARRAY)

   ; reset the slider control variables

   DBAG["MAX"] = nrecords(table())
   if DBAG["MAX"] = 0 then
      return false
   endif
   DBAG["MIN"] = 1
   DBAG["ARROWSTEP"] = 1
   DBAG["PAGESTEP"] = int( DBAG["MAX"] / 20 )
   if DBAG["PAGESTEP"] < Z then
      DBAG["PAGESTEP"] = Z
   endif

   ; set the slider value to current record

   DBAG["WINDOWSLIDER"] = recno()

   ; store the current highlighted row and slider value

   OPTIONS["ACTIVEROW"] = controlvalue(OPTIONS["TAG"])
   OPTIONS["SLIDERVAL"] = controlvalue(OPTIONS["SLIDERTAG"])

   switch
      case OPTIONS["COMMAND"] = "Redisplay" :
         DBOX_TABLE_REDISPLAY()
         return false

      case OPTIONS["COMMAND"] = "Home" :
         skip 1 - OPTIONS["ACTIVEROW"]
         DBAG["WINDOWSLIDER"] = recno()
         resynccontrol OPTIONS["SLIDERTAG"]
         DBAG["Dwindow"] = 1
         resynccontrol OPTIONS["Tag"]
         return false

      case OPTIONS["COMMAND"] = "End" :

         ; don't let the highlight bar go beyond the last record
         ; in those cases where the end of table is in the middle
         ; of the table window

         if DBAG["MAX"] - recno() < Z - OPTIONS["ACTIVEROW"] then
            TEMP = recno()
            end
            DBAG["DWindow"] = OPTIONS["ACTIVEROW"] + DBAG["MAX"] - TEMP
            refreshcontrol OPTIONS["TAG"]
            DBAG["WINDOWSLIDER"] = recno()
            resynccontrol OPTIONS["SLIDERTAG"]
            return false
         else
            skip Z - OPTIONS["ACTIVEROW"]
            DBAG["WINDOWSLIDER"] = recno()
            resynccontrol OPTIONS["SLIDERTAG"]
            DBAG["DWindow"] = Z
            resynccontrol OPTIONS["Tag"]
            return false
         endif

      case OPTIONS["COMMAND"] = "Down" :
         skip
         if eot() = true then
            SHOWMSG("End of table",false)
            skip 0
            return false
         endif
         DBAG["WINDOWSLIDER"] = recno()
         resynccontrol OPTIONS["SLIDERTAG"]
         if OPTIONS["ACTIVEROW"] = Z then
            DBOX_TABLE_REDISPLAY()
         else
            DBAG["DWindow"] = DBAG["Dwindow"] + 1
            resynccontrol OPTIONS["Tag"]
         endif
         return false

      case OPTIONS["COMMAND"] = "Up" :
         skip -1
         if bot() = true then
            SHOWMSG("Beginning of table",false)
            skip 0
            return false
         endif
         DBAG["WINDOWSLIDER"] = recno()
         resynccontrol OPTIONS["SLIDERTAG"]
         if OPTIONS["ACTIVEROW"] = 1 then
            DBOX_TABLE_REDISPLAY()
         else
            DBAG["DWindow"] = DBAG["Dwindow"] - 1
            resynccontrol OPTIONS["Tag"]
         endif
         return false

      case OPTIONS["COMMAND"] = "PgDn" :
         skip ( arraysize(DARRAY) )
         if eot() = true then
            SHOWMSG("End of table",false)
            skip 0
         endif
         DBAG["WINDOWSLIDER"] = recno()
         resynccontrol OPTIONS["SLIDERTAG"]
         DBOX_TABLE_REDISPLAY()
         return false

      case OPTIONS["COMMAND"] = "PgUp" :
         if recno() <= Z then
            SHOWMSG("Beginning of table",false)
            home
            DBAG["DWINDOW"] = 1
            resynccontrol OPTIONS["Tag"]
         else
            skip Z * -1
         endif
         DBAG["WINDOWSLIDER"] = recno()
         resynccontrol OPTIONS["SLIDERTAG"]
         DBOX_TABLE_REDISPLAY()
         return false

      case OPTIONS["COMMAND"] = "CtrlPgDn"  :

         ; don't let the highlight bar go beyond the last record
         ; in those cases where the end of table is in the middle
         ; of the table window

         if DBAG["MAX"] - recno() < Z - OPTIONS["ACTIVEROW"] then
            TEMP = DBAG["MAX"] - recno()
            skip TEMP
            DBAG["DWindow"] = TEMP + OPTIONS["ACTIVEROW"]
         else
            end
            DBAG["DWindow"] = Z
         endif
         DBAG["WINDOWSLIDER"] = recno()
         resynccontrol OPTIONS["SLIDERTAG"]
         resynccontrol OPTIONS["Tag"]
         DBOX_TABLE_REDISPLAY()
         return false

      case OPTIONS["COMMAND"] = "CtrlPgUp" :
         home
         DBAG["WINDOWSLIDER"] = recno()
         resynccontrol OPTIONS["SLIDERTAG"]
         DBAG["DWindow"] = 1
         resynccontrol OPTIONS["Tag"]
         DBOX_TABLE_REDISPLAY()
         return false

      case OPTIONS["COMMAND"] = "PageStepUp"  or
           OPTIONS["COMMAND"] = "PageStepDn" :

         if OPTIONS["COMMAND"] = "PageStepUp" then
            OPTIONS["SLIDERVAL"] = OPTIONS["SLIDERVAL"] - DBAG["PAGESTEP"]
            if OPTIONS["SLIDERVAL"] < 1 then
               OPTIONS["SLIDERVAL"] = 1
            endif
         else
            OPTIONS["SLIDERVAL"] = OPTIONS["SLIDERVAL"] + DBAG["PAGESTEP"]
            if OPTIONS["SLIDERVAL"] > DBAG["MAX"] then
               OPTIONS["SLIDERVAL"] = DBAG["MAX"]
            endif
         endif

         moveto record OPTIONS["SLIDERVAL"]
         DBAG["WINDOWSLIDER"] = recno()
         resynccontrol OPTIONS["SLIDERTAG"]
         DBAG["DWindow"] = 1
         resynccontrol OPTIONS["Tag"]
         DBOX_TABLE_REDISPLAY()
         return false

      case OPTIONS["COMMAND"] = "SliderGoto" :
         if not isblank(OPTIONS["SLIDERVAL"]) then
            moveto record OPTIONS["SLIDERVAL"]
            DBAG["WINDOWSLIDER"] = recno()
            resynccontrol OPTIONS["SLIDERTAG"]
            DBAG["DWindow"] = 1
            resynccontrol OPTIONS["Tag"]
            DBOX_TABLE_REDISPLAY()
         endif
         return false

      case OPTIONS["COMMAND"] = "Select" :

         OPTIONS["AMOUNT"] = controlvalue("DWindow") - DBAG["DWindow"]

         ; at the last record with an attempt to select past the end

         if recno() = DBAG["MAX"] and OPTIONS["AMOUNT"] > 0 then
            resynccontrol OPTIONS["Tag"]
            return false
         endif

         ; before the last record trying to go past the last record

         if recno()+OPTIONS["AMOUNT"] > DBAG["MAX"] then
            OPTIONS["AMOUNT"] = DBAG["MAX"] - recno()
         endif

         skip OPTIONS["AMOUNT"]
         DBAG["WINDOWSLIDER"] = recno()
         DBAG["DWindow"] = DBAG["Dwindow"] + OPTIONS["AMOUNT"]
         resynccontrol OPTIONS["Tag"]
         resynccontrol OPTIONS["SLIDERTAG"]
         return false

   endswitch

   return true

endproc


;----------------------------------------------------------------------
; This proc will redisplay the table window array
; Parameters:        NONE
; Globals Assumed:   All in DBOX_TABLE()
; Procs Called:      None
; Returns:           True
;----------------------------------------------------------------------
proc DBOX_TABLE_REDISPLAY()
   private X, Z
   Z = arraysize(DARRAY)
   OPTIONS["ACTIVEROW"] = controlvalue(OPTIONS["TAG"])

   for X from 1 to Z
      DARRAY[X] = ""
   endfor

   ; skip to the top of the array
   if OPTIONS["ACTIVEROW"] > 1 then
      skip (OPTIONS["ACTIVEROW"]-1) * -1
   endif
   for X from 1 to Z
      execproc OPTIONS["PROC"]
      skip
      if eot() then
         skip 0
         quitloop
      endif
   endfor

   skip ( (X) - OPTIONS["ACTIVEROW"] ) * -1

   refreshcontrol OPTIONS["TAG"]
   refreshcontrol OPTIONS["SLIDERTAG"]
   return true

endproc


;The next procedures save all of a dialog boxes variables, stored in a dynarray, to a memo field so that ;the variables can ge generically saved and restored.
;----------------------------------------------------------------------
; REP_SAVE_QUERY - This proc will save a dynarray into a memo field
; Parameters:      QOPTS[], PARAMS[]
;                  QOPTS[] must have "USERID"  "REPID"  "QUERYNAME"
;                  PARAMS[] - the dynarray to save
; Globals Assumed: None
; Procs Called:    WRITE_TO_TABLE()
; Returns:         True or false
;----------------------------------------------------------------------
proc REP_SAVE_QUERY( QOPTS, PARAMS )
   private  OPTIONS, FVALUES, X, TEMP

   ; initialize variables

   dynarray OPTIONS[]
   dynarray FVALUES[]
   TEMP = ""
   X = ""

   ; step through the dynarray to save

   foreach X in PARAMS
      if isassigned(PARAMS[X]) then
         TEMP = TEMP + X + "" + type(PARAMS[X]) + "" + strval(PARAMS[X]) + ""
      endif
   endforeach

   ; define the dynarray to write to the table

   OPTIONS["TABLE"]        = "Reports\\repspecs"
   OPTIONS["ONWORKSPACE"]  = false

   FVALUES["User_id"]      = QOPTS["USERID"]
   FVALUES["Rep_id"]       = QOPTS["REPID"]
   FVALUES["Qry_nam"]      = QOPTS["QUERYNAME"]
   FVALUES["Parameters"]   = TEMP

   ; overwrite the existing record or save the new record

   return WRITE_TO_TABLE(OPTIONS, FVALUES)

endproc


;----------------------------------------------------------------------
; REP_READ_QUERY - Read saved query parameters into a dynarray
; Parameters:      QOPTS[], PARAMS[]
;                  QOPTS[] must have "USERID"  "REPID"  "QUERYNAME"
;                  PARAMS[] - the dynarray to save
; Globals Assumed: None
; Procs Called:    READ_FROM_TABLE()
; Returns:         True or False
;----------------------------------------------------------------------
proc REP_READ_QUERY( QOPTS, PARAMS )
   private  OPTIONS, FVALUES, KFIELDS, X, TEMP, T, I, J, K, L

   ; initialize variables

   dynarray OPTIONS[]
   dynarray FVALUES[]
   dynarray KFIELDS[]
   dynarray T[]
   TEMP = ""
   X = ""

   ; define dynarray to call standard proc

   OPTIONS["TABLE"]        = "Reports\\repspecs"
   OPTIONS["ONWORKSPACE"]  = false
   OPTIONS["FIELD"]        = "Parameters"
   OPTIONS["SCOPE"]        = "Some"

   FVALUES["User_id"]      = QOPTS["USERID"]
   FVALUES["Rep_id"]       = QOPTS["REPID"]
   FVALUES["Qry_nam"]      = QOPTS["QUERYNAME"]

   KFIELDS["1"]            = "User_id"
   KFIELDS["2"]            = "Rep_id"
   KFIELDS["3"]            = "Qry_nam"

   ; get the saved PARAMS list from the report spec record

   if not READ_FROM_TABLE( OPTIONS, T, FVALUES, KFIELDS ) then
      return false
   endif

   TEMP = T["1"]

   while not isblank(TEMP)
      RV = match( TEMP, ".."+""+".."+""+".."+""+"..", I,J,K,L )
      if RV = True then
         switch
            case J = "N" or J = "$" or J = "S" :
               K = numval(K)
            case J = "L" :
               if upper(K) = upper("True") then
                  K = true
               else
                  K = false
               endif
            case J = "D" :
               K = dateval(K)
         endswitch

         PARAMS[I] = K

      endif

      X = search("",TEMP)
      if X = 0 then
         TEMP = ""
      else
         TEMP = substr( TEMP, X+1, len(TEMP) )
      endif

   endwhile
   return true

endproc










