#include <dos.h>                        // For the REGPACK and intr


/* Define the serial port addresses */

#define COM1_ADDR       0x03F8
#define COM2_ADDR       0x02F8
#define COM3_ADDR       0x03E8
#define COM4_ADDR       0x02E8

/* Define the communications port values for DX in service 14h function 0 */

#define COM1_MASK	0x0000
#define COM2_MASK	0x0001
#define COM3_MASK	0x0002
#define COM4_MASK	0x0003

/* Define the bitmasks for comm. settings for AL in function 0 */

#define B9600           0xE0            // Baud rates.  9600 ...
#define B4800           0xC0            // 4800, etc.
#define B2400           0xA0
#define B1200           0x80
#define B600            0x60
#define B300            0x40
#define B150            0x20
#define B110            0x10

#define NONE            0x00            // Parity bits
#define ODD             0x08
#define EVEN            0x18

#define STOP_1          0x00            // Stop bits
#define STOP_2          0x04

#define CHAR_7          0x02            // Data bits (character size)
#define CHAR_8          0x03

/* Miscellaneous defines */

#define TIME_OUT        0x8000          // Time-out bit in line status
#define DATA_READY      0x0100          // Data ready bit in line status



typedef unsigned char           uchar;
typedef unsigned short int      ushort;
typedef unsigned long int       ulong;


/*

Function:       ushort serial_status (ushort port_id)
Author:         Shawn A. Clifford       sysop@robot.nuceng.ufl.edu
Date:           09-DEC-1993
Purpose:        This function will return the status bytes for port 'port_id'.
		The status bytes are bit masked as follows:

	High Byte (line status)                     Low Byte (modem status)
------------------------------------    ------------------------------------
1 . . . . . . .  Time-out error         1 . . . . . . .  Rcvd line signal
. 1 . . . . . .  Xfr shift reg empty    . 1 . . . . . .  Ring indicator
. . 1 . . . . .  Xfr hold reg empty     . . 1 . . . . .  Data-set-ready (DSR)
. . . 1 . . . .  Break-detect error     . . . 1 . . . .  Clear-to-send (CTS)
. . . . 1 . . .  Framing error          . . . . 1 . . .  Delta rcv line signal
. . . . . 1 . .  Parity error           . . . . . 1 . .  Trailing-edge ring
. . . . . . 1 .  Overrun error          . . . . . . 1 .  Delta data-set ready
. . . . . . . 1  Data ready             . . . . . . . 1  Delta clear-to-send

Usage:  ushort set_port(ushort port_id);

	port_id:        0-3 for COM ports 1-4
				values: COM_PORT_x  where x is 1 through 4

Example:        status = serial_status (COM_PORT_2);

*/

ushort serial_status (ushort port_id)
{
static REGPACK reg;

   reg.r_dx = port_id;                                  // DX = com port
   reg.r_ax = 3<<8;                                     // AH = status servc
   intr(0x14, &reg);                                    // Serial services
   return reg.r_ax;                                     // Return the status

}  /* serial_status() */



/*

Function:       ushort receive_char (ushort port_id, char *c)
Author:         Shawn A. Clifford       sysop@robot.nuceng.ufl.edu
Date:           09-DEC-1993
Purpose:        This function will receive a character from serial port
		'port_id' and return it in AL.  For convenience, this
		character is returned in 'c'.  The service waits for a
		character or any signal that indicates the completion of the
		service, such as a time-out.  AH = 0 if the service is
		successful.  If not, bit 7 of AH reports an error and the
		other bits of AH report the type of error.  Since bit 7
		reports that an error has occurred, it is not available to
		indicate a time-out error.  Consequently, when this service,
		or service 2, reports an error, the simplest and most
		reliable way to check the nature of the error is to use the
		complete status report given by service 3, ie. the serial_status() function.

		The status bytes are bit masked as follows:



AH = High Byte (line status)            AL = Low Byte (modem status)
     Bit                                     Bit
7 6 5 4 3 2 1 0                         7 6 5 4 3 2 1 0
------------------------------------    ------------------------------------
1 . . . . . . .  Error                  1 . . . . . . .  Rcvd line signal
. 1 . . . . . .  Xfr shift reg empty    . 1 . . . . . .  Ring indicator
. . 1 . . . . .  Xfr hold reg empty     . . 1 . . . . .  Data-set-ready (DSR)
. . . 1 . . . .  Break-detect error     . . . 1 . . . .  Clear-to-send (CTS)
. . . . 1 . . .  Framing error          . . . . 1 . . .  Delta rcv line signal
. . . . . 1 . .  Parity error           . . . . . 1 . .  Trailing-edge ring
. . . . . . 1 .  Overrun error          . . . . . . 1 .  Delta data-set ready
. . . . . . . 1  Data ready             . . . . . . . 1  Delta clear-to-send

Usage:  ushort receive_char(ushort port_id, char *c);

	port_id:        0-3 for COM ports 1-4
				values: COM_PORT_x  where x is 1 through 4

	c:              character to be received

Example:        status = receive_char (COM_PORT_2, &c);

*/

ushort receive_char (ushort port_id, char *c)
{
static REGPACK reg;

   reg.r_dx = port_id;                                  // DX = com port
   reg.r_ax = 2<<8;                                     // AH = recv servc
   intr(0x14, &reg);                                    // Serial services
   *c = reg.r_ax & 0x00FF;                              // AL = char recvd
   return reg.r_ax;                                     // Return the status

}  /* receive_char() */




/*

Function:       ushort send_char (ushort port_id, char c)
Author:         Shawn A. Clifford       sysop@robot.nuceng.ufl.edu
Date:           09-DEC-1993
Purpose:        This function will transmit 'character' out serial port
		'port_id'.  AH = 0 if the service is successful.  If not,
		bit 7 of AH reports an error and the other bits of AH report
		the type of error.  Since bit 7 reports that an error has
		occurred, it is not available to indicate a time-out error.
		Consequently, when this service, or service 2, reports an
		error, the simplest and most reliable way to check the nature
		of the error is to use the complete status report given
		by service 3, ie. the serial_status() function.

		The status bytes are bit masked as follows:



AH = High Byte (line status)            AL = Low Byte (modem status)
     Bit                                     Bit
7 6 5 4 3 2 1 0                         7 6 5 4 3 2 1 0
------------------------------------    ------------------------------------
1 . . . . . . .  Error                  1 . . . . . . .  Rcvd line signal
. 1 . . . . . .  Xfr shift reg empty    . 1 . . . . . .  Ring indicator
. . 1 . . . . .  Xfr hold reg empty     . . 1 . . . . .  Data-set-ready (DSR)
. . . 1 . . . .  Break-detect error     . . . 1 . . . .  Clear-to-send (CTS)
. . . . 1 . . .  Framing error          . . . . 1 . . .  Delta rcv line signal
. . . . . 1 . .  Parity error           . . . . . 1 . .  Trailing-edge ring
. . . . . . 1 .  Overrun error          . . . . . . 1 .  Delta data-set ready
. . . . . . . 1  Data ready             . . . . . . . 1  Delta clear-to-send

Usage:  ushort send_char(uchar port_id, char c);

	port_id:        0-3 for COM ports 1-4
				values: COM_PORT_x  where x is 1 through 4

	c:              character to be received

Example:        status = send_char (COM_PORT_2, c);

*/

ushort send_char (uchar port_id, char c)
{
static REGPACK reg;

   reg.r_dx = port_id;                                  // DX = com port
   reg.r_ax = 1<<8;                                     // AH = send servc
   reg.r_ax |= c;                                       // AL = char to send
   intr(0x14, &reg);                                    // Serial services
   return reg.r_ax;                                     // Return the status

}  /* send_char() */



/*

Function:       ushort set_port (port_id, baud, parity, stops, char_size)
Author:         Shawn A. Clifford       sysop@robot.nuceng.ufl.edu
Date:           09-DEC-1993
Purpose:        This function will set port 'port_id' to the settings
		specified by 'baud', 'parity', 'stops', 'char_size'.
		The functions returns the communications status.  See
		serial_status() for details on the status bytes.

Usage:  ushort set_port(uchar port_id, uchar baud, uchar parity,
			uchar stops, uchar char_size);

	port_id:        0-3 for COM ports 1-4
				values: COM_PORT_x  where x is 1 through 4

	baud:           Baudrate
				values: B9600, B4800, B2400, B1200, B600,
					B300, B150, B110

	parity:         Parity
				values: NONE, ODD, EVEN

	stops:          # of Stop Bits
				values: STOP_x or STOP_x  where x is 1 or 2

	char_size:      Character size (word length, or # of data bits)
				values: CHAR_x or CHAR_x  where x is 7 or 8

Example:        set_port(COM_PORT_2, B9600, NONE, STOP_1, CHAR_8);

   This example is the same as the dos command:  MODE COM2:9600,n,8,1
   ie. COM2 is set to 9600 baud, no parity, 8 data bits, and 1 stop bit

*/

ushort set_port(uchar port_id, uchar baud, uchar parity,
		uchar stops, uchar char_size)
{
static REGPACK reg;

/* One method of doing this. */

   reg.r_dx = port_id;                                  // DX = com port
   reg.r_ax = 0;                                        // AH = init port srv
   reg.r_ax |= ( baud | parity | stops | char_size );   // AL = settings
   intr(0x14, &reg);                                    // Serial services
   return reg.r_ax;                                     // Comm status


/* Identical inline assembler code to set COM2:9600,n,8,1 */

//   asm {
//      mov dx, 1               ; hardcode for COM2
//      mov ah, 0               ; select init port service
//      mov al, 0xE3            ; 9600,n,8,1
//      int 14                  ; serial communications services
//      int 20                  ; graceful exit (maybe ret instead ?)
//    }

}  /* set_port() */
