/*
 
Copyright Peter H. Kosel, 1345 56th St., Sacramento, CA 95819, 1988.
Permission to use and retransmit granted by author. This is a free program.

This C program contains and demonstrates algorithms for accessing info
about combinations of k out of n items. In neural network projects, 
information  about the relation of combinations of k out of n bits may be
developed through some learning process and later used in some reasoning 
process. In such cases there must be some way of identifying the bits in 
each combination and of organizing the storage of information about the 
various combinations. The algorithms here assign a unique offset to each
combination of bits which is related to bit "ID" numbers. Bit "ID" numbers
may be assumed to reflect bit position in an input array which varys 
according to some rules of the real world. Bit numbers for a combination
are assumed to always be arrayed in ascending order, bit number sequence
being irrelevant (ie, combinations means combinations, not permutations).
The combination sequencing here is considered rational (by me) but there 
are other ways to do it; don't assume there is only one standard way.


The principle routines presented and demonstrated here are:
    NetSize ( called "Size" in combalg.pas )
    Offset
    BitNumbers
    NextOffset
    PreviousOffset (this is the only one I haven't found a use for yet)

Everything else is just window dressing to demonstrate the routines and
help you understand what they do.

I found that these routines were difficult to write ( Offset and 
BitNumbers require visualizing a k-dimensional non-rectangular array with a
lot of right-triangular or pyramidal faces) and so am passing them along
to save the next guy some effort.

Most of the code here is straight textbook C I think. There may be
one or two passages where the code is appropriate only to Lightspeed C
v2.13 running on an Apple Macintosh during the dark of the moon but
I tried to keep it "stock" as much as I could. On a Macintosh under 
Lightspeed C this file must be in a "project" with the "MacTraps" and
"stdio" libraries; on other machines you must forage for your own libraries
but stdio will probably get it.

*/


/*------------------------------------------------------------------------*/
#include "stdio.h"
/*------------------------------------------------------------------------*/


/*------------------------------------------------------------------------*/
/* These are predeclarations of the algorithm routines... */

unsigned long int 	NetSize();
void 				Offset();
void 				BitNumbers();
unsigned long int 	NextOffset();
unsigned long int 	PreviousOffset();

/* and these are predeclarations of the test routines. */
void				main(),
					TestNetSize(),
					TestOffset(),
					TestBitNumbers(),
					TestNextOffset(),
					TestPreviousOffset(),
					getreturn();

unsigned long int	getvalue();


/* routines in combalg.c... */


/*------------------------------------------------------------------------*/


/*------------------------------------------------------------------------*/
/*
This is a structure used to pass parameters to the algorithm routines.
*/

struct NetInfoBlock
	{
	unsigned long int	offset;			/* C */
	unsigned long int	inputwidth;		/* n */
	unsigned long int	logicdepth;		/* k */
	unsigned long int	*bitnumbers;	/* a[] */
	};

/*------------------------------------------------------------------------*/


/*----------------------------------------------------------------------*/
unsigned long int
NetSize(n,k)
unsigned long int n,k;
/* find number of truth tables given n, k */

{
unsigned long int S,i;

S = 1;
for (i = 1; i <= k; i = i+1 )
    {
    S = ( S * (n+1-i) ) / i ;
    /* Note that the division always comes out integer for some reason. 
      Try it if you don't believe! Not a remainder in a carload!
    */
    }
return S;
}


/*----------------------------------------------------------------------*/


/*----------------------------------------------------------------------*/
void
Offset(input)
struct NetInfoBlock *input;
/* find offset from bit numbers */
{
unsigned long int C,z,nn,kk,i;

unsigned long int n,k,*a;

n = input->inputwidth;
k = input->logicdepth;
a = input->bitnumbers;

z = 0;
C = 0;
kk = k;
nn = n;
for( i = 1 ; i <= k ; i = i+1 )
    {
    C = C + NetSize( nn,kk ) - NetSize( (nn + z + 1 - a[i]) , kk );
    kk = kk - 1;
    z = a[i];
    nn = nn - a[i];
    }
input->offset = C;
}


/*----------------------------------------------------------------------*/


/*----------------------------------------------------------------------*/
void
BitNumbers(input)
struct NetInfoBlock *input;
/* find bit numbers from offset */

{
unsigned long int
i /* points to one of k bit numbers */
,R /* not exactly a remainder but... */
,CC ,nn, /* transdimensional dummy values */
C,n,k,*a;

C = input->offset;
n = input->inputwidth;
k = input->logicdepth;
a = input->bitnumbers;

CC = C;
nn = n;
for( i = 1 ; i <= k ; i++ )
    {
    a[i] = n-k+i; /* start with max value */
    R = NetSize(nn,k-i+1) - CC; /*tip off the triangle, top off the pyramid*/
    while ( NetSize(n-a[i]+1,k-i+1) < R ) 
    	{ a[i] = a[i]-1; } /* find a good one */
    CC = NetSize(n-a[i]+1,k-i+1) - R; /* Warp factor 9, Mr Sulu! */
    nn = n-a[i];
    }
}
/*----------------------------------------------------------------------*/


/*----------------------------------------------------------------------*/
unsigned long int
NextOffset(input)
struct NetInfoBlock *input;
/* increment offset and adjust bit numbers */
{
unsigned long int n,k,*a,i;
unsigned long int UnFeasibilityFlag;

n = input->inputwidth;
k = input->logicdepth;
a = input->bitnumbers;

/* look for largest non-max index */
i = k;
while ((i!=1) && ((n-k+i)==a[i]))
	{ i = i - 1 ;}

/* look over what you found */
if (( n - k + i )==a[i])
	{ UnFeasibilityFlag = 1 ;} /* because a[1] is maxed out */
else	
	{ 
	UnFeasibilityFlag = 0;
	input->offset = input->offset + 1;
	a[i]= a[i] + 1; /* increment index you found */
	i= i+1; /* starting with the next bit number if any... */
	while(i<=k)
		{
		/* ... set any higher bit numbers to min value */
		a[i] = a[i-1]+1;
		i = i+1;
		}
	}
return UnFeasibilityFlag;
}
/*----------------------------------------------------------------------*/


/*------------------------------------------------------------------------*/
unsigned long int
PreviousOffset(input)
struct NetInfoBlock *input;
/* decrement offset and adjust bit numbers */

{
unsigned long int n,k,*a,i;
unsigned long int UnFeasibilityFlag;

n = input->inputwidth;
k = input->logicdepth;
a = input->bitnumbers;

/* look for highest ranking non-min index */
i = k;
while ((i!=1) && ((a[i-1]+1)==a[i]))
	{ i = i - 1; }
	
/* look over what you found */
if (a[i]==1)
	{ UnFeasibilityFlag = 1 ;} /* because a[1] is at min=1 */
else
	{ 
	UnFeasibilityFlag = 0 ;
	input->offset = input->offset - 1;
	a[i] = a[i]-1; /* decrement bit number you found */
	i=i+1; /* starting with the next bit number if any... */
	while (i<=k)
		{
		/* ...set any higher bit numbers to max value */
		a[i]=(n-k+i);
		i=i+1;
		}
	}
return UnFeasibilityFlag;
}
/*----------------------------------------------------------------------*/


/************************************************************************/
/************************************************************************/
/*
From here on, routines are only intended to test demo the routines
above.
*/
/************************************************************************/
/************************************************************************/


/*------------------------------------------------------------------------*/

void
main()
{
unsigned long int choice;
do	{
	printf("\n"); /* blank line */
	printf("Select by number...\n");
	printf("1. Test NetSize routine\n");
	printf("2. Test Offset routine\n");
	printf("3. Test BitNumbers routine\n");
	printf("4. Test NextOffset routine\n");
	printf("5. Test PreviousOffset routine\n");
	printf("6. Quit\n");
	printf("\n");
	printf("Hit a number...");choice = getchar()-'0';
	printf("\n");
	printf("\n");
	switch(choice)
		{
		case 1:	TestNetSize();
				break;
		case 2:	TestOffset();
				break;
		case 3:	TestBitNumbers();
				break;
		case 4:	TestNextOffset();
				break;
		case 5:	TestPreviousOffset();
				break;				
		case 6:	break;				
		}
	}
while (! (choice==6) );
}


/*-------------------------------------------------------------------------*/


/*-------------------------------------------------------------------------*/


void
TestNetSize()
{
unsigned long int n,k,Size;

printf("\n");
printf("Test of NetSize Routine:\n");
printf("\n");
printf("The NetSize routine calculates the number of combinations\n");
printf("of k items which can be drawn from a population of n items\n");
printf("where n is greater than or equal to k. This test asks you for a\n");
printf("value of n and a value of k and prints out the result.\n");
printf("\n");

printf("\n");
printf("enter n..."); n=getvalue();

do
	{
	printf("\n");
	printf("enter k from 0 to %lu ...",n); k=getvalue();
	}
while ((k<0) || (k>n));

Size=NetSize(n,k);
printf("\n");
printf("There are %ld possible combinations of %lu ",Size,k);
printf("out of %lu items.\n",n);

printf("\n");
getreturn();
}



/*------------------------------------------------------------------------*/


/*------------------------------------------------------------------------*/


void
TestNextOffset()
{
unsigned long int i,C,n,k,Size,a[1024],NoMoreOffsets;
struct NetInfoBlock NetInfo;

NetInfo.offset = 0;
NetInfo.bitnumbers = a;

printf("\n");
printf("Test of NextOffset Routine:\n");
printf("\n");
printf("The various combinations of k out of n items are\n");
printf("distinguished by the ID numbers of the items in\n");
printf("the combinations. In neural net projects the items\n");
printf("will probably be bits in an input array. The \n");
printf("NextOffset routine calculates the next set of bit\n");
printf("numbers and next offset given the current bit numbers \n");
printf("and offset. The first set of bit numbers is always\n");
printf("1, 2, 3 ... k  \n");
printf("and the last set is always\n");
printf("'n-k+1 ... n-2, n-1, n. \n");
printf("This test asks you for values for n and k, \n");
printf("starts with the first set of bit numbers, and prints\n");
printf("all sets of bit numbers. Keep n and k small unless\n");
printf("you really like to sit and watch printout scroll by.\n");
printf("\n");

printf("\n");
printf("enter n..."); n=getvalue();

do
	{
	printf("\n");
	printf("enter k from 1 to %lu ...",n); k=getvalue();
	}
while ((k<1) || (k>n));


NetInfo.inputwidth = n;
NetInfo.logicdepth = k;

Size = NetSize(n,k);
printf("\n");
if (Size>2000)
	{
	printf("You do not really want to watch %lu ",Size);
	printf("lines of text scroll by!");
	}
else
	{
	for (i=1;i<=NetInfo.logicdepth;i++) 
		{ a[i] = i; }
	NoMoreOffsets = 0;
	do
		{
		/* print offset and bit numbers */
		printf("offset: %lu", NetInfo.offset);
		printf("  bitnumbers: ");
		for (i= 1; i<= NetInfo.logicdepth; i++) 
			{ printf("%lu ", a[i] ) ;}
		printf("\n");
		NoMoreOffsets = NextOffset(&NetInfo);
		}
	while  (! NoMoreOffsets) ;
	}

printf("\n");
getreturn();
}


/*----------------------------------------------------------------------*/


/*------------------------------------------------------------------------*/

void
TestPreviousOffset()
{
unsigned long int i,C,n,k,a[1024],Size,NoMoreOffsets;

struct NetInfoBlock NetInfo;
NetInfo.bitnumbers = a;

printf("\n");
printf("Test of PreviousOffset Routine:\n");
printf("\n");
printf("The various combinations of k out of n items are\n");
printf("distinguished by the ID numbers of the items in\n");
printf("the combinations. In neural net projects the items\n");
printf("will probably be bits in an input array. The\n");
printf("PreviousOffset routine calculates the previous set of bit\n");
printf("numbers and previous offset given the current bit numbers\n");
printf("and offset. The first set of bit numbers is always \n");
printf("1, 2, 3 ... k  \n");
printf("and the last set is always\n");
printf("n-k+1 ... n-2, n-1, n.\n");
printf("This test asks you for values for n and k,\n");
printf("starts with the last set of bit numbers, and prints\n");
printf("all sets of bit numbers. Keep n and k small unless\n");
printf("you really like to sit and watch printout scroll by.\n");


printf("\n");
printf("enter n..."); n=getvalue();

do
	{
	printf("\n");
	printf("enter k from 1 to %lu ...",n); k=getvalue();
	}
while ((k<1) || (k>n));

NetInfo.inputwidth = n;
NetInfo.logicdepth = k;
NetInfo.offset = NetSize(n,k) -1 ;


Size = NetSize(n,k);
printf("\n");
if (Size>2000)
	{
	printf("You do not really want to watch %lu ",Size);
	printf("lines of text scroll by!");
	}
else
	{
	for (i=1;i<=NetInfo.logicdepth;i++) 
		{ a[i] = NetInfo.inputwidth - NetInfo.logicdepth + i; }
	
	do
		{
		/* print offset and bit numbers */
		printf("offset: %lu  bitnumbers: ", NetInfo.offset);
		for (i= 1; i<= NetInfo.logicdepth; i++) 
			{ printf("%lu ", a[i] ) ;}
		printf("\n");
		NoMoreOffsets=PreviousOffset(&NetInfo); /* prev. offset if possible */
		}
	while (! NoMoreOffsets);
	}
getreturn();
}
/*------------------------------------------------------------------------*/


/*------------------------------------------------------------------------*/

void
TestOffset()
{
unsigned long int i,C,n,k,a[1024],IndexGood,limit;
struct NetInfoBlock NetInfo;

printf("\n");
printf("Test of Offset Routine:\n");
printf("\n");
printf("Consider that there are a large number of data blocks of some \n");
printf("unspecified sort which contain information about the relation of\n");
printf("groups of k bits, each block relating to a different combination\n");
printf("of k bits drawn from a population of n bits which appears in an\n");
printf("input array that somehow transmits the state of the world by\n");
printf("varying according to world events. If we have the ID numbers for\n");
printf("a set of k bits which interest us, where would we look for the\n");
printf("data block that describes their relationship?\n");
printf("The Offset routine calculates a unique offset given a set of bit\n");
printf("numbers and can be used to access the data blocks. This test asks\n");
printf("you for values of n and k and for bit numbers, then calculates\n");
printf("the offset. Keep k small unless you want to get embroiled in\n");
printf("a tedious data entry project furnishing bitnumbers.\n");

printf("\n");
printf("enter n..."); n=getvalue();

do
	{
	printf("\n");
	printf("enter k from 1 to %lu ...",n); k=getvalue();
	}
while ((k<1) || (k>n));


/* get good values for a[1] through a[k] */
for (i=1;i<= k;i++)
    {
	do
		{
		printf("enter a[%lu] from ",i);
		if (i!=1) 
			{ printf("%lu",(a[i-1]+1));} 
		else 
			{printf("%lu",i);}
		limit=n-k+i;
		printf(" to %lu ...",n-k+i);
		a[i]=getvalue();
		if  (
			( (i==1)  &&  (a[i]>=1)         &&  (a[i]<=(n-k+i)) )
            ||
            ( (i>1)   &&  (a[i]>=a[i-1]+1)  &&  (a[i]<=(n-k+i)) )
            )
		{ IndexGood = 1; }
        else 
        { IndexGood = 0; }
        }
    while ( ! IndexGood ) ;
    }

/* set up net info block */
NetInfo.inputwidth = n;
NetInfo.logicdepth = k;
NetInfo.bitnumbers = a;

Offset(&NetInfo);

printf("offset = %lu\n", NetInfo.offset);
getreturn();
}
/*------------------------------------------------------------------------*/


/*------------------------------------------------------------------------*/

void
TestBitNumbers()
{
unsigned long int i,C,n,k,a[1024],Size;
struct NetInfoBlock NetInfo;

printf("\n");
printf("Test of BitNumbers Routine:\n");
printf("\n");
printf("Consider that there are a large number of data blocks of some \n");
printf("unspecified sort which contain information about the relation of\n");
printf("groups of k bits, each block relating to a different combination\n");
printf("of k bits drawn from a population of n bits which appears in an\n");
printf("input array that somehow transmits the state of the world by\n");
printf("varying according to world events. If we select one of the data\n");
printf("blocks by specifying an offset to it, what are the ID numbers\n");
printf("of the bits whose relationship the block describes?\n");
printf("The BitNumbers routine is the inverse of the Offset routine. It\n");
printf("finds the bitnumbers which correspond to a given offset while\n");
printf("the Offset routine finds the offset given the bit numbers. This\n");
printf("test asks you for values for n and k, and for an offset to a\n");
printf("data block, and then uses the Bitnumbers routine to find the\n");
printf("bit ID numbers for the specified data block. Note that the\n");
printf("offset of the first data block is always 0 since it's address\n");
printf("is the same as the address of the collection of data blocks;\n");
printf("offsets are in units of datablock size and not bytes.\n");

printf("\n");
printf("enter n..."); n=getvalue();

do
	{
	printf("\n");
	printf("enter k from 1 to %lu ...",n); k=getvalue();
	}
while ((k<1) || (k>n));

/* get a good value for offset */
do
	{
	Size = NetSize(n,k);
	printf("enter an offset, from 0 to %lu ...", Size-1); 
	C=getvalue();
	}
while ( (C < 0) || (C > NetSize(n,k)-1) );

NetInfo.offset = C;
NetInfo.inputwidth = n;
NetInfo.logicdepth = k;
NetInfo.bitnumbers = a;

BitNumbers(&NetInfo);

printf("bit numbers: ");
for (i= 1 ; i<=k ;i++)
	{ printf ("%lu ",a[i]); }

printf("\n");
getreturn();
}
/*------------------------------------------------------------------------*/


/*----------------------------------------------------------------------*/

void getreturn()
/* let reader read the output */
{
char chr=0;
printf("hit return...") ;/* issue prompt */
while(chr!='\n')chr=getchar();
}

/*------------------------------------------------------------------------*/


/*----------------------------------------------------------------------*/

unsigned long int
getvalue()
/* get multidigit integer entry from keyboard */
{
char chr=0;
unsigned long int value=0;
/* prompt is provided by calling routine */
while(chr!='\n')
	{
	chr=getchar();
	if((chr>='0')&&(chr<='9')) value = 10 * value + chr - '0';
	if( !( ((chr>='0')&&(chr<='9')) || (chr=='\n') ) )
		{
		value = 0;
		printf(" No you don't!\nEnter again from start...");
		}
	}
return value;
}

/*------------------------------------------------------------------------*/
