/*
 * tutor6.c - Turbo C version of Jack Crenshaw's compiler lesson #6.
 *
 */
#include <stdio.h>
#include <conio.h>
#include <ctype.h>
#include <process.h>
#include <string.h>
#include <stdarg.h>

#define TAB	'\t'
#define LF	'\n'
#define CR	'\r'
#define IsAlpha(c) isalpha ((c))
#define IsDigit(c) isdigit ((c))
#define UpCase(c)  toupper ((c))
#define IsAddop(c) (c) == '+' || (c) == '-'
#define IsMulop(c) (c) == '*' || (c) == '/'
#define IsAlNum(c) isalnum ((c))
#define IsWhite(c) (c) == ' ' || (c) == TAB
#define IsBoolean(c) (c) == 't' || (c) == 'f' || (c) == 'T' || (c) == 'F'
#define IsOrOp(c) (c) == '|' || (c) == '~'
#define IsRelop(c) (c) == '=' || (c) == '#' || (c) == '<' || (c) == '>'
#define BOOL int

char Look;	/* lookahead character */
int  LCount;	/* label counter */

/* forward definitions for certain items */
void Block (char *l);
void Expression (void);
void BoolTerm (void);
void BoolFactor (void);

/*
 * Read New Character From Input Stream
 */
void GetChar (void) {
    Look = getchar ();
}

/*
 * Report an Error
 */
void Error (char *fmt, ...) {
    va_list (args);

    printf ("\n\007Error: ");
    va_start (args, fmt);
    vprintf (fmt, args);
    va_end (args);
    printf ("\n");
}

/*
 * Report Error and Halt
 */
void Abort (char *fmt, ...) {
    va_list args;

    va_start (args, fmt);
    Error (fmt, args);
    va_end (args);
    exit (1);
}

/*
 * Report What Was Expected
 */
void Expected (char *fmt, ...) {
    va_list args;
    char f[81];

    va_start (args, fmt);
    strcpy (f, fmt);
    Abort (strcat (f, " Expected"), args);
    va_end (args);
}

/*
 * Match a Specific Input Character
 */
void Match (char x) {
    char s[4];

    if (Look == x)
        GetChar ();
    else {
        sprintf (s, "'%c'", x);
        Expected (s);
    }
}

/*
 * skip over leading white space
 */
void SkipWhite (void) {
    while (IsWhite (Look))
	GetChar ();
}

/*
 * Skip a newline
 */
void Fin (void) {
    if (Look == CR) GetChar ();
    if (Look == LF) GetChar ();
}

/*
 * Get an Identifier
 */
char GetName (void) {
    char c;

    if (! IsAlpha (Look))
        Expected ("Name");
    c = UpCase (Look);
    GetChar ();
    return (c);
}

/*
 * Get a Number
 */
char GetNum (void) {
    char c;
    if (! IsDigit (Look))
        Expected ("Integer");
    c = Look;
    GetChar ();
    return (c);
}

/*
 * Generate a unique label
 */
char *NewLabel (void) {
    char s[80];
    sprintf (s, "L%d", LCount++);
    return (s);
}

/*
 * Post a label to output
 */
void PostLabel (char *L) {
    printf ("%s:\n", L);
}

/*
 * Output a String with Tab
 */
void Emit (char *fmt, ...) {
    va_list args;

    printf ("%c", TAB);
    va_start (args, fmt);
    vprintf (fmt, args);
    va_end (args);
}

/*
 * Parse and Translate an Identifier
 */
void Ident (void) {
    char Name;

    Name = GetName ();
    if (Look == '(') {
	Match ('(');
	Match (')');
	Emit ("CALL %c\n", Name);
    }
    else
	Emit ("MOV AX,%c\n", Name);
}
/*
 * Parse and Translate a Math Factor
 */
void Factor (void) {

    if (Look == '(') {
	Match ('(');
	Expression ();
	Match (')');
    }
    else if (IsAlpha (Look))
	Ident ();
    else
        Emit ("MOV AX,%c\n", GetNum ());
}

/*
 * Parse and Translate the First Math Factor
 */
void SignedFactor (void) {
    if (Look == '+')
	GetChar ();
    if (Look == '-') {
	GetChar ();
	if (IsDigit (Look))
	    Emit ("MOV AX,%c\n", GetNum ());
	else {
	    Factor ();
	    Emit ("NEG AX\n");
	}
    }
    else
	Factor ();
}

/*
 * Recognize and Translate a Multiply
 */
void Multiply (void) {
    Match ('*');
    Factor ();
    Emit ("POP DX\n");
    Emit ("MUL AX,DX\n");
}

/*
 * Recognize and Translate a Divide
 */
void Divide (void) {
    Match ('/');
    Factor ();
    Emit ("POP DX\n");
    Emit ("XCHG AX,DX\n");
    Emit ("DIV AX,DX\n");
}

/*
 * Parse and Translate a Math Term
 */
void Term (void) {
    SignedFactor ();
    while (IsMulop (Look)) {
        Emit ("PUSH AX\n");
	switch (Look) {
	    case '*' : Multiply (); break;
	    case '/' : Divide (); break;
	    default  : Expected ("Mulop"); break;
	}
    }
}

/*
 * Recognize and Translate an Add
 */
void Add (void) {
    Match ('+');
    Term ();
    Emit ("POP DX\n");
    Emit ("ADD AX,DX\n");
}

/*
 * Recognize and Translate a Subtract
 */
void Subtract (void) {
    Match ('-');
    Term ();
    Emit ("POP DX\n");
    Emit ("SUB AX,DX\n");
    Emit ("NEG AX\n");
}

/*
 * Parse and Translate an Expression
 */
void Expression (void) {
    Term ();
    while (IsAddop (Look)) {
        Emit ("PUSH AX\n");
        switch (Look) {
	    case '+' : Add (); break;
	    case '-' : Subtract ();  break;
	    default  : Expected ("Addop"); break;
	}
    }
}

/*
 * Get a Boolean literal
 */
BOOL GetBoolean (void) {
    BOOL good;

    if (!IsBoolean (Look))
	Expected ("Boolean Literal");
    good = (Look == 't' || Look == 'T');
    GetChar ();
    return (good);
}

/*
 * Recognize and Translate a Boolean OR
 */
void BoolOr (void) {
    Match ('|');
    BoolTerm ();
    Emit ("POP DX\n");
    Emit ("OR AX,BX\n");
}

/*
 * Recognize and Translate an Exclusive Or
 */
void BoolXor (void) {
    Match ('~');
    BoolTerm ();
    Emit ("POP DX\n");
    Emit ("XOR AX,DX\n");
}

/*
 * Recognize and Translate a Relational "Equals"
 */
void Equals (void) {
    Match ('=');
    Expression ();
    Emit ("POP DX\n");
    Emit ("CMP AX,DX\n");
    /*
     * All this garbage below is taking the Z flag and moving it to the
     * low-order bit of AH and then producing the Boolean representation of
     * the result.  Not nearly as neat as the 68000, but it works.
     */
    Emit ("LAHF\n");			/* put 8080 flags into AH */
    Emit ("AND AX,0040h\n");		/* mask all but ZF (bit 6) */
    Emit ("ROL AH\n");			/* and move it to low-order position */
    Emit ("ROL AH\n");
    Emit ("NEG AX\n");			/* and produce boolean result */
}

/*
 * Recognize and Translate a Relational "Not Equals"
 */
void NotEquals (void) {
    Match ('#');
    Expression ();
    Emit ("POP DX\n");
    Emit ("CMP AX,DX\n");
    Emit ("LAHF\n");
    Emit ("AND AX,0040h\n");
    Emit ("ROL AH\n");
    Emit ("ROL AH\n");
    Emit ("SUB AX,1\n");
}

/*
 * Recognize and Translate a Relational "Less Than"
 */
void Less (void) {
    Match ('<');
    Expression ();
    Emit ("POP DX\n");
    Emit ("CMP DX,AX\n");
    Emit ("LAHF\n");
    Emit ("AND AX,0001h\n");
    Emit ("NEG AX\n");
}

/*
 * Recognize and Translate a Relational "Greater Than"
 */
void Greater (void) {
    Match ('>');
    Expression ();
    Emit ("POP DX\n");
    Emit ("CMP AX,DX\n");
    Emit ("LAHF\n");
    Emit ("AND AX,0001h\n");
    Emit ("NEG AX\n");
}

/*
 * Parse and Translate a Relation
 */
void Relation (void) {
    Expression ();
    if (IsRelop (Look)) {
	Emit ("PUSH AX\n");
	switch (Look) {
	    case '=' : Equals (); break;
	    case '#' : NotEquals (); break;
	    case '<' : Less (); break;
	    case '>' : Greater (); break;
	}
	Emit ("OR AX,AX\n");
    }
}

/*
 * Parse and Translate a Boolean Factor
 */
void BoolFactor () {
    if (IsBoolean (Look))
	if (GetBoolean ())
	    Emit ("MOV AX,-1\n");
	else
	    Emit ("XOR AX,AX\n");
    else
	Relation ();
}

/*
 * Parse and Translate a Boolean Factor with NOT
 */
void NotFactor (void) {
    if (Look == '!') {
	Match ('!');
	BoolFactor ();
	Emit ("XOR AX,-1");
    }
    else
	BoolFactor ();
}

/*
 * Parse and Translate a Boolean Term
 */
void BoolTerm (void) {
    NotFactor ();
    while (Look == '&') {
	Emit ("PUSH AX\n");
	Match ('&');
	NotFactor ();
	Emit ("POP DX\n");
	Emit ("AND AX,DX\n");
    }
}

/*
 * Parse and Translate a Boolean Expression
 */
void BoolExpression (void) {
    BoolTerm ();
    while (IsOrOp (Look)) {
	Emit ("PUSH AX\n");
	switch (Look) {
	    case '|' : BoolOr (); break;
	    case '~' : BoolXor (); break;
	}
    }
}

/*
 * Recognize and Translate an IF Construct
 */
void DoIf (char *L) {
    char L1[10], L2[10];

    Match ('i');
    BoolExpression ();
    strcpy (L1, NewLabel ());
    strcpy (L2, L1);
    Emit ("JNZ %s\n", L1);
    Block (L);
    if (Look == 'l') {
	Match ('l');
	strcpy (L2, NewLabel ());
	Emit ("JMP %s\n", L2);
	PostLabel (L1);
	Block (L);
    }
    Match ('e');
    PostLabel (L2);
}

/*
 * Parse and Translate a WHILE Statement
 */
void DoWhile (void) {
    char L1[10], L2[10];

    Match ('w');
    strcpy (L1, NewLabel ());
    strcpy (L2, NewLabel ());
    PostLabel (L1);
    BoolExpression ();
    Emit ("JNZ %s\n", L2);
    Block (L2);
    Match ('e');
    Emit ("JMP %s\n", L1);
    PostLabel (L2);
}

/*
 * Parse and Translate a LOOP Statement
 */
void DoLoop (void) {
    char L1[10], L2[10];

    Match ('p');
    strcpy (L1, NewLabel ());
    strcpy (L2, NewLabel ());
    PostLabel (L1);
    Block (L2);
    Match ('e');
    Emit ("JMP %s\n", L1);
    PostLabel (L2);
}

/*
 * Parse and Translate a REPEAT Statement
 */
void DoRepeat (void) {
    char L1[10], L2[10];

    Match ('r');
    strcpy (L1, NewLabel ());
    strcpy (L2, NewLabel ());
    PostLabel (L1);
    Block (L2);
    Match ('u');
    BoolExpression ();
    Emit ("JNZ %s\n", L1);
    PostLabel (L2);
}

/*
 * Parse and Translate a FOR Statement
 */
void DoFor (void) {
    char L1[10], L2[10], Name;

    Match ('f');
    strcpy (L1, NewLabel ());
    strcpy (L2, NewLabel ());
    Name = GetName ();
    Match ('=');
    Expression ();
    Emit ("DEC AX\n");
    Emit ("MOV %c, AX\n", Name);
    Expression ();
    Emit ("PUSH AX\n");
    PostLabel (L1);
    Emit ("INC %c\n", Name);
    Emit ("POP AX\n");
    Emit ("CMP AX, %c\n", Name);
    Emit ("JNC %s\n", L2);
    Emit ("PUSH AX\n");
    Block (L2);
    Match ('e');
    Emit ("JMP %s\n", L1);
    PostLabel (L2);
}

/*
 * Parse and Translate a DO Statement
 */
void DoDo (void) {
    char L1[10], L2[10];

    Match ('d');
    strcpy (L1, NewLabel ());
    strcpy (L2, NewLabel ());
    Expression ();
    Emit ("DEC AX\n");
    PostLabel (L1);
    Emit ("PUSH AX\n");
    Block (L2);
    Emit ("POP AX\n");
    Emit ("DEC AX\n");
    Emit ("JNZ %s\n", L1);
    Emit ("PUSH AX\n");
    PostLabel (L2);
    Emit ("POP AX\n");
}

/*
 * Recognize and Translate a BREAK
 */
void DoBreak (char *L) {
    Match ('b');
    Emit ("JMP %s\n", L);
}

/*
 * Parse and Translate an Assignment Statement
 */
void Assignment (void) {
    char Name;

    Name = GetName ();
    Match ('=');
    BoolExpression ();
    Emit ("MOV %c,AX\n", Name);
}

/*
 * Recognize and Translate a Statement Block
 */
void Block (char *L) {
    while (strchr ("elu", Look) == NULL) {
	Fin ();
	switch (Look) {
	    case 'i' : DoIf (L); 	break;
	    case 'w' : DoWhile (); 	break;
	    case 'p' : DoLoop (); 	break;
	    case 'r' : DoRepeat ();	break;
	    case 'f' : DoFor ();	break;
	    case 'd' : DoDo ();		break;
	    case 'b' : DoBreak (L);	break;
	    default  : Assignment ();	break;
	}
	Fin ();
    }
}

/*
 * Parse and Translate a Program
 */
void DoProgram (void) {
    Block ("");
    if (Look != 'e')
	Expected ("End");
    Emit ("END\n");
}

/*
 * Initialize
 */
void Init (void) {
    LCount = 0;
    GetChar ();
}

/*
 * Main Program
 */
void main (void) {
    Init ();
    DoProgram ();
}
