/*
 * Copyright 1993 Network Computing Devices, Inc.
 * 
 * Permission to use, copy, modify, distribute, and sell this software and its
 * documentation for any purpose is hereby granted without fee, provided that
 * the above copyright notice appear in all copies and that both that
 * copyright notice and this permission notice appear in supporting
 * documentation, and that the name Network Computing Devices, Inc. not be
 * used in advertising or publicity pertaining to distribution of this
 * software without specific, written prior permission.
 * 
 * THIS SOFTWARE IS PROVIDED 'AS-IS'.  NETWORK COMPUTING DEVICES, INC.,
 * DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING WITHOUT
 * LIMITATION ALL IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A
 * PARTICULAR PURPOSE, OR NONINFRINGEMENT.  IN NO EVENT SHALL NETWORK
 * COMPUTING DEVICES, INC., BE LIABLE FOR ANY DAMAGES WHATSOEVER, INCLUDING
 * SPECIAL, INCIDENTAL OR CONSEQUENTIAL DAMAGES, INCLUDING LOSS OF USE, DATA,
 * OR PROFITS, EVEN IF ADVISED OF THE POSSIBILITY THEREOF, AND REGARDLESS OF
 * WHETHER IN AN ACTION IN CONTRACT, TORT OR NEGLIGENCE, ARISING OUT OF OR IN
 * CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 * 
 * $NCDId: @(#)Util.c,v 1.18 1993/11/03 19:23:59 greg Exp $
 */

#define _UTIL_C_

#include <stdio.h>
#include <audio/fileutil.h>
#include "Alibint.h"

AuFlowID
AuGetScratchFlow(aud, ret_status)
AuServer       *aud;
AuStatus       *ret_status;
{
    AuFlowID        flow;
    int             i;

    if (aud->scratch_flows.num_inuse == AU_MAX_SCRATCH_FLOWS)
	return AuCreateFlow(aud, ret_status);

    for (i = 0; i < aud->scratch_flows.total; i++)
	if (!aud->scratch_flows.flows[i].inuse)
	{
	    aud->scratch_flows.flows[i].inuse = AuTrue;
	    aud->scratch_flows.num_inuse++;
	    return aud->scratch_flows.flows[i].flow;
	}

    if ((flow = AuCreateFlow(aud, ret_status)) != AuNone)
    {
	aud->scratch_flows.flows[aud->scratch_flows.total].flow = flow;
	aud->scratch_flows.flows[aud->scratch_flows.total].inuse = AuTrue;
	aud->scratch_flows.total++;
	aud->scratch_flows.num_inuse++;
    }

    return flow;
}

AuFlowID
AuGetScratchFlowToBucket(aud, bucket, import, ret_status)
AuServer       *aud;
AuBucketID      bucket;
int            *import;
AuStatus       *ret_status;
{
    AuElement       elements[2];
    AuFlowID        flow;
    AuBucketAttributes *ba;

    if (!(flow = AuGetScratchFlow(aud, ret_status)))
	return NULL;

    if (!(ba = AuGetBucketAttributes(aud, bucket, ret_status)))
    {
	AuReleaseScratchFlow(aud, flow, ret_status);
	return NULL;
    }

    AuMakeElementImportClient(&elements[0], AuBucketSampleRate(ba),
			      AuBucketFormat(ba), AuBucketNumTracks(ba),
			      AuTrue, 0, 0, 0, NULL);
    AuMakeElementExportBucket(&elements[1], 0, bucket,
			      AuBucketNumSamples(ba), 0, 0, NULL);

    /* set up the flow */
    AuSetElements(aud, flow, AuFalse, 2, elements, ret_status);

    *import = 0;
    AuFreeBucketAttributes(aud, 1, ba);
    return flow;
}

AuFlowID
AuGetScratchFlowFromBucket(aud, bucket, export, ret_status)
AuServer       *aud;
AuBucketID      bucket;
int            *export;
AuStatus       *ret_status;
{
    AuElement       elements[2];
    AuFlowID        flow;
    AuBucketAttributes *ba;

    if (!(flow = AuGetScratchFlow(aud, ret_status)))
	return NULL;

    if (!(ba = AuGetBucketAttributes(aud, bucket, ret_status)))
    {
	AuReleaseScratchFlow(aud, flow, ret_status);
	return NULL;
    }

    AuMakeElementImportBucket(&elements[0], AuBucketSampleRate(ba), bucket,
			      AuBucketNumSamples(ba), 0, 0, NULL);
    AuMakeElementExportClient(&elements[1], 0, AuBucketSampleRate(ba),
			      AuBucketFormat(ba), AuBucketNumTracks(ba),
			      AuTrue, 0, 0, 0, NULL);

    /* set up the flow */
    AuSetElements(aud, flow, AuFalse, 2, elements, ret_status);

    *export = 1;
    AuFreeBucketAttributes(aud, 1, ba);
    return flow;
}

void
AuReleaseScratchFlow(aud, flow, ret_status)
AuServer       *aud;
AuFlowID        flow;
AuStatus       *ret_status;
{
    int             i;

    for (i = 0; i < aud->scratch_flows.total; i++)
	if (aud->scratch_flows.flows[i].flow == flow)
	{
	    aud->scratch_flows.flows[i].inuse = AuFalse;
	    aud->scratch_flows.num_inuse--;
	    return;
	}

    AuDestroyFlow(aud, flow, ret_status);
}

void
AuStartFlow(aud, flow, ret_status)
AuServer       *aud;
AuFlowID        flow;
AuStatus       *ret_status;
{
    AuElementState  states[1];

    AuMakeElementState(&states[0], flow, AuElementAll, AuStateStart);
    AuSetElementStates(aud, 1, states, ret_status);
}

void
AuStopFlow(aud, flow, ret_status)
AuServer       *aud;
AuFlowID        flow;
AuStatus       *ret_status;
{
    AuElementState  states[1];

    AuMakeElementState(&states[0], flow, AuElementAll, AuStateStop);
    AuSetElementStates(aud, 1, states, ret_status);
}

void
AuPauseFlow(aud, flow, ret_status)
AuServer       *aud;
AuFlowID        flow;
AuStatus       *ret_status;
{
    AuElementState  states[1];

    AuMakeElementState(&states[0], flow, AuElementAll, AuStatePause);
    AuSetElementStates(aud, 1, states, ret_status);
}

static struct
{
    int             format;
    char           *string,
                   *define;
}               formats[] =
{
    AuFormatULAW8, "8-bit uLAW", "AuFormatULAW8",
    AuFormatLinearUnsigned8, "8-bit unsigned linear", "AuFormatLinearUnsigned8",
    AuFormatLinearSigned8, "8-bit signed linear", "AuFormatLinearSigned8",
    AuFormatLinearSigned16MSB, "16-bit signed linear (big endian)",
    "AuFormatLinearSigned16MSB",
    AuFormatLinearUnsigned16MSB, "16-bit unsigned linear (big endian)",
    "AuFormatLinearUnsigned16MSB",
    AuFormatLinearSigned16LSB, "16-bit signed linear (little endian)",
    "AuFormatLinearSigned16LSB",
    AuFormatLinearUnsigned16LSB, "16-bit unsigned linear (little endian)",
    "AuFormatLinearUnsigned16LSB",
};

_AuConst char  *
AuFormatToString(format)
unsigned int    format;
{
    int             i;

    for (i = 0; i < sizeof(formats) / sizeof(formats[0]); i++)
	if (formats[i].format == format)
	    return formats[i].string;

    return "Unknown";
}

int
AuStringToFormat(s)
_AuConst char  *s;
{
    int             i;

    for (i = 0; i < sizeof(formats) / sizeof(formats[0]); i++)
	if (!strcasecmp(s, formats[i].string))
	    return formats[i].format;

    return -1;
}

_AuConst char  *
AuFormatToDefine(format)
unsigned int    format;
{
    int             i;

    for (i = 0; i < sizeof(formats) / sizeof(formats[0]); i++)
	if (formats[i].format == format)
	    return formats[i].define;

    return "Unknown";
}

int
AuDefineToFormat(s)
_AuConst char  *s;
{
    int             i;

    for (i = 0; i < sizeof(formats) / sizeof(formats[0]); i++)
	if (!strcasecmp(s, formats[i].define))
	    return formats[i].format;

    return -1;
}

typedef struct
{
    AuFlowID        flow;
    AuPointer       callback_data;
    void            (*callback) (
#if NeedNestedPrototypes
			                    AuServer *, AuEventHandlerRec *,
				                 AuEvent *, AuPointer
#endif
    );
}               MonitorDeviceRec, *MonitorDevicePtr;

static          AuBool
EventHandler(aud, ev, handler)
AuServer       *aud;
AuEvent        *ev;
AuEventHandlerRec *handler;
{
    MonitorDevicePtr priv = (MonitorDevicePtr) handler->data;

    switch (ev->type)
    {
	case AuEventTypeMonitorNotify:
	    if (priv->callback)
		(*priv->callback) (aud, handler, ev, priv->callback_data);
	    break;
	case AuEventTypeElementNotify:
	    {
		AuElementNotifyEvent *event = (AuElementNotifyEvent *) ev;

		if (event->kind == AuElementNotifyKindState &&
		    event->cur_state == AuStateStop)
		{
		    if (priv->callback)
			(*priv->callback) (aud, handler, ev,
					   priv->callback_data);

		    AuUnregisterEventHandler(aud, handler);
		    AuReleaseScratchFlow(aud, priv->flow, NULL);
		    Aufree(priv);
		}
		break;
	    }
    }

    return AuTrue;
}

/* sampling rate of the monitor (in Hz) */
int             AuMonitorRate = 10;
/* format of the monitor data */
int             AuMonitorFormat = AuFormatLinearSigned16MSB;

AuEventHandlerRec *
AuMonitorDevice(aud, rate, in_device, out_device, volume, callback,
		callback_data, pflow, pmultelem, pmonelem, ret_status)
AuServer       *aud;
int             rate;
AuDeviceID      in_device,
                out_device;
AuFixedPoint    volume;
void            (*callback) (
#if NeedFunctionPrototypes
			                     AuServer *, AuEventHandlerRec *,
			                     AuEvent *, AuPointer
#endif
);
AuPointer       callback_data;
AuFlowID       *pflow;
int            *pmultelem,
               *pmonelem;
AuStatus       *ret_status;
{
    AuElement       elements[4];
    AuElementState  states[1];
    AuElementAction actions[1];
    AuEventHandlerRec *handler;
    MonitorDevicePtr priv;
    int             numTracks,
                    i;

    if (!(priv = (MonitorDevicePtr) Aumalloc(sizeof(MonitorDeviceRec))))
	return NULL;

    priv->callback = callback;
    priv->callback_data = callback_data;

    if ((priv->flow = AuGetScratchFlow(aud, NULL)) == AuNone)
    {
	Aufree(priv);
	return NULL;
    }

    if (out_device == AuNone)
    {
	AuDeviceAttributes *d = NULL;

	/* look up the input device attributes */
	for (i = 0; i < AuServerNumDevices(aud); i++)
	{
	    if (AuDeviceIdentifier(AuServerDevice(aud, i)) == in_device)
	    {
		d = AuServerDevice(aud, i);
		break;
	    }
	}

	if (!d)
	{
	    Aufree(priv);
	    return NULL;
	}

	numTracks = AuDeviceNumTracks(d);

	/* look for a matching output device */
	for (i = 0; i < AuServerNumDevices(aud); i++)
	    if ((AuDeviceKind(AuServerDevice(aud, i)) ==
		 AuComponentKindPhysicalOutput) &&
		AuDeviceNumTracks(AuServerDevice(aud, i)) == numTracks)
	    {
		out_device = AuDeviceIdentifier(AuServerDevice(aud, i));
		break;
	    }

	if (out_device == AuNone)
	{
	    Aufree(priv);
	    return NULL;
	}
    }
    else
    {
	AuDeviceAttributes *d;

	if (!(d = AuGetDeviceAttributes(aud, out_device, ret_status)))
	{
	    Aufree(priv);
	    return NULL;
	}

	numTracks = AuDeviceNumTracks(d);
	AuFreeDeviceAttributes(aud, 1, d);
    }

    AuMakeSendNotifyAction(&actions[0], AuStateStop, AuStateAny, AuReasonAny);

    AuMakeElementImportDevice(&elements[0], rate, in_device, AuUnlimitedSamples,
			      1, actions);
    AuMakeElementMultiplyConstant(&elements[1], 0, volume);
    AuMakeElementExportDevice(&elements[2], 1, out_device, rate,
			      AuUnlimitedSamples, 0, NULL);

    if (pmonelem)
    {
	i = 4;
	*pmonelem = 3;
	AuMakeElementExportMonitor(&elements[3], 0, AuMonitorRate,
				   AuMonitorFormat, numTracks);
    }
    else
	i = 3;

    /* set up the flow */
    AuSetElements(aud, priv->flow, AuTrue, i, elements, ret_status);

    if (!(handler = AuRegisterEventHandler(aud, AuEventHandlerIDMask,
					0, priv->flow, EventHandler, priv)))
    {
	AuReleaseScratchFlow(aud, priv->flow, ret_status);
	Aufree(priv);
	return NULL;
    }

    /* start up the components */
    AuMakeElementState(&states[0], priv->flow, AuElementAll, AuStateStart);
    AuSetElementStates(aud, 1, states, ret_status);

    if (pflow)
	*pflow = priv->flow;

    if (pmultelem)
	*pmultelem = 1;

    return handler;
}

#undef convert
#define convert(_type, _convert)					      \
do									      \
{									      \
    _type x;								      \
									      \
    x = *(_type *) s;							      \
    *d-- = _convert;							      \
    s -= sizeof(x);							      \
} while (--samples)

#define ulawToLinear(_x) ulawToLinearTable[_x]

static unsigned short ulawToLinearTable[] =
{
    0x8284, 0x8684, 0x8a84, 0x8e84, 0x9284, 0x9684, 0x9a84, 0x9e84,
    0xa284, 0xa684, 0xaa84, 0xae84, 0xb284, 0xb684, 0xba84, 0xbe84,
    0xc184, 0xc384, 0xc584, 0xc784, 0xc984, 0xcb84, 0xcd84, 0xcf84,
    0xd184, 0xd384, 0xd584, 0xd784, 0xd984, 0xdb84, 0xdd84, 0xdf84,
    0xe104, 0xe204, 0xe304, 0xe404, 0xe504, 0xe604, 0xe704, 0xe804,
    0xe904, 0xea04, 0xeb04, 0xec04, 0xed04, 0xee04, 0xef04, 0xf004,
    0xf0c4, 0xf144, 0xf1c4, 0xf244, 0xf2c4, 0xf344, 0xf3c4, 0xf444,
    0xf4c4, 0xf544, 0xf5c4, 0xf644, 0xf6c4, 0xf744, 0xf7c4, 0xf844,
    0xf8a4, 0xf8e4, 0xf924, 0xf964, 0xf9a4, 0xf9e4, 0xfa24, 0xfa64,
    0xfaa4, 0xfae4, 0xfb24, 0xfb64, 0xfba4, 0xfbe4, 0xfc24, 0xfc64,
    0xfc94, 0xfcb4, 0xfcd4, 0xfcf4, 0xfd14, 0xfd34, 0xfd54, 0xfd74,
    0xfd94, 0xfdb4, 0xfdd4, 0xfdf4, 0xfe14, 0xfe34, 0xfe54, 0xfe74,
    0xfe8c, 0xfe9c, 0xfeac, 0xfebc, 0xfecc, 0xfedc, 0xfeec, 0xfefc,
    0xff0c, 0xff1c, 0xff2c, 0xff3c, 0xff4c, 0xff5c, 0xff6c, 0xff7c,
    0xff88, 0xff90, 0xff98, 0xffa0, 0xffa8, 0xffb0, 0xffb8, 0xffc0,
    0xffc8, 0xffd0, 0xffd8, 0xffe0, 0xffe8, 0xfff0, 0xfff8, 0x0000,
    0x7d7c, 0x797c, 0x757c, 0x717c, 0x6d7c, 0x697c, 0x657c, 0x617c,
    0x5d7c, 0x597c, 0x557c, 0x517c, 0x4d7c, 0x497c, 0x457c, 0x417c,
    0x3e7c, 0x3c7c, 0x3a7c, 0x387c, 0x367c, 0x347c, 0x327c, 0x307c,
    0x2e7c, 0x2c7c, 0x2a7c, 0x287c, 0x267c, 0x247c, 0x227c, 0x207c,
    0x1efc, 0x1dfc, 0x1cfc, 0x1bfc, 0x1afc, 0x19fc, 0x18fc, 0x17fc,
    0x16fc, 0x15fc, 0x14fc, 0x13fc, 0x12fc, 0x11fc, 0x10fc, 0x0ffc,
    0x0f3c, 0x0ebc, 0x0e3c, 0x0dbc, 0x0d3c, 0x0cbc, 0x0c3c, 0x0bbc,
    0x0b3c, 0x0abc, 0x0a3c, 0x09bc, 0x093c, 0x08bc, 0x083c, 0x07bc,
    0x075c, 0x071c, 0x06dc, 0x069c, 0x065c, 0x061c, 0x05dc, 0x059c,
    0x055c, 0x051c, 0x04dc, 0x049c, 0x045c, 0x041c, 0x03dc, 0x039c,
    0x036c, 0x034c, 0x032c, 0x030c, 0x02ec, 0x02cc, 0x02ac, 0x028c,
    0x026c, 0x024c, 0x022c, 0x020c, 0x01ec, 0x01cc, 0x01ac, 0x018c,
    0x0174, 0x0164, 0x0154, 0x0144, 0x0134, 0x0124, 0x0114, 0x0104,
    0x00f4, 0x00e4, 0x00d4, 0x00c4, 0x00b4, 0x00a4, 0x0094, 0x0084,
    0x0078, 0x0070, 0x0068, 0x0060, 0x0058, 0x0050, 0x0048, 0x0040,
    0x0038, 0x0030, 0x0028, 0x0020, 0x0018, 0x0010, 0x0008, 0x0000,
};

int
AuConvertDataToShort(dataFormat, numBytes, data)
int             dataFormat,
                numBytes;
AuPointer       data;
{
    char           *s;
    short          *d;
    int             samples;

    samples = numBytes / AuSizeofFormat(dataFormat);

    if (!samples)
	return 0;

    s = ((char *) data) + numBytes - AuSizeofFormat(dataFormat);
    d = (short *) (((char *) data) + samples * sizeof(short)) - 1;

    switch (dataFormat)
    {
	case AuFormatULAW8:
	    convert(unsigned char, ulawToLinear(x));
	    break;
	case AuFormatLinearUnsigned8:
	    convert(unsigned char, (x - 128) << 8);
	    break;
	case AuFormatLinearSigned8:
	    convert(char, x << 8);
	    break;
	case AuFormatLinearSigned16MSB:
	    if (LITTLE_ENDIAN)
		convert(short, ((((unsigned short) x) >> 8) | (x << 8)));
#if 0
	    else
		convert(short, x);
#endif
	    break;
	case AuFormatLinearUnsigned16MSB:
	    if (LITTLE_ENDIAN)
		convert(short,
			(((((unsigned short) x) >> 8) | (x << 8)) ^ 0x8000));
	    else
		convert(short, x ^ 0x8000);
	    break;
	case AuFormatLinearSigned16LSB:
	    if (BIG_ENDIAN)
		convert(short, ((((unsigned short) x) >> 8) | (x << 8)));
#if 0
	    else
		convert(short, x);
#endif
	    break;
	case AuFormatLinearUnsigned16LSB:
	    if (BIG_ENDIAN)
		convert(short,
			(((((unsigned short) x) >> 8) | (x << 8)) ^ 0x8000));
	    else
		convert(short, x ^ 0x8000);
	    break;
	default:
	    return -1;
    }

    return 0;
}

#undef convert
#define convert(_type, _convert)					      \
{									      \
    _type *_d = (_type *) d;						      \
    short x;								      \
									      \
    do									      \
    {									      \
	x = *s++;							      \
	*_d++ = _convert;						      \
    } while (--samples);						      \
}

/**
 * This routine converts from linear to ulaw.
 *
 * Craig Reese: IDA/Supercomputing Research Center
 * Joe Campbell: Department of Defense
 * 29 September 1989
 *
 * References:
 * 1) CCITT Recommendation G.711  (very difficult to follow)
 * 2) "A New Digital Technique for Implementation of Any
 *     Continuous PCM Companding Law," Villeret, Michel,
 *     et al. 1973 IEEE Int. Conf. on Communications, Vol 1,
 *     1973, pg. 11.12-11.17
 * 3) MIL-STD-188-113,"Interoperability and Performance Standards
 *     for Analog-to_Digital Conversion Techniques,"
 *     17 February 1987
 *
 * Input: Signed 16 bit linear sample
 * Output: 8 bit ulaw sample
 */

#if 0
#define ZEROTRAP				/* turn on the trap as per
						 * the MIL-STD */
#define CLIP 32635
#endif

#define BIAS 0x84				/* define the add-in bias for
						 * 16 bit samples */

static unsigned char
linearToUlaw(sample)
short           sample;
{
    static int      exp_lut[256] =
    {
	0, 0, 1, 1, 2, 2, 2, 2, 3, 3, 3, 3, 3, 3, 3, 3,
	4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4,
	5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5,
	5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5,
	6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6,
	6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6,
	6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6,
	6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6,
	7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7,
	7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7,
	7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7,
	7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7,
	7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7,
	7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7,
	7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7,
	7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7
    };
    short           sign,
                    exponent,
                    mantissa,
                    i;
    unsigned char   ulawbyte;

    /* Get the sample into sign-magnitude. */
    sign = (sample >> 8) & 0x80;/* set aside the sign */

    if (sign)
	sample = -sample;	/* get magnitude */
#ifdef CLIP
    if (sample > CLIP)
	sample = CLIP;		/* clip the magnitude */
#endif
    /* Convert from 16 bit linear to ulaw. */
    sample = sample + BIAS;
    exponent = exp_lut[(sample >> 7) & 0xff];
    mantissa = (sample >> (exponent + 3)) & 0xf;
    ulawbyte = ~(sign | (exponent << 4) | mantissa);
#ifdef ZEROTRAP
    if (!ulawbyte)
	ulawbyte = 0x02;	/* optional CCITT trap */
#endif

    return ulawbyte;
}

int
AuConvertShortToData(dataFormat, numBytes, data)
int             dataFormat,
                numBytes;
AuPointer          data;
{
    char           *d;
    short          *s;
    int             samples;

    samples = numBytes / sizeof(short);

    if (!samples)
	return 0;

    s = (short *) data;
    d = (char *) data;

    switch (dataFormat)
    {
	case AuFormatULAW8:
	    convert(unsigned char, linearToUlaw(x));
	    break;
	case AuFormatLinearUnsigned8:
	    convert(unsigned char, (x >> 8) + 128);
	    break;
	case AuFormatLinearSigned8:
	    convert(char, x >> 8);
	    break;
	case AuFormatLinearSigned16MSB:
	    if (LITTLE_ENDIAN)
		convert(short, ((((unsigned short) x) >> 8) | (x << 8)));
#if 0
	    else
	    {
		convert(short, x);
	    }
#endif
	    break;
	case AuFormatLinearUnsigned16MSB:
	    if (LITTLE_ENDIAN)
	    {
		convert(short,
			(((((unsigned short) x) >> 8) | (x << 8)) ^ 0x8000));
	    }
	    else
	    {
		convert(short, x ^ 0x8000);
	    }
	    break;
	case AuFormatLinearSigned16LSB:
	    if (BIG_ENDIAN)
		convert(short, ((((unsigned short) x) >> 8) | (x << 8)));
#if 0
	    else
	    {
		convert(short, x);
	    }
#endif
	    break;
	case AuFormatLinearUnsigned16LSB:
	    if (BIG_ENDIAN)
	    {
		convert(short,
			(((((unsigned short) x) >> 8) | (x << 8)) ^ 0x8000));
	    }
	    else
	    {
		convert(short, x ^ 0x8000);
	    }
	    break;
	default:
	    return -1;
    }

    return 0;
}
