/**
 * Copyright 1993 Network Computing Devices, Inc.
 *
 * Permission to use, copy, modify, distribute, and sell this software and
 * its documentation for any purpose is hereby granted without fee, provided
 * that the above copyright notice appear in all copies and that both that
 * copyright notice and this permission notice appear in supporting
 * documentation, and that the name Network Computing Devices, Inc. not be
 * used in advertising or publicity pertaining to distribution of this
 * software without specific, written prior permission.
 *
 * THIS SOFTWARE IS PROVIDED `AS-IS'.  NETWORK COMPUTING DEVICES, INC.,
 * DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING WITHOUT
 * LIMITATION ALL IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A
 * PARTICULAR PURPOSE, OR NONINFRINGEMENT.  IN NO EVENT SHALL NETWORK
 * COMPUTING DEVICES, INC., BE LIABLE FOR ANY DAMAGES WHATSOEVER, INCLUDING
 * SPECIAL, INCIDENTAL OR CONSEQUENTIAL DAMAGES, INCLUDING LOSS OF USE, DATA,
 * OR PROFITS, EVEN IF ADVISED OF THE POSSIBILITY THEREOF, AND REGARDLESS OF
 * WHETHER IN AN ACTION IN CONTRACT, TORT OR NEGLIGENCE, ARISING OUT OF OR IN
 * CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 *
 * Author:	Greg Renda <greg@ncd.com>
 * 		Network Computing Devices, Inc.
 * 		350 North Bernardo Ave.
 * 		Mountain View, CA  94043
 *
 * $NCDId: @(#)aupanel.c,v 1.7 1993/11/03 19:23:32 greg Exp $
 */

#include <stdio.h>
#include <string.h>
#include <audio/audiolib.h>
#include <audio/soundlib.h>

#include <X11/Intrinsic.h>
#include <X11/StringDefs.h>
#include <X11/Shell.h>
#include <X11/Xaw/Cardinals.h>
#include <audio/Xtutil.h>

/* widgets */
#include <X11/Xaw/Form.h>
#include <X11/Xaw/Command.h>
#include <X11/Xaw/Label.h>
#include <X11/Xaw/Scrollbar.h>
#include <X11/Xaw/Toggle.h>
#include <X11/Xaw/SimpleMenu.h>
#include <X11/Xaw/MenuButton.h>
#include <X11/Xaw/SmeBSB.h>

#define	APP_CLASS		"Aupanel"
#define	GAIN_FORMAT		"Gain:  %3d%%"
#define	MAX_GAIN		100
#define	MIN_GAIN		1

#define USAGE "\
usage: aupanel [-a audioserver]\
"

#define MakeCommandButton(w, parent, label, callback)			       \
{									       \
    (w) = XtCreateManagedWidget(label, commandWidgetClass, parent, NULL, 0);   \
									       \
    if ((void *) (callback) != NULL)					       \
	XtAddCallback(w, XtNcallback, callback, g);			       \
}

#define MakeLabel(w, parent, label)					       \
{									       \
    (w) = XtCreateManagedWidget(label, labelWidgetClass, parent, NULL, 0);     \
}

#define MakeWidget(w, parent, type, name)				       \
{									       \
    (w) = XtCreateManagedWidget(name, type, parent, NULL, 0);		       \
}

typedef struct
{
    Widget          top,
                    form,
                    quit,
                    query,
                    menu,
                    menuButton,
                    device,
                    gainLabel,
                    gain,
                    modeLabel,
                    mode;
    Display        *dpy;
    AuServer       *aud;
    int             currentGain,
                    numDevices,
                    deviceNum;
    AuDeviceAttributes *da;
}               GlobalDataRec, *GlobalDataPtr;

static String   defaultResources[] =
{
    "*font:                           *courier-medium-r-normal*140*",
    "*Label.borderWidth:              0",
    "*top:                            chainTop",
    "*bottom:                         chainTop",
    "*left:                           chainLeft",
    "*right:                          chainLeft",
    "*query.label:                    Query",
    "*devices.label:                  Devices",
    "*devices.fromHoriz:              query",
    "*quit.label:                     Quit",
    "*quit.fromHoriz:                 devices",
    "*deviceLabel.label:              Stereo Channel Output",
    "*deviceLabel.fromVert:           query",
    "*deviceLabel.font:               *courier-bold-r-normal*140*",
    "*gainLabel.label:                Gain  100%:",
    "*gainLabel.fromVert:             deviceLabel",
    "*gain.fromHoriz:                 gainLabel",
    "*gain.fromVert:                  deviceLabel",
    "*gain.resizable:                 true",
    "*gain.orientation:               horizontal",
    "*modeLabel.label:                Input mode:",
    "*modeLabel.fromVert:             gainLabel",
    "*mode.label:                     Microphone",
    "*mode.fromHoriz:                 modeLabel",
    "*mode.fromVert:                  gainLabel",
    NULL
};

static void
fatalError(message, arg)
char           *message,
               *arg;
{
    fprintf(stderr, message, arg);
    fprintf(stderr, "\n");
    exit(1);
}

static void
quitCB(w, g, call_data)
Widget          w;
GlobalDataPtr   g;
XtPointer       call_data;
{
    exit(0);
}

static void
setDevice(g)
GlobalDataPtr   g;
{
    AuDeviceAttributes *da = &g->da[g->deviceNum];

    AuSetDeviceAttributes(g->aud, AuDeviceIdentifier(da),
			  AuDeviceChangableMask(da) &
			  (AuCompDeviceGainMask | AuCompDeviceLineModeMask),
			  da, NULL);
}

static void
showDevice(g)
GlobalDataPtr   g;
{
    char            buf[20];
    Boolean         modeEnable,
                    gainEnable;
    AuDeviceAttributes *da = &g->da[g->deviceNum];

    XtVaSetValues(g->device, XtNlabel, AuDeviceDescription(da)->data, NULL);

    g->currentGain = AuFixedPointRoundUp(AuDeviceGain(da));
    sprintf(buf, GAIN_FORMAT, g->currentGain);
    XtVaSetValues(g->gainLabel, XtNlabel, buf, NULL);
    XawScrollbarSetThumb(g->gain, (float) g->currentGain / MAX_GAIN, -1.0);

    modeEnable = AuDeviceChangableMask(da) & AuCompDeviceLineModeMask ?
	True : False;

    XtVaSetValues(g->modeLabel, XtNsensitive, modeEnable, NULL);
    XtVaSetValues(g->mode, XtNsensitive, modeEnable, NULL);

    gainEnable = AuDeviceChangableMask(da) & AuCompDeviceGainMask ?
	True : False;

    XtVaSetValues(g->gainLabel, XtNsensitive, gainEnable, NULL);
    XtVaSetValues(g->gain, XtNsensitive, gainEnable, NULL);

    if (modeEnable)
	if (AuDeviceLineMode(da) == AuDeviceLineModeHigh)
	    XtVaSetValues(g->mode, XtNlabel, "Microphone", NULL);
	else
	    XtVaSetValues(g->mode, XtNlabel, "Line", NULL);
}

static void
modeCB(w, g, call_data)
Widget          w;
GlobalDataPtr   g;
XtPointer       call_data;
{
    Boolean         mode;
    AuDeviceAttributes *da = &g->da[g->deviceNum];

    XtCallActionProc(w, "reset", NULL, NULL, 0);

    if (AuDeviceLineMode(da) == AuDeviceLineModeLow)
    {
	XtVaSetValues(g->mode, XtNlabel, "Microphone", NULL);
	AuDeviceLineMode(da) = AuDeviceLineModeHigh;
    }
    else
    {
	XtVaSetValues(g->mode, XtNlabel, "Line", NULL);
	AuDeviceLineMode(da) = AuDeviceLineModeLow;
    }

    setDevice(g);
}

static void
queryCB(w, g, call_data)
Widget          w;
GlobalDataPtr   g;
XtPointer       call_data;
{
    AuFreeDeviceAttributes(g->aud, g->numDevices, g->da);
    g->da = AuListDevices(g->aud, 0, NULL, &g->numDevices, NULL);
    showDevice(g);
}

static void
menuCB(w, g, call_data)
Widget          w;
GlobalDataPtr   g;
XtPointer       call_data;
{
    int             i;
    String          string;

    XtVaGetValues(w, XtNlabel, &string, NULL);
    XtVaSetValues(g->device, XtNlabel, string, NULL);

    for (i = 0; i < g->numDevices; i++)
	if (!strcmp(string, AuDeviceDescription(&g->da[i])->data))
	    break;

    g->deviceNum = i;
    showDevice(g);
}

static void
setGain(g, gain, setThumb)
GlobalDataPtr   g;
int             gain;
Boolean         setThumb;
{
    if (gain < MIN_GAIN)
	gain = MIN_GAIN;
    else if (gain > MAX_GAIN)
	gain = MAX_GAIN;

    if (gain != g->currentGain)
    {
	char            buf[20];

	g->currentGain = gain;
	sprintf(buf, GAIN_FORMAT, g->currentGain);
	XtVaSetValues(g->gainLabel, XtNlabel, buf, NULL);

	if (setThumb)
	    XawScrollbarSetThumb(g->gain, (float) g->currentGain / MAX_GAIN,
				 -1.0);

	AuDeviceGain(&g->da[g->deviceNum]) =
	    AuFixedPointFromSum(g->currentGain, 0);

	setDevice(g);
    }
}

static void
gainScrollCB(w, g, position)
Widget          w;
GlobalDataPtr   g;
int             position;
{
    setGain(g, g->currentGain + (position > 0 ? -1 : 1), True);
}

static void
gainJumpCB(w, g, percent)
Widget          w;
GlobalDataPtr   g;
float          *percent;
{
    setGain(g, (int) (*percent * MAX_GAIN), False);
}

static void
createWidgets(g)
GlobalDataPtr   g;
{
    int             i;
    Widget          w;

    MakeWidget(g->form, g->top, formWidgetClass, "form");

    MakeCommandButton(g->query, g->form, "query", queryCB);

    g->menu = XtCreatePopupShell("menu", simpleMenuWidgetClass, g->form,
				 NULL, 0);

    g->da = AuListDevices(g->aud, 0, NULL, &g->numDevices, NULL);

    if (!g->numDevices)
	fatalError("no devices");

    for (i = 0; i < g->numDevices; i++)
    {
	MakeWidget(w, g->menu, smeBSBObjectClass,
		   AuDeviceDescription(&g->da[i])->data);
	XtAddCallback(w, XtNcallback, menuCB, g);
    }

    MakeWidget(g->menuButton, g->form, menuButtonWidgetClass, "devices");

    MakeCommandButton(g->quit, g->form, "quit", quitCB);

    MakeLabel(g->device, g->form, "deviceLabel");

    MakeLabel(g->gainLabel, g->form, "gainLabel");
    MakeWidget(g->gain, g->form, scrollbarWidgetClass, "gain");
    XtAddCallback(g->gain, XtNscrollProc, gainScrollCB, g);
    XtAddCallback(g->gain, XtNjumpProc, gainJumpCB, g);

    MakeLabel(g->modeLabel, g->form, "modeLabel");
    MakeWidget(g->mode, g->form, toggleWidgetClass, "mode");
    XtAddCallback(g->mode, XtNcallback, modeCB, g);
}

static void
alignWidgets(g)
GlobalDataPtr   g;
{
    Dimension       w,
                    w1;
    Position        x,
                    x1;
    int             d;

    XtVaGetValues(g->mode, XtNwidth, &w, XtNx, &x, NULL);
    XtVaSetValues(g->gain, XtNwidth, w, NULL);

    XtVaGetValues(g->quit, XtNx, &x1, XtNhorizDistance, &d, XtNwidth, &w1,
		  NULL);
    XtVaSetValues(g->quit, XtNhorizDistance, w - w1 - (x1 - d - x), NULL);
}

int
main(argc, argv)
int             argc;
char          **argv;
{
    GlobalDataRec   globals;
    GlobalDataPtr   g = &globals;
    XtAppContext    appContext;
    char           *audioServer = NULL;

    g->top = XtVaAppInitialize(&appContext, APP_CLASS, NULL, ZERO,
			       &argc, argv, defaultResources, NULL, 0);

    if (argc == 3)
	if (!strncmp(argv[1], "-a", 2))
	    audioServer = argv[2];
	else
	    fatalError(USAGE);
    else if (argc != 1)
	fatalError(USAGE);

    if (!(g->aud = AuOpenServer(audioServer, 0, NULL, 0, NULL, NULL)))
	fatalError("Can't connect to audio server");

    g->dpy = XtDisplay(g->top);
    g->currentGain = 0;

    createWidgets(g);
    XtRealizeWidget(g->top);
    alignWidgets(g);

    g->deviceNum = 0;
    showDevice(g);

    AuXtAppAddAudioHandler(appContext, g->aud);
    XtAppMainLoop(appContext);
}
