/***********************************************************
Copyright 1991 by Stichting Mathematisch Centrum, Amsterdam, The
Netherlands.

                        All Rights Reserved

Permission to use, copy, modify, and distribute this software and its 
documentation for any purpose and without fee is hereby granted, 
provided that the above copyright notice appear in all copies and that
both that copyright notice and this permission notice appear in 
supporting documentation, and that the names of Stichting Mathematisch
Centrum or CWI not be used in advertising or publicity pertaining to
distribution of the software without specific, written prior permission.

STICHTING MATHEMATISCH CENTRUM DISCLAIMS ALL WARRANTIES WITH REGARD TO
THIS SOFTWARE, INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND
FITNESS, IN NO EVENT SHALL STICHTING MATHEMATISCH CENTRUM BE LIABLE
FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT
OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.

******************************************************************/

/* Record SGI audio data and output it to Sparc uLAW format,
   converting the sampling size and encoding on the fly.
   This writes to stdout -- use rsh to pipe it to /dev/audio on a Sparc.
   No Sun audio header is added.
   
   Caveats:
   - the program gets the input sampling rate when it starts;
     if you change the sampling rate, better restart this program
   - since it uses an audio port, of which there can only be 4 open
     at a time, running several copies is not a good idea;
     see broadcast.c and radio.c for a distribution mechanism.

  To do:
  - support the other three sampling rates
*/

#include <stdio.h>
#include <audio.h>

int inrate;

#define OUTRATE	8000
#define INBUFSIZE 48000

main(argc, argv)
	int argc;
	char **argv;
{
	short inbuf[INBUFSIZE];
	char outbuf[8000];
	ALconfig c;
	ALport p;
	int n;
	int insamps;
	int sts = 0;
	
	checkinrate();
	insamps = calcinsamps();
	
	c = ALnewconfig();
	if (c == NULL) {
		perror("ALnewconfig");
		exit(1);
	}
	ALsetwidth(c, AL_SAMPLE_16);
	ALsetchannels(c, AL_MONO);
	
	p = ALopenport(argv[0], "r", c);
	if (p == NULL) {
		perror("ALopenport");
		exit(1);
	}

	initcvt();

	for (;;) {
		checkinrate();
		insamps = calcinsamps();
		ALreadsamps(p, (void *)inbuf, insamps);
		n = convert(inbuf, insamps, outbuf);
		if (write(1, outbuf, n) != n) {
			perror("write error");
			sts = 1;
			break;
		}
	}

	exit(sts);
}

int calcinsamps()
{
	int insamps = inrate/2;
	if (insamps > INBUFSIZE)
		insamps = INBUFSIZE;
	return insamps;
}

checkinrate()
{
	inrate = getinrate();
	if ((inrate/OUTRATE)*OUTRATE != inrate) {
		fprintf(stderr,
			"current sampling rate (%d) is not a multiple of %d\n",
			inrate, OUTRATE);
		exit(1);
	}
}

int getinrate()
{
	long PVbuffer[2];
	
	PVbuffer[0] = AL_INPUT_RATE;
	ALgetparams(AL_DEFAULT_DEVICE, PVbuffer, 2);
	return PVbuffer[1];
}

/* Convert a bufferful of data from SGI to uLAW format
   - inrate samples/sec --> 8000 samples/sec
   - 16 bit linear encoding --> 8 bit uLAW encoding
*/

extern unsigned char *cvtvec; /* Forward */

int convert(inbuf, n, outbuf)
	short *inbuf;
	int n;
	unsigned char *outbuf;
{
	register short *inp = inbuf;
	register short *inend = inbuf + n;
	register int di = inrate/OUTRATE;
	register int x;
	register unsigned char *outp = outbuf;

	while (inp < inend) {
		switch (di) {
		case 1:
			*outp++ = cvtvec[*inp++ >> 2];
			break;
		case 2:
			x = *inp++;
			x += *inp++;
			*outp++ = cvtvec[x >> 3];
			break;
		case 4:
			x = *inp++;
			x += *inp++;
			x += *inp++;
			x += *inp++;
			*outp++ = cvtvec[x >> 4];
			break;
		case 6:
			x = *inp++;
			x += *inp++;
			x += *inp++;
			x += *inp++;
			x += *inp++;
			x += *inp++;
			*outp++ = cvtvec[x / 24];
			break;
		}
	}

	return outp - outbuf;
}


/* convert two's complement ch (1+13 bits) into uLAW format (8 bits) */

unsigned int cvt(ch)
int ch;
{
	int mask;

	if (ch < 0) {
		ch = -ch;
		mask = 0x7f;
	} else {
		mask = 0xff;
	}

	if (ch < 32) {
		ch = 0xF0 | 15 - (ch / 2);
	} else if (ch < 96) {
		ch = 0xE0 | 15 - (ch - 32) / 4;
	} else if (ch < 224) {
		ch = 0xD0 | 15 - (ch - 96) / 8;
	} else if (ch < 480) {
		ch = 0xC0 | 15 - (ch - 224) / 16;
	} else if (ch < 992) {
		ch = 0xB0 | 15 - (ch - 480) / 32;
	} else if (ch < 2016) {
		ch = 0xA0 | 15 - (ch - 992) / 64;
	} else if (ch < 4064) {
		ch = 0x90 | 15 - (ch - 2016) / 128;
	} else if (ch < 8160) {
		ch = 0x80 | 15 - (ch - 4064) /	256;
	} else {
		ch = 0x80;
	}

	return (mask & ch);
}

unsigned char cvttab[1<<14];
unsigned char *cvtvec = &cvttab[1<<13];

initcvt()
{
	int i;
	for (i = -(1<<13); i < (1<<13); i++)
		cvtvec[i] = cvt(i);
}
