#import "PSGraphic.h"
#import "GraphicView.h"
#import "draw.h"
#import <appkit/Application.h>
#import <appkit/Panel.h>
#import <appkit/Pasteboard.h>
#import <dpsclient/wraps.h>
#import <dpsclient/dpsclient.h>
#import <mach.h>
#import <math.h>
#import <string.h>

#define Notify(title, msg) NXRunAlertPanel(title, msg, "OK", NULL, NULL)

const char BAD_BBOX_ERROR[] = "Malformed bounding box";
const char NO_BBOX_ERROR[] = "No bounding box";

@implementation PSGraphic
/*
 * This is a simple class to allow arbitrary PostScript as a Graphic object.
 * For efficiency's sake, the image is cached and redrawn only when the
 * image changes size.
 *
 * The cache instance variable is a pointer to the cache window.
 * data is the ASCII PostScript.  length is the number of bytes in data.
 * bbox is the bounding box of the original PostScript.
 */

/* Private methods */

- (BOOL)readPSFromStream:(NXStream *)stream
/*
 * Gets the bounding box from the conforming PostScript comments
 * being careful to skip over included PostScript files.
 */
{
    char *dp;
    int subfiles = 0, matches = 0;

    NXGetMemoryBuffer(stream, &data, &length, &maxlen);

    if (length) {
	dp = data;
	while (dp && !matches) {
	    if (*dp++ == '%' && *dp++ == '%') {
		if (subfiles && *dp == 'E') {
		    if (!strncmp(dp, "EndFile", 7)) subfiles--;
		} else if (*dp == 'B') {
		    if (!strncmp(dp, "BeginFile:", 10)) {
			subfiles++;
		    } else if (!subfiles) {
			matches = sscanf(dp, "BoundingBox: %f %f %f %f\n",
			    &bbox[0], &bbox[1], &bbox[2], &bbox[3]);
			if ((matches && matches != 4) ||
			    (matches == 4 &&
			    (bbox[2] == bbox[0] || bbox[3] == bbox[1]))) {
			    Notify("Load PostScript", BAD_BBOX_ERROR);
			    matches = 1;
			}
		    }
		}
	    }
	    dp = strchr(dp, '\n');
	    if (dp) dp++;
	}
	if (!matches) Notify("Load PostScript", NO_BBOX_ERROR);
    }

    if (matches != 4) data = NULL;

    return data ? YES : NO;
}

/* Factory methods */

+ newFromStream:(NXStream *)stream
/*
 * Sucks the PostScript out of the given stream and into a new Graphic object.
 */
{
    self = [super new];
    if ([self readPSFromStream:stream]) {
	bounds.origin.x = bounds.origin.y = 0.0;
	bounds.size.width = floor(bbox[2] - bbox[0] + 0.99);
	bounds.size.height = floor(bbox[3] - bbox[1] + 0.99);
    } else {
	[self free];
	self = nil;
    }
    return self;
}

- free
/*
 * The data is vm_allocated and vm_deallocated so that when
 * DPSWritePostScript() is called, the data is passed using mach VM
 * primitives (i.e. the data itself is not transferred to the server,
 * just the pointer to the page in memory that the data starts).
 * This makes drawing very large images much faster since much less
 * data movement happens.
 */ 
{
    [cache free];
    if (data && maxlen) vm_deallocate(task_self(), data, maxlen);
    return [super free];
}

/* Methods overridden from superclass */

- (BOOL)isOpaque
{
    return NO;
}

- (float)naturalAspectRatio
{
    return ((bbox[2] - bbox[0]) / (bbox[3] - bbox[1]));
}

#define isBig(length) (length > 8000)

static BOOL fitOnScreen(NXRect *frame)
/*
 * Centers the given frame in the center of the screen.
 * Used to have the imaging window appear there.
 */
{
    NXSize screenSize;

    [NXApp getScreenSize:&screenSize];
    frame->origin.x = floor((screenSize.width - frame->size.width) / 2.0);
    frame->origin.y = floor((screenSize.height - frame->size.height) / 2.0);

    return (frame->origin.x > 0.0 && frame->origin.y > 0.0);
}

- draw
/*
 * This method is not as complicated as it might seem.
 * If the current drawing status resizing, then a box is drawn.
 * It would be far too expensive to continually reimage the PostScript
 * every time the bounding box is change (especially for large PostScript
 * streams).
 *
 * If the status is not resizing, then one of two things happens:
 *
 * 1. If the bounds of the PSGraphic are the same as the cache's, then
 *    all that is necessary is to composite the cache contents.
 * 2. If the bounds is different, then the cache is invalid, and
 *    the PostScript is reimaged into the cache, and draw is called again.
 *
 * If the PostScript is fairly large, then to give the user some useful
 * feedback, the cache window is actually displayed on-screen and, since
 * it is NX_RETAINED, the user can actually watch the PostScript being
 * imaged.
 *
 * There is a slight difference if the NXDrawingStatus is NX_DRAWING
 * since, in that case, the drawing is at origin (0,0) (in the cache).
 * If NXDrawingStatus is not NX_DRAWING then the origin is the origin
 * of the PSGraphic.  Thus the different PSStartFooCustom wraps.
 */
{
    char unique[50];
    BOOL showing = NO;
    NXRect cacheFrame;

    if (bounds.size.width < 1.0 || bounds.size.height < 1.0) return self;

    if (DrawStatus == Resizing) {
	PSsetgray(NX_DKGRAY);
	PSsetlinewidth(0.0);
	PSrectstroke(bounds.origin.x, bounds.origin.y,
		     bounds.size.width, bounds.size.height);
	return self;
    }

    if (cache && NXDrawingStatus == NX_DRAWING) {
	[[cache contentView] getFrame:&cacheFrame];
	if (floor(bounds.size.width+1.0) == cacheFrame.size.width &&
	    floor(bounds.size.height+1.0) == cacheFrame.size.height) {
	    PScomposite(cacheFrame.origin.x, cacheFrame.origin.y,
			floor(bounds.size.width), floor(bounds.size.height),
			[cache gState],
			floor(bounds.origin.x), floor(bounds.origin.y),
			NX_SOVER);
	    return self;
	}
    }

    if (NXDrawingStatus == NX_DRAWING) {
	[cache free];
	cacheFrame.size.width = floor(bounds.size.width+1.0);
	cacheFrame.size.height = floor(bounds.size.height+1.0);
	showing = fitOnScreen(&cacheFrame) && isBig(length);
	cache = [Window newContent:&cacheFrame
			     style:(showing ? NX_TITLEDSTYLE : NX_PLAINSTYLE)
			   backing:NX_RETAINED
			buttonMask:0
			     defer:NO];
	if (showing) {
	    [cache reenableDisplay];
	    [cache setTitle:"Imaging PostScript ..."];
	    [cache orderFront:self];
	}
	[[cache contentView] lockFocus];
    }

    sprintf(unique, "-save-draw%d%d-",
	NXDrawingStatus, [[NXApp pasteboard] changeCount]);

    DPSPrintf(DPSGetCurrentContext(), "%%%%BeginFile: <Draw subdocument>\n");
    if (NXDrawingStatus == NX_DRAWING) {
	PSStartDrawingCustom(unique, bounds.size.width, bounds.size.height,
		      0.0 - bbox[0], 0.0 - bbox[1],
		      bounds.size.width / (bbox[2] - bbox[0]),
		      bounds.size.height / (bbox[3] - bbox[1]));
    } else {
	PSStartCopyingCustom(unique, bounds.size.width, bounds.size.height,
		      bounds.origin.x, bounds.origin.y,
		      0.0 - bbox[0],  0.0 - bbox[1],
		      bounds.size.width / (bbox[2] - bbox[0]),
		      bounds.size.height / (bbox[3] - bbox[1]));
    }
    DPSWritePostScript(DPSGetCurrentContext(), data, length);
    PSEndCustom(unique, bbox[0], bbox[1],
		(bbox[2] - bbox[0]) / bounds.size.width,
		(bbox[3] - bbox[1]) / bounds.size.height);
    DPSPrintf(DPSGetCurrentContext(), "%%%%EndFile\n");

    if (NXDrawingStatus == NX_DRAWING) {
	[[cache contentView] unlockFocus];
	if (showing) [cache orderOut:self];
	[self draw];
    }

    return self;
}

- write:(NXTypedStream *)stream
/*
 * All that is needed to archive the PSGraphic is its bounding box,
 * the length and the data itself.
 */
{
    [super write:stream];
    NXWriteTypes(stream,"ffffi",&bbox[0],&bbox[1],&bbox[2],&bbox[3],&length);
    NXWriteArray(stream, "c", length, data);
    return self;
}

- read:(NXTypedStream *)stream
{
    [super read:stream];
    NXReadTypes(stream,"ffffi",&bbox[0],&bbox[1],&bbox[2],&bbox[3],&length);
    vm_allocate(task_self(), &data, length, YES);
    NXReadArray(stream, "c", length, data);
    return self;
}

@end

