#import "Group.h"
#import <dpsclient/wraps.h>
#import <objc/List.h>

@implementation Group : Graphic
/*
 * This Graphic is used to create heirarchical groups of other Graphics.
 * It simply keeps a list of all the Graphics in the group and resizes
 * and translates them as the Group object itself is resized and moved.
 * It also passes messages sent to the Group onto its members.
 */

/* Factory method */

+ newList:list
/*
 * Creates a new grouping with list containing the list of Graphics
 * in the group.  Groups of Groups is perfectly allowable.  We have
 * to keep track of the largest linewidth in the group as well as
 * whether any of the elements of the group have arrows since both
 * of those attributes affect the extended bounds of the Group.
 */
{
    id g;
    int i;
    NXRect r;

    self = [super new];

    i = [list count];
    g = [list objectAt:--i];
    [g getBounds:&bounds];
    gFlags.arrow = [g lineArrow];
    linewidth = [g lineWidth];
    while (i) {
	g = [list objectAt:--i];
	[g getBounds:&r];
	if (!r.size.width) r.size.width = 1.0;
	if (!r.size.height) r.size.height = 1.0;
	NXUnionRect(&r, &bounds);
	if (!gFlags.arrow && [g lineArrow]) gFlags.arrow = [g lineArrow];
	if ([g lineWidth] > linewidth) linewidth = [g lineWidth];
    }

    components = list;
    lastRect = bounds;

    return self;
}

- free
{
    [components free];
    return [super free];
}

/* Public methods */

- list
{
    return components;
}

- draw
/*
 * Individually scales and translates each Graphic in the group and draws
 * them.  This is done this way so that ungrouping is trivial.
 */
{
    id g;
    int i;
    NXRect b;
    BOOL changed;
    float sx = 1.0, sy = 1.0, tx, ty;

    if (bounds.size.width < 1.0 || bounds.size.height < 1.0) return self;

    changed = (lastRect.origin.x != bounds.origin.x ||
	       lastRect.origin.y != bounds.origin.y ||
	       lastRect.size.width != bounds.size.width ||
	       lastRect.size.height != bounds.size.height);
    if (changed) {
	sx = bounds.size.width / lastRect.size.width;
	sy = bounds.size.height / lastRect.size.height;
    }
    i = [components count];
    while (i) {
	g = [components objectAt:--i];
	if (changed) {
	    [g getBounds:&b];
	    tx = (bounds.origin.x +
		    ((b.origin.x - lastRect.origin.x) /
		     lastRect.size.width * bounds.size.width)) -
		     b.origin.x;
	    ty = (bounds.origin.y +
		    ((b.origin.y - lastRect.origin.y) /
		     lastRect.size.height * bounds.size.height)) -
		     b.origin.y;
	    b.origin.x = b.origin.x + tx;
	    b.origin.y = b.origin.y + ty;
	    b.size.width = b.size.width * sx;
	    b.size.height = b.size.height * sy;
	    [g setBounds:&b];
	}
	[g setGraphicsState];	/* does a gsave */
	[g draw];
	PSgrestore();
    }
    lastRect = bounds;

    return self;
}

- (BOOL)hit:(const NXPoint *)point
/*
 * Gets a hit if any of the items in the group gets a hit.
 */
{
    id g;
    int i;
    NXPoint p;
    float px, py;

    if ([super hit:point]) {
	p = *point;
	px = (p.x - bounds.origin.x) / bounds.size.width;
	p.x = px * lastRect.size.width + lastRect.origin.x;
	py = (p.y - bounds.origin.y) / bounds.size.height;
	p.y = py * lastRect.size.height + lastRect.origin.y;
	i = [components count];
	while (i) {
	    g = [components objectAt:--i];
	    if ([g hit:&p]) return YES;
	}
    }

    return NO;
}

/* Group must override all the setting routines to forward to components */

- setLineWidth:(const float *)value
{
    [components makeObjectsPerform:@selector(setLineWidth:) with:(id)value];
    return self;
}

- setGray:(const float *)value
{
    [components makeObjectsPerform:@selector(setGray:) with:(id)value];
    return self;
}

- setFill:(int)mode
{
    [components makeObjectsPerform:@selector(setFill:) with:(id)mode];
    return self;
}

- setFramed:(BOOL)flag
{
    [components makeObjectsPerform:@selector(setFramed:) with:(id)flag];
    return self;
}

- setLineCap:(int)value
{
    [components makeObjectsPerform:@selector(setLineCap:) with:(id)value];
    return self;
}

- setLineArrow:(int)value
{
    [components makeObjectsPerform:@selector(setLineArrow:) with:(id)value];
    return self;
}

- setLineJoin:(int)value
{
    [components makeObjectsPerform:@selector(setLineJoin:) with:(id)value];
    return self;
}

- write:(NXTypedStream *)stream
/*
 * Just writes out the components.
 */
{
    [super write:stream];
    NXWriteTypes(stream, "@", &components);
    return self;
}

- read:(NXTypedStream *)stream
{
    [super read:stream];
    NXReadTypes(stream, "@", &components);
    lastRect = bounds;
    return self;
}

@end

