#import "DrawApp.h"
#import "DrawDocument.h"
#import "GraphicView.h"
#import "draw.h"
#import <appkit/appkit.h>
#import <appkit/Cursor.h>
#import <appkit/Matrix.h>
#import <appkit/Menu.h>
#import <appkit/MenuCell.h>
#import <appkit/OpenPanel.h>
#import <appkit/Pasteboard.h>
#import <appkit/Text.h>
#import <appkit/defaults.h>
#import <objc/hashtable.h>	/* for NXCopyStringBuffer() */
#import <objc/List.h>
#import <sys/param.h>
#import <ldsyms.h>
#import <sys/loader.h>
#import <libc.h>
#import <string.h>

const int DrawVersion = 30;	/* minor version of the program */

@implementation DrawApp : Application
/*
 * This class is used primarily to handle the opening of new documents
 * and other application-wide activity (such as responding to messages from
 * the tool palette).  It listens for requests from the Workspace Manager
 * to open a draw-format file as well as target/action messages from the
 * New and Open... menu items.  It also keeps the menus in sync by
 * fielding the menu items' updateActions.
 */

/* Private C functions used to implement methods in this class. */

static id initMenu(id menu)
/*
 * Sets the updateAction for every menu item which sends to the
 * First Responder (i.e. their target is nil).  When autoupdate is on,
 * every event will be followed by an update of each of the menu items
 * which is visible.  This keep all unavailable menu items dimmed out
 * so that the user knows what options are available at any given time.
 * Returns the activate menu if is found in this menu.
 */ 
{
    int count;
    const char *s;
    id matrix, cell;
    id matrixTarget, cellTarget;
    id activateMenu = nil;

    matrix = [menu itemList];
    matrixTarget = [matrix target];

    count = [matrix cellCount];
    while (count--) {
	cell = [matrix cellAt:count :0];
	cellTarget = [cell target];
	if (!matrixTarget && !cellTarget) {
	    [cell setUpdateAction:@selector(menuItemUpdate:) forMenu:menu];
	} else if ([cell hasSubmenu]) {
	    initMenu(cellTarget);
	}
	s = [cell title];
	if (s && !strcmp(s, "Activate")) {
	    activateMenu = cellTarget;
	}
    }

    return activateMenu;
}

static id documentInWindow(id window)
/*
 * Checks to see if the passed window's delegate is a DrawDocument.
 * If it is, it returns that document, otherwise it returns nil.
 */
{
    id document = [window delegate];
    return [document isKindOf:[DrawDocument class]] ? document : nil;
}

static id findDocument(const char *name)
/*
 * Searches the window list looking for a DrawDocument with the specified name.
 * Returns the window containing the document if found.
 * If name == NULL then the first document found is returned.
 */
{
    int count;
    id document, window, windowList;

    windowList = [NXApp windowList];
    count = [windowList count];
    while (count--) {
	window = [windowList objectAt:count];
	document = documentInWindow(window);
	if (document && (!name || !strcmp([document filename], name))) {
	    return window;
	}
    }

    return nil;
}

static BOOL loadActivateMenu()
/*
 * Adds all open documents to the activate menu.
 * Returns whether any were actually loaded.
 */
{
    int count;
    BOOL foundOne = NO;
    id document, window, windowList;

    windowList = [NXApp windowList];
    count = [windowList count];
    while (count--) {
	window = [windowList objectAt:count];
	document = documentInWindow(window);
	if (document) {
	    [NXApp addDocument:[document filename]];
	    foundOne = YES;
	}
    }

    return foundOne;
}

static id openFile(const char *directory, const char *name, BOOL display)
/*
 * Opens a file with the given name in the specified directory.
 * If we already have that file open, it is ordered front.
 * Returns the document if successful, nil otherwise.
 */
{
    id window;
    char buffer[MAXPATHLEN+1], path[MAXPATHLEN+1];

    if (name && *name) {
	if (!directory) {
	    directory = ".";
	} else if (*directory != '/') {
	    strcpy(buffer, "./");
	    strcat(buffer, directory);
	    directory = buffer;
	}
	if (!chdir(directory) && getwd(path)) {
	    strcat(path, "/");
	    strcat(path, name);
	    window = findDocument(path);
	    if (window) {
		if (display) [window makeKeyAndOrderFront:window];
		return [window delegate];
	    } else {
		return [DrawDocument newFromFile:path];
	    }
	} else {
	    NXRunAlertPanel("Open", "Invalid path: %s", "OK", NULL, NULL, directory);
	}
    }

    return nil;
}

/* Public methods */

+ initialize
/*
 * Initializes the defaults.
 */
{
    const NXDefaultsVector DrawDefaults = {
	{ "KnobWidth", "5" },
	{ "KnobHeight", "5" },
	{ "KeyMotionDelta", "1"},
	{ "Quit", NULL },
	{ NULL, NULL }
    };

    NXRegisterDefaults("PatchWork", DrawDefaults);

    return self;
}

+ new
/*
 * PSInit() defines all the PostScript defs we use (see draw.psw).
 * setAutoupdate:YES means that updateWindows will be called after
 * every event is processed (this is how we keep our inspector and
 * our menu items up to date).
 */
{
    self = [super new];
    PSInit();
    [self setAutoupdate:YES];
    return self;
}

/* General application status and information querying/modifying methods. */

- currentGraphic
{
    return currentGraphic;
}

- currentDocument
{
    return documentInWindow(mainWindow);
}

/* Application-wide shared panels */

- saveToPanel
/*
 * Returns a SavePanel with the accessory view which allows the user to
 * pick which type of file she wants to save.
 */
{
    id savepanel = [SavePanel new];
    [savepanel setAccessoryView:savePanelAccessory];
    [spamatrix selectCellAt:0 :0];
    [savepanel setRequiredFileType:"PatchWork"];
    return savepanel;
}

- saveAsPanel
/*
 * Returns a regular SavePanel with "draw" as the required file type.
 */
{
    id savepanel = [SavePanel new];
    [savepanel setAccessoryView:nil];
    [savepanel setRequiredFileType:"PatchWork"];
    return savepanel;
}

- gridInspector
/*
 * Returns the application-wide inspector for a document's grid.
 * Note that if we haven't yet loaded the GridView panel, we do it.
 * The instance variable gridInspector is set in setGridInspector:
 * since it is set as an outlet of the owner (self, i.e. DrawApp).
 */
{
    if (!gridInspector) {
	[self loadNibSection:"GridView.nib" owner:self withNames:NO];
    }
    return gridInspector;
}

- inspectorPanel
/*
 * Returns the application-wide inspector for Graphics.
 */
{
    return inspectorPanel;
}

- orderFrontInspectorPanel:sender
/*
 * Creates the inspector panel if it doesn't exist, then orders it front.
 */
{
    if (!inspectorPanel) {
	inspectorPanel =
	    [self loadNibSection:"InspectorPanel.nib" owner:self withNames:NO];
    }
    [inspectorPanel orderFront:self];
    return self;
}

- myDeviceInspectorView /*KHL */
{
	if(!myDeviceInspectorView)
		[self loadNibSection:"deviceInspectorView.nib" owner:self withNames:NO];
	return myDeviceInspectorView;
}

/* Target/Action methods */

- help:sender
{
    int size;
    char *data;
    NXStream *stream;
    id window, document;

    if (!(window = findDocument("/tmp/Help"))) {
	data = getsectdata("HELP", "document", &size);
	if (data) {
	    stream = NXOpenMemory(data, size, NX_READONLY);
	    if (stream) {
		document = [DrawDocument newFromStream:stream];
		[document setName:"Help" andDirectory:"/tmp"];
		[self addDocument:[document filename]];
	    }
	    NXClose(stream);
	}
    } else {
	[window makeKeyAndOrderFront:self];
    }

    return self;
}

- info:sender
/*
 * Brings up the information panel.
 */
{
    char buf[20];

    if (!infoPanel) {
	infoPanel =
	    [self loadNibSection:"InfoPanel.nib" owner:self withNames:NO];
	sprintf(buf, "(v%2d)", DrawVersion);
	[version setStringValue:buf];
    }

    [infoPanel orderFront:self];

    return self;
}

- compileInfo
/*
 * Brings up the compiler information panel.
 */
{
    if (!compileInfoPanel) {
	compileInfoPanel =
	    [self loadNibSection:"compileInfo.nib" owner:self withNames:NO];
    }

//	[[compileInfo docView] setText:""];
    [compileInfoPanel orderFront:self];

    return compileInfo;
}

- print:sender
{
    return [[[self currentDocument] view] printPSCode:sender];
}

- new:sender
/*
 * Called by pressing New in the Window menu.
 */
{
    [DrawDocument new];
    return self;
}

- open:sender
/*
 * Called by pressing Open... in the Window menu.
 */
{
    const char *directory;
    const char *const *files;
    static const char *const drawType[2] = {"PatchWork", NULL};
    id openpanel = [[OpenPanel new] allowMultipleFiles:YES];

    directory = [[self currentDocument] directory];
    if (directory && (*directory == '/')) {
	[openpanel setDirectory:directory];
    }
    if ([openpanel runModalForTypes:drawType]) {
	files = [openpanel filenames];
	directory = [openpanel directory];
	while (files && *files) {
	    openFile(directory, *files, YES);
	    files++;
	}
    }

    return self;
}

- terminate:sender
/*
 * Overridden to be sure all documents get an opportunity to be saved
 * before exiting the program.
 */
{
    int count;
    id window, document;

    count = [windowList count];
    while (count--) {
	window = [windowList objectAt:count];
 	document = [window delegate];
	if ([document respondsTo:@selector(windowWillClose:)]) {
	    if ([document windowWillClose:window]) {
		[window close];
	    } else {
		return self;
	    }
	}
    }

    return [super terminate:sender];
}
    
/*
 * Application object delegate methods.
 * Since we don't have an application delegate, messages that would
 * normally be sent there are sent to the Application object itself instead.
 */

- appDidInit:sender
/*
 * Makes the tool palette not ever become the key window.
 * Check for files to open specified on the command line.
 * Initialize the menus and get a handle on the activate menu.
 * Add all opened documents to the activate menu.
 * If there are no open documents, then open a blank one.
 */
{
    int i;
    char buffer[MAXPATHLEN+1];
    char *directory, *name, *ext;

    [[tools window] removeFromEventMask:NX_KEYDOWNMASK|NX_KEYUPMASK];
    [[tools window] orderFront:self];
    [[cSoundTools window] removeFromEventMask:NX_KEYDOWNMASK|NX_KEYUPMASK];

    if (NXArgc > 1) {
	for (i = 1; i < NXArgc; i++) {
	    strcpy(buffer, NXArgv[i]);
	    ext = strrchr(buffer, '.');
	    if (!ext || strcmp(ext, ".PatchWork")) strcat(buffer, ".PatchWork");
	    name = strrchr(buffer, '/');
	    if (name) {
		*name++ = '\0';
		directory = buffer;
	    } else {
		name = buffer;
		directory = NULL;
	    }
	    openFile(directory, name, YES);
	}
    }

    activateMenu = initMenu([NXApp mainMenu]);
    if (!loadActivateMenu()) [self new:self];	/* if none opened, open one */

    if (NXGetDefaultValue([self appName], "Quit")) {
	[self activateSelf:YES];
	NXPing();
	[self terminate:self];
    }

    return self;
}

- (int)appOpenFile:(const char *)path type:(const char *)type
/*
 * This method is performed whenever a user double-clicks on an icon
 * in the Workspace Manager representing a Draw program document.
 */
{
    char *name;
    char directory[MAXPATHLEN+1];

    if (type && !strcmp(type, "PatchWork")) {
	strcpy(directory, path);
	name = strrchr(directory, '/');
	if (name) {
	    *name++ = '\0';
	    return (openFile(directory, name, YES) ? YES : NO);
	}
    }

    return NO;
}

- (BOOL)appAcceptsAnotherFile:sender
/*
 * We accept any number of appOpenFile:type: messages.
 */
{
    return YES;
}

/* Listener/Speaker remote methods */

- (int)msgDirectory:(const char **)fullPath ok:(int *)flag
{
    *fullPath = [[self currentDocument] directory];
    if (!*fullPath) *fullPath = [[OpenPanel new] directory];
    if (*fullPath) {
	*flag = YES;
    } else {
	*fullPath = "";
	*flag = NO;
    }
    return 0;
}

- (int)msgVersion:(const char **)aString ok:(int *)flag
{
    char buf[20];

    sprintf(buf, "1.0 (v%2d)", DrawVersion);
    *aString = NXCopyStringBuffer(buf);
    *flag = YES;

    return 0;
}

- (int)msgFile:(const char **)fullPath ok:(int *)flag
{
    const char *file;

    file = [[self currentDocument] filename];
    if (file) {
	*fullPath = file;
	*flag = YES;
    } else {
	*fullPath = "";
	*flag = NO;
    }

    return 0;
}

- (BOOL)shouldRunPrintPanel:sender
/*
 * When NXApp prints, don't bring up the PrintPanel.
 */
{
    return NO;
}

- (int)msgPrint:(const char *)fullPath ok:(int *)flag
{
    BOOL close;
    id document = nil;
    char *directory, *name;
    char path[MAXPATHLEN+1];
    char buffer[MAXPATHLEN+1];

    InMsgPrint = YES;
    strcpy(buffer, fullPath);
    name = strrchr(buffer, '/');
    if (name) {
	*name++ = '\0';
	directory = buffer;
    } else {
	name = buffer;
	directory = NULL;
    }
    if (!chdir(directory) && getwd(path)) {
	strcat(path, "/");
	strcat(path, name);
	document = [findDocument(path) delegate];
    }
    if (document) {
	close = NO;
    } else {
	document = openFile(directory, name, NO);
	close = YES;
    }
    if (document && ![[document view] isEmpty]) {
	[NXApp setPrintInfo:[document printInfo]];
	[[document view] printPSCode:self];
	if (close) [[[document view] window] close];
	*flag = YES;
    } else {
	*flag = NO;
    }
    InMsgPrint = NO;

    return 0;
}

- (int)msgSelection:(const char **)bytes length:(int *)len
    asType:(const char *)aType ok:(int *)flag
{
    int maxlen;
    NXStream *stream;

    if (!strcmp(aType, DrawPBType)) {
	stream = NXOpenMemory(NULL, 0, NX_WRITEONLY);
	if ([[[self currentDocument] view] copySelectionToStream:stream]) {
	    NXGetMemoryBuffer(stream, bytes, len, &maxlen);
	    *flag = YES;
	} else {
	    *flag = NO;
	}
	NXClose(stream);
    } else if (!strcmp(aType, NXPostScriptPboard)) {
	stream = NXOpenMemory(NULL, 0, NX_WRITEONLY);
	if ([[[self currentDocument] view] copySelectionAsPS:stream]) {
	    NXGetMemoryBuffer(stream, bytes, len, &maxlen);
	    *flag = YES;
	} else {
	    *flag = NO;
	}
	NXClose(stream);
    } else if (!strcmp(aType, NXTIFFPboard)) {
	stream = NXOpenMemory(NULL, 0, NX_WRITEONLY);
	if ([[[self currentDocument] view] copySelectionAsTIFF:stream]) {
	    NXGetMemoryBuffer(stream, bytes, len, &maxlen);
	    *flag = YES;
	} else {
	    *flag = NO;
	}
	NXClose(stream);
    } else {
	*flag = NO;
    }

    if (!*flag) {
	*bytes = "";
	*len = 0;
    }

    return 0;
}

- (int)msgCopyAsType:(const char *)aType ok:(int *)flag
{
    if (!strcmp(aType, NXPostScriptPboard) ||
	!strcmp(aType, NXTIFFPboard) ||
	!strcmp(aType, DrawPBType)) {
	*flag = ([[[self currentDocument] view] copy:self] ? YES : NO);
    } else {
	*flag = NO;
    }
    return 0;
}

- (int)msgCutAsType:(const char *)aType ok:(int *)flag
{
    if (!strcmp(aType, NXPostScriptPboard) ||
	!strcmp(aType, NXTIFFPboard) ||
	!strcmp(aType, DrawPBType)) {
	*flag = ([[[self currentDocument] view] cut:self] ? YES : NO);
    } else {
	*flag = NO;
    }
    return 0;
}

- (int)msgPaste:(int *)flag;
{
    *flag = ([[[self currentDocument] view] paste:self] ? YES : NO);
    return 0;
}

/* Global cursor setting */

- cursor
/*
 * This is called by DrawDocument objects who want to set the cursor
 * depending on what the currently selected tool is (as well as on whether
 * the Control key has been pressed indicating that the select tool is
 * temporarily set--see sendEvent:).
 */
{
    id theCursor = nil;
    if (!cursorPushed) theCursor = [[self currentGraphic] cursor];
    return theCursor ? theCursor : NXArrow;
}

- sendEvent:(NXEvent *)event
/*
 * We override this because we need to find out when the control key is down
 * so we can set the arrow cursor so the user knows she is (temporarily) in
 * select mode.
 */
{
    if (event && event->type < NX_KITDEFINED) {	/* mouse or keyboard event */
	if (event->flags & NX_CONTROLMASK) {
	    if (!cursorPushed && currentGraphic) {
		cursorPushed = YES;
		[[self currentDocument] resetCursor];
	    }
	} else if (cursorPushed) {
	    cursorPushed = NO;
	    [[self currentDocument] resetCursor];
	}
    }

    return [super sendEvent:event];
}

/*
 * Target/Action messages to affect the field editor in the main window.
 * This is the places where messages from the Text menu are handled.
 */

static id fieldEditorInMainWindow()
{
    id fe = nil, mainWindow;
    mainWindow = [NXApp mainWindow];
    if (documentInWindow(mainWindow)) {
	fe = [mainWindow getFieldEditor:NO for:NXApp];
    }
    return fe;
}

- setBlackText:sender
{
    [fieldEditorInMainWindow() setSelGray:NX_BLACK];
    return self;
}

- setWhiteText:sender
{
    [fieldEditorInMainWindow() setSelGray:NX_WHITE];
    return self;
}

- setDarkGrayText:sender
{
    [fieldEditorInMainWindow() setSelGray:NX_DKGRAY];
    return self;
}

- setLightGrayText:sender
{
    [fieldEditorInMainWindow() setSelGray:NX_LTGRAY];
    return self;
}

static void setAlignment(int alignment)
{
    NXRect before, after;
    id fe = fieldEditorInMainWindow();
    [fe setAlignment:alignment];
    [fe getFrame:&before];
    [fe calcLine];
    [fe getFrame:&after];
    NXUnionRect(&before, &after);
    [[fe superview] displayFromOpaqueAncestor:&after :1 :NO];
}

- centerText:sender
{
    setAlignment(NX_CENTERED);
    return self;
}

- leftJustifyText:sender
{
    setAlignment(NX_LEFTALIGNED);
    return self;
}

- rightJustifyText:sender
{
    setAlignment(NX_RIGHTALIGNED);
    return self;
}

/* Activate menu updating */

static int cellcmp(const void *first, const void *second)
{
    const char *s1 = [*((id *)first) title];
    const char *s2 = [*((id *)second) title];
    return (s1 && s2) ? strcmp(s1, s2) : 0;
}

- addDocument:(const char *)name
/*
 * Adds the name passed in to the Activate menu (if it isn't already there).
 */
{
    int i;
    id cells;
    const char *s;

    if (!name) return self;

    cells = [[activateMenu itemList] cellList];
    for (i = [cells count] - 1; i >= 0; i--) {
	s = [[cells objectAt:i] title];
	if (s && !strcmp(s, name)) return self;
    }

    [activateMenu addItem:name
	action:@selector(activateDocument:)
	keyEquivalent:0];
    if (cells) {
	qsort(NX_ADDRESS(cells), [cells count], sizeof(id *), cellcmp);
	[activateMenu sizeToFit];
    }

    return self;
}

- removeDocument:(const char *)name
/*
 * Removes the passed name from the Activate menu.
 */
{
    int i, count;
    const char *s;
    id matrix, cells;

    if (!name) return self;
    matrix = [activateMenu itemList];
    cells = [matrix cellList];
    count = [cells count];
    for (i = 0; i < count; i++) {
	s = [[cells objectAt:i] title];
	if (s && !strcmp(s, name)) {
	    [matrix removeRowAt:i andFree:YES];
	    break;
	}
    }
    if (i != count) {
	qsort(NX_ADDRESS(cells), [cells count], sizeof(id *), cellcmp);
	[activateMenu sizeToFit];
    }

    return self;
}

- renameDocument:(const char *)oldName to:(const char *)newName
/*
 * Finds the passed oldName in the Activate menu and changes the
 * title of that menu entry to newName.
 */
{
    int i, count;
    const char *s;
    id cell, cells;

    if (!oldName) return self;
    cells = [[activateMenu itemList] cellList];
    count = [cells count];
    for (i = 0; i < count; i++) {
	cell = [cells objectAt:i];
	s = [cell title];
	if (s && !strcmp(s, oldName)) {
	    [cell setTitle:newName];
	    break;
	}
    }
    if (i != count) {
	qsort(NX_ADDRESS(cells), [cells count], sizeof(id), cellcmp);
	[activateMenu sizeToFit];
    }

    return self;
}

- activateDocument:sender
/*
 * Sent by items in the Activate menu.  Queries back to the sender to
 * find out which document to activate, then activates it.
 */
{
    [findDocument([[sender selectedCell] title]) makeKeyAndOrderFront:self];
    return self;
}

/* Automatic update methods */

- (BOOL)menuItemUpdate:menuCell
/*
 * Method called by all menu items which send their actions to the
 * First Responder.  First, if the object which would respond were the
 * action sent down the responder chain also responds to the message
 * validateCommand:, then it is sent validateCommand: to determine
 * whether that command is valid now, otherwise, if there is a responder
 * to the message, then it is assumed that the item is valid.
 * The method returns YES if the cell has changed its appearance (so that
 * the caller (a Menu) knows to redraw it).
 */
{
    SEL action;
    id responder, target;
    BOOL enable;

    target = [menuCell target];
    enable = [menuCell isEnabled];

    if (!target) {
	action = [menuCell action];
	responder = [self calcTargetForAction:action];
	if ([responder respondsTo:@selector(validateCommand:)]) {
	    enable = [responder validateCommand:menuCell];
	} else {
	    enable = responder ? YES : NO;
	}
    }

    if ([menuCell isEnabled] != enable) {
	[menuCell setEnabled:enable];
	return YES;
    } else {
	return NO;
    }
}

- (BOOL)validateCommand:menuCell
/*
 * NXApp itself is the responder for the Text menu messages, therefore
 * we must validate them here.
 */
{
    id view;
    SEL action = [menuCell action];

    if (action == @selector(centerText:) ||
	action == @selector(leftJustifyText:) ||
	action == @selector(rightJustifyText:) ||
	action == @selector(setBlackText:) ||
	action == @selector(setDarkGrayText:) ||
	action == @selector(setLightGrayText:) ||
	action == @selector(setWhiteText:)) {
	return [fieldEditorInMainWindow() superview] ? YES : NO;
    } else if (action == @selector(print:)) {
	view = [[self currentDocument] view];
	return (view && ![view isEmpty]) ? YES : NO;
    }

    return YES;
}

/*
 * This is a very funky method and tricks of this sort are not generally
 * recommended, but this hack is done so that new Graphic subclasses can
 * be added to the program without changing even one line of code (except,
 * of course, to implement the subclass itself).
 *
 * The objective-C method findClass: is used to find the factory object
 * corresponding to the name of the cell sending the setCurrentGraphic:
 * message (this is set in InterfaceBuilder using the Miscellaneous panel
 * in the Inspector window).
 *
 * Again, this is not recommended procedure, but it illustrates how
 * objective-C can be used to make some funky runtime dependent decisions.
 */

- setCurrentGraphic:sender
/*
 * The sender's name is queried.  If that name corresponds to the name
 * of a class, then that class is set as the currentGraphic.  If not,
 * then the select tool is put into effect.
 */
{
    id cell;
    const char *className;

    cell = [sender selectedCell];
    if (cell) {
	className = NXGetObjectName(cell);
	if (className) {
	    currentGraphic = [self findClass:(char *)className];
	} else {
	    currentGraphic = nil;
	}
	if (!currentGraphic) [tools selectCellAt:0 :0];
	if (!currentGraphic) [cSoundTools selectCellAt:0 :0];
	[[self currentDocument] resetCursor];
    }

    return self;
}

/* Methods used by InterfaceBuilder to set the values of outlets. */

- setTools:anObject
{
    tools = anObject;
    return self;
}

- setCSoundTools:anObject
{
    cSoundTools = anObject;
    return self;
}

- setSavePanelAccessory:anObject
{
    savePanelAccessory = anObject;
    return self;
}

- setSpaMatrix:anObject
{
    spamatrix = anObject;
    return self;
}

- setGridInspector:anObject
{
    gridInspector = anObject;
    return self;
}

- setVersion:anObject
{
    version = anObject;
    return self;
}

- setCompileInfo:anObject
{
    compileInfo = anObject;
    return self;
}

- setMyDeviceInspectorView:anObject /*KHL */
{
    myDeviceInspectorView = anObject;
    return self;
}


@end

