#import "Tiff.h"
#import "GraphicView.h"
#import <appkit/Panel.h>
#import <appkit/nextstd.h>
#import <dpsclient/wraps.h>

#define Notify(title, msg) NXRunAlertPanel(title, msg, "OK", NULL, NULL)

@implementation Tiff
/*
 * This class is much like PSGraphic except that it deals with TIFF
 * (i.e. bits).  See that class for more in-depth comments.
 * Note that although the tiff data is internally dealt with with
 * alpha (if it has any), alpha is never drawn to the screen since
 * the user can't get that effect when she prints the document.
 * HOWEVER, if the view is saved out as TIFF or is put in the
 * Pasteboard, then the alpha can be maintained (this is controlled
 * via the drawWithAlpha flag).
 */
 
/* Private methods and functions */

static BOOL hasAlpha(const NXImageInfo *info)
/*
 * Returns whether the image has alpha in it.
 */
{
    return (info->photoInterp & (NX_ALPHAMASK|NX_MONOTONICMASK));
}

static int sizeOfImage(const NXImageInfo *info)
/*
 * Returns the number of bytes in the image.
 */
{
    return info->height * (info->samplesPerPixel) *
	((info->bitsPerSample * info->width + 7) / 8);
}

static void handleTIFFError(int error)
/*
 * Puts up an Alert to report any errors reading the TIFF data in the stream.
 */
{
    switch (error) {
	case NX_BAD_TIFF_FORMAT:
	    Notify("Load TIFF", "Bad TIFF format.");
	    break;
	case NX_IMAGE_NOT_FOUND:
	    Notify("Load TIFF", "No image found.");
	    break;
	case NX_ALLOC_ERROR:
	    Notify("Load TIFF", "Can't allocate memory.");
	    break;
	case NX_FORMAT_NOT_YET_SUPPORTED:
	    Notify("Load TIFF", "Unsupported format.");
	    break;
	case NX_FILE_IO_ERROR:
	    Notify("Load TIFF", "I/O error.");
	    break;
	case NX_COMPRESSION_NOT_YET_SUPPORTED:
	    Notify("Load TIFF", "Compression format not yet supported.");
	    break;
    }
}

static id createCache(const NXSize *size)
{
    NXRect cacheRect;

    cacheRect.origin.x = 0.0;
    cacheRect.origin.y = 0.0;
    cacheRect.size = *size;

    return [Window newContent:&cacheRect
    			style:NX_PLAINSTYLE
		      backing:NX_RETAINED
		   buttonMask:0
			defer:NO];
}

static BOOL drawWithAlpha = NO;

+ setDrawWithAlpha:(BOOL)flag
/*
 * Sets whether TIFF objects should be composited with or without alpha.
 * Normally, we composite without alpha since printers can't handle alpha.
 * But, if we are writing the TIFF to a file, we might as well include
 * the alpha.
 */
{
    drawWithAlpha = flag;
    return self;
}

- createCache
/*
 * If drawWithAlpha is not true, a trick of compositing the image onto
 * itself with NX_DOVER is used to get rid of any alpha that is in the image.
 */
{
    int length;
    NXRect imageRect;
    unsigned char *alpha;

    if (data) {
	if (!bounds.size.width && !bounds.size.height) {
	    bounds.size.width = (NXCoord)info.width;
	    bounds.size.height = (NXCoord)info.height;
	}
	if (NXDrawingStatus == NX_DRAWING) {
	    [cache free];
	    cache = createCache(&bounds.size);
	    [[cache contentView] lockFocus];
	    imageRect.origin.x = imageRect.origin.y = 0.0;
	} else {
	    imageRect.origin = bounds.origin;
	}

	imageRect.size = bounds.size;
	length = sizeOfImage(&info);
	alpha = (info.photoInterp & NX_ALPHAMASK) ?
	    (unsigned char *)data + (length >> 1) : NULL;
	if (NXDrawingStatus == NX_DRAWING) {
	    NXImageBitmap(&imageRect, info.width, info.height,
			  info.bitsPerSample, info.samplesPerPixel,
			  info.planarConfig, info.photoInterp,
			  (unsigned char *)data, alpha,
			  NULL, NULL, NULL);
	} else {
	    NXImageBitmap(&imageRect, info.width, info.height,
			  info.bitsPerSample, 1,
			  info.planarConfig, NX_MONOTONICMASK,
			  (unsigned char *)data,
			  NULL, NULL, NULL, NULL);
	}

	if (NXDrawingStatus == NX_DRAWING) {
	    lastSize = bounds.size;
	    if (hasAlpha(&info)) {
		if (drawWithAlpha) {
		    lastSize.width = lastSize.height = 0.0;
		} else {
		    PSsetgray(1.0);
		    PSsetalpha(1.0);
		    PScompositerect(0.0, 0.0,
			bounds.size.width, bounds.size.height,
			NX_DOVER);
		}
	    }
	    [[cache contentView] unlockFocus];
	}

	
    }

    return self;
}

- (BOOL)readTIFFFromStream:(NXStream *)stream
{
    NXTIFFInfo tiffInfo;

    if (data = NXReadTIFF(0, stream, &tiffInfo, NULL)) {
	info = tiffInfo.image;
	return YES;
    } else {
	handleTIFFError(tiffInfo.error);
	return NO;
    }
}

/* Factory method */

+ newFromStream:(NXStream *)stream
{
    self = [super new];
    if ([self readTIFFFromStream:stream]) {
	bounds.origin.x = bounds.origin.y = 0.0;
	bounds.size.width = (NXCoord)info.width;
	bounds.size.height = (NXCoord)info.height;
    } else {
	[self free];
	self = nil;
    }
    return self;
}

- free
{
    [cache free];
    NX_FREE(data);
    return [super free];
}

- (float)naturalAspectRatio
{
    return (float)info.width / (float)info.height;
}

- (BOOL)isOpaque
{
    return YES;
}

- draw
{
    NXRect source;

    if (bounds.size.width < 1.0 || bounds.size.height < 1.0) return self;

    if (NXDrawingStatus != NX_DRAWING ||
	lastSize.width != bounds.size.width ||
	lastSize.height != bounds.size.height ||
	drawWithAlpha) {
	[self createCache];
    }

    if (cache != nil) {
	if (NXDrawingStatus == NX_DRAWING) {
	    [[cache contentView] getFrame:&source];
	    PScomposite(source.origin.x, source.origin.y,
			bounds.size.width, bounds.size.height,
			[cache gState],
			bounds.origin.x, bounds.origin.y,
			NX_SOVER);
	}
    }

    return self;
}

- write:(NXTypedStream *)stream
/*
 * All the image info and the image data itself is written out during
 * archiving.
 */
{
    int length;

    [super write:stream];
    NXWriteTypes(stream, "iiiiii", &info.width, &info.height,
	&info.bitsPerSample, &info.samplesPerPixel,
	&info.planarConfig, &info.photoInterp);
    length = sizeOfImage(&info);
    NXWriteArray(stream, "c", length, data);

    return self;
}

- read:(NXTypedStream *)stream
{
    int length;

    [super read:stream];
    NXReadTypes(stream, "iiiiii", &info.width, &info.height,
	&info.bitsPerSample, &info.samplesPerPixel,
	&info.planarConfig, &info.photoInterp);
    length = sizeOfImage(&info);
    NX_MALLOC(data, char, length+1);
    NXReadArray(stream, "c", length, data);
    data[length] = '\0';
    [self createCache];

    return self;
}

@end


