#import "TextGraphic.h"
#import "GraphicView.h"
#import <appkit/Application.h>
#import <appkit/Cell.h>
#import <appkit/Cursor.h>
#import <appkit/Text.h>
#import <appkit/nextstd.h>
#import <dpsclient/wraps.h>

#define	DOTSIZE 4.0
extern void KDrawBlackDot(float h,float v);
extern BOOL KIsNear(float h1,float v1,float h2,float v2);


@implementation TextGraphic
/*
 * This uses a text object to draw and edit text.
 *
 * The one quirky thing to understand here is that growable Text objects
 * in 1.0 must be subviews of flipped view.  Since a GraphicView is not
 * flipped, we must have a flipped view into the view heirarchy when we
 * edit (this editing view is permanently installed as a subview of the
 * GraphicView--see GraphicView's newFrame: method).
 */

static id drawText = nil;	/* shared Text object used for drawing */

+ initialize
{
    [self setVersion:1];	/* class version */
    return self;
}

+ (BOOL)isEditable
{
    return YES;
}

+ cursor
{
    NXPoint spot;
    static id cursor = nil;

    if (!cursor) {
	cursor = [Cursor newFromMachO:"TextGraphic.tiff"];
	spot.x = 4.0; spot.y = 8.0;
	[cursor setHotSpot:&spot];
    }

    return cursor ? cursor : [super cursor];
}

static void initClassVars()
/*
 * Create the class variable drawText here.
 */
{
    if (!drawText) {
	drawText = [Text new];
	[drawText setMonoFont:NO]; 
//	[drawText setMonoFont:YES];   /* KHL */
	[drawText setEditable:NO];
	[drawText setSelectable:NO];
	[drawText setFlip:YES];
    }
}

+ new
{
    initClassVars();
    return [super new];
}

- awake
{
    initClassVars();
    return [super awake];
}

- (BOOL)create:(NXEvent *)event in:view
 /*
  * We are only interested in where the mouse goes up, that's
  * where we'll start editing.
  */
{
    NXRect viewBounds;
    NXPoint pt;

    event = [NXApp getNextEvent:NX_MOUSEUPMASK];
    bounds.size.width = bounds.size.height = 0.0;
    bounds.origin.x = -99999.;
    pt = event->location;
    [view convertPoint:&pt fromView:nil];
    if( ![view solder:&pt] )   /*KHL*/
    	[view grid:&pt];  /*KHL*/
    bounds.origin = pt;
    [view getBounds:&viewBounds];
    gFlags.selected = NO;

    return NXMouseInRect(&bounds.origin, &viewBounds, NO);
}

- (BOOL)edit:(NXEvent *)event in:view
/*
 * Here we are going to use the shared field editor for the window to
 * edit the text in the TextGraphic.  First, we must end any other editing
 * that is going on with the field editor in this window using endEditingFor:.
 * Next, we get the field editor from the window.  Normally, the field
 * editor ends editing when carriage return is pressed.  This is due to
 * the fact that its character filter is NXFieldFilter.  Since we want our
 * editing to be more like an editor (and less like a Form or TextField),
 * we set the character filter to be NXEditorFilter.  What is more, normally,
 * you can't change the font of a TextField or Form with the FontPanel
 * (since that might interfere with any real editable Text objects), but
 * in our case, we do want to be able to do that.  We also want to be
 * able to edit rich text, so we issue a setMonoFont:NO.  Editing is a bit
 * more efficient if we set the Text object to be opaque.  Note that
 * in textDidEnd:endChar: we will have to set the character filter,
 * FontPanelEnabled and mono-font back so that if there were any forms
 * or TextFields in the window, they would have a correctly configured
 * field editor.
 *
 * To let the field editor know exactly where editing is occurring and how
 * large the editable area may grow to, we must calculate and set the frame
 * of the field editor as well as its minimum and maximum size.
 *
 * We load up the field editor with our rich text (if any).
 *
 * Finally, we set self as the delegate (so that it will receive the
 * textDidEnd:endChar: message when editing is completed) and either
 * pass the mouse-down event onto the Text object, or, if a mouse-down
 * didn't cause editing to occur (i.e. we just created it), then we
 * simply put the blinking caret at the beginning of the editable area.
 *
 * The line marked with the "ack!" is kind of strange, but is necessary
 * since in 1.0, growable Text objects only work when they are subviews
 * of a flipped view.
 *
 * This is why GraphicView has an "editView" which is a flipped view that it
 * inserts as a subview of itself for the purposes of providing a superview
 * for the Text object.  The "ack!" line converts the bounds of the TextGraphic
 * (which are in GraphicView coordinates) to the coordinates of the Text
 * object's superview (the editView).  This limitation of the Text object
 * will be fixed post-1.0.  Note that the "ack!" line is the only one
 * concession we need to make to this limitation in this method (there are
 * two more such lines in textDidEnd:endChar:).
 */
{
    id fe;
    NXSize maxSize;
    NXStream *stream;
    NXRect viewBounds, frame;
	NXCoord  leftMar,rightMar,topMar,bottomMar; /*KHL*/

    /* Get the field editor in this window. */

    [[view window] endEditingFor:self];
    fe = [[view window] getFieldEditor:YES for:self];
    if (!fe) return NO;

    /* Modify it so that it will edit Rich Text and use the FontPanel. */

    [fe setCharFilter:NXFieldFilter];  /* KHL */
    [fe setFontPanelEnabled:YES];
    [fe setMonoFont:NO]; 
//    [fe setMonoFont:YES]; /* KHL */
    [fe setOpaque:YES]; 
	[fe getMarginLeft:&leftMar right:&rightMar top:&topMar bottom:&bottomMar];   /* KHL */
	leftMar = DOTSIZE*3.;
	[fe setMarginLeft:leftMar right:rightMar top:topMar bottom:bottomMar];   /* KHL */

    /*
     * Determine the minimum and maximum size that the Text object can be.
     * We let the Text object grow out to the edges of the GraphicView,
     * but no further.
     */

    [view getBounds:&viewBounds];
    maxSize.width = viewBounds.origin.x+viewBounds.size.width-bounds.origin.x;
    maxSize.height = bounds.origin.y+bounds.size.height-viewBounds.origin.y;
    if (!bounds.size.height && !bounds.size.width) {
//	bounds.origin.y -= floor([fe lineHeight] / 2.0);
	bounds.origin.y -= DOTSIZE;   /* KHL */ 
	bounds.origin.x -= DOTSIZE;   /* KHL */ 
    }
    bounds.size.height = MAX(bounds.size.height, [fe lineHeight]);
//    bounds.size.width = MAX(bounds.size.width, 5.0); 
    bounds.size.width = MAX(bounds.size.width, DOTSIZE*5.); /* KHL */
    frame = bounds;
    [view convertRect:&frame fromView:[view superview]];	// ack!
    [fe setMinSize:&bounds.size];
    [fe setMaxSize:&maxSize];
    [fe setFrame:&frame];
    [fe setVertResizable:YES]; 
//    [fe setVertResizable:NO]; /* KHL */

    /*
     * If we already have text, then put it in the Text object (allowing
     * the Text object to grow downward if necessary), otherwise, put
     * no text in, set some initial parameters, and allow the Text object
     * to grow horizontally as well as vertically
     */

    if (data) {
//	[fe setHorizResizable:NO]; 
	[fe setHorizResizable:YES]; /* KHL */
	stream = NXOpenMemory(data, length, NX_READONLY);
	[fe readRichText:stream];
	NXCloseMemory(stream, NX_SAVEBUFFER);
    } else {
	[fe setHorizResizable:YES]; /* KHL */
	[fe setText:""];
	[fe setAlignment:NX_LEFTALIGNED];
	[fe setSelGray:NX_BLACK];
	[fe unscript:self];
    }

    /*
     * Add the Text object to the view heirarchy and set self as its delegate
     * so that we will receive the textDidEnd:endChar: message when editing
     * is finished.
     */

    [fe setDelegate:self];
    [view addSubview:fe];

    /*
     * Make it the first responder and either pass the mouse-down event
     * on to the Text object, or set the selection at the beginning of
     * the text.
     */

    [[view window] makeFirstResponder:fe];

    if (event) {
	[fe selectNull];	/* eliminates any existing selection */
	[fe mouseDown:event];
    } else {
	[fe setSel:0:0];
    }

    return YES;
}

#define BUF_MAX 500

- textDidEnd:textObject endChar:(unsigned short)endChar
/*
 * This method is called when ever first responder is taken away from a
 * currently editing TextGraphic (i.e. when the user is done editing and
 * chooses to go do something else).  We must extract the rich text the user
 * has typed from the Text object, and store it away.  We also need to
 * get the frame of the Text object and make that our bounds (but,
 * remember, since the Text object must be a subview of a flipped view,
 * we need to convert the bounds rectangle to the coordinates of the
 * unflipped GraphicView).  If the Text object is empty, then we remove
 * this TextGraphic from the GraphicView and delayedFree: it.
 * We must remove the Text object from the view heirarchy and, since
 * this Text object is going to be reused, we must set its delegate
 * back to nil.
 *
 * For further explanation of the two "ack!" lines, see edit:in: above.
 */
{
    int maxlen;
    char *buffer;
    NXStream *stream;
    NXRect oldBounds;
    id editView, graphicView;

    if (data) {
	NX_FREE(data);
	data = NULL;
    }

    editView = [textObject superview];
    graphicView = [editView superview];				// ack!

    if ([textObject textLength]) {
	stream = NXOpenMemory(NULL, 0, NX_WRITEONLY);
	[textObject writeRichText:stream];
	NXGetMemoryBuffer(stream, &buffer, &length, &maxlen);
	NX_MALLOC(data, char, length);
	bcopy(buffer, data, length);
	NXCloseMemory(stream, NX_FREEBUFFER);
	oldBounds = bounds;
	[textObject getFrame:&bounds];
	[editView convertRect:&bounds toView:graphicView];	// ack!
	NXUnionRect(&bounds, &oldBounds);
	[graphicView cache:&oldBounds];
	[[graphicView window] flushWindow];
	[graphicView dirty];
     } else {
	[graphicView removeGraphic:self];
    }

   [textObject removeFromSuperview];
   [textObject setDelegate:nil];
	[[graphicView window] makeFirstResponder:graphicView];  /* KHL  */
 
    return self;
}

- (BOOL)isOpaque
 /*
  * We are never opaque.
  */
{
    return NO;
}

- (BOOL)isValid
 /*
  * Any size TextGraphic is valid (since we fix up the size if it is
  * to small in our override of create:in:).
  */
{
    return YES;
}

- draw
 /*
  * If the region has already been created, then we must draw the text.
  * To do this, we first load up the shared drawText Text object with
  * our rich text.  We then set the frame of the drawText object
  * to be our bounds.  Finally, we add the Text object as a subview of
  * the view that is currently being drawn in ([NXApp focusView])
  * and tell the Text object to draw itself.  We then remove the Text
  * object view from the view heirarchy.
  */
{
    NXStream *stream;

    if (data) {
	stream = NXOpenMemory(data, length, NX_READONLY);
	[drawText readRichText:stream];
	NXCloseMemory(stream, NX_SAVEBUFFER);
	[drawText setFrame:&bounds];
	[[NXApp focusView] addSubview:drawText];
	[drawText display];
	[drawText removeFromSuperview];
	if (DrawStatus == Resizing) {
	    PSsetgray(NX_LTGRAY);
	    NXFrameRect(&bounds);
	}
	KDrawBlackDot(bounds.origin.x+DOTSIZE,bounds.origin.y+DOTSIZE); /*KHL*/
    }

    return self;
}

- write:(NXTypedStream *)stream
 /*
  * Writes the TextGraphic out to the typed stream.
  */
{
    [super write:stream];
    NXWriteTypes(stream, "i", &length);
    NXWriteArray(stream, "c", length, data);
    return self;
}

- read:(NXTypedStream *)stream
 /*
  * Reads the TextGraphic in from the typed stream.
  */
{
    id cell;
    int maxlen;
    NXStream *s;
    char *buffer;

    [super read:stream];
    if (NXTypedStreamClassVersion(stream, [self name]) < 1) {
	NXReadTypes(stream, "@", &cell);
	[drawText setText:[cell stringValue]];
	[drawText setFont:[cell font]];
	[drawText setTextGray:gray];
	s = NXOpenMemory(NULL, 0, NX_WRITEONLY);
	[drawText writeRichText:s];
	NXGetMemoryBuffer(s, &buffer, &length, &maxlen);
	NX_MALLOC(data, char, length);
	bcopy(buffer, data, length);
	NXCloseMemory(s, NX_FREEBUFFER);
    } else {
	NXReadTypes(stream, "i", &length);
	NX_MALLOC(data, char, length);
	NXReadArray(stream, "c", length, data);
    }

    return self;
}

- (BOOL)solderTo:(NXPoint *)p     /*KHL*/
{
	if(KIsNear(p->x,p->y,bounds.origin.x+DOTSIZE,bounds.origin.y+DOTSIZE)) {
				p->x =bounds.origin.x+DOTSIZE;
				p->y =bounds.origin.y+DOTSIZE;
				return	YES;
			}
	return	NO;
}

char		theWireName[500];
- (char *) theWireOrTextName
{
    NXStream *stream;

	stream = NXOpenMemory(data, length, NX_READONLY);
	[drawText readRichText:stream];
	NXCloseMemory(stream, NX_SAVEBUFFER);
	[drawText  getSubstring:theWireName start:0 length:500];
	return(theWireName);
}
	
@end

