#import "InspectorPanel.h"
#import "Graphic.h"
#import "GraphicView.h"
#import "DrawApp.h"
#import "DrawDocument.h"
#import <appkit/Button.h>
#import <appkit/Form.h>
#import <appkit/PopUpList.h>
#import <appkit/Slider.h>
#import <appkit/Text.h>
#import <math.h>

@implementation InspectorPanel
/*
 * The interface is simple.  The "load" method is called with the
 * current GraphicView.  The InspectorPanel is loaded from there, and
 * performs an appropriate graphicsPerform:... method on the view.
 */

- reloadGraphic:graphic
/*
 * Loads up the size fields if they have changed since last time
 * we loaded up the panel with this graphic.  This is used since we
 * know that none of the things controlled by the InspectorPanel can
 * change except the size from event to event.
 */
{
    NXRect bounds;

    if (!graphic) return self;
    [graphic getBounds:&bounds];
    if (lastSize.width != bounds.size.width) {
	[widthForm setFloatValue:bounds.size.width];
	lastSize.width = bounds.size.width;
    }
    if (lastSize.height != bounds.size.height) {
	[heightForm setFloatValue:bounds.size.height];
	lastSize.height = bounds.size.height;
    }

    return self;
}

static void setTitle(id button, int row)
{
    [button setTitle:[[[[button target] itemList] cellAt:row :0] title]];
}

- loadGraphic:graphic
/*
 * Loads up the InspectorPanel with a new graphic's attributes.
 */
{
    NXRect bounds;

    if (selectedGraphic == graphic) {
	return [self reloadGraphic:graphic];
    }
    selectedGraphic = graphic;
    if (!selectedGraphic) {
	return self;
    }

    [lineWidthField setFloatValue:[graphic lineWidth]];
    [lineWidthSlider setFloatValue:[graphic lineWidth]];
    [grayField setFloatValue:[graphic gray]];
    [graySlider setFloatValue:[graphic gray]];
    [graphic getBounds:&bounds];
    [widthForm setFloatValue:bounds.size.width];
    [heightForm setFloatValue:bounds.size.height];
    lastSize = bounds.size;
    setTitle(filledButton, [graphic fill]);
    setTitle(framedButton, ([graphic isFramed] ? 1 : 0));
    setTitle(lineCapButton, [graphic lineCap]);
    setTitle(lineArrowButton, [graphic lineArrow]);
    setTitle(lineJoinButton, [graphic lineJoin]);

    return self;
}

- load:view
/*
 * If the view has only one selected graphic, then the panel is loaded with it.
 */
{
    if (![self isVisible]) {
	selectedGraphic = nil;
	return self;
    }
    graphicView = view;
    [self loadGraphic:[view selectedGraphic]];
    return self;
}

- initializeGraphic:graphic
/*
 * Goes the opposite way of loadGraphic.  Gives the Graphic the attributes
 * which are in the InspectorPanel.
 */
{
    float value;
    const char *s;

    if (!initialized) return self;

    s = [lineWidthField stringValue];
    if (s && s[0] && (value = atof(s))) {
	[graphic setLineWidth:&value];
    }
    s = [grayField stringValue];
    if (s && s[0] && (value = atof(s))) {
	[graphic setGray:&value];
    }
    [graphic setFill:[[[filledButton target] itemList] selectedRow]];
    [graphic setFramed:[[[framedButton target] itemList] selectedRow]];
    [graphic setLineCap:[[[lineCapButton target] itemList] selectedRow]];
    [graphic setLineArrow:[[[lineArrowButton target] itemList] selectedRow]];
    [graphic setLineJoin:[[[lineJoinButton target] itemList] selectedRow]];

    return self;
}

/* Overridden from superclass */

- orderWindow:(int )place relativeTo:(int )otherWin
/*
 * Since we don't update the panel when it isn't visible, we must update
 * it when it becomes visible.
 */
{
    if (!initialized) {
	[self attachPULs];
	initialized = YES;
    }
    if (place == NX_ABOVE) {
	graphicView = [[NXApp currentDocument] view];
	[self loadGraphic:[graphicView selectedGraphic]];
    }
    return [super orderWindow:place relativeTo:otherWin];
}

- update
/*
 * Called each time an event occurs.  Loads up the panel.
 */
{
    [self load:[[NXApp currentDocument] view]];
    return [super update];
}

/* Target/Action methods */

- changeFramed:sender
{
    [graphicView takeFrameValueFrom:[[framedButton target] itemList]];
    [[graphicView window] makeKeyWindow];
    return self;
}

- changeFill:sender
{
    [graphicView takeFillValueFrom:[[filledButton target] itemList]];
    [[graphicView window] makeKeyWindow];
    return self;
}

- changeLineCap:sender
{
    [graphicView takeLineCapFrom:[[lineCapButton target] itemList]];
    [[graphicView window] makeKeyWindow];
    return self;
}

- changeLineArrow:sender
{
    [graphicView takeLineArrowFrom:[[lineArrowButton target] itemList]];
    [[graphicView window] makeKeyWindow];
    return self;
}

- changeLineJoin:sender
{
    [graphicView takeLineJoinFrom:[[lineJoinButton target] itemList]];
    [[graphicView window] makeKeyWindow];
    return self;
}

- changeLineWidth:sender
{
    float linewidth;

    linewidth = [sender floatValue];
    if (sender == lineWidthSlider) {
	if ([NXApp currentEvent]->type == NX_MOUSEDRAGGED) {
	    [graphicView graphicsPerform:@selector(deselect) andDraw:NO];
	} else {
	    [graphicView graphicsPerform:@selector(select) andDraw:NO];
	    [[graphicView window] makeKeyWindow];
	}
	[lineWidthField setFloatValue:linewidth];
    } else {
	if ([lineWidthSlider maxValue] < linewidth) {
	    [lineWidthSlider setMaxValue:linewidth];
	}
	[lineWidthSlider setFloatValue:linewidth];
	[[graphicView window] makeKeyWindow];
    }
    [graphicView takeLineWidthFrom:lineWidthField];
    return self;
}

- changeGray:sender
{
    id fe;
    float gray;

    gray = [sender floatValue];
    if (sender == graySlider) {
	if ([NXApp currentEvent]->type == NX_MOUSEDRAGGED) {
	    [graphicView graphicsPerform:@selector(deselect) andDraw:NO];
	} else {
	    [graphicView graphicsPerform:@selector(select) andDraw:NO];
	    [[graphicView window] makeKeyWindow];
	}
	[grayField setFloatValue:gray];
    } else {
	[graySlider setFloatValue:gray];
	[[graphicView window] makeKeyWindow];
    }
    [graphicView takeGrayValueFrom:grayField];
    fe = [[graphicView window] getFieldEditor:NO for:self];	/* hack! */
    [fe setSelGray:gray];					/* hack! */
    return self;
}


- changeDimensions:sender
{
    NXSize size;

    size.width = [widthForm floatValue];
    size.height = [heightForm floatValue];
    [graphicView graphicsPerform:@selector(sizeTo:)
	with:(id)&size andDraw:YES];
    [[[graphicView window] flushWindow] makeKeyWindow];

    return self;
}

/* NIB outlet-setting methods */

- setLineWidthField:anObject
{
    lineWidthField = anObject;
    [lineWidthField setFloatingPointFormat:NO left:3 right:2];
    [lineWidthField setAlignment:NX_CENTERED];
    return self;
}

- setLineWidthSlider:anObject
{
    lineWidthSlider = anObject;
    return self;
}

- setGraySlider:anObject
{
    graySlider = anObject;
    return self;
}

- setGrayField:anObject
{
    grayField = anObject;
    [grayField setFloatingPointFormat:NO left:1 right:3];
    [grayField setAlignment:NX_CENTERED];
    return self;
}

- setWidthForm:anObject
{
    widthForm = anObject;
    [widthForm setFloatingPointFormat:NO left:4 right:0];
    return self;
}

- setHeightForm:anObject
{
    heightForm = anObject;
    [heightForm setFloatingPointFormat:NO left:4 right:0];
    return self;
}

- setFilledButton:anObject
{
    filledButton = anObject;
    return self;
}

- setFramedButton:anObject
{
    framedButton = anObject;
    return self;
}

- setLineCapButton:anObject
{
    lineCapButton = anObject;
    return self;
}

- setLineArrowButton:anObject
{
    lineArrowButton = anObject;
    return self;
}

- setLineJoinButton:anObject
{
    lineJoinButton = anObject;
    return self;
}

/* Attaching the PopUpLists to the buttons */

- attachFillPUL:button
{
    id pul;

    pul = [PopUpList new];
    [pul addItem:"Unfilled"];
    [pul addItem:"Even/Odd"];
    [pul addItem:"Non-Zero"];
    NXAttachPopUpList(button, pul);

    return self;
}

- attachFramePUL:button
{
   id pul;

    pul = [PopUpList new];
    [pul addItem:"Unframed"];
    [pul addItem:"Framed"];
    NXAttachPopUpList(button, pul);

    return self;
}

- attachLineCapPUL:button
{
    id pul;

    pul = [PopUpList new];
    [pul addItem:"Chopped"];
    [pul addItem:"Round"];
    [pul addItem:"Square"];
    NXAttachPopUpList(button, pul);

    return self;
}

- attachLineArrowPUL:button
{
    id pul;

    pul = [PopUpList new];
    [pul addItem:"None"];
    [pul addItem:"At Start"];
    [pul addItem:"At End"];
    [pul addItem:"Both Ends"];
    NXAttachPopUpList(button, pul);

    return self;
}

- attachLineJoinPUL:button
{
    id pul;

    pul = [PopUpList new];
    [pul addItem:"Miter"];
    [pul addItem:"Round"];
    [pul addItem:"Bevel"];
    NXAttachPopUpList(button, pul);

    return self;
}

- attachPULs
{
    [self attachLineJoinPUL:lineJoinButton];
    [self attachLineCapPUL:lineCapButton];
    [self attachLineArrowPUL:lineArrowButton];
    [self attachFillPUL:filledButton];
    [self attachFramePUL:framedButton];
    return self;
}

@end
