#import "DrawDocument.h"
#import "DrawApp.h"
#import "GraphicView.h"
#import "DrawPageLayout.h"
#import "GridView.h"
#import <appkit/Cursor.h>
#import <appkit/Listener.h>
#import <appkit/Matrix.h>
#import <appkit/PrintInfo.h>
#import <appkit/SavePanel.h>
#import <appkit/ScrollView.h>
#import <appkit/Speaker.h>
#import <appkit/nextstd.h>
#import <appkit/publicWraps.h>
#import <objc/hashtable.h>	/* for NXCopyStringBuffer() */
#import <string.h>

#define Notify(title, msg) NXRunAlertPanel(title, msg, "OK", NULL, NULL)

@implementation DrawDocument
/*
 * This class is used to keep track of a Draw document.
 *
 * Its view and window instance variables keep track of the GraphicView
 * comprising the document as well as the window it is in.
 * The printInfo instance variable is used to allow the user to control
 * how the printed page is printed.  It is an instance of a PrintInfo
 * object (which is edited via the PageLayout and PrintPanels).
 * The listener is used to allow the user to drag an icon representing
 * a PostScript or TIFF file into the document.  The iconPathList is the
 * list of files which was last dragged into the document.
 * The name and directory specify where the document is to be saved.
 * haveSavedDocument keeps track of whether a disk file is associated
 * with the document yet (i.e. if it has ever been saved).
 *
 * The DrawDocument class's responsibilities:
 *
 * 1. Manage the window (including the scrolling view) which holds the
 *    document's GraphicView.  This includes constraining the resizing of
 *    the window so that it never becomes larger than the GraphicView, and
 *    ensuring that if the window contains an unsaved document and the user
 *    tries to close it, the user gets an opportunity to save her changes.
 * 2. Handle communication with the Workspace Manager which allows icons
 *    for PostScript and TIFF files to be dragged into the document window
 *    and be assimilated into the document.
 * 3. Saving the document to a disk file.
 * 4. Provide an external interface to saving the contents of the GraphicView
 *    as a PostScript or TIFF file.
 */

#define MIN_WINDOW_WIDTH 50.0
#define MIN_WINDOW_HEIGHT 75.0
#define SCROLLVIEW_BORDER NX_NOBORDER

static const char DEFAULT_TITLE[] = "Untitled%d.PatchWork";

static NXRect *calcFrame(id printInfo, NXRect *viewRect)
/*
 * Calculates the size of the page the user has chosen minus its margins.
 */
{
    float lm, rm, bm, tm;
    const NXRect *paperRect;

    viewRect->origin.x = viewRect->origin.y = 0.0;
    paperRect = [printInfo paperRect];
    [printInfo getMarginLeft:&lm right:&rm top:&tm bottom:&bm];
    viewRect->size = paperRect->size;
    viewRect->size.width -= lm + rm;
    viewRect->size.height -= tm + bm;

    return viewRect;
}

static void getContentSizeForView(id view, NXSize *contentSize)
/*
 * Calculates the size of the window's contentView by accounting for the
 * existence of the ScrollView around the GraphicView.  No scrollers are
 * assumed since we are interested in the minimum size need to enclose
 * the entire view and, if the entire view is visible, we don't need
 * scroll bars!
 */
{
    NXRect viewFrame;

    [view getFrame:&viewFrame];
    [ScrollView getFrameSize:contentSize
	      forContentSize:&viewFrame.size
	       horizScroller:NO vertScroller:NO
		  borderType:SCROLLVIEW_BORDER];
}

#define WINDOW_MASK (NX_MOUSEUPMASK|NX_MOUSEDRAGGEDMASK|NX_FLAGSCHANGEDMASK)

static id createWindowFor(id view, NXRect *windowContentRect)
/*
 * Creates a window for the specified view.
 * If windowContentRect is NULL, then a window big enough to fit the whole
 * view is created (unless that would be too big to comfortably fit on the
 * screen, in which case a smaller window may be allocated).
 * If windowContentRect is not NULL, then it is used as the contentView of
 * the newly created window.
 */
{
    NXSize screenSize;
    id scrollView, window;
    NXRect defaultWindowContentRect;

    if (!windowContentRect) {
	windowContentRect = &defaultWindowContentRect;
	getContentSizeForView(view, &windowContentRect->size);
	[NXApp getScreenSize:&screenSize];
	if (windowContentRect->size.width > screenSize.width / 2.0) {
	    windowContentRect->size.width = floor(screenSize.width / 2.0);
	}
	if (windowContentRect->size.height > screenSize.height - 60.0) {
	    windowContentRect->size.height = screenSize.height - 60.0;
	}
	windowContentRect->origin.x = screenSize.width - 85.0 -
	    windowContentRect->size.width;
	windowContentRect->origin.y = floor((screenSize.height -
	    windowContentRect->size.height) / 2.0);
    }

    window = [Window newContent:windowContentRect
			  style:NX_TITLEDSTYLE
		        backing:(InMsgPrint ? NX_NONRETAINED : NX_BUFFERED)
		     buttonMask:NX_ALLBUTTONS
			  defer:(InMsgPrint ? NO : YES)];

    scrollView = [ScrollView newFrame:windowContentRect];
    [scrollView setVertScrollerRequired:NO];
    [scrollView setHorizScrollerRequired:NO];
    [scrollView setBorderType:SCROLLVIEW_BORDER];
    [scrollView setDocView:view];
    [window setContentView:scrollView];
    [window addToEventMask:WINDOW_MASK];
    [window makeFirstResponder:view];

    return window;
}

/* Very private instance method needed by factory methods */

- (BOOL)loadDocument:(NXStream *)stream frameSize:(NXRect *)frame
/*
 * Loads an archived document from the specified filename.
 * Loads the window frame specified in the archived document into the
 * frame argument (if the frame argument is NULL, then the frame in
 * the archived document is ignored).  Returns YES if the document
 * has been successfully loaded, NO otherwise.  Note that this method
 * destroys the receiving document, so use with extreme care
 * (essentially, this should only be called when a new document is
 * being created or an existing one is being reverted to its form
 * on disk).
 *
 * An NX_DURING handler is needed around the NXTypedStream operations because
 * if the user has asked that a bogus file be opened, the NXTypedStream will
 * raise an error.  To handle the error, the NXTypedStream must be closed.
 */
{
    int version;
    NXRect dummyRect;
    NXTypedStream *volatile ts = NULL;

    if (!frame) frame = &dummyRect;

    NX_DURING
	ts = NXOpenTypedStream(stream, NX_READONLY);
	if (ts) {
	    NXReadType(ts, "i", &version);	/* historical */
	    printInfo = NXReadObject(ts);
	    NXReadRect(ts, frame);
	    view = NXReadObject(ts);
	} else {
	    return NO;
	}
    NX_HANDLER
	NXCloseTypedStream(ts);
	return NO;
    NX_ENDHANDLER

    NXCloseTypedStream(ts);

    return YES;
}

/* Factory methods */

+ new
/*
 * Creates a new, empty, document.
 *
 * Creates a PrintInfo object; creates a view whose size depends on the
 * default PrintInfo created; creates a window for that view; sets self
 * as the window's delegate; orders the window front; registers the window
 * with the Workspace Manager.  Note that the default margins are set
 * to 1/2 inch--that's more appropriate for a draw program than 1 or 1.25
 * inches.
 */
{
    NXRect frameRect;

    self = [super new];
    printInfo = [PrintInfo new];
    [printInfo setMarginLeft:36.0 right:36.0 top:36.0 bottom:36.0];
    calcFrame(printInfo, &frameRect);
    view = [GraphicView newFrame:&frameRect];
    [view setClipping:NO];			/* since it is in a ClipView */
    window = createWindowFor(view, NULL);
    [window setDelegate:self];
    [self resetScrollers];
    [self setName:NULL andDirectory:NULL];
    [window makeKeyAndOrderFront:self];
//    [self registerWindow];
    [NXApp addDocument:[self filename]];

    return self;
}

+ newFromStream:(NXStream *)stream
/*
 * Creates a new document from what is in the passed stream.
 */
{
    NXRect contentViewFrame;

    self = [super new];
    if (stream && [self loadDocument:stream frameSize:&contentViewFrame]) {
	window = createWindowFor(view, &contentViewFrame);
	[window setDelegate:self];
	[self resetScrollers];
	if (!InMsgPrint) {
	    [window makeKeyAndOrderFront:self];
//	    [self registerWindow];
	}
	haveSavedDocument = YES;
	return self;
    } else {
	Notify("Open PatchWork Document", "I/O error.  Can't open file.");
	[self free];
	return nil;
    }
}

+ newFromFile:(const char *)file
/*
 * Opens an existing document from the specified file.
 */
{
    NXStream *stream;

    stream = NXMapFile(file, NX_READONLY);
    if (stream) {
	if (self = [self newFromStream:stream]) {
	    [self setName:file];
	    [NXApp addDocument:[self filename]];
	}
	NXClose(stream);
	return self;
    } else {
	return nil;
    }
}

- free
{
    [printInfo free];
    [window free];
    NX_FREE(name);
    NX_FREE(directory);
    NX_FREE(iconPathList);
    return [super free];
}

- loadPostScriptFile:(const char *)file at:(const NXPoint *)p
/*
 * Maps in the specified file and asks the view to load the PostScript in
 * from the resulting NXStream.  The PostScript image will be centered at
 * the point p (in the GraphicView's coordinate system).  This is called
 * from the icon-dragging mechanism (icons dragged from the Workspace into
 * the document--see registerWindow).
 */
{
    NXStream *stream;

    stream = NXMapFile(file, NX_READONLY);
    [view loadPostScriptFromStream:stream at:p];
    NXClose(stream);

    return self;
}

- loadTIFFFile:(const char *)file at:(const NXPoint *)p
{
    NXStream *stream;

    stream = NXMapFile(file, NX_READONLY);
    [view loadTIFFFromStream:stream at:p];
    NXClose(stream);

    return self;
}

- resetScrollers
/*
 * Checks to see if the new window size requires a vertical or horizontal
 * scroll bar.  Note that if a scroll bar is required, the window may have
 * to be resized to accomodate it.  Called whenever the page layout (either
 * by user action or by the opening or reverting of a file) is changed or
 * the user resizes the window.
 */
{
    id scrollView;
    NXSize scrollViewSize, contentSize;
    BOOL resizeHoriz = NO, resizeVert = NO;
    NXRect contentRect, windowFrame, viewFrame;

    if (window) {
	[window getFrame:&windowFrame];
	[[window class] getContentRect:&contentRect
			  forFrameRect:&windowFrame
				 style:[window style]];
	scrollView = [window contentView];
	[scrollView setHorizScrollerRequired:YES];
	[scrollView setVertScrollerRequired:YES];
	getContentSizeForView(view, &contentSize);
	if (contentRect.size.width >= contentSize.width ||
	    contentRect.size.height >= contentSize.height) {
	    if (contentRect.size.width >= contentSize.width) {
		[scrollView setHorizScrollerRequired:NO];
		resizeHoriz = YES;
	    }
	    if (contentRect.size.height >= contentSize.height) {
		[scrollView setVertScrollerRequired:NO];
		resizeVert = YES;
	    }
	    [view getFrame:&viewFrame];
	    [[scrollView class] getFrameSize:&scrollViewSize
			      forContentSize:&viewFrame.size
			       horizScroller:!resizeHoriz
				vertScroller:!resizeVert
				  borderType:[scrollView borderType]];
	    if (!resizeHoriz) scrollViewSize.width = contentRect.size.width;
	    if (!resizeVert) scrollViewSize.height = contentRect.size.height;
	    [window sizeWindow:scrollViewSize.width :scrollViewSize.height];
	}
    }

    return self;
}

- view
/*
 * Returns the GraphicView associated with this document.
 */
{
    return view;
}

- printInfo
{
    return printInfo;
}

/* Target/Action methods */

- changeLayout:sender
/*
 * Puts up a PageLayout panel and allows the user to pick a different
 * size paper to work on.  After she does so, the view is resized to the
 * new paper size, and resetScrollers is performed to ensure that the window
 * is not bigger or smaller than the view (thus requiring scroll bars).
 * Since the PrintInfo is effectively part of the document, we dirty
 * the view (by performing the dirty method).
 */
{
    NXRect frame;

    if ([[DrawPageLayout new] runModal] == NX_OKTAG) {
	calcFrame(printInfo, &frame);
	[view sizeTo:frame.size.width :frame.size.height];
	[self resetScrollers];
	[view display];
	[view dirty];
    }

    return self;
}

- changeGrid:sender
/*
 * Changes the grid by putting up a modal panel asking the user what
 * she wants the grid to look like.
 */
{
    [[NXApp gridInspector] runModalForGraphicView:view];
    return self;
}

- save:sender
/*
 * Saves the file.  If this document has never been saved to disk,
 * then a SavePanel is put up to ask the user what file name she
 * wishes to use to save the document.
 */
{
    id savepanel;

    if (!haveSavedDocument) {
	savepanel = [NXApp saveAsPanel];
	if ([savepanel runModalForDirectory:directory file:name]) {
	    [self setName:[savepanel filename]];
	} else {
	    return self;
	}
    }

    [self save];

    return self;
}

- saveAs:sender
{
    [view dirty];
    haveSavedDocument = NO;
    return [self save:sender];
}

- changeSaveType:sender
/*
 * Called by the SavePanel accessory view whenever the user chooses
 * a different type of file to save to.  The window of the sender
 * is, of course, the SavePanel itself.  setRequiredFileType: does
 * not affect the SavePanel while it is running.  It only has effect
 * when the user has chosen a file, and the SavePanel ensures that it
 * has the correct extension by adding it if it doesn't have it already.
 * This message gets here via the Responder chain from the SavePanel.
 */
{
    switch ([sender selectedRow]) {
	case 0: [[sender window] setRequiredFileType:"PatchWork"]; break;
	case 1: [[sender window] setRequiredFileType:"eps"]; break;
	case 2: [[sender window] setRequiredFileType:"tiff"]; break;
    }
    return self;
}

- saveTo:sender
/*
 * This takes the document and saves it as a Draw document file, PostScript
 * file, or TIFF file.  If the document type chosen is Draw document, then
 * this saves the file, but DOES NOT make that file the currently edited
 * file (this makes it easy to save your document elsewhere as a backup
 * and keep on going in the current document).
 *
 * If PostScript or TIFF is selected, then the document is written out
 * in the appropriate format.  In the case of PostScript and TIFF, the
 * actual saving is done using the more general method saveAs:using:.
 */
{
    id savepanel;
    BOOL viewIsDirty;
    const char *file;
    char *type, *savedName, *savedDirectory;

    savepanel = [NXApp saveToPanel];
    if (![savepanel runModalForDirectory:directory file:name]) return self;

    file = [savepanel filename];
    type = strrchr(file, '.');
    if (type) {
	if (!strcmp(type, ".eps")) {
	    [self saveTo:file using:@selector(writePSToStream:)];
	} else if (!strcmp(type, ".tiff")) {
	    [self saveTo:file using:@selector(writeTIFFToStream:)];
	} else if (!strcmp(type, ".PatchWork")) {
	    viewIsDirty = [view isDirty];	/* is it really dirty? */
	    [view dirty];			/* temporarily dirty it */
	    savedName = name;			/* save current name */
	    savedDirectory = directory;		/* save current directory */
	    name = NULL; directory = NULL;	/* clear current filename */
	    [self setName:file];		/* temporarily change name */
	    [self save];			/* save, then restore name */
	    [self setName:savedName andDirectory:savedDirectory];
	    if (viewIsDirty) [view dirty];	/* restore correct dirtiness */
	}
    }

    return self;
}


- revertToSaved:sender
/*
 * Revert the document back to what is on the disk.
 */ 
{
    NXStream *stream;
    NXRect viewFrame, visibleRect;

    if (!haveSavedDocument || ![view isDirty] || (NXRunAlertPanel("Revert", "%s has been edited.  Are you sure you want to undo changes?", "Revert", "Cancel", NULL, name) != NX_ALERTDEFAULT)) {
	return self;
    }

    [view getVisibleRect:&visibleRect];
    [window endEditingFor:self];

    stream = NXMapFile([self filename], NX_READONLY);
    if (stream && [self loadDocument:stream frameSize:NULL]) {
	[[[window contentView] setDocView:view] free];
	calcFrame(printInfo, &viewFrame);
	[window disableFlushWindow];
	[view sizeTo:viewFrame.size.width :viewFrame.size.height];
	[self resetScrollers];
	[view scrollRectToVisible:&visibleRect];
	[view display];
	[window reenableFlushWindow];
	[window flushWindow];
	[window makeFirstResponder:view];
	[window setDocEdited:NO];
	NXClose(stream);
    } else {
	if (stream) NXClose(stream);
	Notify("Revert", "I/O error.  Can't revert.");
    }

    return self;
}

/* Methods related to naming/saving this document. */

- (const char *)filename
/*
 * Gets the fully specified file name of the document.
 * If directory is NULL, then the /tmp is used.
 * If name is NULL, then the default title is used.
 */
{
    static char filenamebuf[MAXPATHLEN+1];

    if (!directory && !name) {
	[self setName:NULL andDirectory:NULL];
    }
    if (directory) {
	strcpy(filenamebuf, directory);
	strcat(filenamebuf, "/");
    } else {
	filenamebuf[0] = '\0';
    }
    if (name) {
	strcat(filenamebuf, name);
    }

    return filenamebuf;
}

- (const char *)directory
{
    return directory;
}

- (const char *)name
{
    return name;
}

- setName:(const char *)newName andDirectory:(const char *)newDirectory
/*
 * Updates the name and directory of the document.
 * newName or newDirectory can be NULL, in which case the name or directory
 * will not be changed (unless one is currently not set, in which case
 * a default name will be used).
 */
{
    char title[MAXPATHLEN+5];
    char oldName[MAXPATHLEN+1];
    char newNameBuf[MAXPATHLEN+1];
    static int uniqueCount = 1;

    if (directory && name) {
	strcpy(oldName, [self filename]);
    } else {
	oldName[0] = '\0';
    }

    if ((newName && *newName) || !name) {
 	if (!newName || !*newName) {
	    sprintf(newNameBuf, DEFAULT_TITLE, uniqueCount++);
	    newName = newNameBuf;
	} else if (name) {
	    NX_FREE(name);
	}
	name = NXCopyStringBuffer(newName);
    }

    if ((newDirectory && (*newDirectory == '/')) || !directory) {
 	if (!newDirectory || (*newDirectory != '/')) {
	    newDirectory = "/tmp";
	} else if (directory) {
	    NX_FREE(directory);
	}
	directory = NXCopyStringBuffer(newDirectory);
    }

    if (strcmp(oldName, [self filename])) {
	strcpy(title, name);
	strcat(title, " \320 ");	/* \320 is a "long dash" */
	strcat(title, directory);
	[window setTitle:title];
	[NXApp renameDocument:oldName to:[self filename]];
    }

    return self;
}

- setName:(const char *)file
/*
 * If file is a full path name, then both the name and directory of the
 * document is updated appropriately, otherwise, only the name is changed.
 */
{
    char *lastComponent;
    char path[MAXPATHLEN+1];

    if (file) {
	strcpy(path, file);
	lastComponent = strrchr(path, '/');
	if (lastComponent) {
	    *lastComponent++ = '\0';
	    return [self setName:lastComponent andDirectory:path];
	} else {
	    return [self setName:file andDirectory:NULL];
	}
    }

    return self;
}

- saveTo:(const char *)file using:(SEL)streamWriter
/*
 * Performed by the saveTo: method, this method uses the streamWriter method
 * to have the GraphicView write itself in some foreign format (i.e., not
 * in Draw archive format).  It does some work to make the default name
 * of the file being saved to be the name of the document with the appropriate
 * extension.  It brings up the SavePanel so the user can specify the name
 * of the file to save to.
 */
{
    NXStream *stream;

    if (!file || !streamWriter) return self;

    stream = NXOpenMemory(NULL, 0, NX_WRITEONLY);
    if (stream) {
	[view perform:streamWriter with:(id)stream];
	NXSaveToFile(stream, file);
	NXCloseMemory(stream, NX_FREEBUFFER);
    }

    return self;
}


- save
/*
 * Writes out the document in three steps:
 *
 * 1. Write the printInfo object
 * 2. Write the rectangle of the contentView of the window
 * 3. Write the GraphicView itself.
 *
 * See GraphicView's write: method for more details on how the GraphicView
 * is archived.
 */
{
    int version;
    NXTypedStream *ts;
    NXRect windowFrame, contentViewRect;

    if ([view isDirty]) {
	ts = NXOpenTypedStreamForFile([self filename], NX_WRITEONLY);
	if (ts) {
	    haveSavedDocument = YES;
	    NXWriteType(ts, "i", &version);	/* historical */
	    NXWriteRootObject(ts, printInfo);
	    [window getFrame:&windowFrame];
	    [[window class] getContentRect:&contentViewRect
			      forFrameRect:&windowFrame
				     style:[window style]];
	    NXWriteRect(ts, &contentViewRect);
	    NXWriteRootObject(ts, view);
	    NXCloseTypedStream(ts);
	} else {
	    Notify("Save", "Can't create file.");
	}
    }

    return self;
}

/* Window delegate methods. */

- windowDidBecomeMain:sender
/*
 * Switch the Application's PrintInfo to the document's when the document
 * window becomes the main window.  Also set the cursor appropriately
 * depending on which tool is currently selected.
 */
{
    [NXApp setPrintInfo:printInfo];
    [self resetCursor];
    return self;
}

- windowWillClose:sender
/*
 * If the GraphicView has been edited, then this asks the user if she
 * wants to save the changes before closing the window.  When the window
 * is closed, the DrawDocument itself must be freed.  This is accomplished
 * via Application's delayedFree: mechanism.  Unfortunately, by the time
 * delayedFree: frees the DrawDocument, the window and view instance variables
 * will already have automatically been freed by virtue of the window's being
 * closed.  Thus, those instance variables must be set to nil to avoid their
 * being freed twice.
 *
 * Returning nil from this method informs the caller that the window should
 * NOT be closed.  Anything else implies it should be closed.
 */
{
    int save;

    if ([view isDirty] && (haveSavedDocument || ![view isEmpty])) {
	save = NXRunAlertPanel("Close", "%s has changes. Save them?", "Save", "Don't Save", "Cancel", name);
	if (save != NX_ALERTDEFAULT && save != NX_ALERTALTERNATE) {
	    return nil;
	} else {
	    [sender endEditingFor:self];	/* terminate any editing */
	    if (save == NX_ALERTDEFAULT) {
		[self save:nil];
	    }
	}
    }

//    [self unregisterWindow];
    window = nil;
    view = nil;
    [NXApp setPrintInfo:nil];
    [NXApp delayedFree:self];
    [NXApp removeDocument:[self filename]];

    return self;
}

- windowWillResize:sender toSize:(NXSize *)size
/*
 * Constrains the size of the window to never be larger than the
 * GraphicView inside it (including the ScrollView around it).
 */
{
    NXRect frameRect, contentRect;

    getContentSizeForView(view, &contentRect.size);
    [[sender class] getFrameRect:&frameRect
		  forContentRect:&contentRect
			   style:[sender style]];

    if (size->width > frameRect.size.width)
	size->width = frameRect.size.width;
    if (size->height > frameRect.size.height)
	size->height = frameRect.size.height;
    size->width = MAX(MIN_WINDOW_WIDTH, size->width);
    size->height = MAX(MIN_WINDOW_HEIGHT, size->height);

    return self;
}

- windowDidResize:sender
/*
 * Ensures that if the view is bigger than the window's content view,
 * scroll bars are added to the ScrollView.
 */
{
    return [self resetScrollers];
}

/* Icon dragging methods */
//
//- registerWindow
///*
// * Registers the document window with the Workspace Manager so that when the
// * user picks up an icon in the Workspace and drags it over our document window
// * and lets go, iconEntered:... and iconReleasedAt::ok: messages will be
// * sent to the DrawDocument from the Workspace Manager.  Allows the user to
// * drag PostScript and TIFF files into the document.
// */
//{
//    unsigned int windowNum;
//    id speaker = [NXApp appSpeaker];
//
//    listener = [Listener new];
//    [listener setDelegate:self];
//    [listener privatePort];
//    [listener addPort];
//    NXConvertWinNumToGlobal([window windowNum], &windowNum);
//    [speaker setSendPort:NXPortFromName(NX_WORKSPACEREQUEST, NULL)];
//    [speaker registerWindow:windowNum toPort:[listener listenPort]];
//
//    return self;
//}
//
//- unregisterWindow
///*
// * Undoes what registerWindow does.
// */
//{
//    unsigned int windowNum;
//    id speaker = [NXApp appSpeaker];
//
//    if (listener) {
//	[speaker setSendPort:NXPortFromName(NX_WORKSPACEREQUEST, NULL)];
//	NXConvertWinNumToGlobal([window windowNum], &windowNum);
//	[speaker unregisterWindow:windowNum];
//	[listener free];
//    }
//
//    return self;
//}
//
//- (int)iconEntered:(int)windowNum at:(double)x :(double)y
//    iconWindow:(int)iconWindowNum iconX:(double)iconX iconY:(double)iconY
//    iconWidth:(double)iconWidth iconHeight:(double)iconHeight
//    pathList:(char *)pathList
///*
// * Called whenever an icon is dragged from the Workspace over the document
// * window.  At this point, all that is done is to salt away the list of files
// * represented by the icon.  All the real work is done in iconReleasedAt::ok:.
// */
//{
//    if (!iconPathList || strcmp(iconPathList, pathList)) {
//	NX_FREE(iconPathList);
//	NX_MALLOC(iconPathList, char, strlen(pathList)+1);
//	strcpy(iconPathList, pathList);
//    }
//    return 0;
//}
//
//- (int)iconReleasedAt:(double)x :(double)y ok:(int *)flag
///*
// * Goes through the list of files associated with the icon dragged
// * from the Workspace and checks if any of them are PostScript or TIFF.
// * If any are, then the GraphicView is asked to load those in as objects.
// * Very important: an NX_DURING handler is required around all the processing
// * of this method since an uncaught raised error will cause this method not
// * to return and thus hang the Workspace Manager for a while.
// */
//{
//    NXPoint p;
//    volatile int foundOne = NO;
//    char *file, *tab, *extension;
//
//    p.x = x; p.y = y;
//    [window convertScreenToBase:&p];
//    [view convertPoint:&p fromView:nil];
//
//    NX_DURING
//	file = iconPathList;
//	while (file) {
//	    tab = strchr(file, '\t');
//	    if (tab) *tab = '\0';
//	    extension = strrchr(file, '.');
//	    if (extension) {
//		if (!strcmp(extension, ".ps") || !strcmp(extension, ".eps")) {
//		    [self loadPostScriptFile:file at:&p];
//		    foundOne = YES;
//		} else if (!strcmp(extension,".tiff") ||
//			   !strcmp(extension,".tif")) {
//		    [self loadTIFFFile:file at:&p];
//		    foundOne = YES;
//		}
//	    }
//	    file = tab ? tab++ : NULL;
//	}
//	if (foundOne) {
//	    [NXApp activateSelf:YES];
//	    [window makeKeyAndOrderFront:self];
//	}
//    NX_HANDLER
//    NX_ENDHANDLER
//
//    *flag = foundOne;
//
//    return 0;
//}
//
/* Validates whether a menu command makes sense now */

- (BOOL)validateCommand:menuCell
/*
 * Validates whether a menu command that DrawDocument responds to
 * is valid at the current time.
 */
{
    SEL action = [menuCell action];

    if (action == @selector(save:)) {
	return [view isDirty];
    } else if (action == @selector(revertToSaved:)) {
	return ([view isDirty] && haveSavedDocument);
    } else if (action == @selector(saveAs:)) {
	return (haveSavedDocument || ![view isEmpty]);
    } else if (action == @selector(saveTo:)) {
	return ![view isEmpty];
    }

    return YES;
}

/* Cursor-setting method */

- resetCursor
/*
 * Sets the document's cursor according to whatever the current graphic is.
 * Makes the graphic view the first responder if there isn't one or if
 * no tool is selected (the cursor is the normal one).
 */
{
    id fr, cursor = [NXApp cursor];
    id scrollview = [window contentView];

    [scrollview setDocCursor:cursor];
    fr = [window firstResponder];
    if (!fr || fr == window || cursor == NXArrow) {
	[window makeFirstResponder:view];
    }

    return self;
}

@end

