;;Written by J. Laroche at the Center for Music Experiment at UCSD, San Diego ;;California. December 1990.

;; This program receives data from the host using the host receive
;; interrupt (an interrupt $10 is issued each time the DSP's HRX register is
;; written by the host.) The data are supposed to be one byte Mu_Law encoded
;; samples. The data are decoded on the DSP and sent back to the host using
;; classical DMA protocol. This is similar (but with a poorer quality) to the
;; way sndplay works when playing Mu-Law encoded sounds (CODEC microphone.)
;; The host->DSP flow is a NON DMA stream. The program implements a simple 
;; LOW_MARK / HIGH_MARK schedule to control the rate of sample input.
;; Since the CODEC microphone samples at 8012.821 Hz, and the output DACs work
;; with either 22050 Hz stereo or 44100 Hz stereo, we must not only decode the
;; one byte Mu-Law samples into 2 byte linear samples, but also perform some
;; sort of sample conversion. If you consider that 44100 * 2 / 8012.821
;; is about 11.0073, you can create 11 16 bit linear samples at 44100 Hz stereo
;; with one mono 8012.821 Hz sample. The pitch error is about 0.06 % which is
;; quite reasonable. There might be a problem though if the imput to this 
;; program was real time CODEC and the output real time 44100 Hz since after
;; a while, the DACs would miss a couple of samples...
;; There are many ways to resample a sound. We used a very simple one which
;; consists in linearly interpolating the 10 missing samples between 2
;; original successive samples. The correct way would be convolve the original
;; signal with a sin x / x function to retrieve the missing ones, or in a more
;; realistic way, perform polynomial interpolation instead of just linear.
;; The result of linear interpolation is an audible high frequency (8012 Hz)
;; "whistling" due to the discontinuity of the derivative each 11 samples...
;; This whistling could be easily filtered out, but this hasn't been
;; implemented here...


	include "ioequ.asm"

IW_Buff		equ	8192		;Start address of input buffer
Buff_size	equ	8191
DMA_SIZE	equ	4096
LOW_MARK	equ	2000
HIGH_MARK	equ	7000

DM_R_REQ	equ	$050001		;message -> host to request dma
VEC_R_DONE	equ	$0024		;host command indicating dma complete


;;;------------------------- Variable locations
;;;

x_sFlags	equ	$00fd		;dspstream flags
DMA_DONE	equ	0		;  indicates that dma is complete
bull		equ	$01
save_a		equ	$02		; Where to save register a
save_b		equ	$03		; Where to save register b


	org	p:$0			
	jmp	reset

	org	p:VEC_R_DONE
	bset	#DMA_DONE,x:x_sFlags
	nop
	
	org	p:$20			; Host received interrupt
	jsr	input

	org	p:100
	
reset
	movec   #6,omr			; data rom enabled, mode 2
	bset    #0,x:m_pbc		; host port
	bset	#3,x:m_pcddr		; pc3 is an output with value
	bclr	#3,x:m_pcd		; zero to enable the external ram
	movep   #>$000000,x:m_bcr	; no wait states on the external sram
        movep   #>$00BC00,x:m_ipr  	; intr levels: SSI=2, SCI=1, HOST=2
	clr	a
	move	a,x:x_sFlags		; clear flags
	bset    #m_hcie,x:m_hcr		; host command interrupts (for DMA!)
	move	#0,sr			; enable interrupts
	
;;	Configure variables

	move	#>IW_Buff,R1		; Pointer to incoming sample
	move	#>IW_Buff,R7		; pointer to output sample
	move	#>Buff_size,M7		; The input buffer length is Buff_size
	move	#>Buff_size,M1
	jmp	main
	
		
main
	bset	#m_hrie,x:m_hcr		; enables host receive interrupts.
	move	#>$100,R5		; Mu_Law table in the DSP's x ROM
	move	#>$7F,M5		; The table is $80 (128) words long.
	move	#>0,R3			; R3 counts the number of non processed
					; samples in the input buffer.

_main_loop
	jclr	#m_htde,x:m_hsr,_main_loop	; (optional!)
	movep	#DM_R_REQ,x:m_htx		; send "DSP_dm_R_REQ" to host
_ackBegin	
	jclr	#m_hf1,x:m_hsr,_ackBegin	; wait for HF1 to go high (DMA)
	move	#>DMA_SIZE,b
	do	b,_prodDMA
_send
	move	R3,b			; Do we have a sample to output? (R3>0)
	tst	b		
	jeq	_send			; If not, just wait.
	move	y:(R7)+,X0		
	move	#>$7FFF,X1		; Rescaling! The Mu_Law table contains
	mpyr	X0,X1,a (R3)-		; 3 byte words, we only want two.
		
_wait
	jclr	#m_htde,x:m_hsr,_wait
	move	a,x:m_htx		; Output the value.
	
	move	#>LOW_MARK,a
	cmp	b,a			; is LOW_MARK < R3 ?
	jlt	_too_many_samples	; if yes, do not re-enable interrupt.
	bset	#m_hrie,x:m_hcr		; it not, accept more input samples.
_too_many_samples
	nop				; You can't jump to the end of a do!!!
_prodDMA
	btst	#DMA_DONE,x:x_sFlags
	jcs	_endDMA
	jclr	#m_htde,x:m_hsr,_prodDMA
	movep	#0,x:m_htx		;send zeros until noticed
	jmp	_prodDMA
_endDMA
	bclr	#DMA_DONE,x:x_sFlags	
_ackEnd
	jmp	_main_loop


;; This is the subroutine that's called every time a sample is received...
;; The MU_law table is 128 words long, and only gives the positive part of
;; the MU_Law coding curve. Each CODEC sample has 7 bits for the value, and one
;; for the sign (the highest bit.) For example, a positive value near zero will
;; be coded by a positive value near but less than 127. A small negative value
;; will be coded by a negative value close to zero. To find the value
;; corresponding to a coded byte, you must get rid of the bit #7, fetch the 
;; sample in the table whose address is the remaining 7 bits (modulo 128!)
;; then multiply it by -1 if the bit #7 is one.

input
	move	a,x:save_a		; Store a and b registers
	move	b,x:save_b
	movep	x:m_hrx,x:bull
	move	x:bull,a
	move	#>$7F,Y1
	and	Y1,a			; Get rid of the highest bit.
	move	(R1)-
	move	a,N5			; Put the rest in the address register
	move	y:(R1)+,b		; Fetch the last ouput sample.
	move	x:(R5+N5),a		; Fetch the value in the table.
	jclr	#7,x:bull,_ooo
	neg	a			; and negate it if necessary
_ooo
	sub	b,a #>1./11.,Y1		; Find Delta_x = x(n) - x(n-1)
	move	a,Y0
	
	do	#11,_endInterpol	; Compute 11 samples
	mac	Y0,Y1,b (R3)+		; by adding delta_x to x(n),
	move	b,y:(R1)+		; and saving the result, eleven times.
_endInterpol

	move	R3,a			; Do we have too many output samples?
	move	#>HIGH_MARK,b
	cmp	a,b
	jgt	_continue_input
	bclr	#m_hrie,x:m_hcr		; If so, block the receive interrupt.
_continue_input	
	move	x:save_a,a		; restore a and b registers.
	move	x:save_b,b
	rti				; return from interrupt...
	
	
	

