;;Written by J. Laroche at the Center for Music Experiment at UCSD, San Diego ;;California. December 1990.

;; simple program implementing host->DSP-> DMA to play sound files through the 
;; DSP. When the host sends a host command 21, the DSP knows that the sound is
;; mono, and therefore, for each DMA in, copies two samples instead of one, and
;; sends two DMA buffers out.
;; DMA In is DSP-initiated and is done the following way:
;;  The DSP sends the driver a DM_W_REQ request (a simple integer with the 
;; last few bits indicating the number of the DMA channel.)
;;  The driver performs all sorts of initializations and sends a DMA-accepted
;; host command (address $2C.) when it's ready to start, along with an integer.
;;  The DSP can start reading data from the host, and continues reading after
;; one DMA buffer has been read until HF1 is reset.
;;  When the host has finished sending a buffer, it resets HF1 and sends a 
;; host command to the DSP (address $24.)

;; To pass the DSP the value of the volume, the host sends two integers, the
;; actual value, and a header to tell the DSP that the value is a volume.
;; If we needed other values (filter coeff etc...) we would have different
;; headers... The DSP accumulates the incoming values in a queue indexed by R2
;; and dispatch the values at the end of the DMA-out.


	include "ioequ.asm"

IW_Buff		equ	8192	;Start address of input buffer
Buff_size	equ	8191
Control_Queue	equ	0	; Start address of the control value queue
Control_Size	equ	99	; Size of that queue.
DMA_SIZE	equ	4096

DM_R_REQ	equ	$050001		;message to host to request dma-OUT
DM_W_REQ	equ	$040002		;message to host to request dma-IN 
VEC_R_DONE	equ	$0024		;host command: dma-OUT complete
VEC_W_DONE	equ	$0028		;host command: dma-IN complete

Vol_Header	equ	$01		; Signals that next value is a volume.
Stuff_Header	equ	$02		; Signals that next value is something.


;;;------------------------- Variable locations
;;;

x_sFlags	equ	$00fd		;dspstream flags
DMA_DONE	equ	0		;  indicates that dma is complete
DMA_ACCEPTED 	equ	1
Stop_Flag	equ	$00		; Stop DMA flag
bull		equ	$02
volume		equ	$03


writeHost macro source
_one	
	jclr	#m_htde,x:m_hsr,_one	
	movep	source,x:m_htx
	endm
	
readHost macro	dest
_two
	jclr	#m_hrdf,x:m_hsr,_two	
	movep	x:m_hrx,dest
	endm
	


	org	p:$0			
	jmp	reset

	org	p:$20
	movep	x:m_hrx,y:(R2)+
	nop

	org	p:$2A
	move	#>2,N1			; When the sound is mono.
	nop

	org	p:VEC_R_DONE		; DMA-OUT completed.
	bset	#DMA_DONE,x:x_sFlags
		
	org	p:$2C			; DMA-IN accepted: start reading.
	jsr	startDMA_In		

	org	p:100
	
reset
	movec   #6,omr			;data rom enabled, mode 2
	bset    #0,x:m_pbc		;host port
	bset	#3,x:m_pcddr		;   pc3 is an output with value
	bclr	#3,x:m_pcd		;   zero to enable the external ram
	movep   #>$000000,x:m_bcr	;no wait states on the external sram
        movep   #>$00BC00,x:m_ipr  	;intr levels: SSI=2, SCI=1, HOST=2
	clr	a
	move	a,x:x_sFlags		;clear flags
	bset    #m_hcie,x:m_hcr		;host command interrupts
	move	#0,sr			;enable interrupts
	
	move	#>1,N1			; Stereo sound by default.
	jmp	main
	
		
main
	move	#>IW_Buff,R0
	move	#>Buff_size,M0
	move	#>0,R1
	move	#>DMA_SIZE-1,M1
	move	#>Control_Queue,R2
	move	#>Control_Queue,R3
	move	#>Control_Size,M2
	move	#>Control_Size,M3
	move	#>.9,a
	move	a,x:volume
	clr	a 
	move	a,x:x_sFlags


_main_loop
	jsr	Read_DMA_Buffer		; Get a buffer from the host
	jsr	Write_DMA_Buffer	; Send it back!
	jsr	update_para		; dispatch the received control values.
	jmp	_main_loop		; Until the next earthquake...
	
	
;; Subroutine that reads one complete DMA from the host, and puts it in the 
;; input buffer. If the sound is mono, then two samples are copied instead
;; of just one.

Read_DMA_Buffer
	jset	#m_hf1,x:m_hsr,Read_DMA_Buffer
	move	#>IW_Buff,R0
	bclr	#m_hrie,x:m_hcr		; Disable the host receive interrupt.
				; since the following values are samples...
	writeHost #DM_W_REQ
	move	#>IW_Buff,R0
	jclr	#m_hf0,x:m_hsr,_ready		
_ready
	btst 	#DMA_ACCEPTED,x:x_sFlags
	jcc	_ready
	
	move	#DMA_SIZE,b
	do	b,_end_DMA_loop
_clear
	jclr	#m_hrdf,x:m_hsr,_clear	
	movep	x:m_hrx,a
	move	a,x:bull
	jclr	#15,x:bull,_no_correct	; This is a modification which corrects
	move	#>$FF,a2		; the driver's bug. It sign-extends the
	move	#>$FF0000,X1		; received short value, if necessary
	or	X1,a	
_no_correct
	rep	N1			; If mono, copies samples twice.
	move	a,y:(R0)+		; for left and right channels.
_end_DMA_loop
	jclr	#m_hrdf,x:m_hsr,_then	
	move	x:m_hrx,X0		; Continue reading incoming data...
_then
	jset	#m_hf1,x:m_hsr,_end_DMA_loop	; until HF1 is reset.
	rts



;; Subroutine that sends a DMA buffer to the host.
;; This is a classical DMA out routine...

Write_DMA_Buffer	
	bset	#m_hrie,x:m_hcr		; enable reception of control values.
	move	#>IW_Buff,R0
	do	N1,_ackEnd		; If mono, we need to send two buffer
_DMA_out					; for each received one...
	jclr	#m_htde,x:m_hsr,_DMA_out
	movep	#DM_R_REQ,x:m_htx		
	
_ackBegin
	jclr	#m_hf1,x:m_hsr,_ackBegin	;    wait for HF1 to go high
	move	#>DMA_SIZE,b

	do	b,_prodDMA
_ddd	
	move	y:(R0)+,X1
	move	x:volume,X0
	mpyr	X0,X1,a
	writeHost a
	
_prodDMA
	btst	#DMA_DONE,x:x_sFlags
	jcs	_endDMA
	jclr	#m_htde,x:m_hsr,_prodDMA
	movep	#0,x:m_htx		    ; send zeros until noticed
	jmp	_prodDMA
_endDMA
	bclr	#DMA_DONE,x:x_sFlags	; Clear the flag for next buffer!
_ackEnd
	rts


;; Subroutine called when the host is ready to send the samples. It reads an
;; integer.

startDMA_In
	readHost X0				; The host sends a integer.
	bset	#DMA_ACCEPTED,x:x_sFlags	; But we don't really need it.
	rti

;; Subroutine that checks the control values queue, and dispatches the received
;; values to the corresponding parameters (here, only the volume...)


update_para
	move	R2,a
	move	R3,b		
	cmp	a,b #>Vol_Header,b	; Is the queue empty?
	jeq	_end
	move	y:(R3)+,a		; If not, what's the header?
	cmp	a,b #>Stuff_Header,b	; Is it a volume header?
	jeq	_update_vol		; YES: update the volume
	cmp	a,b 			; Is it a stuff header?
	jeq	_update_stuff		; YES: update the stuff, etc...
	jmp	update_para		; do it again Sam
_end
	rts
	
_update_vol				; Updates the value of the volume
	move	y:(R3)+,a		; The next value is the volume.
	move	a,x:volume
	jmp	update_para

_update_stuff				; would update the value of another
	jmp	update_para		; parameter (amount of reverb etc...)
