/*
 * dsp_dma_stream - Example of streaming into and out of the DSP using DMA
 * in both directions.  This is very similar to dsp_example_3.  In fact, the
 * stream from the DSP is identical even though it has a different name.
 */

#import <sound/sound.h>
#import <sound/sounddriver.h>
#import <mach.h>
#import <stdlib.h>
#import <stdio.h>
#import "sd_error.h"	/* should be <sound/snddriver_error.h> */

#define DATA_SIZE (32*1024)

/* The following must agree with dsp_dma_stream.asm */
#define DSP_DMA_BUFFER_SIZE 2048 /* 16-bit words */

#define READ_TAG 1
#define WRITE_TAG 2
#define FLUSH_TAG 3

#define VIEW_SIZE 4

static port_t cmd_port;
static int done = 0;
static unsigned short *read_data;
static int bytes_read;

static port_t reply_port, write_port;
static int sd_err;

static void write_completed(void *arg, int tag)
    /* Called when the input to the DSP has been written. */
{
    printf("write done.\n");
}


static void read_completed(void *arg, int tag, void *p, int nbytes)
/* This gets called when the entire result array has been read from the DSP. */
{
    if (tag == READ_TAG) {
	read_data = (unsigned short *)p;
	printf("read done\n");
	bytes_read = nbytes;
	done++;
    }
}


main (int argc, char *argv[])
{
    int i, j, s_err, size, protocol;
    kern_return_t k_err;
    port_t dev_port=0, owner_port=0, temp_port, read_port;
    SNDSoundStruct *dspStruct;
    snddriver_handlers_t handlers = { 0, 0, 
    		0, write_completed, 0, 0, 0, 0, read_completed};
    msg_header_t *reply_msg;
    int low_water = 48*1024;
    int high_water = 64*1024;
    int read_width = 2;		/* bytes per sample */
    int read_buf_size = DSP_DMA_BUFFER_SIZE;
    unsigned short *write_data;
    int write_count;
    int read_count;
    int write_width = 2;	/* bytes per sample */
    int write_buf_size = DSP_DMA_BUFFER_SIZE;
    int dsp_scale_factor = 0x400000; /* 1/2 in DSP number format */

    //
    // initialize input buffer with some arbitrary but known data
    //
    write_count = DATA_SIZE+DSP_DMA_BUFFER_SIZE;
    read_count = DATA_SIZE;

    /* vm_allocate() always grabs whole pages.  We need the alignment. */
    vm_allocate(task_self(),(vm_address_t *)(&write_data),
		write_count*write_width,TRUE);
    
    srand(time(0));
    for (i=0; i<DATA_SIZE; i++) write_data[i] = (rand()<<1) & 0xFFFF;

    //
    // print first few elements of input
    //
    printf("--- \n");
    for (i=0; i<VIEW_SIZE; i++) \
	printf("input[%d] = 0x%x\n",i,(unsigned short)write_data[i]);
    printf("...\n");
    for (i=DATA_SIZE-VIEW_SIZE; i<DATA_SIZE; i++) \
	printf("input[%d] = 0x%x\n",i,(unsigned short)write_data[i]);
    printf("--- \n");
    
    //
    // get the device port for the sound/dsp driver on the local machine
    // and try to become owner of the dsp resource.  Note that it fails
    // if you are logged into a remote machine which is not a "public sound
    // server" (as set in the UNIX section of the Preferences application).
    //
    s_err = SNDAcquire(SND_ACCESS_DSP, 0, 0, 0, NULL_NEGOTIATION_FUN,
			0, &dev_port, &owner_port);
    if (s_err != 0) {
	fprintf(stderr,"*** Could not acquire DSP\n");
	exit(1);
    }

    //
    // get the command port
    //
    sd_err = snddriver_get_dsp_cmd_port(dev_port,owner_port,&cmd_port);
    if (sd_err != 0) {
	sd_error("Cannot acquire command port", sd_err);
	exit(1);
    }

    //
    // set up the reading and writing streams, and set the DSP protocol
    //
    protocol = 0;
    sd_err = snddriver_stream_setup(dev_port, owner_port,
    				 SNDDRIVER_DMA_STREAM_FROM_DSP,
				 read_buf_size, read_width,
				 low_water, high_water,
				 &protocol, &read_port);
    if (sd_err != 0) {
	sd_error("Cannot set up stream from DSP", sd_err);
	exit(1);
    }
    
    sd_err = snddriver_stream_setup(dev_port, owner_port,
    				 SNDDRIVER_DMA_STREAM_TO_DSP,
				 write_buf_size, write_width, 
				 low_water, high_water,
				 &protocol, &write_port);
    if (sd_err != 0) {
	sd_error("Cannot set up stream to DSP", sd_err);
	exit(1);
    }

    sd_err = snddriver_dsp_protocol(dev_port, owner_port, protocol);
    if (sd_err != 0) {
	sd_error("Cannot set up DSP protocol", sd_err);
	exit(1);
    }

    //
    // allocate a port for the replies
    //
    k_err = port_allocate(task_self(),&reply_port);
    if (k_err != KERN_SUCCESS) {
	mach_error("Cannot allocate reply port", k_err);
	exit(1);
    }

    //
    // enqueue region read request, asking only for a completion message
    //
    sd_err = snddriver_stream_start_reading(read_port,0,read_count,READ_TAG,
					 	0,1,0,0,0,0, reply_port);
    if (sd_err != 0) {
	sd_error("Cannot enqueue read request", sd_err);
	exit(1);
    }

    //
    // enqueue the region write request. Request only the termination message..
    //
    sd_err = snddriver_stream_start_writing(write_port,
    					 (void *)write_data,write_count,
					 WRITE_TAG,
					 0,0,
					 0,1,0,0,0,0, reply_port);

    if (sd_err != 0) {
	sd_error("Cannot enqueue write request", sd_err);
	exit(1);
    }

    //
    // get the DSP core image ready for booting
    //
    s_err = SNDReadDSPfile("dsp_dma_stream.lod", &dspStruct, NULL);
    if (s_err != SND_ERR_NONE) {
	fprintf(stderr,"Cannot parse DSP load image : %s\n",
					SNDSoundError(s_err));
	exit(1);
    }

    //
    // boot the dsp
    //
    s_err = SNDBootDSP(dev_port, owner_port, dspStruct);
    if (s_err != SND_ERR_NONE) {
	fprintf(stderr,"Cannot boot dsp : %s\n", SNDSoundError(s_err));
	exit(1);
    }

    //
    // communicate the scale factor
    //
    sd_err = snddriver_dsp_write(cmd_port,&dsp_scale_factor,1,4,
				 SNDDRIVER_MED_PRIORITY);
    if (sd_err != 0) {
	sd_error("Could not send scale factor to DSP", sd_err);
	exit(1);
    }

    //
    // receive reply messages (the completion messages) and dispatch.
    //
    reply_msg = (msg_header_t *)malloc(MSG_SIZE_MAX);
    done = 0;

    while (!done) {
	//
	// get a message from DSP
	//
	reply_msg->msg_size = MSG_SIZE_MAX;
	reply_msg->msg_local_port = reply_port;
	k_err = msg_receive(reply_msg, MSG_OPTION_NONE, 0);
	if (k_err != KERN_SUCCESS) {
	    mach_error("Cannot receive message from DSP", k_err);
	    exit(1);
	}
    
	//
	// dispatch to the appropriate handler
	//
	sd_err = snddriver_reply_handler(reply_msg,&handlers);
	if (sd_err != 0) {
	    sd_error("Cannot parse message from DSP", sd_err);
	    exit(1);
	}
    }
    
    //
    // print first few elements of result
    //
    for (i=0; i<VIEW_SIZE; i++)
        printf("output[%d] = 0x%x\n",i,(unsigned short)read_data[i]);
    printf("...\n");
    for (i=read_count-VIEW_SIZE; i<read_count; i++)
	printf("output[%d] = 0x%x\n",i,(unsigned short)read_data[i]);
    printf("--- \n");

    //
    // check correctness
    //
    for (i=0; i<DATA_SIZE; i++)
      if (read_data[i] != (write_data[i] >> 1)) {
	  printf("*** output[%d] = 0x%x but expected 0x%x.\n",
		 i,read_data[i],write_data[i]>>1);
	  if(j++ > 10) {
	      printf(" . . . \n");
	      exit(1);
	  }
      }
    printf("Data received from DSP is correct.\n");

    //
    // Data received in the read_handler should be deallocated
    //
    vm_deallocate(task_self(),(pointer_t)read_data,bytes_read);
    
    exit(0);
}
