//    This is part of the iostream library, providing input/output for C++.
//    Copyright (C) 1991 Per Bothner.
//
//    This library is free software; you can redistribute it and/or
//    modify it under the terms of the GNU Library General Public
//    License as published by the Free Software Foundation; either
//    version 2 of the License, or (at your option) any later version.
//
//    This library is distributed in the hope that it will be useful,
//    but WITHOUT ANY WARRANTY; without even the implied warranty of
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
//    Library General Public License for more details.
//
//    You should have received a copy of the GNU Library General Public
//    License along with this library; if not, write to the Free
//    Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

#ifndef _STREAMBUF_H
#define _STREAMBUF_H
#ifdef __GNUG__
#pragma interface
#endif

#ifndef atarist
/* KLUDGES!! */
#ifndef size_t
#define size_t unsigned long
#endif
#ifndef fpos_t	/* this is a very questionable define */
#define fpos_t long
#endif

#ifndef EOF
#define EOF (-1)
#endif
#ifndef NULL
#define NULL ((void*)0)
#endif
#ifndef BUFSIZ
#define BUFSIZ 1024
#endif

#else /* atari */

#include <stddef.h>
#include <stdio.h>
extern "C" unsigned long __DEFAULT_BUFSIZ__;
#endif

class ostream; class streambuf;
extern streambuf* __stream_list; /* List of open FILEs. */

typedef long streamoff, streampos;

class ios
{
  public:
    enum io_state { goodbit=0, eofbit=1, failbit=2, badbit=4 };
    enum open_mode {
	in=1,		// on the atari the default bin/text mode
	out=2,		// for these are set from default mode
	ate=4,
	app=8,
	trunc=16,
	nocreate=32,
	noreplace=64
#ifdef atarist
        ,
	in_bin=1|128,	// force bin mode regardless of default mode
        out_bin=2|128,
        app_bin=8|128,

        in_text=1|256,	// force text mode
	out_text=2|256,
        app_text=8|256
#endif
        };
    enum seek_dir { beg, cur, end};
    enum { skipws=01, left=02, right=04, internal=010,
	   dec=020, oct=040, hex=0100,
	   showbase=0200, showpoint=0400, uppercase=01000, showpos=02000,
	   scientific=04000, fixed=0100000, unitbuf=020000, stdio=040000,
	   dont_close=0x80000000 //Don't close streambuf when destroying stream
	   };
    streambuf *_strbuf;
    char _fill;
    unsigned char _state;
    unsigned short _precision;
    long _width;
    unsigned long _flags;
    ostream* _tie;
    ostream* tie() { return _tie; }
    ostream* tie(ostream* val) { ostream* save=_tie; _tie=val; return save; }

    // Methods to change the format state.
    char fill() { return _fill; }
    char fill(char newf) { char oldf = _fill; _fill = newf; return oldf; }
    unsigned long flags() { return _flags; }
    unsigned long flags(unsigned long new_val) {
	unsigned long old_val = _flags; _flags = new_val; return old_val; }
    unsigned short precision() { return _precision; }
    unsigned short precision(int newp) {
	unsigned short oldp = _precision; _precision = (unsigned short)newp;
	return oldp; }
    unsigned long setf(unsigned long mask) {
	unsigned long oldbits = _flags & mask;
	_flags |= _flags; return oldbits; }
    unsigned long setf(unsigned long val, unsigned long mask) {
	unsigned long oldbits = _flags & mask;
	_flags &= ~mask; _flags |= val&mask; return oldbits; }
    unsigned long unsetf(unsigned long mask) {
	unsigned long oldbits = _flags & mask;
	_flags &= ~mask; return oldbits; }
    long width() { return _width; }
    long width(long val) { long save = _width; _width = val; return save; }

    static const unsigned long basefield;
    static const unsigned long adjustfield;
    static const unsigned long floatfield;

    streambuf* rdbuf() { return _strbuf; }
    void clear(int state = 0) { _state = state; }
    int good() { return _state == 0; }
    int eof() { return _state & ios::eofbit; }
    int fail() { return _state & (ios::badbit|ios::failbit); }
    int bad() { return _state & ios::badbit; }
    int rdstate() { return _state; }
    operator void*() { return fail() ? (void*)0 : (void*)this; }
    int operator!() { return fail(); }
    int is_open();
    int readable();
    int writable();

  protected:
    ios(streambuf*sb) { _strbuf=sb; _state=0; _width=0; _fill=' ';
			_flags=ios::skipws; _precision=6; }
};

// Magic numbers and bits for the _flags field.
// The magic numbers use the high-order bits of _flags;
// the remaining bits are abailable for variable flags.
// Note: The magic numbers must all be negative ...

#define _IO_MAGIC 0xFBAD0000 /* Magic number */
#define _OLD_STDIO_MAGIC 0xFABC0000 /* Emulate old stdio. */
#define _IO_MAGIC_MASK 0xFFFF0000
#define _S_USER_BUF 1 /* User owns buffer; don't delete it on close. */
#define _S_UNBUFFERED 2
#define _S_CAN_READ 4
#define _S_CAN_WRITE 8
#define _S_EOF_SEEN 16
#define _S_ERR_SEEN 32
#define _S_DELETE_DONT_CLOSE 64
#ifdef atarist
#define _S_IS_BINARY 128	// stream is binary/text
#endif
#define _S_LINE_BUF 0x4000
#define _S_IS_FILEBUF 0x8000

struct __streambuf {
    // NOTE: If this is changed, also change __FILE in stdio.h!
    unsigned long _flags;/* High-order word is _IO_MAGIC; rest is flags. */
    char* _gptr;	/* Current get pointer */
    char* _egptr;	/* End of get area. */
    char* _eback;	/* Start of putback+get area. */
    char* _pbase;	/* Start of put area. */
    char* _pptr;	/* Current put pointer. */
    char* _epptr;	/* End of put area. */
    char* _base;	/* Start of reserve area. */
    char* _ebuf;	/* End of reserve area. */
    struct streambuf *_chain;
};

struct streambuf : private __streambuf {
  protected:
    char* gptr() const { return _gptr; }
    char* pptr() const { return _pptr; }
    char* egptr() const { return _egptr; }
    char* epptr() const { return _epptr; }
    char* pbase() const { return _pbase; }
    char* eback() const { return _eback; }
    char* ebuf() const { return _ebuf; }
    char* base() const { return _base; }
#ifdef atarist
    int  is_binary() { return _flags & _S_IS_BINARY ? 1 : 0; }
    void set_binary() { _flags |= _S_IS_BINARY; }
#endif
    void gbump(int n) { _gptr += n; }
    void pbump(int n) { _pptr += n; }
    void setb(char* b, char* eb, int a=0);
    void setp(char* p, char* ep) { _pbase=_pptr=p; _epptr=ep; }
    void setg(char* eb, char* g, char *eg) { _eback=eb; _gptr=g; _egptr=eg; }
  public:
    virtual int underflow(); // Leave public for now
    virtual int overflow(int c = EOF); // Leave public for now
    virtual int doallocate();
    virtual streamoff seekoff(streamoff, seek_dir, int mode=ios::in|ios::out);
    virtual streamoff seekpos(streampos pos, int mode = ios::in|ios::out);
//    extern "C" friend int __underflow(FILE*);
//    extern "C" friend int __overflow(FILE*, int);
    int sputbackc(char c);
    streambuf();
    virtual ~streambuf();
//    int handle_overflow(int c = EOF); // Used by fflush().
    int unbuffered() { return _flags & _S_UNBUFFERED ? 1 : 0; }
    int linebuffered() { return _flags & _S_LINE_BUF ? 1 : 0; }
    void unbuffered(int i)
	{ if (i) _flags |= _S_UNBUFFERED; else _flags &= ~_S_UNBUFFERED; }
    void linebuffered(int i)
	{ if (i) _flags |= _S_LINE_BUF; else _flags &= ~_S_LINE_BUF; }
    int allocate() {
	if (base() || unbuffered()) return 0;
	else return doallocate(); }
    virtual int sync();
    virtual int pbackfail(int c);
    virtual streambuf* setbuf(char* p, size_t len);
    int in_avail() { return _egptr - _gptr; }
    int out_waiting() { return _pptr - _pbase; }
    size_t sputn(const char* s, size_t n);
    size_t sgetn(char* s, size_t n);
    long sgetline(char* buf, size_t n, char delim, int putback_delim);
    int sbumpc() {
	if (_gptr >= _egptr && underflow() == EOF) return EOF;
	else
#ifndef atarist
	    return *(unsigned char*)_gptr++;
#else
        {
	    int ch = *(unsigned char*)_gptr++;
	    if((!is_binary()) && (ch == '\r'))
		return sbumpc();
	    return ch;
	}
#endif
    }
    
    int snextc() {
	if (++_gptr >= _egptr && underflow() == EOF) return EOF;
	else
#ifndef atarist
	    return *(unsigned char*)_gptr;
#else
        {
    	    int ch = *(unsigned char *)_gptr;
	    if((!is_binary()) && (ch == '\r'))
		return snextc();
	    return ch;
	}
#endif
     }
	   
    int sgetc() {
	if (_gptr >= _egptr && underflow() == EOF) return EOF;
	else
#ifndef atarist
	    return *(unsigned char*)_gptr;
#else
        {
	    int ch = *(unsigned char *)_gptr;
	    if((!is_binary()) && (ch == '\r'))
		return snextc();
	    return ch;
	}
#endif	
    }

#ifdef atarist
    int _atari_putc(int c) {
	if ((!is_binary()) && ((c == '\n') || (c == '\r')))
	{
	    if(c == '\r')
		return (int)c;
	    if (_pptr >= _epptr)
	    {
		if(overflow((int)'\r') == EOF) return EOF;
	    }
	    else
		*_pptr++ = '\r';
	}
	if (_pptr >= _epptr) return overflow(c);
	return *_pptr++ = c, (unsigned char)c;
    }
#endif

    int sputc(int c) {
#ifndef atarist
	if (_pptr >= _epptr) return overflow(c);
	return *_pptr++ = c, (unsigned char)c;
#else
	return _atari_putc(c);
#endif	
    }
};

struct __file_fields {
#ifdef LITTLE_ENDIAN
    char _fake;
    char _shortbuf[1];
    short _fileno;
#else
    short _fileno;
    char _shortbuf[1];
    char _fake;
#endif
    int _blksize;
    char* _save_gptr;
    char* _save_egptr;
#ifndef atarist
    fpos_t _offset;
#else
    long _offset;
#endif
};

class filebuf : public streambuf {
    struct __file_fields _fb;
    void init();
  public:
    filebuf();
    filebuf(int fd);
    filebuf(int fd, char* p, size_t len);
    ~filebuf();
    filebuf* attach(int fd);
    filebuf* open(const char *filename, const char *mode);
    filebuf* open(const char *filename, int mode, int prot = 0664);
    virtual int underflow();
    virtual int overflow(int c = EOF);
    int is_open() { return _fb._fileno >= 0; }
    int fd() { return is_open() ? _fb._fileno : EOF; }
    filebuf* close();
    virtual int doallocate();
    virtual streamoff seekoff(streamoff, seek_dir, int mode=ios::in|ios::out);
  protected: // See documentation in filebuf.C.
    virtual int pbackfail(int c);
    virtual int sync();
    int is_reading() { return eback() != egptr(); }
    int is_writing() { return pbase() != NULL; }
    char* cur_ptr() { return is_reading() ?  gptr() : pptr(); }
    /* System's idea of pointer */
    char* file_ptr() { return _fb._save_gptr ? _fb._save_egptr : egptr(); }
    int do_flush();
    // Low-level operations (Usually invoke system calls.)
#ifndef atarist
    virtual int sys_read(char* buf, size_t size);
    virtual fpos_t sys_seek(fpos_t, seek_dir);
#else
    virtual long sys_read(char* buf, size_t size);
    virtual long sys_seek(long, seek_dir);
#endif
    virtual long sys_write(const void*, long);
    virtual int sys_stat(void*); // Actually, a (struct stat*)
    virtual int sys_close();
};

inline int ios::readable() { return rdbuf()->_flags & _S_CAN_READ; }
inline int ios::writable() { return rdbuf()->_flags & _S_CAN_WRITE; }
inline int ios::is_open() {return rdbuf()->_flags & _S_CAN_READ+_S_CAN_WRITE;}

#endif _STREAMBUF_H
