/* Copyright (C) 1989, 1992, 1993 Aladdin Enterprises.  All rights reserved.

This file is part of Ghostscript.

Ghostscript is distributed in the hope that it will be useful, but
WITHOUT ANY WARRANTY.  No author or distributor accepts responsibility
to anyone for the consequences of using it or for whether it serves any
particular purpose or works at all, unless he says so in writing.  Refer
to the Ghostscript General Public License for full details.

Everyone is granted permission to copy, modify and redistribute
Ghostscript, but only under the conditions described in the Ghostscript
General Public License.  A copy of this license is supposed to have been
given to you along with Ghostscript so you can know your rights and
responsibilities.  It should be in a file named COPYING.  Among other
things, the copyright notice and this notice must be preserved on all
copies.  */

/* gp_atar3.c */

/*
 * This file contains general utility routines for the Atari platform.
 */

#include <gemdefs.h>
#include <aesbind.h>
#include <vdibind.h>
#include <osbind.h>
#include <string.h>
#include <stdlib.h>

#include "stdio_.h"
#include "gx.h"
#include "stream.h"

#include "st_rsc.h"
#include "st_defs.h"
#include "gp_atar1.h"
#include "gp_atar2.h"
#include "gp_atar3.h"
#include "gp_atar4.h"

/* Initialize the text window structures. */

void
TextWinInit(WINDOW *twin, WINTEXT *text, int gadgets, char *title)
{

  char *TitleString;

  twin->handle = -1;
  twin->type = TEXT;
  twin->obj = (void *)text;
  twin->gadgets = gadgets;
  twin->opened = 0;
  twin->iconified = 0;

  if ((TitleString = gs_malloc(MAXTITLE, 1, "Window Title")) == NULL) {
    twin->title = title;
  }
  else {
    strncpy(TitleString, title, MAXTITLE-1);
    twin->title = TitleString;
  }

  twin->canvas.g_x = 0; twin->canvas.g_y = 0;
  twin->canvas.g_w = 0; twin->canvas.g_h = 0;

  twin->frame.g_x = 0; twin->frame.g_y = 0;
  twin->frame.g_w = 0; twin->frame.g_h = 0;

  twin->oframe.g_x = 0; twin->oframe.g_y = 0;
  twin->oframe.g_w = 0; twin->oframe.g_h = 0;

  twin->mframe.g_x = 0; twin->mframe.g_y = 0;
  twin->mframe.g_w = 0; twin->mframe.g_h = 0;

  twin->redraw = txtw_redraw;
  twin->move   = txtw_move;
  twin->size   = txtw_size;
  twin->arrow  = txtw_arrow;
  twin->hslide = txtw_hslide;
  twin->vslide = txtw_vslide;

  text->bw   = COLUMNS;
  text->bh   = LINES;
  text->top  = text->edge = 0;
  text->xoff = text->yoff = 0;
  text->cx   = text->cy   = 0;
  text->cstate = 0;
  text->wc   = text->hc   = 0;
  text->lins = text->cols = 0;
  text->ln   = text->cn   = 0;
  text->fl   = 0;
  text->ll   = LINES-1;
  text->fdl  = 0;
  text->ldl  = LINES-1;
  text->fdc  = text->ldc  = 0;
  text->scrolled = 0;
}

/* Initialize the bit window structures. */

void
BitWinInit(WINDOW *bwin, GRAPHIC *gr, int gadgets, char *title)
{

  char *TitleString;

  if ((TitleString = gs_malloc(MAXTITLE, 1, "Window Title")) == NULL) {
    bwin->title = title;
  }
  else {
    strncpy(TitleString, title, MAXTITLE);
    bwin->title = TitleString;
  }

  bwin->handle = -1;
  bwin->type = BIT;
  bwin->obj = (void *)gr;
  bwin->gadgets = gadgets;
  bwin->title = title;
  bwin->opened = 0;
  bwin->iconified = 0;

  bwin->redraw = bitw_redraw;
  bwin->move   = bitw_move;
  bwin->size   = bitw_size;
  bwin->arrow  = bitw_arrow;
  bwin->hslide = bitw_hslide;
  bwin->vslide = bitw_vslide;

  bwin->frame.g_x = imagwin.frame.g_x + 2 * VWork.Wchar;
  bwin->frame.g_y = imagwin.frame.g_y + VWork.Hchar;
  bwin->frame.g_w = imagwin.frame.g_w;
  bwin->frame.g_h = imagwin.frame.g_h;

  memcpy(gr, imagwin.obj, sizeof(GRAPHIC));

}

/* Initialize the object window structures. */

void
ObjWinInit(WINDOW *owin, OBJECT *object, int gadgets, char *title)
{
  owin->handle = -1;
  owin->type = OBJ;
  owin->obj = (void *)object;
  owin->gadgets = gadgets;
  owin->title = title;
  owin->opened = 0;
  owin->iconified = 0;

  owin->canvas.g_x = 0; owin->canvas.g_y = 0;
  owin->canvas.g_w = 0; owin->canvas.g_h = 0;

  owin->frame.g_x = 0; owin->frame.g_y = 0;
  owin->frame.g_w = 0; owin->frame.g_h = 0;

  owin->oframe.g_x = 0; owin->oframe.g_y = 0;
  owin->oframe.g_w = 0; owin->oframe.g_h = 0;

  owin->mframe.g_x = 0; owin->mframe.g_y = 0;
  owin->mframe.g_w = 0; owin->mframe.g_h = 0;

  owin->redraw = objw_redraw;
  owin->move   = objw_move;
  owin->size   = NULL;
  owin->arrow  = NULL;
  owin->hslide = NULL;
  owin->vslide = NULL;

}

/* Open a window containing a bitmapped image. */

int
BitWinOpen(WINDOW *bwin, VWRK *vw, PSTATE *st)
{
  GRAPHIC *gr = bwin->obj;

  /* Open a window for a bitmap image. */

  if (bwin->iconified) {
    wind_open(bwin->handle,
	      bwin->frame.g_x, bwin->frame.g_y,
	      MIN(bwin->frame.g_w, bwin->mframe.g_w),
	      MIN(bwin->frame.g_h, bwin->mframe.g_h));

    bwin->opened = 1;
    bwin->iconified = 0;

    if ((bwin != &iconwin) && (iconwin.obj != iconobj)) {
      WinClose(&iconwin);
      UpdateIconList(iconwin.obj, vw);
      ObjWinOpen(iconwin.obj, &iconwin, vw, st);
    }

  }
  else if (!bwin->opened) {

    wind_calc(WC_BORDER, bwin->gadgets,
	      vw->full.g_x, vw->full.g_y,
	      gr->Width, gr->Height,
	      &bwin->mframe.g_x, &bwin->mframe.g_y,
	      &bwin->mframe.g_w, &bwin->mframe.g_h);

    MoveIntoLimits(&bwin->mframe, vw);
	
    if ((bwin->frame.g_w + bwin->frame.g_h) == 0) {
      bwin->frame.g_x = bwin->mframe.g_x;
      bwin->frame.g_y = bwin->mframe.g_y;
      bwin->frame.g_w = bwin->mframe.g_w;
      bwin->frame.g_h = bwin->mframe.g_h;
    }

    MoveIntoLimits(&bwin->frame, vw);

    bwin->handle = wind_create(
	    bwin->gadgets + (st->HasIconifier ? SMALLER : 0),
	    bwin->mframe.g_x, bwin->mframe.g_y,
	    bwin->mframe.g_w, bwin->mframe.g_h);

    if (bwin->handle < 0) {
      FormAlert(1, "[1][Ghostscript Error!|No More Windows.][Continue]");
      WinListDelete(WList, bwin);
      return (-1);
    }

    wind_set(bwin->handle, WF_NAME, bwin->title, bwin->title, 0, 0);

    wind_open(bwin->handle,
	      bwin->frame.g_x, bwin->frame.g_y,
	      MIN(bwin->frame.g_w, bwin->mframe.g_w),
	      MIN(bwin->frame.g_h, bwin->mframe.g_h));

    wind_get(bwin->handle, WF_CURRXYWH,
	     &bwin->frame.g_x, &bwin->frame.g_y,
	     &bwin->frame.g_w, &bwin->frame.g_h);
    
    wind_get(bwin->handle, WF_CURRXYWH,
	     &bwin->oframe.g_x, &bwin->oframe.g_y,
	     &bwin->oframe.g_w, &bwin->oframe.g_h);

    wind_get(bwin->handle, WF_WORKXYWH,
	     &bwin->canvas.g_x, &bwin->canvas.g_y,
	     &bwin->canvas.g_w, &bwin->canvas.g_h);

    gr->StepX = .9 * bwin->canvas.g_w;
    gr->StepY = .9 * bwin->canvas.g_h;

    bwin->opened = 1;
    UpdateScroll(bwin->handle);
  }

  return 0;
}

/* Open a help window. */

int
HelpWinOpen(VWRK *vw, PSTATE *st, char *HelpFile)
{
  FILE *helpf;
  WINDOW *hwin;
  WINTEXT *text;
  WINLIST *wl=WList;
  extern FILE *lib_fopen(const char *);

  char line[MAXLEN], helpfile[MAXLEN];
  char *firstline;

  int count, twincount=0, length, longest=0;

  if ((hwin = WinAlloc()) == NULL) {
    FormAlert(1, "[1][Ghostscript Error!|No More Memory.][Continue]");
    return (-1);
  }

  if ((text = WinTextAlloc(LINES)) == NULL) {
    FormAlert(1, "[1][Ghostscript Error!|No More Memory.][Continue]");
    WinFree(hwin);
    return (-1);
  }

  TextWinInit(hwin, text, H_GADGETS, H_TITL);

  /* Read help file into the text buffer. */

  strcpy(helpfile, HelpFile);

  if ((helpf = lib_fopen(helpfile)) == NULL) {
    sprintf(line, "[1][Could not open %s!][OK]", helpfile);
    FormAlert(1, line);
    return -1;
  }

  for (count=0; (fgets(line, MAXLEN-1, helpf) != NULL); ++count) {

    length = strlen(line);
    line[length-2] = '\0';

    if (length-2 > longest) longest = length-2;

    if (count >= text->bsize - 1) {
      int NewSize = text->bsize + LINES;
      if (TextBuffRealloc(text, NewSize) == NULL) {
	FormAlert(1, "[1][There is only enough ram to|\
		    load a portion of this file.][Continue]");
	break;
      }
    }

    strcpy(TEXTLINE(text, count), line);

  }

  fclose(helpf);

  /* Fix the window title if possible. */

  firstline = TEXTLINE(text, 0);
  
  if (strlen(firstline)) {
    strcpy(hwin->title, " ");
    strncat(hwin->title, firstline+strspn(firstline, " "), MAXTITLE-3);
    strcat(hwin->title, " ");
  }

  if (hwin->frame.g_w + hwin->frame.g_h == 0) {
    wind_calc(WC_BORDER, hwin->gadgets, vw->full.g_x, vw->full.g_y,
	      (longest + 4)*vw->Wchar, (count + 1)*vw->Hchar,
	      &hwin->frame.g_x, &hwin->frame.g_y,
	      &hwin->frame.g_w, &hwin->frame.g_h);

    /* Count the number of open text windows. */

    do {
      if (wl->Win->type == TEXT){
	++twincount;
      }
      wl = wl->Next;
    }
    while (wl != WList);

    /* Place the window according to how many text windows are open. */

    hwin->frame.g_x = 2 * twincount * vw->Wchar;
    hwin->frame.g_y = vw->full.g_y + twincount * vw->Hchar;
    hwin->frame.g_w = MIN(hwin->frame.g_w, vw->full.g_w - hwin->frame.g_x);
    hwin->frame.g_h = MIN(hwin->frame.g_h,
			  vw->full.g_h - twincount * vw->Hchar);
  }

  text->bw = longest;
  text->bh = count;

  WinListAdd(WList, hwin);
  TextWinOpen(hwin, vw, st);

  text->ln = count-1;
    
  return(0);
}

/* Open a text window. */

int
TextWinOpen(WINDOW *twin, VWRK *vw, PSTATE *st)
{

  WINTEXT *text = twin->obj;

  /* Create, open, and initialize a text window. */

  if (twin->iconified) {
    wind_open(twin->handle,
	      twin->frame.g_x, twin->frame.g_y,
	      MIN(twin->frame.g_w, twin->mframe.g_w),
	      MIN(twin->frame.g_h, twin->mframe.g_h));

    twin->opened = 1;
    twin->iconified = 0;
  }
  else if (!twin->opened) {

    int dummy;

    text->wc = vw->Wchar;
    text->hc = vw->Hchar;

    wind_calc(WC_BORDER, twin->gadgets, vw->full.g_x, vw->full.g_y,
	      (text->bw + 4)*text->wc, (text->bh + 1)*text->hc,
	      &twin->mframe.g_x, &twin->mframe.g_y,
	      &twin->mframe.g_w, &twin->mframe.g_h);

    twin->mframe.g_x = MAX(twin->mframe.g_x, vw->full.g_x);
    twin->mframe.g_y = MAX(twin->mframe.g_y, vw->full.g_y);
    twin->mframe.g_w = MIN(twin->mframe.g_w, vw->full.g_w);
    twin->mframe.g_h = MIN(twin->mframe.g_h, vw->full.g_h);

    MoveIntoLimits(&twin->mframe, vw);

    twin->handle = wind_create(
	    twin->gadgets + (st->HasIconifier ? SMALLER : 0),
	    twin->mframe.g_x, twin->mframe.g_y,
	    twin->mframe.g_w, twin->mframe.g_h);

    if (twin->handle < 0) {
      FormAlert(1, "[1][Ghostscript Error!|No More Windows.][Continue]");
      WinListDelete(WList, twin);
      return (-1);
    }

    wind_set(twin->handle, WF_NAME,
	     twin->title, twin->title, 0, 0);

    if ((twin->frame.g_w + twin->frame.g_h) == 0) {

      wind_calc(WC_BORDER, twin->gadgets + (st->HasIconifier ? SMALLER : 0),
		vw->full.g_x, vw->full.g_y,
		(text->bw + 2)*text->wc, (text->bh + 1)*text->hc,
		&twin->frame.g_x, &twin->frame.g_y,
		&twin->frame.g_w, &twin->frame.g_h);

      twin->frame.g_x = vw->full.g_x;
      twin->frame.g_y = vw->full.g_y;
      twin->frame.g_w = MIN(twin->frame.g_w, vw->full.g_w);
      twin->frame.g_h = MIN(twin->frame.g_h, vw->full.g_h);

    }

    MoveIntoLimits(&twin->frame, vw);

    wind_open(twin->handle,
	      twin->frame.g_x, twin->frame.g_y,
	      twin->frame.g_w, twin->frame.g_h);

    wind_get(twin->handle, WF_WORKXYWH,
	     &twin->canvas.g_x, &twin->canvas.g_y,
	     &twin->canvas.g_w, &twin->canvas.g_h);

    wind_get(twin->handle, WF_CURRXYWH,
	     &twin->oframe.g_x, &twin->oframe.g_y,
	     &twin->oframe.g_w, &twin->oframe.g_h);

    ClearWin(&twin->canvas);

    twin->opened = 1;

    text->edge = twin->canvas.g_x;
    text->top = twin->canvas.g_y;

    text->xoff = text->wc/2;
    text->yoff = text->hc/2;

    text->lins = VisibleLines(twin);
    text->cols = VisibleCols(twin);
    text->ll   = text->bh - 1;
    text->ldl  = text->fdl + text->lins - 1;

    text->cx = align(text->edge + text->xoff);
    text->cy = text->top + text->yoff;

    UpdateScroll(twin->handle);
    vst_alignment(vw->VdiHandle, 0, 5, &dummy, &dummy);

  }

  return 0;
}

/* Open a window containing a GEM object. */

int
ObjWinOpen(OBJECT obj[], WINDOW *owin, VWRK *vw, PSTATE *st)
{

  int tempx, tempy;

  /* Create and open a window for a GEM object. */

  if (owin->iconified) {

    obj[0].ob_flags ^= HIDETREE; 	/* Make the object tree visible. */

    wind_open(owin->handle,
	      owin->frame.g_x, owin->frame.g_y,
	      owin->frame.g_w, owin->frame.g_h);

    owin->opened = 1;
    owin->iconified = 0;

    if ((owin != &iconwin) && (iconwin.obj != iconobj)) {
      WinClose(&iconwin);
      UpdateIconList(iconwin.obj, vw);
      ObjWinOpen(iconwin.obj, &iconwin, vw, st);
    }

  }
  else if (!owin->opened) {

    owin->handle = wind_create(owin->gadgets,
			       vw->full.g_x, vw->full.g_y,
			       vw->full.g_w, vw->full.g_h);

    if (owin->handle < 0) {
      FormAlert(1,
		"[1][Ghostscript Error!|No More Windows.][Continue]");
      WinListDelete(WList, owin);
      return (-1);
    }

    wind_set(owin->handle, WF_NAME, owin->title, owin->title, 0, 0);

    /* Make the object tree visible. */

    obj[0].ob_flags ^= HIDETREE;

    /* If the window has never been opened, width and height will
     * be zero. In this case, center the object before opening
     * the window.
     */

    if ((owin->frame.g_w + owin->frame.g_h) == 0) {

      form_center(obj, &owin->canvas.g_x, &owin->canvas.g_y,
		       &owin->canvas.g_w, &owin->canvas.g_h);

      wind_calc(WC_BORDER, owin->gadgets,
		owin->canvas.g_x, owin->canvas.g_y,
		owin->canvas.g_w, owin->canvas.g_h,
		&tempx, &tempy,
		&owin->frame.g_w, &owin->frame.g_h);

      if (!(owin->frame.g_x + owin->frame.g_y)) {
	owin->frame.g_x = tempx;
	owin->frame.g_y = tempy;
      }

    }
    else {
      tempx = owin->frame.g_x;
      tempy = owin->frame.g_y;
    }

    /* Make sure the window in entirely on the screen. */

    MoveIntoLimits(&owin->frame, vw);

    /* Calculate the new canvas area. */

    wind_calc(WC_WORK, owin->gadgets,
	      owin->frame.g_x, owin->frame.g_y,
	      owin->frame.g_w, owin->frame.g_h,
	      &owin->canvas.g_x, &owin->canvas.g_y,
	      &owin->canvas.g_w, &owin->canvas.g_h);

    /* Move the object to the window location. */

    obj[0].ob_x += owin->frame.g_x - tempx;
    obj[0].ob_y += owin->frame.g_y - tempy;

    /* The object is drawn by the redraw message which is
     * automatically sent when the window is opened.
     */

    wind_open(owin->handle,
	      owin->frame.g_x, owin->frame.g_y,
	      owin->frame.g_w, owin->frame.g_h);

    owin->opened = 1;

  }
  else {
    wind_set(owin->handle, WF_TOP, 0, 0, 0, 0);
  }

  return 0;

}

/* Close a window of any type. */

int
WinClose(WINDOW *win)
{
  VWRK *vw = &VWork;
  PSTATE *st = &State;

  /* Close and delete the window; hide the object. */
    
  if (win->iconified) {

    wind_delete(win->handle);

    win->handle = -1;
    win->iconified = 0;

    if (iconwin.obj != iconobj) {
      WinClose(&iconwin);
      UpdateIconList(iconwin.obj, vw);
      ObjWinOpen(iconwin.obj, &iconwin, vw, st);
    }

  }
  else if (win->opened) {

    wind_get(win->handle, WF_CURRXYWH,
	     &win->frame.g_x, &win->frame.g_y,
	     &win->frame.g_w, &win->frame.g_h);

    wind_get(win->handle, WF_CURRXYWH,
	     &win->oframe.g_x, &win->oframe.g_y,
	     &win->oframe.g_w, &win->oframe.g_h);

    wind_close(win->handle);
    wind_delete(win->handle);

    if (win->type == OBJ) {
      OBJECT *obj = win->obj;
      obj[0].ob_flags ^= HIDETREE;
    }

    win->handle = -1;
    win->opened = 0;
  }

  return 0;
}

/* Arrow-button callback for bit windows. */

void
bitw_arrow(PSTATE *st)
{
  WINDOW *bwin = WinFindH(st->Event->Message[3]);
  GRAPHIC *gr = bwin->obj;
  VWRK *vw = &VWork;

  switch (st->Event->Message[4]) {

  case WA_UPLINE:	/* up line or top of page */
    if (st->Event->KeyState)
      gr->PlotY = 0;
    else
      gr->PlotY = MAX(gr->PlotY - vw->Hchar, 0);
    break;

  case WA_DNLINE:	/* down line or bottom of page */
    if (st->Event->KeyState)
      gr->PlotY = gr->Height - bwin->canvas.g_h;
    else
      gr->PlotY = MIN(gr->PlotY + vw->Hchar,
		      gr->Height - bwin->canvas.g_h);
    break;

  case WA_LFLINE:	/* left line or left page edge */
    if (st->Event->KeyState)
      gr->PlotX = 0;
    else
      gr->PlotX = MAX(gr->PlotX - vw->Wchar, 0);
    break;

  case WA_RTLINE:	/* right page edge */
    if (st->Event->KeyState)
      gr->PlotX = gr->Width - bwin->canvas.g_w;
    else
      gr->PlotX = MIN(gr->PlotX + vw->Wchar, 
		      gr->Width - bwin->canvas.g_w);
    break;

  case WA_UPPAGE:	/* up page */
    gr->PlotY = MAX(gr->PlotY - gr->StepY, 0);
    break;

  case WA_DNPAGE:	/* down page */
    gr->PlotY = MIN(gr->PlotY + gr->StepY,
		    gr->Height - bwin->canvas.g_h);
    break;

  case WA_LFPAGE:	/* left page */
    gr->PlotX = MAX(gr->PlotX - gr->StepX, 0);
    break;

  case WA_RTPAGE:	/* right page */
    gr->PlotX = MIN(gr->PlotX + gr->StepX,
		    gr->Width - bwin->canvas.g_w);
    break;

  }

  UpdateScroll(st->Event->Message[3]);
  Cursor(OFF);
  HandleRedraw(FULL_WIN, &State);
  Cursor(ON);

}

/* Arrow-button callback for text windows. */

void
txtw_arrow(PSTATE *st)
{
  int ChangeX, ChangeY;
  int handle = st->Event->Message[3];

  WINDOW *twin  = WinFindH(handle);
  WINTEXT *text = twin->obj;

  switch (st->Event->Message[4]) {

  case WA_UPLINE:	/* top line or up line */
    if (st->Event->KeyState) {
      ChangeY = twin->canvas.g_y - text->top;
      text->top += ChangeY;
      text->cy  += ChangeY;
      text->fdl = text->fl;
      text->ldl = text->fl + text->lins - 1;
      if (text->ldl >= text->bh) text->ldl -= text->bh;
    }
    else {
      if (text->fdl != text->fl) {
	text->top += text->hc;
	text->cy += text->hc;
	if (--text->fdl < 0) text->fdl += text->bh;
	if (--text->ldl < 0) text->fdl += text->bh;
      }
    }
    break;

  case WA_DNLINE:	/* bottom line or down line */
    if (st->Event->KeyState) {
      ChangeY = twin->canvas.g_y
	- (text->bh - text->lins) * text->hc - text->top;
      text->top += ChangeY;
      text->cy  += ChangeY;
      text->fdl = text->ll - text->lins + 1;
      text->ldl = text->ll;
      if (text->fdl < 0) text->fdl += text->bh;
    }
    else {
      if (text->ldl != text->ll) {
	text->top -= text->hc;
	text->cy  -= text->hc;
	if (++text->fdl >= text->bh) text->fdl = 0;
	if (++text->ldl >= text->bh) text->ldl = 0;
      }
    }
    break;

  case WA_LFLINE:	/* left line */
    if (st->Event->KeyState) {
      ChangeX = twin->canvas.g_x - text->edge;
      text->edge += ChangeX;
      text->cx   += ChangeX;
      text->fdc = 0;
      text->ldc = text->cols - 1;
    }
    else {
      if (text->fdc > 0) {
	text->edge += text->wc;
	text->cx  += text->wc;
	text->fdc--;
	text->ldc--;
      }
    }
    break;

  case WA_RTLINE:	/* right line */
    if (st->Event->KeyState) {
      ChangeX = twin->canvas.g_x
	- (text->bw + 2 - text->cols) * text->wc - text->edge;
      text->edge += ChangeX;
      text->cx   += ChangeX;
      text->fdc = (text->bw + 2 - text->cols);
      text->ldc = text->fdc + text->cols - 1;
    }
    else {
      if ((text->fdc + text->cols) < (text->bw + 2)) {
	text->edge -= text->wc;
	text->cx  -= text->wc;
	text->fdc++;
	text->ldc++;
      }
    }
    break;

  case WA_UPPAGE:	/* up page */
    {
      int to_top = text->fdl - text->fl;
      to_top = (to_top >= 0) ? to_top : to_top + text->bh;
      
      if (to_top >= (text->lins - 1)) {
	text->top += (text->lins - 1) * text->hc;
	text->cy  += (text->lins - 1) * text->hc;
	text->fdl -= (text->lins - 1);
	text->ldl  -= (text->lins - 1);
	if (text->fdl < 0) text->fdl += text->bh;
	if (text->ldl < 0) text->ldl += text->bh;
      }
      else {
	ChangeY = twin->canvas.g_y - text->top;
	text->top += ChangeY;
	text->cy  += ChangeY;
	text->fdl = text->fl;
	text->ldl = text->fl + text->lins - 1;
	if (text->ldl >= text->bh) text->ldl -= text->bh;
      }
      break;
    }

  case WA_DNPAGE:	/* down page */
    {
      int to_bot = text->ll - text->ldl;
      to_bot = (to_bot >= 0) ? to_bot : to_bot + text->bh;
      
      if (to_bot >= (text->lins - 1)) {
	text->top -= (text->lins - 1) * text->hc;
	text->cy  -= (text->lins - 1) * text->hc;
	text->fdl += (text->lins - 1);
	text->ldl += (text->lins - 1);
	if (text->fdl >= text->bh) text->fdl -= text->bh;
	if (text->ldl >= text->bh) text->ldl -= text->bh;
      }
      else {
	ChangeY = twin->canvas.g_y
	  - (text->bh - text->lins) * text->hc - text->top;
	text->top += ChangeY;
	text->cy  += ChangeY;
	text->fdl = text->ll - text->lins + 1;
	text->ldl = text->ll;
	if (text->fdl < 0) text->fdl += text->bh;
      }
      break;
    }

  case WA_LFPAGE:	/* left page */
    if (text->fdc  >= (text->cols - 1)) {
      text->edge += (text->cols - 1) * text->wc;
      text->cx   += (text->cols - 1) * text->wc;
      text->fdc  -= (text->cols - 1);
      text->ldc = text->fdc + text->cols - 1;
    }
    else {
      ChangeX = twin->canvas.g_x - text->edge;
      text->edge += ChangeX;
      text->cx   += ChangeX;
      text->fdc = 0;
      text->ldc = text->cols - 1;
    }
    break;

  case WA_RTPAGE:	/* right page */
    if (text->fdc <= (text->bw - 2*text->cols + 3)) {
      text->edge -= (text->cols - 1) * text->wc;
      text->cx  -= (text->cols - 1) * text->wc;
      text->fdc += (text->cols - 1);
      text->ldc += (text->cols - 1);
    }
    else {
      ChangeX = twin->canvas.g_x
	- (text->bw + 2 - text->cols) * text->wc - text->edge;
      text->edge += ChangeX;
      text->cx   += ChangeX;
      text->fdc = (text->bw + 2 - text->cols);
      text->ldc = text->fdc + text->cols - 1;
    }
    break;
    
  }

  UpdateScroll(st->Event->Message[3]);
  Cursor(OFF);
  HandleRedraw(FULL_WIN, &State);
  Cursor(ON);
  
}

/* Horizontal slider callback for text windows. */

void
txtw_hslide(PSTATE *st)
{
  int ChangeX;
  int handle = st->Event->Message[3];

  WINDOW  *twin = WinFindH(handle);
  WINTEXT *text = twin->obj;

  text->fdc = (text->bw + 2 - text->cols) * st->Event->Message[4]/1000;
  text->ldc = text->fdc + text->cols - 1;

  ChangeX = twin->canvas.g_x - text->fdc * text->wc - text->edge;

  text->edge += ChangeX;
  text->cx   += ChangeX;

  UpdateScroll(handle);
  Cursor(OFF);
  HandleRedraw(FULL_WIN, st);
  Cursor(ON);
}

/* Horizontal slider callback for bitw windows. */

void
bitw_hslide(PSTATE *st)
{
  WINDOW *bwin = WinFindH(st->Event->Message[3]); 
  GRAPHIC *gr = bwin->obj;

  gr->PlotX = (gr->Width - bwin->canvas.g_w) * st->Event->Message[4]/1000;

  UpdateScroll(st->Event->Message[3]);
  Cursor(OFF);
  HandleRedraw(FULL_WIN, st);
  Cursor(ON);
}

/* Vertical slider callback for text windows. */

void
txtw_vslide(PSTATE *st)
{
  int ChangeY;
  WINTEXT *text = ObjFindH(st->Event->Message[3]);

  int OldOffset = text->fdl - text->fl;
  int NewOffset = (text->bh - text->lins) * st->Event->Message[4]/1000;

  if (OldOffset < 0) OldOffset += text->bh;

  text->fdl = text->fl + NewOffset;
  if (text->fdl >= text->bh) text->fdl -= text->bh;

  text->ldl = text->fdl + text->lins - 1;
  if (text->ldl >= text->bh) text->ldl -= text->bh;

  ChangeY = (OldOffset - NewOffset) * text->hc;
  text->top += ChangeY;
  text->cy  += ChangeY;

  UpdateScroll(st->Event->Message[3]);
  Cursor(OFF);
  HandleRedraw(FULL_WIN, st);
  Cursor(ON);
}

/* Vertical slider callback for bit windows. */

void
bitw_vslide(PSTATE *st)
{
  WINDOW *bwin = WinFindH(st->Event->Message[3]);
  GRAPHIC *gr = bwin->obj;

  gr->PlotY = (gr->Height - bwin->canvas.g_h) * st->Event->Message[4]/1000;

  UpdateScroll(st->Event->Message[3]);
  Cursor(OFF);
  HandleRedraw(FULL_WIN, st);
  Cursor(ON);
}

/* Move callback for object windows. */

void
objw_move(PSTATE *st)
{
  int ChangeX, ChangeY;

  WINDOW *owin = WinFindH(st->Event->Message[3]);
  OBJECT *obj = owin->obj;

  wind_set(owin->handle, WF_CURRXYWH,
	   st->Event->Message[4], st->Event->Message[5],
	   st->Event->Message[6], st->Event->Message[7]);

  ChangeX = st->Event->Message[4] - owin->frame.g_x;
  ChangeY = st->Event->Message[5] - owin->frame.g_y;

  obj[0].ob_x += ChangeX;
  obj[0].ob_y += ChangeY;

  wind_get(owin->handle, WF_CURRXYWH,
	   &owin->frame.g_x, &owin->frame.g_y,
	   &owin->frame.g_w, &owin->frame.g_h);

  wind_get(owin->handle, WF_WORKXYWH,
	   &owin->canvas.g_x, &owin->canvas.g_y,
	   &owin->canvas.g_w, &owin->canvas.g_h);

}

/* Move callback for bit windows. */

void
bitw_move(PSTATE *st)
{

  WINDOW *bwin = WinFindH(st->Event->Message[3]);

  wind_set(bwin->handle, WF_CURRXYWH,
	   st->Event->Message[4], st->Event->Message[5],
	   MIN(bwin->mframe.g_w, st->Event->Message[6]),
	   MIN(bwin->mframe.g_h, st->Event->Message[7]));

  wind_get(bwin->handle, WF_CURRXYWH,
	   &bwin->frame.g_x, &bwin->frame.g_y,
	   &bwin->frame.g_w, &bwin->frame.g_h);

  wind_get(bwin->handle, WF_WORKXYWH,
	   &bwin->canvas.g_x, &bwin->canvas.g_y,
	   &bwin->canvas.g_w, &bwin->canvas.g_h);
}

/* Move callback for text windows. */

void
txtw_move(PSTATE *st)
{
  int ChangeX, ChangeY;

  WINDOW *twin = WinFindH(st->Event->Message[3]);
  WINTEXT *text = twin->obj;

  int oldcanvas_x = twin->canvas.g_x;
  int oldcanvas_y = twin->canvas.g_y;

  wind_set(twin->handle, WF_CURRXYWH,
	   align(st->Event->Message[4]), st->Event->Message[5],
	   MIN(twin->mframe.g_w, st->Event->Message[6]),
	   MIN(twin->mframe.g_w, st->Event->Message[7]));

  wind_get(twin->handle, WF_CURRXYWH,
	   &twin->frame.g_x, &twin->frame.g_y,
	   &twin->frame.g_w, &twin->frame.g_h);

  wind_get(twin->handle, WF_WORKXYWH,
	   &twin->canvas.g_x, &twin->canvas.g_y,
	   &twin->canvas.g_w, &twin->canvas.g_h);

  ChangeX = twin->canvas.g_x - oldcanvas_x;
  ChangeY = twin->canvas.g_y - oldcanvas_y;

  text->edge += ChangeX;
  text->top  += ChangeY;

  text->cx += ChangeX;
  text->cy += ChangeY;
}

/* Size callback for text windows. */

void
txtw_size(PSTATE *st)
{

  int handle = st->Event->Message[3];

  WINDOW *twin = WinFindH(handle);
  WINTEXT *text = twin->obj;

  int offset, diff, ChangeX;

  int old_x = twin->canvas.g_x;
  int old_y = twin->canvas.g_y;
  int old_w = twin->canvas.g_w;
  int old_h = twin->canvas.g_h;

  wind_set(twin->handle, WF_CURRXYWH,
	   st->Event->Message[4], st->Event->Message[5],
	   MIN(twin->mframe.g_w, st->Event->Message[6]),
	   MIN(twin->mframe.g_h, st->Event->Message[7]));

  wind_get(twin->handle, WF_CURRXYWH,
	   &twin->frame.g_x, &twin->frame.g_y,
	   &twin->frame.g_w, &twin->frame.g_h);

  wind_get(twin->handle, WF_WORKXYWH,
	   &twin->canvas.g_x, &twin->canvas.g_y,
	   &twin->canvas.g_w, &twin->canvas.g_h);

  text->lins = VisibleLines(twin);
  text->cols = VisibleCols(twin);

  /* For the console window, put the current line at the bottom
   * of the window and adjust the first and last lines accordingly.
   * For other text windows, leave the first line unchanged if
   * possible.
   */

  if (twin->handle == conswin.handle) {
    text->fdl = text->ln - text->lins + 1;

    if (text->fdl > text->bh) {
      text->fdl -= text->bh;
    }
    else if (text->fdl < 0 && !text->scrolled) {
      text->fdl = 0;
    }
    else if (text->fdl < 0 && text->scrolled) {
      text->fdl += text->bh;
    }
  }

  text->ldl = text->fdl + text->lins - 1;

  if (text->ldl >= text->bh) {
    if (text->scrolled) {
      text->ldl -= text->bh;
    }
    else {		/* First displayed line must be changed. */
      text->ldl = text->bh - 1;
      text->fdl = text->ldl - text->lins + 1;
    }
  }

  /* Keep the first column the same, if possible, and adjust the
   * last column appropriately.
   */

  text->ldc = text->fdc + text->cols - 1;
  ChangeX = twin->canvas.g_x - old_x;

  if ((diff = text->ldc - text->bw - 1) > 0) {  /* Change first column. */
    ChangeX += diff * text->wc;
    text->ldc = text->bw + 1;
    text->fdc = text->ldc - text->cols + 1;
  }

  /* Adjust the cursor position. */

  text->cx += ChangeX;

  offset = text->ln - text->fdl;
  if (offset < 0) offset += text->bh;
  if (offset >= text->bh) offset -= text->bh;
  text->cy = twin->canvas.g_y + text->yoff + offset * text->hc;

  /* Adjust the left edge and top of the page. */

  text->edge += ChangeX;

  offset = text->fdl - text->fl;
  if (offset < 0) offset += text->bh;
  if (offset >= text->bh) offset -= text->bh;
  text->top = twin->canvas.g_y - offset * text->hc;

  /* Redraw the window if the OS won't. */

  if (old_x <= twin->canvas.g_x &&
      old_y <= twin->canvas.g_y &&
      old_w >= twin->canvas.g_w &&
      old_h >= twin->canvas.g_h &&
      twin->opened) {

    Cursor(OFF);
    HandleRedraw(FULL_WIN, st);
    Cursor(ON);
  }

  UpdateScroll(handle);
}

/* Size callback for bit windows. */

void
bitw_size(PSTATE *st)
{

  WINDOW *bwin = WinFindH(st->Event->Message[3]);
  GRAPHIC *gr = bwin->obj;

  wind_set(bwin->handle, WF_CURRXYWH,
	   st->Event->Message[4], st->Event->Message[5],
	   MIN(bwin->mframe.g_w, st->Event->Message[6]),
	   MIN(bwin->mframe.g_h, st->Event->Message[7]));

  wind_get(bwin->handle, WF_CURRXYWH,
	   &bwin->frame.g_x, &bwin->frame.g_y,
	   &bwin->frame.g_w, &bwin->frame.g_h);

  wind_get(bwin->handle, WF_WORKXYWH,
	   &bwin->canvas.g_x, &bwin->canvas.g_y,
	   &bwin->canvas.g_w, &bwin->canvas.g_h);

  gr->StepX = .9 * bwin->canvas.g_w;
  gr->StepY = .9 * bwin->canvas.g_h;

  UpdateScroll(st->Event->Message[3]);
}

/* Redraw callback for text windows. */

void
txtw_redraw(int full_win, PSTATE *st)
{
  int handle = st->Event->Message[3];

  WINDOW *twin = WinFindH(handle);
  WINTEXT *text = twin->obj;

  int i, cnt;
  int x = align(twin->canvas.g_x + text->xoff);
  int fcol = text->fdc;
  int y, TopLineY, ln;

  GRECT dirty, rect;

  if (full_win) {
    dirty.g_x = twin->canvas.g_x;
    dirty.g_y = twin->canvas.g_y;
    dirty.g_w = twin->canvas.g_w;
    dirty.g_h = twin->canvas.g_h;
  }
  else {
    dirty.g_x = st->Event->Message[4];
    dirty.g_y = st->Event->Message[5];
    dirty.g_w = st->Event->Message[6];
    dirty.g_h = st->Event->Message[7];
  }

  TopLineY = twin->canvas.g_y + text->yoff;

  cnt = text->ln - text->fdl + 1;
  cnt = (cnt > 0) ? cnt : cnt + text->bh;
  cnt = MIN(cnt, text->lins);

  wind_get(twin->handle, WF_FIRSTXYWH,
	   &rect.g_x, &rect.g_y,
	   &rect.g_w, &rect.g_h);

  graf_mouse(M_OFF, 0L);

  while (rect.g_w && rect.g_h) {

    if (rc_intersect(&dirty, &rect)) {
      if (rc_intersect(&twin->canvas, &rect) &&
	  rc_intersect(&VWork.full, &rect)) {
	
	y = TopLineY;

	grect_to_array(&rect, st->pxy);
	vs_clip(VWork.VdiHandle, 1, st->pxy);
		
	ClearWin(&rect);

	for (ln=text->fdl, i=cnt; i; i--, ln++, y+=text->hc) {
	  if (ln >= text->bh) ln -= text->bh;
	  if (strlen(TEXTLINE(text, ln)) > fcol)
	    v_gtext(VWork.VdiHandle, x, y, TEXTLINE(text, ln)+fcol);
	}

	vs_clip(VWork.VdiHandle, 0, st->pxy);
      }
    }

    wind_get(twin->handle, WF_NEXTXYWH,
	     &rect.g_x, &rect.g_y,
	     &rect.g_w, &rect.g_h);
  }

  graf_mouse(M_ON, 0L);

}

/* Redraw callback for object windows. */

void
objw_redraw(int full_win, PSTATE *st)
{

  int handle = st->Event->Message[3];

  WINDOW *owin = WinFindH(handle);
  OBJECT *obj = owin->obj;

  GRECT dirty, rect;

  if (full_win) {
    dirty.g_x = owin->canvas.g_x;
    dirty.g_y = owin->canvas.g_y;
    dirty.g_w = owin->canvas.g_w;
    dirty.g_h = owin->canvas.g_h;
  }
  else {
    dirty.g_x = st->Event->Message[4];
    dirty.g_y = st->Event->Message[5];
    dirty.g_w = st->Event->Message[6];
    dirty.g_h = st->Event->Message[7];
  }

  wind_get(handle, WF_FIRSTXYWH,
	   &rect.g_x, &rect.g_y,
	   &rect.g_w, &rect.g_h);

  graf_mouse(M_OFF, 0L);

  while (rect.g_w && rect.g_h) {

    if (rc_intersect(&dirty, &rect)) {
      if (rc_intersect(&owin->canvas, &rect)) {
	if (rc_intersect(&VWork.full, &rect)) {
	  objc_draw(obj, 0, 3,
		    rect.g_x, rect.g_y,
		    rect.g_w, rect.g_h);
	}
      }
    }

    wind_get(handle, WF_NEXTXYWH,
	     &rect.g_x, &rect.g_y,
	     &rect.g_w, &rect.g_h);

  }

  graf_mouse(M_ON, 0L);

}

/* Redraw callback for bit windows. */

void
bitw_redraw(int full_win, PSTATE *st)
{

  WINDOW *bwin = WinFindH(st->Event->Message[3]);
  GRAPHIC *gr  = bwin->obj;

  GRECT dirty, rect;

  if (full_win) {
    dirty.g_x = bwin->canvas.g_x;
    dirty.g_y = bwin->canvas.g_y;
    dirty.g_w = bwin->canvas.g_w;
    dirty.g_h = bwin->canvas.g_h;
  }
  else {
    dirty.g_x = st->Event->Message[4];
    dirty.g_y = st->Event->Message[5];
    dirty.g_w = st->Event->Message[6];
    dirty.g_h = st->Event->Message[7];
  }

  gr->PlotX = MIN(gr->PlotX, gr->Width - bwin->canvas.g_w);
  gr->PlotY = MIN(gr->PlotY, gr->Height - bwin->canvas.g_h);

  wind_get(bwin->handle, WF_FIRSTXYWH,
	   &rect.g_x, &rect.g_y,
	   &rect.g_w, &rect.g_h);

  graf_mouse(M_OFF, 0L);

  while (rect.g_w && rect.g_h) {
    if (rc_intersect(&dirty, &rect)) {
      if (rc_intersect(&bwin->canvas, &rect) &&
	  rc_intersect(&VWork.full, &rect)) {
	
	st->pxy[0] = gr->PlotX + (rect.g_x - bwin->canvas.g_x);
	st->pxy[1] = gr->PlotY + (rect.g_y - bwin->canvas.g_y);
	st->pxy[2] = st->pxy[0] + rect.g_w - 1;
	st->pxy[3] = st->pxy[1] + rect.g_h - 1;

	grect_to_array(&rect, &st->pxy[4]);

	vs_clip(VWork.VdiHandle, 1, &st->pxy[4]);
	vro_cpyfm(VWork.VdiHandle, 3, st->pxy,
		  &gr->image, &gr->screen);
	vs_clip(VWork.VdiHandle, 0, &st->pxy[4]);

      }
    }

    wind_get(bwin->handle, WF_NEXTXYWH,
	     &rect.g_x, &rect.g_y,
	     &rect.g_w, &rect.g_h);
  }

  graf_mouse(M_ON, 0L);

}

/* Button callback for the resolution dialog. */

int
res_button(OBJECT object[], int index, WINDOW *owin, VWRK *vw,
	PSTATE *st, GRAPHIC *gr, stream_cursor_write *pw, byte **dest)
{

  char nl[2]="";

  byte cone, cten, chun;

  short  cindex=0, done=0, redraw=1;
  static short applied=0;

  int Resolution;

  static unsigned long oldone=0x30FF1100L;
  static unsigned long oldten=0x38FF1100L;
  static unsigned long oldhun=0x30FF1100L;

  switch (index) {

  case RES_OUP:
    if (!cindex) cindex = RES_ONE;

  case RES_TUP:
    if (!cindex) cindex = RES_TEN;

  case RES_HUP:
    if (!cindex) cindex = RES_HUN;

    if (object[cindex].ob_spec < 0x39000000L) {
      object[cindex].ob_spec += 0x01000000L;
    }
    else {
      object[cindex].ob_spec -= 0x09000000L;
    }

    ObjcDraw(object, cindex, 0, owin);

    applied = 0;
    break;

  case RES_ODN:
    if (!cindex) cindex = RES_ONE;

  case RES_TDN:
    if (!cindex) cindex = RES_TEN;

  case RES_HDN:
    if (!cindex) cindex = RES_HUN;

    if (object[cindex].ob_spec > 0x31000000L) {
      object[cindex].ob_spec -= 0x01000000L;
    }
    else {
      object[cindex].ob_spec += 0x09000000L;
    }

    ObjcDraw(object, cindex, 0, owin);

    applied = 0;
    break;

  case RES_DEF:
    object[RES_ONE].ob_spec = 0x30FF1100L;
    object[RES_TEN].ob_spec = 0x38FF1100L;
    object[RES_HUN].ob_spec = 0x30FF1100L;

    ObjcDraw(object, RES_HUN, 0, owin);
    ObjcDraw(object, RES_TEN, 0, owin);
    ObjcDraw(object, RES_ONE, 0, owin);

    applied = 0;
    break;

  case RES_CAN:
    object[RES_ONE].ob_spec = oldone;
    object[RES_TEN].ob_spec = oldten;
    object[RES_HUN].ob_spec = oldhun;
    WinClose(owin);
    WList = WinListDelete(WList, owin);
    redraw = 0;
    break;

  case RES_DONE:
    if (applied) {
      WinClose(owin);
      WList = WinListDelete(WList, owin);
      redraw = 0;
      break;
    }
    done = 1;

  case RES_APP:
    if (done) {
      WinClose(owin);
      WList = WinListDelete(WList, owin);
      redraw = 0;
    }

    oldone = object[RES_ONE].ob_spec;
    oldten = object[RES_TEN].ob_spec;
    oldhun = object[RES_HUN].ob_spec;

    cone = *(byte *)(&object[RES_ONE].ob_spec);
    cten = *(byte *)(&object[RES_TEN].ob_spec);
    chun = *(byte *)(&object[RES_HUN].ob_spec);

    Resolution = 100 * (chun-'0') + 10 * (cten-'0') + (cone-'0');

    gr->XRes = Resolution;
    gr->YRes = nint(Resolution * vw->AspectRatio);

    sprintf(st->Command,
      "%smark /HWResolution [%d %d] currentdevice putdeviceprops setdevice\n",
      nl, gr->XRes, gr->YRes);

    SendCommand(st, pw, dest);

    applied = 1;
    break;

  }

  ObjcChange(object, index, 0, owin,
	     (object[index].ob_state ^ SELECTED), redraw);

  return 0;
}

/* Button callback for the about window. */

int
abo_button(OBJECT object[], int index, WINDOW *owin, VWRK *vw,
	PSTATE *st, GRAPHIC *gr, stream_cursor_write *pw, byte **dest)
{

  short redraw=1;

  ObjcChange(object, index, 0, owin,
	     (object[index].ob_state ^ SELECTED), redraw);

  switch (index) {

  case ABO_GEN:
    HandleHelp(GENHELP, vw, st, pw, dest);
    break;

  case ABO_FONT:
    HandleHelp(FONTHELP, vw, st, pw, dest);
    break;

  case ABO_ERR:
    HandleHelp(ERRHELP, vw, st, pw, dest);
    break;

  case ABO_FEAT:
    HandleHelp(FEATHELP, vw, st, pw, dest);
    break;
  }

  return 0;
}

/* Button callback for the device window. */

int
dev_button(OBJECT object[], int index, WINDOW *owin, VWRK *vw,
	PSTATE *st, GRAPHIC *gr, stream_cursor_write *pw, byte **dest)
{

  char nl[2]="";

  short done=0, redraw=1;
  static short applied=0;
  unsigned short state, screendev;

  switch (index) {

  case DEV_DEF:

    ObjcChange(object, index, 0, owin,
	       (object[index].ob_state ^ SELECTED), 1);

    index = FIRSTDEV;

    ObjcChange(object, index, 0, owin,
	       (object[index].ob_state ^ SELECTED), 1);

    ObjcChange(object, st->Device, 0, owin,
	       (object[st->Device].ob_state ^ SELECTED), 1);

    st->Device = index;
    applied = 0;
    break;

  default:
		
    ObjcChange(object, st->Device, 0, owin,
	       (object[st->Device].ob_state ^ SELECTED), 1);

    st->Device = index;
    applied = 0;
    break;

  case DEV_CAN:

    ObjcChange(object, st->Device, 0, owin,
	       (object[st->Device].ob_state ^ SELECTED), 1);

    ObjcChange(object, st->LastDevice, 0, owin,
	       (object[st->LastDevice].ob_state ^ SELECTED), 1);

    ObjcChange(object, index, 0, owin,
	       (object[index].ob_state ^ SELECTED), 0);

    WinClose(owin);
    WList = WinListDelete(WList, owin);
    st->Device = st->LastDevice;

    break;

  case DEV_DONE:
    if (applied) {
      WinClose(owin);
      WList = WinListDelete(WList, owin);

      ObjcChange(object, index, 0, owin,
		 (object[index].ob_state ^ SELECTED), 0);
      
      break;
    }
    done = 1;

    /* Intentional fall-through. */

  case DEV_APP:

    /* Close the window if we are done. */

    if (done) {
      WinClose(owin);
      WList = WinListDelete(WList, owin);
      redraw = 0;
    }

    /* Update the print dialog and redraw if it is open. */
    
    strcpy(st->DevString, (char *)object[st->Device].ob_spec);

    if (printwin.opened) {
      st->Event->Message[3] = printwin.handle;

      objc_offset(printobj, PR_DSTR,
		  &st->Event->Message[4], &st->Event->Message[5]);
      st->Event->Message[6] = printobj[PR_DSTR].ob_width;
      st->Event->Message[7] = printobj[PR_DSTR].ob_height;
      objw_redraw(0, st);

    }

    if (LastPrnOutButton) {
      ObjcChange(printobj, LastPrnOutButton, 0, &printwin,
		 (printobj[LastPrnOutButton].ob_state ^ SELECTED),
		 printwin.opened);
    }

    screendev = !strcmp(st->DevString, "stvdi");

    state = printobj[PR_PRN].ob_state;
    ObjcChange(printobj, PR_PRN, 0, &printwin,
	       ((screendev) ? state | DISABLED : state & ~ DISABLED),
	       printwin.opened);

    state = printobj[PR_CEN].ob_state;
    ObjcChange(printobj, PR_CEN, 0, &printwin,
	       ((screendev) ? state | DISABLED : state & ~ DISABLED),
	       printwin.opened);

    state = printobj[PR_OFILE].ob_state;
    ObjcChange(printobj, PR_OFILE, 0, &printwin,
	       ((screendev) ? state | DISABLED : state & ~ DISABLED),
	       printwin.opened);

    LastPrnOutButton = 0;

    /* Send the command to change the device to the interpreter. */

    sprintf(st->Command, "%s(%s) selectdevice\n",
	    nl, (char *)object[st->Device].ob_spec);

    SendCommand(st, pw, dest);

    redraw = 1;
    applied = 1;
    st->LastDevice = st->Device;

    ObjcChange(object, index, 0, owin,
	       (object[index].ob_state ^ SELECTED), redraw);

    break;

  }

  return 0;

}

/* Button callback for the page size window. */

int
siz_button(OBJECT object[], int index, WINDOW *owin, VWRK *vw,
	PSTATE *st, GRAPHIC *gr, stream_cursor_write *pw, byte **dest)
{

  char nl[2]="";

  short done=0, redraw=1;
  static short applied=0, select=FIRSTSIZ, oldselect=FIRSTSIZ;

  switch (index) {

  case SIZ_DEF:

    ObjcChange(object, index, 0, owin,
	       (object[index].ob_state ^ SELECTED), 1);

    index = FIRSTSIZ;

    ObjcChange(object, index, 0, owin,
	       (object[index].ob_state ^ SELECTED), 1);

    ObjcChange(object, select, 0, owin,
	       (object[select].ob_state ^ SELECTED), 1);

    select = index;
    applied = 0;

    break;

  default:
		
    ObjcChange(object, select, 0, owin,
	       (object[select].ob_state ^ SELECTED), 1);

    select = index;
    applied = 0;

    break;

  case SIZ_CAN:

    ObjcChange(object, select, 0, owin,
	       (object[select].ob_state ^ SELECTED), 1);

    ObjcChange(object, oldselect, 0, owin,
	       (object[oldselect].ob_state ^ SELECTED), 1);

    ObjcChange(object, index, 0, owin,
	       (object[index].ob_state ^ SELECTED), 0);

    WinClose(owin);
    WList = WinListDelete(WList, owin);
    select = oldselect;

    break;

  case SIZ_DONE:
    if (applied) {
      WinClose(owin);
      WList = WinListDelete(WList, owin);

      ObjcChange(object, index, 0, owin,
		(object[index].ob_state ^ SELECTED), 0);

      break;
    }
    done = 1;

  case SIZ_APP:

    if (done) {
      WinClose(owin);
      WList = WinListDelete(WList, owin);
      redraw = 0;
    }

    sprintf(st->Command, "%s%s\n",
	    nl, (char *)object[select].ob_spec);

    SendCommand(st, pw, dest);

    redraw = 1;
    applied = 1;
    oldselect = select;

    ObjcChange(object, index, 0, owin,
	       (object[index].ob_state ^ SELECTED), redraw);

    break;

  }

  return 0;
}

/* Button callback for the print window. */

int
prn_button(OBJECT object[], int index, WINDOW *owin, VWRK *vw,
	PSTATE *st, GRAPHIC *gr, stream_cursor_write *pw, byte **dest)
{

  char nl[2]="";

  int Select=1, Deselect=0;

  switch (index) {

  case PR_DEF:

    *st->Command = '\0';

    if (st->Device != FIRSTDEV + 1) {
      st->Device = FIRSTDEV + 1;
      strcpy(st->DevString, devices[1]);
      sprintf(st->Command, "%s(%s) selectdevice\n", nl, devices[1]);
    }

    if (printwin.opened) {
      st->Event->Message[3] = printwin.handle;
      objc_offset(printobj, PR_DSTR,
		  &st->Event->Message[4], &st->Event->Message[5]);
      st->Event->Message[6] = printobj[PR_DSTR].ob_width;
      st->Event->Message[7] = printobj[PR_DSTR].ob_height;

      objw_redraw(0, st);
    }

    if (printobj[PR_PRN].ob_state & DISABLED) {

      ObjcChange(printobj, PR_PRN, 0, &printwin,
		 (printobj[PR_PRN].ob_state & ~ DISABLED),
		 printwin.opened);

      ObjcChange(printobj, PR_CEN, 0, &printwin,
		 (printobj[PR_CEN].ob_state & ~ DISABLED),
		 printwin.opened);

      ObjcChange(printobj, PR_OFILE, 0, &printwin,
		 (printobj[PR_OFILE].ob_state & ~ DISABLED),
		 printwin.opened);
    }

    if (LastPrnOutButton != PR_CEN) {

      if (LastPrnOutButton) {
	ObjcChange(object, LastPrnOutButton, 0, owin,
		   (object[LastPrnOutButton].ob_state ^ SELECTED), 1);
      }

      ObjcChange(object, PR_CEN, 0, owin,
		 (object[PR_CEN].ob_state ^ SELECTED), 1);

      LastPrnOutButton = PR_CEN;

      strcpy(st->Outfile, "CEN:");

      strcat(st->Command,
	"mark /OutputFile (CEN:) currentdevice putdeviceprops setdevice\n");

    }

    if (strlen(st->Command)) {
      SendCommand(st, pw, dest);
    }

    DevObjUpdate(st);
    Deselect = 1;
    break;

  case PR_CAN:
    WinClose(owin);
    WList = WinListDelete(WList, owin);
    Deselect = 1;
    break;

  case PR_PR:
    sprintf(st->Command, "%s(%s%s) run\n", nl, st->InDir, st->Infile);
    SendCommand(st, pw, dest);
    Deselect = 1;
    break;

  case PR_DEV:
    WinListAdd(WList, &devwin);
    ObjWinOpen(devobj, &devwin, vw, st);
    Deselect = 1;
    break;

  case PR_IFILE:
    /* Use any previously selected input file as the default. */

    if (strlen(st->Infile)) {
      strcpy(st->FileSel, st->Infile);
    }
    else {
      strcpy(st->FileSel, "");
    }

    if ((Select = GetFile(st, "*.*")) != 0) {
      strcpy(st->InDir, st->DirSpec);
      strcpy(st->Infile, st->FileSel);
    }

    if (printwin.opened) {
      st->Event->Message[3] = printwin.handle;
      objc_offset(printobj, PR_FSTR,
		  &st->Event->Message[4], &st->Event->Message[5]);
      st->Event->Message[6] = printobj[PR_FSTR].ob_width;
      st->Event->Message[7] = printobj[PR_FSTR].ob_height;

      objw_redraw(0, st);
    }

    Deselect = 1;
    break;

  case PR_CEN:	/* directly to centronics hardware */

    if (LastPrnOutButton != PR_CEN) {

      if (LastPrnOutButton) {
	ObjcChange(object, LastPrnOutButton, 0, owin,
		   (object[index].ob_state ^ SELECTED), 1);
      }
	  
      LastPrnOutButton = PR_CEN;
    }
    else {
      ObjcChange(object, PR_CEN, 0, owin,
		 (object[index].ob_state ^ SELECTED), 1);
    }

    if (strcmp(st->Outfile, "CEN:")) {
      strcpy(st->Outfile, "CEN:");
      sprintf(st->Command,
	"%smark /OutputFile (CEN:) currentdevice putdeviceprops setdevice\n",
	 nl);
      SendCommand(st, pw, dest);
    }

    break;

  case PR_PRN:	/* centronics port via gemdos */

    if (LastPrnOutButton != PR_PRN) {

      if (LastPrnOutButton) {
	ObjcChange(object, LastPrnOutButton, 0, owin,
		   (object[index].ob_state ^ SELECTED), 1);
      }

      LastPrnOutButton = PR_PRN;

    }
    else {
      ObjcChange(object, PR_PRN, 0, owin,
		 (object[index].ob_state ^ SELECTED), 1);
    }
	
    if (strcmp(st->Outfile, "PRN:")) {
      strcpy(st->Outfile, "PRN:");
      sprintf(st->Command,
	"%smark /OutputFile (PRN:) currentdevice putdeviceprops setdevice\n",
	 nl);
      SendCommand(st, pw, dest);
    }

    break;

  case PR_OFILE:	/* output to 'Outfile' */

    if (LastPrnOutButton != PR_OFILE) {

      if (LastPrnOutButton) {
	ObjcChange(object, LastPrnOutButton, 0, owin,
		   (object[index].ob_state ^ SELECTED), 1);
      }

      LastPrnOutButton = PR_OFILE;
    }
    else {
      ObjcChange(object, PR_OFILE, 0, owin,
		 (object[index].ob_state ^ SELECTED), 1);
    }

    /* Use any previously selected output file as the default. */

    if (strlen(st->Outfile)) {
      strcpy(st->FileSel, st->Outfile);
    }
    else {
      strcpy(st->FileSel, "");
    }

    if ((Select = GetFile(st, "*.*")) != 0) {
      strcpy(st->OutDir, st->DirSpec);
      strcpy(st->Outfile, st->FileSel);
      sprintf(st->Command,
	"%smark /OutputFile (%s%s) currentdevice putdeviceprops setdevice\n",
	 nl, st->OutDir, st->Outfile);
      SendCommand(st, pw, dest);
    }

    break;
  }

  if (Deselect) {
    ObjcChange(object, index, 0, owin,
	       (object[index].ob_state ^ SELECTED), 1);
  }

  return 0;
}

/* Button callback for the page select window. */

int
pag_button(OBJECT object[], int index, WINDOW *owin, VWRK *vw,
	PSTATE *st, GRAPHIC *gr, stream_cursor_write *pw, byte **dest)
{
  ObjcChange(object, index, 0, owin,
	     (object[index].ob_state ^ SELECTED), 1);

  graf_mouse(BUSY_BEE, 0L);
  HandlePageSelect(vw, st, pw, dest, index);
  graf_mouse(ARROW, 0L);

  return 0;
}

/* Button callback for the icon window. */

int
ico_button(OBJECT object[], int index, WINDOW *owin, VWRK *vw,
	PSTATE *st, GRAPHIC *gr, stream_cursor_write *pw, byte **dest)
{

  short Count=0, Found=0;
  
  OBJECT *obj = object;

  WINLIST *wl = WList, *fwl = WList;

  ObjcChange(object, index, 0, owin,
	     (object[index].ob_state ^ SELECTED), 1);

  if (!st->IconManager) {
    return 0;
  }

  if (iconwin.opened) {

    /* Close the icon window */
    WinClose(&iconwin);

    if (obj == iconobj) {

      /* Add a closer gadget for the icon list window. */
      iconwin.gadgets |= CLOSER;

      /* Change to the icon list object. */
      obj = (OBJECT *)iconwin.obj = CreateIconListObj();

      UpdateIconObj(obj, vw);

    }
    else {

      /* Find the iconified window corresponding to the clicked button. */

      do {
	if (wl->Win->iconified) {

	  if (++Count == index) {

	    Found = 1;
	    fwl = wl;

	    wl->Win->iconified = 0;
	  }
	}
	wl = wl->Next;
      }
      while (wl != WList);

      UpdateIconList(obj, vw);

    }

    if (Count == 1 && Found) {

      /* The last iconified window has been opened --
       * close the icon window.
       */

      WList = WinListDelete(WList, &iconwin);

    }
    else {

      ObjWinOpen(iconwin.obj, &iconwin, vw, st);

    }

    if (Found) {

      /* Uniconify the selected window. */

      wind_open(fwl->Win->handle,
		fwl->Win->frame.g_x, fwl->Win->frame.g_y,
		fwl->Win->frame.g_w, fwl->Win->frame.g_h);

      fwl->Win->opened = 1;

      if (fwl->Win->type == OBJ) {
	OBJECT *obj = fwl->Win->obj;
	obj[0].ob_flags ^= HIDETREE;
      }

    }

  }

  return 0;
}

/* Button callback to ignore the button click. */

int
ign_button(OBJECT object[], int index, WINDOW *owin, VWRK *vw,
	PSTATE *st, GRAPHIC *gr, stream_cursor_write *pw, byte **dest)
{
  ObjcChange(object, index, 0, owin,
	     (object[index].ob_state ^ SELECTED), 1);
  
  return 0;
}

/* Update the scroll bars. */

int
UpdateScroll(int handle)
{
  int hslide_pos, hslide_siz, vslide_pos, vslide_siz;

  WINDOW *win = WinFindH(handle);

  switch (win->type) {

  case BIT:
    {
      GRAPHIC *gr = win->obj;

      if (gr->Width != win->canvas.g_w) {
	hslide_pos = 1000 * gr->PlotX/(gr->Width - win->canvas.g_w);
      }
      else {
	hslide_pos = 0;
      }

      if (gr->Height != win->canvas.g_h) {
	vslide_pos = 1000 * gr->PlotY
	  / (gr->Height - win->canvas.g_h);
      }
      else {
	vslide_pos = 0;
      }

      hslide_siz = 1000 * win->canvas.g_w/gr->Width;
      vslide_siz = 1000 * win->canvas.g_h/gr->Height;

    }
    break;

  case TEXT:
    {
      WINTEXT *text = win->obj;

      int offset = text->fdl - text->fl;
      
      if (offset < 0) offset += text->bh;

      if (text->cols != text->bw + 2) {
	hslide_pos = 1000 * text->fdc/(text->bw + 2 - text->cols);
      }
      else {
	hslide_pos = 0;
      }

      if (text->lins < text->bh) {
	vslide_pos = 1000 * offset/(text->bh - text->lins);
      }
      else {
	vslide_pos = 0;
      }

      hslide_siz = 1000 * text->cols/(text->bw + 2);
      vslide_siz = 1000 * text->lins/text->bh;

    }
    break;

  default:
    hslide_pos = vslide_pos = 0;
    hslide_siz = vslide_siz = 1000;
    break;
  }

  wind_set(handle, WF_HSLIDE,  hslide_pos);
  wind_set(handle, WF_VSLIDE,  vslide_pos);
  wind_set(handle, WF_HSLSIZE, hslide_siz);
  wind_set(handle, WF_VSLSIZE, vslide_siz);

  return 0;
}

/* Find a window from x-y coordinates. */

WINDOW *
WinFindXY(int MouseX, int MouseY)
{
  int handle = wind_find(MouseX, MouseY);

  WINLIST *wl = WList;

  do {
    if (handle == wl->Win->handle) {
      return wl->Win;
    }
    wl = wl->Next;
  }
  while (wl != WList);
  
  return NULL;

}

/* Find a window from the handle. */

WINDOW *
WinFindH(int handle)
{
  WINLIST *wl = WList;

  do {
    if (handle == wl->Win->handle) {
      return wl->Win;
    }
    wl = wl->Next;
  }
  while (wl != WList);

  return NULL;

}

/* WinListAdd adds a new window to the window list. */

WINLIST *
WinListAdd(WINLIST *wlist, WINDOW *win)
{
  WINLIST *wl = WinListAlloc();

  if (win->opened || win->iconified) return wlist;

  if (wlist == NULL) {	/* Begin a new list. */

    wl->Prev = wl;
    wl->Next = wl;

  }
  else {			/* Add a new list element. */

    wl->Prev = wlist->Prev;
    wl->Next = wlist;

    wlist->Prev->Next = wl;
    wlist->Prev = wl;

  }

  wl->Win = win;		/* Put the window in the list. */
  return wl;
}


/* Delete a window from the window list and free its associated memory. */

WINLIST *
WinListDelete(WINLIST *wlist, WINDOW *win)
{
  WINLIST *ret, *wl = wlist;

  /*
   * Return the new head of the window list. If the head was not
   * deleted, it remains the head. If the head is deleted, make
   * the next list element the head. If deleting the head leaves
   * the list empty, then return NULL.
   */

  ret = (win == wlist->Win) ?
    ((wlist->Next == wlist) ? NULL : wlist->Next) : wlist;

  /* Delete the specified list element. */

  do {
    if (win == wl->Win) {
      wl->Prev->Next = wl->Next;
      wl->Next->Prev = wl->Prev;
      break;
    }
    wl = wl->Next;
  }
  while (wl != wlist);

  /* Free the memory for the window structure. */
  if ((win->type == TEXT) && (win != &conswin)) {
    WinTextFree(win->obj);
    WinTitleFree(win->title);
    WinFree(win);
  }
  else if ((win->type == BIT) && (win != &imagwin)) {
    GRAPHIC *gr = win->obj;
    BitBufferFree(gr);
    GraphicFree(gr);
    WinTitleFree(win->title);
    WinFree(win);
  }

  /* Free the memory used for the window list. */
  WinListFree(wl);

  return ret;
}

/* Open all the windows in the window list, leaving the list intact. */

void
WinListOpen(WINLIST *wlist)
{
  WINLIST *wl = wlist;

  do {
    if (wl->Win->iconified) {
      wind_open(wl->Win->handle,
		wl->Win->frame.g_x, wl->Win->frame.g_y,
		wl->Win->frame.g_w, wl->Win->frame.g_h);

      wl->Win->opened = 1;
      wl->Win->iconified = 0;

      if (wl->Win->type == OBJ) {
	OBJECT *obj = wl->Win->obj;
	obj[0].ob_flags ^= HIDETREE;
      }
    }

    wl = wl->Next;
  }
  while (wl != wlist);
}

/* Close all the windows in the window list, leaving the list intact. */

void
WinListClose(WINLIST *wlist)
{
  WINLIST *wl = wlist;

  do {
    if (wl->Win->opened) {
      
      wind_get(wl->Win->handle, WF_CURRXYWH,
	       &wl->Win->oframe.g_x, &wl->Win->oframe.g_y,
	       &wl->Win->oframe.g_w, &wl->Win->oframe.g_h);

      wind_get(wl->Win->handle, WF_CURRXYWH,
	       &wl->Win->frame.g_x, &wl->Win->frame.g_y,
	       &wl->Win->frame.g_w, &wl->Win->frame.g_h);

      wind_close(wl->Win->handle);
      wl->Win->opened = 0;
      wl->Win->iconified = 1;

      if (wl->Win->type == OBJ) {
	OBJECT *obj = wl->Win->obj;
	obj[0].ob_flags ^= HIDETREE;
      }

    }

    wl = wl->Next;
  }
  while (wl != wlist);
}

/* Redraw all windows in the list. */

void
WinListRedraw(WINLIST *wlist, PSTATE *st)
{
  WINLIST *wl = wlist;

  do {
    if (wl->Win->opened) {
      st->Event->Message[3] = wl->Win->handle;
      HandleRedraw(FULL_WIN, st);
    }

    wl = wl->Next;
  }
  while (wl != wlist);
}

/* Ensure that the size of a window is within the limits of
 * the hardware.
 */

void
EnforceLimits(GRECT *rect, VWRK *vw)
{

  /* Is the x coordinate of the window off of the screen? */

  if ((rect->g_x < vw->full.g_x) || (rect->g_x >= vw->full.g_w)) {
    rect->g_x = vw->full.g_x;
  }

  /* Is the y coordinate of the window off of the screen? */
  
  if ((rect->g_y < vw->full.g_y) || (rect->g_y >= vw->full.g_h)) {
    rect->g_y = vw->full.g_y;
  }

  /* Is the width positive? */

  if (rect->g_w < 0) {
    rect->g_w = vw->full.g_w - rect->g_x + vw->full.g_x;
  }

  /* Is the height positive? */

  if (rect->g_h < 0) {
    rect->g_h = vw->full.g_h - rect->g_y + vw->full.g_y;
  }
   	
  /* Clip portions of the window which are off the screen. */

  if ((rect->g_x + rect->g_w) > (vw->full.g_x + vw->full.g_w)) {
    rect->g_w = vw->full.g_w - rect->g_x + vw->full.g_x;
  }

  if ((rect->g_y + rect->g_h) > (vw->full.g_y + vw->full.g_h)) {
    rect->g_h = vw->full.g_h - rect->g_y + vw->full.g_y;
  }
}

/* Push the window around to assure that it is all on the screen. */

void
MoveIntoLimits(GRECT *rect, VWRK *vw)
{

  int HorizEnd, VertEnd;

  /* Make sure the window is on the screen in the horizontal direction. */

  if ((rect->g_x < vw->full.g_x) || (rect->g_x >= vw->full.g_w)) {
    rect->g_x = vw->full.g_x;
  }
  else if ((HorizEnd = rect->g_x+rect->g_w) > (vw->full.g_x + vw->full.g_w)) {
    rect->g_x -= HorizEnd - vw->full.g_x - vw->full.g_w + 2;
  }

  /* Make sure the window is on the screen in the vertical direction. */

  if ((rect->g_y < vw->full.g_y) || (rect->g_y >= vw->full.g_h)) {
    rect->g_y = vw->full.g_y;
  }
  else if ((VertEnd = rect->g_y + rect->g_h) > (vw->full.g_y + vw->full.g_h)) {
    rect->g_y -= VertEnd - vw->full.g_y - vw->full.g_h + 2;
  }

}
