/* Copyright (C) 1989, 1992, 1993 Aladdin Enterprises.  All rights reserved.

This file is part of Ghostscript.

Ghostscript is distributed in the hope that it will be useful, but
WITHOUT ANY WARRANTY.  No author or distributor accepts responsibility
to anyone for the consequences of using it or for whether it serves any
particular purpose or works at all, unless he says so in writing.  Refer
to the Ghostscript General Public License for full details.

Everyone is granted permission to copy, modify and redistribute
Ghostscript, but only under the conditions described in the Ghostscript
General Public License.  A copy of this license is supposed to have been
given to you along with Ghostscript so you can know your rights and
responsibilities.  It should be in a file named COPYING.  Among other
things, the copyright notice and this notice must be preserved on all
copies.  */

/* gp_atar1.c */

/*
 * Atari-specific routines for Ghostscript. This file contains the
 * standard suite of gp_* routines called by Ghostscript plus
 * a few utility functions used by them. TOS file system functions
 * are in gp_tosfs.c.
 *
 * GEM initialization is performed in gp_init.
 */

#include <stdlib.h>
#include <gemdefs.h>
#include <aesbind.h>
#include <vdibind.h>
#include <mintbind.h>
#include <support.h>
#include <unistd.h> /* for getcwd() */

#include "string_.h"
#include "gx.h"
#include "gsexit.h"
#include "gp.h"
#include "time_.h"
#include "gxdcolor.h"
#include "gserrors.h"
#include "math_.h"
#include "stream.h"

#include "st_rsc.h"
#include "st_rsc.c"	/* File containing resource definitions. */
#include "st_defs.h"
#include "gdevvdi.h"
#include "gp_atar1.h"
#include "gp_atar2.h"
#include "gp_atar3.h"
#include "gp_atar4.h"

/* Device dialog object. */

char devices[DEVICES][DNAMELEN] = {
    "stvdi", "djet500",
    "bj10e", "necp6",
    "cdj500", "cdj550",
    "epson",  "slm"
};

/* Page size dialog object. */

char sizes[SIZES][DNAMELEN] = {
    "letter", "legal",
    "11x17",  "ledger",
    "a1",     "a2",
    "a3",     "a4"
};

char saveline[MAXLEN];

OBJECT *menuobj, *aboutobj, *iconobj;
OBJECT *resobj, *devobj, *sizobj;
OBJECT *printobj, *savemsg;

/* External routines */

extern FILE *popen(P2(const char *, const char *));
extern int pclose(P1(FILE *));
extern void gs_exit(P1(int exit_status));	/* from gsmain.c */

/* Global Variables */

/* Virtual Workstation Structure. */

VWRK VWork = {
    0,				/* VdiHandle */
    0,				/* XRes */
    0,				/* YRes */
    0,				/* AspectRatio */
    {0, 0, 0, 0},		/* full */
    0,				/* ColorBits */
    0,				/* MaxRGB */
    0,				/* TrueColor */
    0,				/* PaletteSize */
    0,				/* GSPalette */
    0,				/* Palette */
    0,				/* OldPalette */
    0,				/* ColorReg */
    0,				/* Character cell width */
    0,				/* Character cell height */
    0,				/* Character box width */
    0				/* Character box height */
};

DOCUMENT document;		/* Contains PS document information. */

/* Program State Structure. */

PSTATE State = {
    0,				/* ApId */
    0,				/* MultiTOS */
    0,				/* HasIconifier */
    1,                          /* IconManager */
    1,				/* Windows */
    0,				/* TopWin */
    0,				/* PSHelp */
    0,				/* SelectPages */
    FIRSTDEV,                   /* Device */
    FIRSTDEV,                   /* LastDevice */
    "",				/* CurrentDir[] */
    "",				/* DirSpec[] */
    "",				/* FileSel[] */
    "",                         /* InDir[] */
    "",                         /* OutDir[] */
    "",				/* Infile[] */
    "",				/* Outfile[] */
    "",				/* Command[] */
    "",                         /* DevString[] */
    1,				/* PaletteCount */
    0,				/* Chunky8 */
    0,				/* Gdebug */
    0,				/* Debug8 */
    0,				/* Debug16 */
    {0, 0, 0, 0, 0, 0, 0, 0},	/* pxy[] */
    &Event,			/* Event */
    &document			/* Doc */
};

/* Private Functions */

private void GemInit(VWRK *, PSTATE *);
private int InitPalette(VWRK *, PSTATE *);
private void AdjustObjects(VWRK *);

/* Gp_init does Atari-dependent initialization. It reads
 * Atari-specific environment variables, initializes the
 * AES, opens a virtual workstation, creates the windows,
 * displays the menu, opens the console, and performs various
 * other initializations.
 */

void
gp_init(void)
{
    char *env, temp[MAXLEN];
    char *TitleString;
    static char PrgName[] = "  ST-GS";		/* CF */

    int i, print_disable=0;

    WINTEXT *text = conswin.obj;

    /* Check the GS_WIN environment variable to decide what to
     * do about windows. Windows are used unless GS_WIN contains
     * the string "off".
     */

    if ((env = getenv("GS_WIN")) != NULL) {
	if (!strcmp(env, "off") || !strcmp(env, "OFF")) {
	    State.Windows = 0;		
        }
    }

    /* Find the current directory. */

    getcwd(temp, MAXLEN);
    unx2dos(temp, State.CurrentDir);

    strcat(State.CurrentDir, "\\");
    strcpy(State.InDir, State.CurrentDir);
    strcpy(State.OutDir, State.CurrentDir);

    /* Initialize the AES and assign most of the VWork structure. */

    GemInit(&VWork, &State);

    if (State.Windows) {

      /* Initialization for GEM objects. */
	
      gp_atrsc_rsc_load();
	
      gp_atrsc_rsc_gaddr(R_TREE, MENUOBJ,  &menuobj);
      gp_atrsc_rsc_gaddr(R_TREE, ABOUTOBJ, &aboutobj);
      gp_atrsc_rsc_gaddr(R_TREE, RESOBJ,   &resobj);
      gp_atrsc_rsc_gaddr(R_TREE, ICONOBJ,  &iconobj);
      gp_atrsc_rsc_gaddr(R_TREE, DEVOBJ,   &devobj);
      gp_atrsc_rsc_gaddr(R_TREE, SIZOBJ,   &sizobj);
      gp_atrsc_rsc_gaddr(R_TREE, PROBJ,   &printobj);
      gp_atrsc_rsc_gaddr(R_TREE, SAVEMSG,  &savemsg);
	
      aboutwin.obj = aboutobj;
      reswin.obj   = resobj;
      devwin.obj   = devobj;
      sizwin.obj   = sizobj;
      iconwin.obj  = iconobj;
      printwin.obj = printobj;
      pagewin.obj  = NULL;
	
      /* Fix object sizes for current resolution. */
	
      AdjustObjects(&VWork);

      /* Set the string pointers for the device dialog. */

      for (i = 0; i < DEVICES; i++)
	devobj[i+FIRSTDEV].ob_spec = UL devices[i];

      /* Set the string pointers for the page size dialog. */

      for (i = 0; i < SIZES; i++)
	sizobj[i+FIRSTSIZ].ob_spec = UL sizes[i];
	
      savemsg[SAVELINE].ob_spec = UL saveline;
	
      /* Set the string pointers for the print dialog. */

      printobj[PR_FSTR].ob_spec = UL State.Infile;
      printobj[PR_DSTR].ob_spec = UL State.DevString;

      /* Allocate and initialize the title string for the image window. */

      if ((TitleString = gs_malloc(MAXTITLE, 1, "Window Title")) != NULL) {
	imagwin.title = TitleString;
	strcpy(imagwin.title, B_TITL);
      }

      /* Initialize the data structures describing the console window.
       * This must be done before calling LoadConfig to prevent the
       * settings in the configuration file from being overridden.
       */

      TextWinInit(&conswin, text, T_GADGETS, T_TITL);

      /* LoadConfig must come before the environment variables are
       * checked, since these variables can be set from the
       * configuration file.
       */

      LoadConfig(&VWork, &State);

    }

    /* Some environment variables override the values
     * returned by the system in GemInit.
     */

    if ((env = getenv("GS_DISPLAY")) != NULL) {

      if (!strcmp(env, "truecolor")) {
	VWork.TrueColor = 1;		
      }
      else if (!strcmp(env, "chunky8") || !strcmp(env, "CHUNKY8")) {
	State.Chunky8 = 1;
      }

    }

    if ((env = getenv("GS_DEBUG")) != NULL) {
	if (!strcmp(env, "gdebug")) {
	    State.Gdebug = 1;
	}
	else if (!strcmp(env, "debug8")) {
	    State.Debug8 = 1;
	    VWork.ColorBits = 8;
	    VWork.TrueColor = 0;		
	}
	else if (!strcmp(env, "debug16")) {
	    State.Debug16 = 1;
	    VWork.ColorBits = 16;
	}
    }

    /* The default device is the stvdi screen device unless indicated
     * otherwise by the GS_DEVICE environment variable.
     */

    if ((env = getenv("GS_DEVICE")) != NULL) {
      for (i=0; (strcmp(env, devices[i])) && (i<DEVICES); ++i);
      
      if (i < DEVICES) {
        State.Device = i;
	strcpy(State.DevString, devices[i]);
	if (!strcmp(State.DevString, "stvdi")) {
	  print_disable = 1;
	}
      }

    }
    else {
      strcpy(State.DevString, "stvdi");
      print_disable = 1;
    }
    
    /* Check whether the current video hardware is supported. */

    if (VWork.ColorBits != 1  && VWork.ColorBits != 2 &&
	VWork.ColorBits != 4  && VWork.ColorBits != 8 &&
	VWork.ColorBits != 16 && VWork.ColorBits != 24) {

	eprintf1("stvdi_open: %d bit color is not supported.\n",
	    VWork.ColorBits);

	gs_exit(-1);

    }

    VWork.MaxRGB = DITH_RGB(VWork.ColorBits) - 1;
	
    /* Initialize the palette. */

    if ((VWork.ColorBits > 1) && !VWork.TrueColor) {
	InitPalette(&VWork, &State);
    }

    if (State.Windows) {

        if (print_disable) {
          ObjcChange(printobj, PR_PRN, 0, &printwin,
              (printobj[PR_PRN].ob_state |= DISABLED),	    
              printwin.opened);

          ObjcChange(printobj, PR_CEN, 0, &printwin,
	      (printobj[PR_CEN].ob_state |= DISABLED),
	      printwin.opened);

          ObjcChange(printobj, PR_OFILE, 0, &printwin,
	      (printobj[PR_OFILE].ob_state |= DISABLED),
	      printwin.opened);
        }

	if (!conswin.opened) {
	    extern int __mint;
		
	    /* Display the menu and open the console window. */

	    RestoreBG(&VWork);
	    graf_mouse(M_OFF, 0L );

	    menu_bar(menuobj, 1);

	    if (State.MultiTOS && __mint) {
	      menu_register(gl_apid, PrgName);
	    }

	    /* Create the head of the window list. */

	    WList = WinListAdd(NULL, &conswin);

	    /* Open the console window. */
            
	    if (TextBuffRealloc(text, LINES) != NULL ) {
		TextWinOpen(&conswin, &VWork, &State);
	    }
	    else {
		graf_mouse(M_ON, 0L);
		/* Window mode turned off, else message will never appear */
		State.Windows = 0;
		eprintf("gp_init: No memory for the console text buffer!\n");
		gs_exit(-1);
	    }

	    graf_mouse(BUSY_BEE, 0L);
	    graf_mouse(M_ON, 0L);

	}
    }
    
    if (State.Gdebug) {
	eprintf1("Color Bits   = %d\n", VWork.ColorBits);
	eprintf1("True Color   = %d\n", VWork.TrueColor);
	eprintf1("Max RGB      = %d\n", VWork.MaxRGB);
	eprintf1("Palette Size = %d\n", VWork.PaletteSize);
	eprintf1("XRes         = %d\n", VWork.XRes);
	eprintf1("YRes         = %d\n", VWork.YRes);
	eprintf1("Wchar        = %d\n", VWork.Wchar);
	eprintf1("Hchar        = %d\n", VWork.Hchar);
	eprintf1("Wbox         = %d\n", VWork.Wbox);
	eprintf1("Hbox         = %d\n", VWork.Hbox);
	eprintf1("AspectRatio  = %f\n", VWork.AspectRatio);
    }

}

private void
GemInit(VWRK *vw, PSTATE *st)
{

    int GemHandle;
    int WorkOut[57];
    int WorkIn[11] = {1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 2};
    int word1, word2, word3, word4;
    
    extern short _global[];
    
    /* Call appl_init() to initialize the aes before using any aes
     * functions.
     */

    if ((st->ApId = appl_init()) == -1) {
        eprintf("stvdi_open: appl_init() failed!\n");
        gs_exit(-1);
    }
    
    /* Find out about special OS features. */

    st->MultiTOS = _global[1] != 1;
    st->HasIconifier = ApplGetinfo(11, &word1, &word2, &word3, &word4) != 0
      && (word3 & 1);
    
    /* Open a virtual workstation. */

    GemHandle = graf_handle(&vw->Wchar, &vw->Hchar, &vw->Wbox, &vw->Hbox);
    vw->VdiHandle = GemHandle;
    v_opnvwk(WorkIn, &VWork.VdiHandle, WorkOut);

    if (vw->VdiHandle == 0) {
	eprintf("stvdi_open: Could not open virtual screen workstation");
	gs_exit(-1);
    }

    /* Get the screen resolution, aspect ratio, and palette size. */

    vw->XRes = WorkOut[0];
    vw->YRes = WorkOut[1];

#if !FIXED_CONSOLE_SIZE
    vw->lines = (vw->YRes + 1) / vw->Hchar;
    if (vw->lines > MAX_LINES)
    	vw->lines = MAX_LINES;
    vw->columns = (vw->XRes + 1) / vw->Wchar;
    if (vw->columns > MAX_COLUMNS)
    	vw->columns = MAX_COLUMNS;
#endif
    
    /* Calculate the aspect ratio. */

    vw->AspectRatio = (double) WorkOut[3] / WorkOut[4];
	
    /* Get the number of color planes. */

    vq_extnd(vw->VdiHandle, 1, WorkOut);

    vw->ColorBits = WorkOut[4];
    vw->TrueColor = (!WorkOut[5] || (vw->ColorBits >= 16));

    /* Get the size of the root window. */

    if (st->Windows) {
	wind_get(0, WF_WORKXYWH,
	    &vw->full.g_x, &vw->full.g_y,
	    &vw->full.g_w, &vw->full.g_h);
    }
}

private int
InitPalette(VWRK *vw, PSTATE *st)
{

    int psize;

    psize = (int)(pow(2.0, (double)vw->ColorBits) + .5);
    vw->PaletteSize = 3 * psize;

    /* Set up the palette--a sequence of rgb values. */


    if ((vw->Palette = (int *)gs_malloc((uint)vw->PaletteSize,
      sizeof(int), "Palette")) == NULL) {
      return_error(gs_error_VMerror);
    }

    if ((vw->OldPalette = (int *)gs_malloc((uint)vw->PaletteSize,
      sizeof(int), "OldPalette")) == NULL) {
      return_error(gs_error_VMerror);
    }

    if ((vw->ColorReg = (int *)gs_malloc((uint)psize,
      sizeof(int), "ColorReg")) == NULL) {
      return_error(gs_error_VMerror);
    }

    if (st->Debug8) {
	int i;

	vw->Palette[0] = 0;
	vw->Palette[1] = 0;
	vw->Palette[2] = 0;

	for (i=3; i<vw->PaletteSize; i++) {
	    vw->Palette[i] = 1;
	}

    }
    else {

	/* Fill the array ColorReg. The value of ColorReg[i]
	 * will be the hardware color register which corresponds
	 * to the ith vdi color index.
	 */

	IndexToPixel((gx_color_index *)vw->ColorReg, psize);

	GetPalette(VWork.Palette, &VWork);	/* Get current palette. */
	GetPalette(VWork.OldPalette, &VWork);	/* Save for restoration. */
	VWork.GSPalette = 1;

    }

    return 0;
}

/* Make resolution-dependent adjustments to the objects. */

private void
AdjustObjects(VWRK *vw)
{
    static char line1[] = "ST Ghostscript 3.33";
    static char line2[] = "Tim Gallivan, 1995";

    /* The icon and the banner bitmaps must be adjusted for the
     * strange aspect ratio in ST medium (and perhaps other)
     * resolutions.
     */

    if (vw->AspectRatio < .6) {
      iconobj[ICONFULL].ob_flags |= HIDETREE;
      iconobj[ICONHALF].ob_flags &= ~HIDETREE;
      iconobj[0].ob_height = iconobj[ICONHALF].ob_height;

      aboutobj[ABO_FULL].ob_flags |= HIDETREE;
      aboutobj[ABO_HALF].ob_flags &= ~HIDETREE;
    } else {
      iconobj[0].ob_height = iconobj[ICONFULL].ob_height;
    }

    /* The About object must be changed to fit on the screen in
     * low resolution. This is a major change to the object.
     */

    if (vw->full.g_w <= aboutobj[0].ob_width) {
	aboutobj[0].ob_width = 28 * vw->Wchar;
	aboutobj[0].ob_height = 15 * vw->Hchar;

	aboutobj[ABO_GEN].ob_y = 7 * vw->Hchar;
	aboutobj[ABO_FONT].ob_y = 9 * vw->Hchar;

	aboutobj[ABO_ERR].ob_x = 2 * vw->Wchar;
	aboutobj[ABO_ERR].ob_y = 11 * vw->Hchar;

	aboutobj[ABO_FEAT].ob_x = 2 * vw->Wchar;
	aboutobj[ABO_FEAT].ob_y = 13 * vw->Hchar;

	aboutobj[ABO_BOX].ob_state = SHADOWED;
	aboutobj[ABO_BOX].ob_spec = 0xFF1111L;
	aboutobj[ABO_BOX].ob_width = 24 * vw->Wchar;
	aboutobj[ABO_BOX].ob_height = 5 * vw->Hchar;

	aboutobj[ABO_FULL].ob_type = G_STRING;
	aboutobj[ABO_FULL].ob_spec = UL line1;
	aboutobj[ABO_FULL].ob_x = 2 * vw->Wchar;
	aboutobj[ABO_FULL].ob_y = 1 * vw->Hchar;
	aboutobj[ABO_FULL].ob_width = 20 * vw->Wchar;
	aboutobj[ABO_FULL].ob_height = 1 * vw->Hchar;

	aboutobj[ABO_LINE].ob_spec = UL line2;
	aboutobj[ABO_LINE].ob_x = 3 * vw->Wchar;
	aboutobj[ABO_LINE].ob_y = 3 * vw->Hchar;
	aboutobj[ABO_LINE].ob_width = 18 * vw->Wchar;

    }
}

/* Gp_exit does Atari-dependent cleanup. It frees memory and AES
 * resources, closes the virtual workstation, and calls appl_exit.
 */

#define EXITMSG "[3][Ghostscript Error %d!|See the console window\
|for more details.][Abort]"

void
gp_exit(int exit_status, int code)
{
    WINLIST *wl = WList;
    WINTEXT *text = conswin.obj;

    char ErrorMsg[MAXLEN];

    if (exit_status != 0 && State.Windows) {
        graf_mouse(ARROW, 0L);

        sprintf(ErrorMsg, EXITMSG, exit_status);
	FormAlert(1, ErrorMsg);

        graf_mouse(BUSY_BEE, 0L);
    }

    /* Restore the palette and free the allocated memory. */

    if (VWork.ColorBits > 1 && !VWork.TrueColor) {

	if (VWork.GSPalette) {
	    SetPalette(VWork.OldPalette, &VWork);

	    gs_free((char *)VWork.Palette, VWork.PaletteSize,
		    sizeof(int), "Palette");

	    gs_free((char *)VWork.OldPalette, VWork.PaletteSize,
		    sizeof(int), "OldPalette");

	    gs_free((char *)VWork.ColorReg, VWork.PaletteSize / 3,
		    sizeof(int), "ColorReg");

	    VWork.GSPalette = 0;
	}

    }

    /* Free the resources used by the aes. */

    if (State.Windows) {
        menu_bar(menuobj, 0);

	/* Free the console character buffer. I probably should treat
	 * the console window the same as other text windows.
	 */

	gs_free((char *)text->buff, text->bsize, (COLUMNS+1), "charbuff");
	text->buff = NULL;
	
	/*
	 * Delete any temporary files for extracted pages and
	 * free any memory allocated for document page information.
	 */

	if (State.Doc->pageinf) {
	    char filename[MAXLEN], temp[10];
	    int i;

	    remove(PROLOG);

	    for (i=1; i <= State.Doc->maxpages; ++i) {
		if (State.Doc->pageinf[i].extracted != 0) {
		    itoa(i, temp);

		    strcpy(filename, PAGE_PREFIX);
		    strcat(filename, temp);
		    strcat(filename, ".ps");
		
		    remove(filename);
		}
	    }

	    gs_free((char *)State.Doc->pageinf, State.Doc->maxpages,
		sizeof(PAGEINFO), "page info");
	}

	/* If memory was allocated for the page dialog, free it. */

	if (pagewin.obj) {
	    DeletePageObj(&State, &pagewin);
	}

	/*
	 * Close and delete all open windows. Delete all window
	 * list entries.
	 */

	do {
	    if (!(wl->Win->iconified)) {
		wind_close(wl->Win->handle);
	    }
	    wind_delete(wl->Win->handle);
	}
	while ((wl = WinListDelete(wl, wl->Win)));

    }

    /* Close the virtual workstation and call appl_exit */

    if (VWork.VdiHandle > 0) {
      v_clsvwk(VWork.VdiHandle);
    }

    if (State.Windows) {
    	gp_atrsc_rsc_free();
        graf_mouse(ARROW, 0L);
    }

    if (appl_exit() == 0) {
        eprintf("stvdi_close: appl_exit() failed!\n");
        exit_status = -1;
    }
}


/* The routines below were copied (and perhaps slightly modified)
 * from the Unix Ghostscript source.
 */

/* ------ Miscellaneous ------ */

/* Get the string corresponding to an OS error number. */
/* Unix systems support this so inconsistently that we don't attempt */
/* to figure out whether it's available. */
const char *
gp_strerror(int errnum)
{	return NULL;
}

/* ------ Date and time ------ */

/* Read the current date (in days since Jan. 1, 1980) */
/* and time (in milliseconds since midnight). */
void
gp_get_clock(long *pdt)
{	long secs_since_1980;
	struct timeval tp;
	time_t tsec;
	struct tm *tm, *localtime(P1(const time_t *));

#if gettimeofday_no_timezone
	{	if ( gettimeofday(&tp) == -1 )
		  {	lprintf("Ghostscript: gettimeofday failed!\n");
			gs_exit(1);
		  }
		secs_since_1980 = 0;
	}
#else
	{	struct timezone tzp;
		if ( gettimeofday(&tp, &tzp) == -1 )
		  {	lprintf("Ghostscript: gettimeofday failed!\n");
			gs_exit(1);
		  }
		/*
		 * Don't adjust for timezone!  That structure returns the
		 * kernel's notion of the timezone, which may be different
		 * from the one specified in the TZ environment variable.
		 * In particular, the tz_minuteswest value is documented to
		 * need adjustment according to the daylight savings time
		 * rules indicated by tz_dsttime.  The latter indicates
		 * only the applicable rules and doesn't answer the
		 * question whether DST currently is in effect or not.
		 */
		/*secs_since_1980 = -(tzp.tz_minuteswest * 60);*/
		secs_since_1980 = 0;
	}
#endif

	/* tp.tv_sec is #secs since Jan 1, 1970 */

	/* subtract off number of seconds in 10 years */
	/* leap seconds are not accounted for */
	secs_since_1980 += tp.tv_sec - (long)(60 * 60 * 24 * 365.25 * 10);

	/* adjust for daylight savings time - assume dst offset is 1 hour */
	tsec = tp.tv_sec;
	tm = localtime(&tsec);
	if ( tm->tm_isdst )
		secs_since_1980 += (60 * 60);

	/* divide secs by #secs/day to get #days (integer division truncates) */
	pdt[0] = secs_since_1980 / (60 * 60 * 24);
	/* modulo + microsecs/1000 gives number of millisecs since midnight */
	pdt[1] = (secs_since_1980 % (60 * 60 * 24)) * 1000;
	/* Some Unix systems (e.g., Interactive 3.2 r3.0) return garbage */
	/* in tp.tv_usec.  Try to filter out the worst of it here. */
	if ( tp.tv_usec >= 0 && tp.tv_usec < 1000000 )
		pdt[1] += tp.tv_usec / 1000;

#ifdef DEBUG_CLOCK
	printf("tp.tv_sec = %d  tp.tv_usec = %d  pdt[0] = %ld  pdt[1] = %ld\n",
		tp.tv_sec, tp.tv_usec, pdt[0], pdt[1]);
#endif
}

/* ------ Screen management ------ */

/* Get the environment variable that specifies the display to use. */
const char *
gp_getenv_display(void)
{	return getenv("DISPLAY");
}

/* ------ Check whether a file is the console. ------ */

int
gp_file_is_console(FILE *f)
{
    if (!f) return 0;
    if (fileno(f) >= 0 && fileno(f) <= 2) return 1;

    return 0;
}

/* ------ Printer accessing ------ */

/* Open a file or a printing device. A null file name will cause
 * output to be sent to "PRN:". If gp_open_printer returns
 * a NULL, the file will be sent directly to the centronics port.
 * This happens if fname = "CEN:" or if gp_open_scratch_file()
 * fails. Usually, GS wants to interpret a NULL return value as
 * an error, so this is slightly incompatible.
 * "|command" opens an output pipe.
 */

FILE *
gp_open_printer(char *fname, int binary_mode)
{
	if (!strcmp(fname, "CEN:")) {	/* Direct Centronics printing. */
	   return NULL;
	}
	else {
	   return
	     (strlen(fname) == 0 ?
	      gp_open_scratch_file(gp_scratch_file_name_prefix, fname, "wb") :
	      fname[0] == '|' ?
	      popen(fname + 1, "wb") :
	      fopen(fname, "wb"));
	}
}

/* Close the connection to the printer. */
void
gp_close_printer(FILE *pfile, const char *fname)
{	if ( fname[0] == '|' )
		pclose(pfile);
	else
		fclose(pfile);
}



