/* Copyright (C) 1989, 1992, 1993 Aladdin Enterprises.  All rights reserved.

This file is part of Ghostscript.

Ghostscript is distributed in the hope that it will be useful, but
WITHOUT ANY WARRANTY.  No author or distributor accepts responsibility
to anyone for the consequences of using it or for whether it serves any
particular purpose or works at all, unless he says so in writing.  Refer
to the Ghostscript General Public License for full details.

Everyone is granted permission to copy, modify and redistribute
Ghostscript, but only under the conditions described in the Ghostscript
General Public License.  A copy of this license is supposed to have been
given to you along with Ghostscript so you can know your rights and
responsibilities.  It should be in a file named COPYING.  Among other
things, the copyright notice and this notice must be preserved on all
copies.  */

/* gp_atar4.c */

/*
 * This file contains callback functions for the user interface of
 * the Atari platform.
 */

#include <gemdefs.h>
#include <aesbind.h>
#include <vdibind.h>
#include <osbind.h>
#include <stdlib.h>
#include <support.h> /* for putenv() */

#include "memory_.h"
#include "string_.h"
#include "gx.h"
#include "gp.h"
#include "stream.h"
#include "math_.h"

#include "st_rsc.h"
#include "st_defs.h"
#include "gp_atar1.h"
#include "gp_atar2.h"
#include "gp_atar3.h"
#include "gp_atar4.h"

int LastPrnOutButton=0;

/* Redraw an object. */

void
ObjcDraw(OBJECT *obj, int index, int max_depth, WINDOW *win)
{
  int handle = win->handle;

  GRECT dirty, rect;

  dirty.g_x = win->canvas.g_x;
  dirty.g_y = win->canvas.g_y;
  dirty.g_w = win->canvas.g_w;
  dirty.g_h = win->canvas.g_h;

  wind_get(handle, WF_FIRSTXYWH,
	   &rect.g_x, &rect.g_y,
	   &rect.g_w, &rect.g_h);

  graf_mouse(M_OFF, 0L);

  while (rect.g_w && rect.g_h) {

    if (rc_intersect(&dirty, &rect)) {
      if (rc_intersect(&VWork.full, &rect)) {
	objc_draw(obj, index, max_depth,
		  rect.g_x, rect.g_y,
		  rect.g_w, rect.g_h);
      }
    }

    wind_get(handle, WF_NEXTXYWH,
	     &rect.g_x, &rect.g_y,
	     &rect.g_w, &rect.g_h);

  }

  graf_mouse(M_ON, 0L);

}

/* Change and object. */

void
ObjcChange(OBJECT *obj, int index, int max_depth,
        WINDOW *win, int NewState, int Redraw)
{
  int handle = win->handle;

  GRECT dirty, rect;
    
  if (win->opened && Redraw)
    {
      dirty.g_x = win->canvas.g_x;
      dirty.g_y = win->canvas.g_y;
      dirty.g_w = win->canvas.g_w;
      dirty.g_h = win->canvas.g_h;
    
      wind_get(handle, WF_FIRSTXYWH,
	       &rect.g_x, &rect.g_y,
	       &rect.g_w, &rect.g_h);
    
      graf_mouse(M_OFF, 0L);
    
      while (rect.g_w && rect.g_h) {
    
	if (rc_intersect(&dirty, &rect)) {
	  if (rc_intersect(&VWork.full, &rect)) {
	    objc_change(obj, index, max_depth,
                        rect.g_x, rect.g_y,
                        rect.g_w, rect.g_h,
                        NewState, Redraw);
	  }
	}
    
	wind_get(handle, WF_NEXTXYWH,
		 &rect.g_x, &rect.g_y,
		 &rect.g_w, &rect.g_h);
    
      }
    
      graf_mouse(M_ON, 0L);
    }
  else {
    obj[index].ob_state = NewState;
  }
}

/* WinAlloc allocates memory for a new window. */

WINDOW *
WinAlloc(void)
{
  return (WINDOW *) gs_malloc((uint)sizeof(WINDOW), 1, "window");
}

/* Free the memory allocated for a window. */

void
WinFree(WINDOW *win)
{
  gs_free((char *)win, 1, sizeof(WINDOW), "window");
}

/* WinTextAlloc allocates memory for a window text structure. */

WINTEXT *
WinTextAlloc(int Lines)
{
  WINTEXT *text;

  /* Allocate the WINTEXT structure. */

  if ((text = (WINTEXT *) gs_malloc((uint)sizeof(WINTEXT), 1, "wintext"))
      == NULL) {
    return text;
  }

  /* Allocate the character buffer. */

#if FIXED_CONSOLE_SIZE
  if ((text->buff = (char (*)[COLUMNS+1])gs_malloc((uint)(COLUMNS+1),
		    (uint)Lines, "charbuff")) == NULL)
#else
  if ((text->buff = (char *)gs_malloc((uint)(COLUMNS+1),
		    (uint)Lines, "charbuff")) == NULL)
#endif
    {
      gs_free((char *)text, 1, sizeof(WINTEXT), "wintext");
      return (WINTEXT *)NULL;
    }

  text->bsize = Lines;

  return text;
}

char
#if FIXED_CONSOLE_SIZE
(*TextBuffRealloc(WINTEXT *text, int NewSize))[COLUMNS+1]
#else
*TextBuffRealloc(WINTEXT *text, int NewSize)
#endif
{
#if FIXED_CONSOLE_SIZE
  char (*temp)[COLUMNS+1];
#else
  char *temp;
#endif

  /* Allocate another character buffer of the new size. */

#if FIXED_CONSOLE_SIZE
  if ((temp = (char (*)[COLUMNS+1])gs_malloc((uint)(COLUMNS+1),
				(uint)NewSize, "charbuff")) == NULL)
#else
  if ((temp = (char *)gs_malloc((uint)(COLUMNS+1),
				(uint)NewSize, "charbuff")) == NULL)
#endif
    {
      return temp;
    }

  memset(temp, 0, (uint)(COLUMNS+1) * NewSize);
	
  if (text->buff != 0) {

    /* Copy the old buffer to the new one. */
    memcpy(temp, text->buff, (uint)((MIN(text->bsize, NewSize))*(COLUMNS+1)));

    /* Free the old character buffer. */
    gs_free((char *)text->buff, text->bsize, (COLUMNS+1), "charbuff");

  }

  text->bsize = NewSize;
  text->buff = temp;
  return temp;
}

/* Free the memory allocated for a window text structure. */

void
WinTextFree(WINTEXT *text)
{
  gs_free((char *)text->buff, text->bsize, (COLUMNS+1), "charbuff");
  gs_free((char *)text, 1, sizeof(WINTEXT), "wintext");
}

/* Free the space allocated for a window title. */

void
WinTitleFree(char *title)
{
  gs_free((char *)title, MAXTITLE, 1, "Window Title");
}

/* GraphicAlloc allocates memory for a graphic structure. */

GRAPHIC *
GraphicAlloc(void)
{
  return (GRAPHIC *) gs_malloc((uint)sizeof(GRAPHIC), 1, "graphic");
}

/* Free the memory allocated for a graphic structure. */

void
GraphicFree(GRAPHIC *gr)
{
  gs_free((char *)gr, 1, sizeof(GRAPHIC), "graphic");
}

/* Free the memory allocated for a bitmap buffer. */

void
BitBufferFree(GRAPHIC *gr)
{
  char *buff = (char *)gr->image.fd_addr;

  gs_free((char *)buff, (ulong)(gr->image.fd_w * gr->image.fd_h),
	  1, "bitbuffer");
}

/* ListAlloc allocates memory for a new element of the window list. */

WINLIST *
WinListAlloc(void)
{
  return (WINLIST *) gs_malloc((uint)sizeof(WINLIST), 1, "winlist");
}

/* Free the memory for the winlist structure */

void
WinListFree(WINLIST *wl)
{
  gs_free((char *)wl, 1, sizeof(WINLIST), "winlist");
}

/* Allocate memory for an object which lists the available pages. */

OBJECT *
CreatePageObj(int pages)
{
  char (*str)[4];

  int i, j, count, rows, cols, prod;

  OBJECT *pobj;
  TEDINFO *tinf;

  rows = cols = (int)(pow((double) pages, .5) + .1);

  while (rows*cols < pages) {
    ++cols;
  }

  prod = rows * cols;

  /* Put some error checks here! */

  pobj = (OBJECT *)gs_malloc((uint)(prod+1), sizeof(OBJECT), "page object");
  tinf = (TEDINFO *)gs_malloc((uint)pages, sizeof(TEDINFO), "page tedinfo");
  str  = (char (*)[4])gs_malloc((uint)pages, 4, "page strings");

  AssignObj(&pobj[0], -1, 1, prod, G_BOX, HIDETREE, OUTLINED, 0x11100L,
	    0, 0, 4*cols, 2*rows);

  for (j=0, count=0; j<rows; ++j) {
    for (i=0; i<cols; ++i) {
      if (++count <= pages) {
	itoa(count, str[count-1]);

	AssignTinf(&tinf[count-1], str[count-1] , "", "",
		   3, 6, 2, 0x1100, 0x0, 1, 2, 1);

	AssignObj(&pobj[count], count+1, -1, -1, G_BOXTEXT, SELECTABLE,
		  NORMAL, (unsigned long)&tinf[count-1], 4*i, 2*j, 4, 2);
      }
      else {
	AssignObj(&pobj[count], count+1, -1, -1, G_BOX, NONE,
		  NORMAL, 0x11131L, 4*i, 2*j, 4, 2);
      }
    }
  }

  pobj[prod].ob_next = 0;
  pobj[prod].ob_flags |= LASTOB;

  i=-1;
  do {
    ++i;
    rsrc_obfix(pobj, i);
  } while (!(pobj[i].ob_flags & LASTOB));

  return pobj;
}

/* Free memory for the page object. */

int
DeletePageObj(PSTATE *st, WINDOW *pwin)
{
  int rows, cols, prod;
  OBJECT *pobj = (OBJECT *)pwin->obj;
  TEDINFO *tinf = (TEDINFO *)pobj[1].ob_spec;

  rows = cols = (int)(pow((double) st->Doc->numpages, .5) + .1);

  while (rows*cols < st->Doc->numpages) {
    ++cols;
  }

  prod = rows * cols;

  if (pwin->opened || pwin->iconified) {
    WinClose(pwin);
    WList = WinListDelete(WList, pwin);
  }

  gs_free((char *)tinf->te_ptext, st->Doc->numpages, 4, "page strings");

  gs_free((char *)tinf, st->Doc->numpages, sizeof(TEDINFO), "page tedinfo");

  gs_free((char *)pobj, prod+1, sizeof(OBJECT), "page object");

  pwin->obj = NULL;
  pwin->frame.g_x = 0;
  pwin->frame.g_y = 0;
  pwin->frame.g_w = 0;
  pwin->frame.g_h = 0;

  st->Doc->name[0] = '\0';
  st->Doc->maxpages = 0;
  st->Doc->numpages = 0;
  st->Doc->currentpage = 0;

  return 0;
}

/* Assign values to a newly-allocated object. */

void
AssignObj(OBJECT *obj, int next, int head, int tail, int type, int flags,
	int state, unsigned long spec, int x, int y, int width, int height)
{
  obj->ob_next = (short)next;
  obj->ob_head = (short)head;
  obj->ob_tail = (short)tail;
  obj->ob_type = (unsigned short)type;
  obj->ob_flags = (unsigned short)flags;
  obj->ob_state = (unsigned short)state;
  obj->ob_spec = spec;
  obj->ob_x = (short)x;
  obj->ob_y = (short)y;
  obj->ob_width = (short)width;
  obj->ob_height = (short)height;
}

/* Assign values to a newly-allocated TEDINFO structure. */

void
AssignTinf(TEDINFO *inf, char *ptext, const char *ptmplt, const char *pvalid,
	int font, int junk1, int just, int color, int junk2,
	int thickness, int txtlen, int tmplen)
{
  inf->te_ptext = ptext;
  inf->te_ptmplt = (char *)ptmplt;
  inf->te_pvalid = (char *)pvalid;
  inf->te_font = (short)font;
  inf->te_junk1 = (short)junk1;
  inf->te_just = (short)just;
  inf->te_color = (short)color;
  inf->te_junk2 = (short)junk2;
  inf->te_thickness = (short)thickness;
  inf->te_txtlen = (short)txtlen;
  inf->te_tmplen = (short)tmplen;
}

/* Scan a postscript file and sort out the pages. */

int
PageFilter(char* InputName, DOCUMENT *doc)
{
  char line[MAXLEN];

  int PageCount=0, length=0, maxpages=20;

  FILE *InFile;

  PAGEINFO *temp;

  if ((InFile = fopen(InputName, "r")) == NULL) {
    return -1;
  }

  strcpy((char *)doc->name, InputName);
  doc->maxpages = maxpages;

  if (doc->pageinf == NULL) {
    doc->pageinf = (PAGEINFO *)gs_malloc((uint)maxpages, sizeof(PAGEINFO),
					 "page info");
    memset(doc->pageinf, 0, maxpages*sizeof(PAGEINFO));
  }

  fgetpos(InFile, &doc->pageinf[PageCount].start);

  while (fgets(line, MAXLEN, InFile) != NULL) {
    length += strlen(line);
    if (strncmp(line, "%%", 2) == 0) {

      if (strncmp(line, "%%Page:", 7) == 0 ||
	  strncmp(line, "%%Trailer", 9) == 0) {
	
	if (doc->pageinf[PageCount].extracted == 1) {
	  doc->pageinf[PageCount].extracted = -1;
	}

	doc->pageinf[PageCount].length = length;
	length = 0;

	if (++PageCount >= maxpages) {
	  int oldsize = maxpages;
	  maxpages += 20;

	  /* Allocate a larger buffer. */
	  temp = (PAGEINFO *)gs_malloc((uint)maxpages, sizeof(PAGEINFO),
				       "page info");

	  memset(temp, 0, maxpages*sizeof(PAGEINFO));

	  /* Copy the old buffer to the new one. */
	  memcpy(temp, doc->pageinf,
		 (MIN(oldsize, maxpages))*sizeof(PAGEINFO));

	  /* Free the old buffer. */
	  gs_free((char *)doc->pageinf, oldsize, sizeof(PAGEINFO),
		  "page info");

	  doc->pageinf = temp;
	  doc->maxpages = maxpages;
	}

	fgetpos(InFile, &doc->pageinf[PageCount].start);
      }
    }
  }

  doc->pageinf[PageCount].length = length;

  doc->numpages = PageCount-1;

  fclose(InFile);

  return PageCount-1;
}


/* Allocate and initialize a list of iconified windows. */

OBJECT *
CreateIconListObj(void)
{
  short i, Count=0, IconCount=0, longest=0, length;

  OBJECT *pobj;
  
  WINLIST *wl = WList;

  /* Count the number of iconified windows and find the longest`
   * title string.
   */

  do {
    if (wl->Win->iconified) {
      ++IconCount;

      length = strlen(wl->Win->title);

      if (length > longest) {
	longest = length;
      }

    }
    wl = wl->Next;
  }
  while (wl != WList);

  /* Allocate enough memory for the object. */
  /* Put some error checks here! */

  pobj = (OBJECT *)gs_malloc((uint)(IconCount+1), sizeof(OBJECT),
			     "icon list object");

  AssignObj(&pobj[0], -1, 1, IconCount, G_BOX, HIDETREE, OUTLINED, 0x11100L,
	0, 0, longest+2, IconCount);

  do {
    if (wl->Win->iconified) {
      ++Count;
      AssignObj(&pobj[Count], Count+1, -1, -1, G_BUTTON, SELECTABLE,
		NORMAL, UL wl->Win->title, 0, Count-1, longest+2, 1);
    }

    if (Count > IconCount) {
      break;
    }

    wl = wl->Next;
  }
  while (wl != WList);

  pobj[IconCount].ob_next = 0;
  pobj[IconCount].ob_flags |= LASTOB;

  /* Fix the new object for the current resolution. */

  i=-1;
  do {
    ++i;
    rsrc_obfix(pobj, i);
  } while (!(pobj[i].ob_flags & LASTOB));

  return pobj;

}

/* Free the icon-list memory. */

void
IconListFree(OBJECT *obj)
{
  if (obj != iconobj) {
    gs_free((char *)obj, obj->ob_height+1, sizeof(OBJECT),
	    "icon list object");
  }
}

/* Update the icon list by freeing the old one and reallocating
 * a new one. Then adjust the window to the correct size.
 */

void
UpdateIconList(OBJECT *obj, VWRK *vw)
{

  if (obj != iconobj) {

    /* Free the icon list. */
      
    IconListFree(obj);

    /* Fix the icon list to reflect the current state and open it. */

    obj = (OBJECT *)iconwin.obj = CreateIconListObj();

  }

  UpdateIconObj(obj, vw);

}

/* Update the icon window size and position. */

void
UpdateIconObj(OBJECT *obj, VWRK *vw)
{

  /* Calculate the new window parameters. */

  iconwin.canvas.g_w = obj[0].ob_width;
  iconwin.canvas.g_h = obj[0].ob_height;

  wind_calc(WC_BORDER, iconwin.gadgets,
	    iconwin.canvas.g_x, iconwin.canvas.g_y,
	    iconwin.canvas.g_w, iconwin.canvas.g_h,
	    &iconwin.frame.g_x, &iconwin.frame.g_y,
	    &iconwin.frame.g_w, &iconwin.frame.g_h);
  
  /* Move the window onto the screen if necessary. */

  MoveIntoLimits(&iconwin.frame, vw);

  /* Recalculate the work area in case the window was moved. */
  
  wind_calc(WC_WORK, iconwin.gadgets,
	    iconwin.frame.g_x, iconwin.frame.g_y,
	    iconwin.frame.g_w, iconwin.frame.g_h,
	    &iconwin.canvas.g_x, &iconwin.canvas.g_y,
	    &iconwin.canvas.g_w, &iconwin.canvas.g_h);

  /* Move the object to the window location. */

  obj[0].ob_x = iconwin.canvas.g_x;
  obj[0].ob_y = iconwin.canvas.g_y;
}

/* Copy a PostScript page into a temporary file. */

int
PageCopy(int page, char *OutName, DOCUMENT *doc)
{
  char temp[MAXLEN], *buffer;

  int ch, ret;
  uint count;
    
  FILE *InFile, *OutFile;

  if (doc->numpages <= 1) {
    return -1;
  }

  if (page != 0) {
    itoa(page, temp);
    strcat(OutName, temp);
  }

  strcat(OutName, ".ps");

  if (doc->pageinf[page].extracted == 1) {
    return 0;
  }

  if ((InFile = fopen(doc->name, "r")) == NULL) {
    return -1;
  }

  if ((OutFile = fopen(OutName, "wb")) == NULL) {
    return -1;
  }

  fsetpos(InFile, &doc->pageinf[page].start);

  count = doc->pageinf[page].length;

  if ((buffer = (char *)gs_malloc(count, 1, "copy buffer")) != NULL) {

    if ((ret = fread(buffer, 1, count, InFile)) != count)
      eprintf("PageCopy: Read Error!");
    
    if ((ret = fwrite(buffer, 1, count, OutFile)) != count) 
      eprintf("PageCopy: Write Error!");

    gs_free(buffer, count, 1, "copy buffer");

  }
  else {
    
    while (count--) {
      ch = fgetc(InFile);
      fputc(ch, OutFile);
    }
    
  }

  doc->pageinf[page].extracted = 1;

  fclose(InFile);
  fclose(OutFile);
  return(0);
}

/* Find an object from the window handle. */

void *
ObjFindH(int handle)
{
  WINLIST *wl = WList;

  do {
    if (handle == wl->Win->handle) {

      if (wl->Win->type == TEXT) {
	return (WINTEXT *)wl->Win->obj;
      }
      else if (wl->Win->type == OBJ) {
	return (OBJECT *)wl->Win->obj;
      }
      else if (wl->Win->type == BIT) {
	return (GRAPHIC *)wl->Win->obj;
      }

    }
    wl = wl->Next;
  }
  while (wl != WList);

  return NULL;

}

/* Send a command to the PostScript interpreter. */

int
SendCommand(PSTATE *st, stream_cursor_write *pw, byte **dest)
{
  char *sptr;

  /* Change back slashes to forward. */

  while ((sptr = strchr(st->Command,'\\')) != NULL) {
    *sptr = '/';
  }

  /* Copy the command into Ghostscript's input buffer. */

  for (sptr = st->Command; *sptr; sptr++) {
    *++(*dest) = *sptr;
  }

  pw->ptr = *dest;
  
  /* Send a message to output the command after
   * all pending events have been dispatched.
   */

  st->Event->SendMsg[0] = COMMAND;
  appl_write(st->ApId, 16, st->Event->SendMsg);

  return 0;
}

/* Call the file selector and return the selected file. */

int
GetFile(PSTATE *st, const char *template)
{
  int ExitButton;
  char *sptr;
	
  strcpy(st->DirSpec, st->CurrentDir);
  strcat(st->DirSpec, template);

  if (VWork.GSPalette) {   /* Restore system palette. */
    SetPalette(VWork.OldPalette, &VWork);
  }

  fsel_input(st->DirSpec, st->FileSel, &ExitButton);

  if (VWork.GSPalette) {   /* Restore GS palette. */
    SetPalette(VWork.Palette, &VWork);
  }

  if (ExitButton && strlen(st->FileSel)) {
    sptr = strrchr(st->DirSpec, '\\');
    *++sptr = '\0';
    strcpy(st->CurrentDir, st->DirSpec);
  }

  return (ExitButton && strlen(st->FileSel));
}

/*
 * Return index of title belonging to menu entry.
 */

private int
TitleFind(OBJECT *menu, int index)
{
  int title, menui, idx, idx2, titlebox;
	
  titlebox = menu[menu[ROOT].ob_head].ob_head;
  title = menu[titlebox].ob_head;
  menui = menu[menu[ROOT].ob_tail].ob_head;

  do {
    idx = menu[menui].ob_head;

    do {
      if (idx == index) {
	return(title);
      }

      if (menu[idx].ob_head != -1) {
	idx2 = menu[menu[idx].ob_head].ob_head;

	do {
	  if (idx2 == index) {
	    return(title);
	  }
	  idx2 = menu[idx2].ob_next;
	} while (idx2 != menu[idx].ob_head);

      }

      idx = menu[idx].ob_next;

    } while (idx != menui);

    menui = menu[menui].ob_next;
    title = menu[title].ob_next;

  } while (title != titlebox);
  return(-1);
}

/* Fake a menu selection. */

void
MenuWrite(PSTATE *st, int title, int entry)
{

  if (title == -1) {
    title = TitleFind(menuobj, entry);
  }

  st->Event->SendMsg[0] = MN_SELECTED;
  st->Event->SendMsg[1] = st->ApId;
  st->Event->SendMsg[2] = 0;
  st->Event->SendMsg[3] = title;
  st->Event->SendMsg[4] = entry;
  st->Event->SendMsg[5] = 0;
  st->Event->SendMsg[6] = 0;
  st->Event->SendMsg[7] = 0;

  if (title != -1) {
    menu_tnormal(menuobj, title, 0);
  }

  appl_write(st->ApId, 16, st->Event->SendMsg);

}

#ifndef K_CAPSLOCK
# define K_CAPSLOCK 0x10
#endif

/* Test whether a key is a menu hot key. */

private bool
TestEntry(char *str, char chr, int scan, unsigned int state)
{
  char *pchar;
  char vchr;
  int zahl;
	
  pchar = str;

  /* Skip to the end of the string and back up to the last non-blank
   * character.
   */

  while (*pchar)
    pchar++;

  while (*--pchar == ' ')
    ;

  vchr = *pchar;

  if (vchr == '\'' && pchar[-2] == '\'') {
    vchr = *--pchar;
  }

  /* Check for ordinary hot key combinations. */

  if (vchr >= 'a' && vchr <= 'z') {
    vchr += 'A' - 'a';
  }

  if (vchr == chr) {
    vchr = pchar[-1];

    if ((vchr == '^' && (state & K_CTRL)) ||
	((vchr == 7 || vchr == 5) && (state & K_ALT)) ||
	(vchr == ' ' && !(state & (K_CTRL | K_ALT)))) {
      return(1);
    }

    vchr = *pchar;
  }
  
  /* Check for function keys. */

  if (vchr >= '0' && vchr <= '9') {
    zahl = vchr - '0';
    vchr = pchar[-1];

    if (vchr >= '0' && vchr <= '9') {
      zahl += (vchr - '0') * 10;
      vchr = pchar[-2];
    }

    if (vchr == 'F') {
      if (zahl >= 1 && zahl <= 10 && zahl == (scan - 0x3b + 1))
	return(1);
      if (zahl >= 11 && zahl <= 20 && zahl == (scan - 0x54 + 11))
	return(1);
    }
  }

  /* Check for special keys. */

  if ((pchar - str) >= 3) {
    if ((stricmp(pchar-3, "undo") == 0 && scan == 0x61) ||
/*	(stricmp(pchar-3, "help") == 0 && scan == 0x62) || */
	(stricmp(pchar-2, "esc") == 0 && scan == 0x01) ||
	(stricmp(pchar-3, "home") == 0 && scan == 0x47))
      return(1);
  }
  return(0);
}


#define G_STRINGOF(tree, obj) \
	((char *)(tree[obj].ob_spec))

/* Walk the menu structure testing testing for hot keys. */

bool
MenuSearch(PSTATE *st, OBJECT *m_tree)
{
    _KEYTAB *keytbl;

    bool desk = 1;
    
    char chr;

    int scan;
    int key = st->Event->Key;
    int mother_title, child_title;
    int mother_entry, child_entry;

    unsigned int state = st->Event->KeyState;
    
    keytbl = Keytbl((void *)-1l, (void *)-1l, (void *)-1l);

    scan = (key / 256) & 0xff;

    if (Kbshift(-1) & K_CAPSLOCK) {
      chr = ((char *)(keytbl->caps))[scan];
    }
    else if (state & (K_LSHIFT | K_RSHIFT)) {
      chr = ((char *)(keytbl->shift))[scan];
    }
    else {
      chr = ((char *)(keytbl->unshift))[scan];
    }

    if (chr >= 'a' && chr <= 'z') {
      chr += 'A' - 'a';
    }
    
    state &= K_ALT | K_CTRL;
    
    mother_title = m_tree[m_tree[0].ob_head].ob_head;
    child_title  = m_tree[mother_title].ob_head;
    mother_entry = m_tree[m_tree[0].ob_tail].ob_head;

    while (child_title != mother_title && child_title != -1) {
      child_entry = m_tree[mother_entry].ob_head;

      if (!(m_tree[child_title].ob_state & DISABLED) &&
	  !(m_tree[child_title].ob_flags & HIDETREE)) {

	while (child_entry != mother_entry && child_entry != -1) {

	  if (!(m_tree[child_entry].ob_state & DISABLED) &&
	      !(m_tree[child_entry].ob_flags & HIDETREE)) {

	    if (m_tree[child_entry].ob_type == G_STRING ||
		m_tree[child_entry].ob_type == G_BUTTON) {

	      if (TestEntry(G_STRINGOF(m_tree, child_entry), chr,
			     scan, state)) {
		MenuWrite(st, child_title, child_entry);
		return(1);
	      }

	    }

	  }

	  child_entry = m_tree[child_entry].ob_next;

	  /*
	   * don't scan the desktop accessories
	   */

	  if (desk) {

	    if ((m_tree[child_entry].ob_state & DISABLED) &&
		m_tree[child_entry].ob_type == G_STRING &&
		G_STRINGOF(m_tree, child_entry)[0] == '-') {
	      child_entry = mother_entry;
	    }

	    desk = 0;

	  }

	}

      }

      child_title = m_tree[child_title].ob_next;
      mother_entry = m_tree[mother_entry].ob_next;

    }

    return(0);
}

/* Get the color palette. */

void
GetPalette(int *Palette, VWRK *vw)
{
  int ColorIndex;
  int ColorEntry[3];
  unsigned int DefinedColors = vw->PaletteSize/3;

  for (ColorIndex = 0; ColorIndex < DefinedColors; ColorIndex++) {

    vq_color(vw->VdiHandle, ColorIndex, DEFINED_INTENSITY, ColorEntry);

    Palette[3*ColorIndex]   = ColorEntry[0];
    Palette[3*ColorIndex+1] = ColorEntry[1];
    Palette[3*ColorIndex+2] = ColorEntry[2];
      
  }
}

/* Set the color palette. */

void
SetPalette(int *Palette, VWRK *vw)
{ 
  
  int ColorIndex;
  int ColorEntry[3];
  unsigned int DefinedColors = vw->PaletteSize/3;

  for (ColorIndex = 0; ColorIndex < DefinedColors; ColorIndex++) {

    ColorEntry[0] = Palette[3*ColorIndex];
    ColorEntry[1] = Palette[3*ColorIndex+1];
    ColorEntry[2] = Palette[3*ColorIndex+2];
  
    vs_color(vw->VdiHandle, ColorIndex, ColorEntry);
  }
}

/* Set the position of an object window. */

void
SetWinPosition(WINDOW *win, VWRK *vw, PSTATE *st, double x, double y)
{

  GRECT itemp;
  OBJECT *object = win->obj;

  itemp.g_x = nint(x * vw->full.g_w);
  itemp.g_y = nint(y * vw->full.g_h);

  EnforceLimits(&itemp, vw);

  if (win->opened) {
    st->Event->SendMsg[0] = WM_MOVED;
    st->Event->SendMsg[3] = win->handle;
    st->Event->SendMsg[4] = itemp.g_x;
    st->Event->SendMsg[5] = itemp.g_y;
    st->Event->SendMsg[6] = win->frame.g_w;
    st->Event->SendMsg[7] = win->frame.g_h;
    appl_write(st->ApId, 16, st->Event->SendMsg);

  }
  else {
    int ChangeX, ChangeY;

    wind_calc(WC_BORDER, win->gadgets,
	      object[0].ob_x, object[0].ob_y,
	      object[0].ob_width, object[0].ob_height,
	      &win->frame.g_x, &win->frame.g_y,
	      &win->frame.g_w, &win->frame.g_h);

    ChangeX = itemp.g_x - win->frame.g_x;
    ChangeY = itemp.g_y - win->frame.g_y;

    win->frame.g_x = itemp.g_x;
    win->frame.g_y = itemp.g_y;

    win->canvas.g_x += ChangeX;
    win->canvas.g_y += ChangeY;

    object[0].ob_x += ChangeX;
    object[0].ob_y += ChangeY;

  }
}

/* Set the window frame size. */

void
SetWinFrame(WINDOW *win, VWRK *vw, PSTATE *st,
	    double x, double y, double w, double h)
{
  GRECT itemp;

  if (win->opened) {

    itemp.g_x = nint(x * vw->full.g_w);
    itemp.g_y = nint(y * vw->full.g_h);
    itemp.g_w = nint(w * vw->full.g_w);
    itemp.g_h = nint(h * vw->full.g_h);

    EnforceLimits(&itemp, vw);

    st->Event->SendMsg[0] = WM_SIZED;
    st->Event->SendMsg[3] = win->handle;
    st->Event->SendMsg[4] = itemp.g_x;
    st->Event->SendMsg[5] = itemp.g_y;
    st->Event->SendMsg[6] = itemp.g_w;
    st->Event->SendMsg[7] = itemp.g_h;
    appl_write(st->ApId, 16, st->Event->SendMsg);

  }
  else {

    win->frame.g_x = nint(x * vw->full.g_w);
    win->frame.g_y = nint(y * vw->full.g_h);
    win->frame.g_w = nint(w * vw->full.g_w);
    win->frame.g_h = nint(h * vw->full.g_h);

    EnforceLimits(&win->frame, vw);
    
    wind_calc(WC_WORK, win->gadgets,
	      win->frame.g_x, win->frame.g_y,
	      win->frame.g_w, win->frame.g_h,
	      &win->canvas.g_x, &win->canvas.g_y,
	      &win->canvas.g_w, &win->canvas.g_h);
  }

}

/* Load the configuration file. */

int
LoadConfig(VWRK *vw, PSTATE *st)
{

  FILE *config;
  const char *sptr;
  char line[MAXLEN], envvar[MAXLEN];

  short DevCount=1;
  short SizCount=0;
  int itemp;

  double tempx, tempy, tempw, temph;

  /* Look for the configuration file in the current and HOME directories. */

  if ((sptr = PathFind("HOME", CONFIGFILE, ",")) != NULL) {
    strcpy(line, sptr);
  }
  else {    
    return -1;
  }

  if ((config = fopen(line, "r")) == NULL) {
    sprintf(line, "[1][Could not open %s!][OK]", CONFIGFILE);
    FormAlert(1, line);
    return -1;
  }

  /* Process the information in the config file. */

  while (fgets(line, MAXLEN-1, config) != NULL) {

    line[strlen(line)-1] = '\0';

    if (!strncmp(line, "consolewin ", 11)) {
      sscanf(line+11, "%lf %lf %lf %lf", &tempx, &tempy, &tempw, &temph);
      SetWinFrame(&conswin, vw, st, tempx, tempy, tempw, temph);
    }
    else if (!strncmp(line, "imagewin ", 9)) {
      sscanf(line+9, "%lf %lf %lf %lf", &tempx, &tempy, &tempw, &temph);
      SetWinFrame(&imagwin, vw, st, tempx, tempy, tempw, temph);
    }
    else if (!strncmp(line, "aboutwin ", 9)) {
      sscanf(line+9, "%lf %lf", &tempx, &tempy);
      SetWinPosition(&aboutwin, vw, st, tempx, tempy);
    }
    else if (!strncmp(line, "reswin ", 7)) {
      sscanf(line+7, "%lf %lf", &tempx, &tempy);
      SetWinPosition(&reswin, vw, st, tempx, tempy);
    }
    else if (!strncmp(line, "devwin ", 7)) {
      sscanf(line+7, "%lf %lf", &tempx, &tempy);
      SetWinPosition(&devwin, vw, st, tempx, tempy);
    }
    else if (!strncmp(line, "sizwin ", 7)) {
      sscanf(line+7, "%lf %lf", &tempx, &tempy);
      SetWinPosition(&sizwin, vw, st, tempx, tempy);
    }
    else if (!strncmp(line, "iconwin ", 8)) {
      sscanf(line+8, "%lf %lf", &tempx, &tempy);
      SetWinPosition(&iconwin, vw, st, tempx, tempy);
    }
    else if (!strncmp(line, "printwin ", 9)) {
      sscanf(line+9, "%lf %lf", &tempx, &tempy);
      SetWinPosition(&printwin, vw, st, tempx, tempy);
    }
    else if (!strncmp(line, "device ", 7)) {
      if (DevCount < DEVICES) {
	sscanf(line+7, "%15s", devices[DevCount]);
	++DevCount;
      }
    }
    else if (!strncmp(line, "pagesize ", 9)) {
      if (SizCount < SIZES) {
	sscanf(line+9, "%15s", sizes[SizCount]);
	++SizCount;
      }
    }
    else if (!strncmp(line, "pshelp ", 7)) {
      sscanf(line+7, "%d", &itemp);
      st->PSHelp = itemp;
      menu_icheck(menuobj, PSHELP, st->PSHelp);
    }
    else if (!strncmp(line, "iconmanager ", 12)) {
      sscanf(line+12, "%d", &itemp);
      st->IconManager = itemp;
      menu_icheck(menuobj, MANAG, st->IconManager);
    }
    else if (!strncmp(line, "setenv ", 7)) {
      sscanf(line+7, "%s", envvar);
      putenv(envvar);
    }
    
  }

  fclose(config);
    
  return(0);
}

/* Save the configuration file. */

int
SaveConfig(VWRK *vw, PSTATE *st)
{

  FILE *config, *backup;
  
  char *sptr;
  char line[MAXLEN], configfile[MAXLEN], backupfile[MAXLEN];
  
  short i, back=0;
  int fx, fy, fw, fh;
  
  double hnorm, vnorm;
  
  if ((vw->full.g_w == 0) || (vw->full.g_h == 0)) {
    FormAlert(1, "[1][Could not save!|Internal Error.][OK]");
    return -1;
  }
  
  /* Look for the configuration file in the current and HOME directories. */
  
  if ((sptr = (char *)PathFind("HOME", CONFIGFILE, ",")) != NULL) {
    strcpy(configfile, sptr);
    strcpy(backupfile, sptr);
    
    if ((sptr = strrchr(backupfile, '\\')) != NULL) {
      strcpy(sptr+1, CONFIGBAK);
    }
    else if ((sptr = strrchr(backupfile, '/')) != NULL) {
      strcpy(sptr+1, CONFIGBAK);
    }
    else {
      strcpy(backupfile, CONFIGBAK);
    }
    
    back = 1;
    
  }
  
  if (!strlen(configfile)) {
    if ((sptr = getenv("HOME")) != NULL) {
      strcpy(configfile, sptr);
      strcat(configfile, CONFIGFILE);
    }
    else {
      strcat(configfile, CONFIGFILE);
    }
  }
  
  /* Make the object containing the save message visible. */

  savemsg[0].ob_flags ^= HIDETREE;

  sprintf(saveline, " Saving Configuration to %s ... ", configfile);
  savemsg[0].ob_width = strlen(saveline) * vw->Wchar;
  savemsg[SAVELINE].ob_width = strlen(saveline) * vw->Wchar;
  
  form_center(savemsg, &fx, &fy, &fw, &fh);
  form_dial(FMD_START, 0, 0, vw->Wchar, vw->Hchar, fx, fy, fw, fh);
  objc_draw(savemsg, 0, 2, fx, fy, fw, fh);
  
  /* Back up the config file. */
  
  if (back) {
    
    if ((config = fopen(configfile, "r")) == NULL) {
      sprintf(line, "[1][Could not open %s!][OK]", configfile);
      form_dial(FMD_FINISH, 0, 0, vw->Wchar, vw->Hchar, fx, fy, fw, fh);
      FormAlert(1, line);
      return -1;
    }
    
    if ((backup = fopen(backupfile, "w")) == NULL) {
      sprintf(line, "[1][Could not open %s!][OK]", backupfile);
      form_dial(FMD_FINISH, 0, 0, vw->Wchar, vw->Hchar, fx, fy, fw, fh);
      FormAlert(1, line);
      return -1;
    }
    
    while (fgets(line, MAXLEN-1, config) != NULL) {
      fputs(line, backup);
    }
    
    fclose(backup);
    fclose(config);
    
  }
  
  if ((config = fopen(configfile, "w")) == NULL) {
    sprintf(line, "[1][Could not open %s!][OK]", configfile);
    form_dial(FMD_FINISH, 0, 0, vw->Wchar, vw->Hchar, fx, fy, fw, fh);
    FormAlert(1, line);
    return -1;
  }
  
  hnorm = 1.0 / ((double)vw->full.g_w);
  vnorm = 1.0 / ((double)vw->full.g_h);
  
  /* Write information to the config file. */
  
  fputs("# Configuration file for Atari ST Ghostscript.\n", config);
  fputs("# Window x, y, w, and h are specified as fractions\n", config);
  fputs("# of the screen size.\n\n", config);
  
#undef fprintf
  fprintf(config, "consolewin %f %f %f %f\n",
	  conswin.frame.g_x * hnorm,
	  conswin.frame.g_y * vnorm,
	  conswin.frame.g_w * hnorm,
	  conswin.frame.g_h * vnorm);

  fprintf(config, "imagewin %f %f %f %f\n",
	  imagwin.frame.g_x * hnorm,
	  imagwin.frame.g_y * vnorm,
	  imagwin.frame.g_w * hnorm,
	  imagwin.frame.g_h * vnorm);

  fputs("\n# Only x and y for the following windows.\n\n", config);
  
  fprintf(config, "aboutwin %f %f\n",
	  aboutwin.frame.g_x * hnorm,
	  aboutwin.frame.g_y * vnorm);

  fprintf(config, "reswin %f %f\n",
	  reswin.frame.g_x * hnorm,
	  reswin.frame.g_y * vnorm);
  
  fprintf(config, "devwin %f %f\n",
	  devwin.frame.g_x * hnorm,
	  devwin.frame.g_y * vnorm);
  
  fprintf(config, "sizwin %f %f\n",
	  sizwin.frame.g_x * hnorm,
	  sizwin.frame.g_y * vnorm);

  fprintf(config, "iconwin %f %f\n",
	  iconwin.frame.g_x * hnorm,
	  iconwin.frame.g_y * vnorm);

  fprintf(config, "printwin %f %f\n",
	  printwin.frame.g_x * hnorm,
	  printwin.frame.g_y * vnorm);

  fputs("\n# Devices available from the device dialog.\n\n", config);
  
  for (i=1; i<DEVICES; ++i) {
    fprintf(config, "device %s\n", devices[i]);
  }

  fputs("\n# Page sizes available from the size dialog.\n\n", config);
  
  for (i=0; i<SIZES; ++i) {
    fprintf(config, "pagesize %s\n", sizes[i]);
  }

  fputs("\n# Options: 1 = selected, 0 = unselected.\n\n", config);
  fprintf(config, "pshelp %d\n", st->PSHelp);
  fprintf(config, "iconmanager %d\n", st->IconManager);
  
  /* Copy any environment variables from the backup file. */

#if 0
  if (back) {
    
    fputs("\n# Ghostscript Environment Variables.\n\n", config);

    if ((backup = fopen(backupfile, "r")) == NULL) {
      sprintf(line, "[1][Could not open %s!][OK]", backupfile);
      form_dial(FMD_FINISH, 0, 0, vw->Wchar, vw->Hchar, fx, fy, fw, fh);
      FormAlert(1, line);
      return -1;
    }

    while (fgets(line, MAXLEN-1, backup) != NULL) {
      if (!strncmp(line, "setenv ", 7)) {
	fputs(line, config);
      }
    }
    
    fclose(backup);
  }
#endif

  fclose(config);
  
  form_dial(FMD_FINISH, 0, 0, vw->Wchar, vw->Hchar, fx, fy, fw, fh);

  /* Hide the object. */

  savemsg[0].ob_flags ^= HIDETREE;
	
  return(0);
}

/* Enter supervisor mode to read the cookie jar. */

#define cookie_init() \
COOKIE *p; \
long ssp = -1; \
\
if (Super((void *)1) == 0) \
  ssp = Super((void *)0); \
p = *((COOKIE **)0x5a0l)
     
/* Leave supervisor mode after reading the cookie jar. */

#define cookie_end() \
if (ssp != -1) \
  (void)Super((void *)ssp)
     
/* Search the cookie jar. */

private COOKIE *
SearchJar(const char *id)
{
  cookie_init();
  while (p != NULL)
    {
      if (p->id == 0)
	p = NULL;
      else if (p->id == *(long *)id)
	break;
      else
	p++;
    }
  cookie_end();
  return(p);
}

/* Read the cookie jar. */

private bool
CK_ReadJar(const char *id, long *value)
{
  COOKIE *p;
	
  p = SearchJar(id);
  if (p != NULL)
    {
      if (value != NULL)
	*value = p->value;
      return 1;
    }
  return 0;
}

extern short _intin[];
extern short _intout[];
#define _int_in		(&_intin[0])
#define _int_out	(&_intout[0])
extern int __aes__(unsigned long);

/* Get information about the environment under which GS is running. */

int
ApplGetinfo(int type, int *out1, int *out2, int *out3, int *out4)
{
  long l;
  static int has_appl_getinfo = -1;
  int retval;
    
#define WF_WINX 22360

  /* check for appl_getinfo() being present */
  if (has_appl_getinfo < 0) {
    has_appl_getinfo = 0;
    /* AES 4.0? */
    if (gl_ap_version >= 0x400)
      has_appl_getinfo = 1;
    else
      /* Mag!X 2.0? */
      if (CK_ReadJar("MagX", &l) != NULL && ((short **)l)[2][24] >= 0x200)
	has_appl_getinfo = 1;
      else
        if (appl_find( "?AGI") == 0)
	  has_appl_getinfo = 1;
        else
	  /* WiNX >= 2.2 ? */
	  if (wind_get( 0, WF_WINX, out1, out2, out3, out4) == WF_WINX)
            has_appl_getinfo = 1;
  }

  /* no appl_getinfo? return error code */
  if (!has_appl_getinfo)
    {
      *out1 = 0;
      *out2 = 0;
      *out3 = 0;
      *out4 = 0;
      return( 0);
    }
        
  /* fill parameter blocks */
  _int_in[ 0] = type;

  /* call AES */
#define AES_CONTROL_ENCODE(A, B, C, D) (unsigned long)  \
    		( 				    \
       		     (((unsigned long)A) << 24) |   \
       		     (((unsigned long)B) << 16) |   \
       		     (((unsigned long)C) <<  8) |   \
       		     (((unsigned long)D)      )     \
		)
  
  retval = __aes__(AES_CONTROL_ENCODE(130, 1, 5, 0));
 
  /* get return codes */
  *out1 = _int_out[ 1];
  *out2 = _int_out[ 2];
  *out3 = _int_out[ 3];
  *out4 = _int_out[ 4];
  return retval;
}

/* Wrapper for the AES form_alert() function. */

int
FormAlert(int default_button, const char *msg)
{
  int button;
    
  if (VWork.GSPalette) {   /* Restore system palette. */
    SetPalette(VWork.OldPalette, &VWork);
  }
	
  button = form_alert(default_button, (char *)msg);
	
  if (VWork.GSPalette) {   /* Restore GS palette. */
    SetPalette(VWork.Palette, &VWork);
  }

  return(button);
}

/* Update the device object. */

void
DevObjUpdate(PSTATE *st)
{
  int i;

  /* Ensure that the button for the current device is selected. */

  for (i=FIRSTDEV; !(devobj[i].ob_state & SELECTED) && (i<=LASTDEV); ++i);

  if ((i <= LASTDEV) && (i != st->Device)) {

    ObjcChange(devobj, i, 0, &devwin,
		(devobj[i].ob_state ^ SELECTED), devwin.opened);

    ObjcChange(devobj, st->Device, 0, &devwin,
		(devobj[st->Device].ob_state ^ SELECTED), devwin.opened);

    st->LastDevice = st->Device;

  }
}

/* Copy the current image window. */

int
CopyImage(GRAPHIC *sgr, VWRK *vw, PSTATE *st)
{
  WINDOW *bwin;
  GRAPHIC *gr;		/* new graphic structure */
  
  WINLIST *wl = WList;

  char *TitleString;
  byte *buff;

  ulong buff_size = sgr->image.fd_w * sgr->image.fd_h;

  if ((bwin = WinAlloc()) == NULL) {
    FormAlert(1, "[1][Ghostscript Error!|No More Memory.][Continue]");
    return (-1);
  }

  if ((gr = GraphicAlloc()) == NULL) {
    FormAlert(1, "[1][Ghostscript Error!|No More Memory.][Continue]");
    WinFree(bwin);
    return (-1);
  }

  if ((buff = (byte *)gs_malloc((uint)buff_size, 1, "bitbuffer")) == NULL) {
    FormAlert(1, "[1][Ghostscript Error!|No More Memory.][Continue]");
    WinFree(bwin);
    GraphicFree(gr);
    return (-1);
  }

  if ((TitleString = gs_malloc(MAXTITLE, 1, "Window Title")) != NULL) {
    strncpy(TitleString, (strstr(imagwin.title, B_TITL)) ?
	    C_TITL : imagwin.title, MAXTITLE-1);
  }

  BitWinInit(bwin, gr, C_GADGETS, (TitleString) ? TitleString : C_TITL);
  
  gr->image.fd_addr = (unsigned long) buff;

  memcpy((void *)gr->image.fd_addr, (void *)sgr->image.fd_addr, buff_size);

  WinListAdd(wl, bwin);
  BitWinOpen(bwin, vw, st);
  
  return(0);
}

/*
 * Pathfind(path, file, sep) searches for 'file' in the path given
 * by the envionment variable named in 'env'. Sep specifies the
 * path separator. It returns a pointer to the filename if it is
 * found, and a NULL if not.
 */

const char *
PathFind(const char *env, const char *file, const char *sep)
{
  FILE *fp;

  char path[2*MAXLEN];
  static char name[MAXLEN];

  char *ptr;

  if ((fp = fopen(file, "r")) == NULL) {

    if ((ptr = getenv(env)) != NULL) {
      strcpy(path, ptr);
    }
    else {
      return NULL;
    }

    if ((ptr = presub(path, sep)) == NULL) {
      return NULL;
    }

    do {
      strcpy(name, ptr);
      
      if (name[strlen(name)-1] != '\\' ||
	  name[strlen(name)-1] != '/') strcat(name, "\\");

      strcat(name, file);

      if ((fp = fopen(name, "r")) != NULL) {
	fclose(fp);
	return name;
      }

    } while ((ptr = presub(NULL, sep)) != NULL);
    return(NULL);
  }
  else {
    fclose(fp);
    return file;
  }
}

/* Presub(s, subs) searches the string s for the substring subs. Presub
 * returns a pointer to the NULL terminated substring which immediately
 * preceeds subs. If presub is called again with a NULL pointer for s, 
 * it returns a pointer to the substring between the previous subs and
 * the current subs. If subs is a null character, presub returns a pointer
 * to the portion of s between the previous subs and the end of s.
 */

char *presub(char *s, const char *subs)
{
  int sublen;
  char *mptr;
  static int count;
  static char match[MAXLEN], *ptr;

  if (s != NULL) {
    ptr = s;
    count = 0;
  }

  mptr = match + count;
  if ((sublen = strlen(subs)) == 0) {
    strcpy(mptr, ptr);
    return (mptr);
  }

  while (*ptr != '\0') {
    if (strncmp(ptr, subs, (unsigned)sublen) == 0) {
      match[count] = '\0';
      ptr += sublen;
      ++count;
      return (mptr);
    }
    match[count] = *ptr;
    ++ptr;
    ++count;
		
  }

  if ((match + count) > mptr) {
    match[count] = *ptr;
    ++count;
    return (mptr);
    
  }

  return (NULL);

}

/* Routine itoa(num, s) converts an integer, num, to an ascii string 
 * representation of the integer, pointed to by s. The integer 100 is 
 *  converted to the string "100", etc.
 */

char *itoa(int number, char *s)
{

/* Local variables:
 *	i:		loop counter,
 *	j:		number of loops needed to transpose the string in s,
 *	body:		integer part of num/10,
 *	remain:		remainder of num/10,
 *	sign:		sign of the integer,
 *	count:		number of times num is divisible by 10,
 *	temp:		temporary character storage.
 */

  int i, j, num, body, remain, sign, count;
  char temp;

  count = 0;
  num = number;
  sign = 1;

  if (num == 0) {
    s[count] = '0';
    ++count;
  } else {

    if (num < 0) {
      sign = -1;
      num = -num;
    }

    while (num > 0) {		/* Divide by 10, convert */
      body = (num/10);	/* remainder to ascii    */
      remain = num - body*10;
      s[count] = remain + '0';
      num = body;
      ++count;
    }

    if (sign < 0) {
      s[count] = '-';
      ++count;
    }

    /* Ascii representation is transposed in s, so put it in 
     * the right order.
     */

    j = count/2;	

    for (i = 0; i < j; ++i) {
      temp = s[i];
      s[i] = s[count-(i+1)];
      s[count-(i+1)] = temp;
    }

  }

  s[count] = '\0';
  return(s);

}

/* Round to nearest integer. */

int
nint(double Num)
{

  int Floor = floor(Num);
  int Ceil  = ceil(Num);

  return (fabs(Num - Floor) > fabs(Num - Ceil)) ? Ceil : Floor;

}

/* Align an x coordinate to the nearest word. */

int align(int x)
{
  return ((x & 0xfffffff0) + ((x & 0xf) ? 0x10 : 0));
  /*	return ((x & 0xfffffff8) + ((x & 0x07) ? 0x08 : 0)); */
}
