/* Copyright (C) 1989, 1992, 1993 Aladdin Enterprises.  All rights reserved.

This file is part of Ghostscript.

Ghostscript is distributed in the hope that it will be useful, but
WITHOUT ANY WARRANTY.  No author or distributor accepts responsibility
to anyone for the consequences of using it or for whether it serves any
particular purpose or works at all, unless he says so in writing.  Refer
to the Ghostscript General Public License for full details.

Everyone is granted permission to copy, modify and redistribute
Ghostscript, but only under the conditions described in the Ghostscript
General Public License.  A copy of this license is supposed to have been
given to you along with Ghostscript so you can know your rights and
responsibilities.  It should be in a file named COPYING.  Among other
things, the copyright notice and this notice must be preserved on all
copies.  */

/* gp_atar2.c */

#include <stdarg.h>
#include <gemdefs.h>
#include <aesbind.h>
#include <vdibind.h>
#include <osbind.h>
#include <stdlib.h>
#include <support.h> /* for putenv */


#include "stdio_.h"
#include "string_.h"
#include "memory_.h"
#include "malloc_.h"
#include "gx.h"
#include "gp.h"
#include "gpcheck.h"
#include "gserrors.h"
#include "gsexit.h"
#include "gxdevice.h"

/* for imitation pipes */
#include "stream.h"
#include "gxiodev.h"			/* must come after stream.h */

#include "st_rsc.h"
#include "st_defs.h"
#include "gdevvdi.h"
#include "gp_atar1.h"
#include "gp_atar2.h"
#include "gp_atar3.h"
#include "gp_atar4.h"

/*
 * This file contains most of the routines that deal with windows on
 * the Atari platform.
 */

WINLIST *WList;

/* Declare and assign window structures. See ataridef.h for the
 * DEFINE_XXXWIN macro.
 */

#if FIXED_CONSOLE_SIZE
WINTEXT Cons = DEFINE_WINTEXT;
#else
WINTEXT Cons;
#endif

WINDOW aboutwin = DEFINE_OBJWIN(NULL, A_GADGETS, A_TITL, NULL, abo_button);
WINDOW reswin   = DEFINE_OBJWIN(NULL, R_GADGETS, R_TITL, NULL, res_button);
WINDOW devwin   = DEFINE_OBJWIN(NULL, D_GADGETS, D_TITL, NULL, dev_button);
WINDOW sizwin   = DEFINE_OBJWIN(NULL, S_GADGETS, S_TITL, NULL, siz_button);
WINDOW iconwin  = DEFINE_OBJWIN(NULL, I_GADGETS, I_TITL, NULL, ico_button);
WINDOW printwin = DEFINE_OBJWIN(NULL, O_GADGETS, O_TITL, NULL, prn_button);
WINDOW pagewin  = DEFINE_OBJWIN(NULL, P_GADGETS, P_TITL, NULL, pag_button);

WINDOW imagwin = DEFINE_BITWIN(&Graphic, B_GADGETS, B_TITL, NULL, ign_button);

WINDOW conswin = DEFINE_TEXTWIN(&Cons, T_GADGETS, T_TITL, NULL, ign_button);

EVENT Event = DEFINE_EVENT;

#ifndef GS_HELPDIR
#  define GS_HELPDIR "f:\\gs\\help\\"
#endif

private char gs_helpdir[MAXLEN] = GS_HELPDIR;

private void gemputc(uint);

/*
 * Use a pseudo file device to get win_stdio_init called
 * at the right time.
 */

private iodev_proc_init(win_stdio_init);
private stream_proc_process(win_std_read_process);
private stream_proc_process(win_std_write_process);

/*
 * The following method for redirecting stdio to a window was
 * stolen from the Microsoft Windows driver.
 */

gx_io_device gs_iodev_wstdio = {
	"wstdio", "Special",
	{ win_stdio_init, iodev_no_open_device,
	  iodev_no_open_file, iodev_no_fopen, iodev_no_fclose,
	  iodev_no_delete_file, iodev_no_rename_file,
	  iodev_no_file_status, iodev_no_enumerate_files
	}
};


/* Define alternate 'open' routines for our stdin/out/err streams. */

extern int iodev_stdin_open(P4(gx_io_device *, const char *, stream **,
			       gs_memory_t *));
private int
win_stdin_open(gx_io_device *iodev, const char *access, stream **ps,
  gs_memory_t *mem)
{
  int code = iodev_stdin_open(iodev, access, ps, mem);
  stream *s = *ps;
  if ( code != 1 ) {
    return code;
  }
  s->procs.process = win_std_read_process;
  s->file = NULL;
  return 0;
}

extern int iodev_stdout_open(P4(gx_io_device *, const char *, stream **,
				gs_memory_t *));
private int
win_stdout_open(gx_io_device *iodev, const char *access, stream **ps,
  gs_memory_t *mem)
{
  int code = iodev_stdout_open(iodev, access, ps, mem);
  stream *s = *ps;
  if ( code != 1 ) {
    return code;
  }
  s->procs.process = win_std_write_process;
  s->file = NULL;
  return 0;
}

extern int iodev_stderr_open(P4(gx_io_device *, const char *, stream **,
				gs_memory_t *));
private int
win_stderr_open(gx_io_device *iodev, const char *access, stream **ps,
  gs_memory_t *mem)
{
  int code = iodev_stderr_open(iodev, access, ps, mem);
  stream *s = *ps;
  if ( code != 1 ) {
    return code;
  }
  s->procs.process = win_std_write_process;
  s->file = NULL;
  return 0;
}

/* Do one-time initialization */
/* reinitialize stdin/out/err to use windows */
/* assume stream has already been initialized for the real stdin */

private int
win_stdio_init(gx_io_device *iodev, gs_memory_t *mem)
{
  /* If stdxxx is the console, replace the 'open' routines, */
  /* which haven't gotten called yet. */

  if ( gp_file_is_console(gs_stdin) )
    gs_findiodevice((const byte *)"%stdin", 6)->procs.open_device =
      win_stdin_open;

  if ( gp_file_is_console(gs_stdout) )
    gs_findiodevice((const byte *)"%stdout", 7)->procs.open_device =
      win_stdout_open;

  if ( gp_file_is_console(gs_stderr) )
    gs_findiodevice((const byte *)"%stderr", 7)->procs.open_device =
      win_stderr_open;

  return 0;
}

#define ERASELINE 21		/* ^U */
#define ERASECHAR1 8		/* ^H */
#define ERASECHAR2 127		/* DEL */

/*
 * Handle all user input. This contains the main event loop for
 * Atari windows.
 */

private int
win_std_read_process(stream_state *sst, stream_cursor_read *ignore_pr,
  stream_cursor_write *pw, bool last)
{

  byte *buf = pw->ptr;
  byte *dest = buf;

  int loop_end=0;
  
  /* Main Event Loop */

  if (State.Windows) {
    State.Event->WaitEvent = MU_MESAG | MU_KEYBD | MU_BUTTON;
    State.Event->WaitButtonState = 1;
  }

  while (!loop_end && State.Windows) {

    /* Find the top window. */

    int empty;
    WINDOW *tpwin;

    wind_get(0, WF_TOP, &State.TopWin, &empty, &empty, &empty);
    tpwin = WinFindH(State.TopWin);

    graf_mouse(ARROW, 0L);

    /* Ensure mouse and menus are in a reasonable state. */

    menu_ienable(menuobj, COPY, imagwin.opened);
    menu_ienable(menuobj, CLOSE, tpwin != NULL && (tpwin->gadgets & CLOSER));

    if (State.Event->KeyScroll >= 0) {
      WINDOW *win = WinFindXY(State.Event->MouseX, State.Event->MouseY);

      State.Event->Event = 0;

      if (win != NULL) {
	if (win->handle != conswin.handle) {
	  State.Event->Event = MU_MESAG;
	  State.Event->Message[0] = WM_ARROWED;
	  State.Event->Message[3] = win->handle;
	  State.Event->Message[4] = State.Event->KeyScroll;
	}
      }

      State.Event->KeyScroll = -1;

    }
    else {

      State.Event->Event = evnt_multi(
		State.Event->WaitEvent,
		2,
		1,
		State.Event->WaitButtonState,
		1,
		State.Event->MouseX,
		State.Event->MouseY,
		1,
		1,
		0,
		0,
		0,
		0,
		0,
		State.Event->Message,
		0L,
		&State.Event->MouseX,
		&State.Event->MouseY,
		&State.Event->ButtonState,
		&State.Event->KeyState,
		&State.Event->Key,
		&State.Event->Clicks);

    }

    if (State.Event->Event & MU_BUTTON) {		/* button events */

      loop_end = HandleButton(&VWork, &State, &Graphic, pw, &dest);
      
    }

    if (State.Event->Event & MU_KEYBD) {		/* keyboard input */
    
      loop_end = HandleKeybd(&VWork, &State, pw, &dest);

    }

    if (State.Event->Event & MU_MESAG) {		/* message events */

      switch (State.Event->Message[0]) {

      case MN_SELECTED:
	loop_end = HandleMenu(&VWork, &State, &Graphic, pw, &dest);
	break;

      case WM_REDRAW:
	{
	  static int FirstConsRedraw = 1;

	  /* Ignore the very first console redraw message. */
		
	  if ((State.Event->Message[3] == conswin.handle) &&
	      FirstConsRedraw) {
	    FirstConsRedraw = 0;
	  }
	  else {
	    Cursor(OFF);
	    HandleRedraw(DIRT_RECT, &State);
	    Cursor(ON);
	  }
	}
	break;

      case WM_ARROWED:
	HandleArrow(&State);
	break;

      case WM_HSLID:
	HorizontalSlider(&State);
	break;

      case WM_VSLID:
	VerticalSlider(&State);
	break;

      case WM_TOPPED:
	wind_set(State.Event->Message[3], WF_TOP, 0, 0, 0, 0);
	HandleOntop(&VWork);
	break;

      case WM_ONTOP:
	HandleOntop(&VWork);
	break;

      case WM_UNTOPPED:
	HandleUntop(&VWork);
	break;

      case WM_CLOSED:
	loop_end = HandleClose(&State, pw, &dest);
	break;

      case WM_FULLED:
	HandleFull(&State, &VWork);
	break;

      case WM_MOVED:
	HandleMove(&State);
	break;

      case WM_SIZED:
	HandleSize(&State);
	break;

      case WM_BOTTOMED:					/* CF */
	wind_set(State.Event->Message[3], WF_BOTTOM, 0, 0, 0, 0);
	break;
		
      case WM_ICONIFY:
      case WM_ALLICONIFY:
	HandleIcon(&VWork, &State);
	break;

      case WM_UNICONIFY:
	HandleUnIcon(&VWork, &State);
	break;

      case COMMAND:
	dprintf1("%s", State.Command);
	graf_mouse(BUSY_BEE, 0L);
	loop_end = 1;
	break;

      case PGFILTER:
	graf_mouse(BUSY_BEE, 0L);
	HandlePageFilter(&VWork, &State, pw, &dest);
	graf_mouse(ARROW, 0L);
	break;

      case PGSELECT:
	graf_mouse(BUSY_BEE, 0L);
	HandlePageSelect(&VWork, &State, pw, &dest, State.Event->Message[1]);
	graf_mouse(ARROW, 0L);
	break;
	
      }
      
    }
    
  }

  return 0;
}

/*
 * Read from Ghostscript's i/o buffer and write output to
 * the console window.
 */

private int
win_std_write_process(stream_state *sst, stream_cursor_read *pr,
  stream_cursor_write *ignore_pw, bool last)
{

    WINTEXT *text = conswin.obj;

    char string[MAX_COLUMNS+1];
    char *ptr = (char *)pr->ptr + 1;

    int i, j, tempcol;
    int oldcol = text->cn;
    uint count = pr->limit - pr->ptr;
		
    Cursor(OFF);

    State.Event->Message[3] = conswin.handle;

    for (i=0, tempcol=0; i<count; i++) {

	switch(*ptr) {

	case '\r':
	case '\n':
	newline:

	    string[tempcol] = '\0';
	    strcat(TEXTLINE(text, text->ln), string);
	    NextLine(oldcol);

	    oldcol = 0;
	    tempcol = 0;

	    ptr++;
	    break;

	case '\t':

	    for(j=0; j<8; j++) {		
		string[tempcol++] = ' ';
		text->cn++;
	    }
		
	    ptr++;
	    break;

	case '\b':
	    --tempcol;
	    if (--text->cn < 0) text->cn = 0;
	    else --ptr;
	    break;

	default:

	    string[tempcol++] = *ptr;
	    if (++text->cn >= text->bw) goto newline;

	    ptr++;
	    break;

	}

    }

    if (tempcol) {
	int dirtw;

	string[tempcol] = '\0';
	strcat(TEXTLINE(text, text->ln), string);

	dirtw = strlen(string) * text->wc;
	State.Event->Message[4] = text->cx;
	State.Event->Message[5] = text->cy;
	State.Event->Message[6] = dirtw;
	State.Event->Message[7] = text->hc;

	HandleRedraw(DIRT_RECT, &State);

	text->cx += dirtw;

    }

    Cursor(ON);

    pr->ptr = pr->limit;

    return 0;
}

/* Handle menu selection events. */

int
HandleMenu(VWRK *vw, PSTATE *st, GRAPHIC *gr, stream_cursor_write *pw, byte **dest)
{

    byte res[4];

    byte cone, cten, chun;
    long diffone=0, difften=0, diffhun=0;

    int loop_end=0;

    if (st->Event->Message[3] != -1) {
	menu_tnormal(menuobj, st->Event->Message[3], 1);
    }

    switch (st->Event->Message[4]) {

    case ABOUT:
	WinListAdd(WList, &aboutwin);
	ObjWinOpen(aboutobj, &aboutwin, vw, st);
	break;

    case OPEN:
	if (st->Event->KeyState) {
	    HandleHelp(OPENHELP, vw, st, pw, dest);
	}
	else {
	    st->SelectPages = 0;
	    menu_ienable(menuobj, PREV, 0);
	    menu_ienable(menuobj, NEXT, 1);
	    loop_end = HandleOpen(vw, st, pw, dest);
	}
	break;

    case PAGES:
	if (st->Event->KeyState) {
	    HandleHelp(PAGHELP, vw, st, pw, dest);
	}
	else {
	    st->SelectPages = 1;
	    menu_ienable(menuobj, PREV, 0);
	    menu_ienable(menuobj, NEXT, 0);
	    loop_end = HandleOpen(vw, st, pw, dest);
	}
	break;

    case PRINT:
	if (st->Event->KeyState) {
	    HandleHelp(OUTHELP, vw, st, pw, dest);
	}
	else {
	    WinListAdd(WList, &printwin);
	    ObjWinOpen(printobj, &printwin, vw, st);
	}
	break;

    case PREV:
	if (st->Event->KeyState) {
	    HandleHelp(PREVHELP, vw, st, pw, dest);
	}
	else {
	    loop_end = HandlePrev(st);
	}
	break;


    case NEXT:
	if (st->Event->KeyState) {
	    HandleHelp(NEXTHELP, vw, st, pw, dest);
	}
	else {
	    loop_end = HandleNext(st, pw, dest);
	}
	break;

    case LOAD:
	if (st->Event->KeyState) {
	    HandleHelp(LOADHELP, vw, st, pw, dest);
	}
	else {
	    LoadConfig(vw, st);
	}
	break;

    case SAVE:
	if (st->Event->KeyState) {
	    HandleHelp(SAVEHELP, vw, st, pw, dest);
	}
	else {
	    SaveConfig(vw, st);
	}
	break;

    case QUIT:
        graf_mouse(BUSY_BEE, 0L);
	gs_exit(0);	/* Cleanup is done in gp_exit. */
	break;

    case PSHELP:
	if (st->Event->KeyState) {
	    HandleHelp(PSFHELP, vw, st, pw, dest);
	}
	else {
	    menu_icheck(menuobj, PSHELP,
		(menuobj[PSHELP].ob_state ^ CHECKED));

	    st->PSHelp = !st->PSHelp;
	}
	break;

    case NOPAUSE:
	if (st->Event->KeyState) {
	    HandleHelp(PAUSEHELP, vw, st, pw, dest);
	}
	else {
	    menu_icheck(menuobj, NOPAUSE,
		(menuobj[NOPAUSE].ob_state ^ CHECKED));

	    if (menuobj[NOPAUSE].ob_state & CHECKED) {
		strcpy(st->Command, "/NOPAUSE {true} def\n");
	    }
	    else {
		strcpy(st->Command, "/NOPAUSE {false} def\n");
	    }

	    SendCommand(st, pw, dest);

	}
	break;

    case MANAG:
	if (st->Event->KeyState) {
	  HandleHelp(MANAGHELP, vw, st, pw, dest);
	}
	else {
	  menu_icheck(menuobj, MANAG,
		      (menuobj[MANAG].ob_state ^ CHECKED));

	  st->IconManager = !st->IconManager;

	  /* Close the icon manager if it is open. */

	  if (iconwin.obj != iconobj) {

	    WinClose(&iconwin);
	    iconwin.gadgets &= ~CLOSER;

	    IconListFree(iconwin.obj);
	    iconwin.obj = iconobj;

	    UpdateIconObj(iconwin.obj, vw);
	    ObjWinOpen(iconwin.obj, &iconwin, vw, st);
	  }

	}
	break;


    case DEVICE:
	if (st->Event->KeyState) {
	    HandleHelp(DEVHELP, vw, st, pw, dest);
	}
	else {
	    DevObjUpdate(st);
	    WinListAdd(WList, &devwin);
	    ObjWinOpen(devobj, &devwin, vw, st);

	}
	break;

    case RES:
	if (st->Event->KeyState) {
	    HandleHelp(RESHELP, vw, st, pw, dest);
	}
	else {

	    /* Get an ascii representation of the dialog setting. */

	    cone = *(byte *)(&resobj[RES_ONE].ob_spec);
	    cten = *(byte *)(&resobj[RES_TEN].ob_spec);
	    chun = *(byte *)(&resobj[RES_HUN].ob_spec);

	    /* Get and ascii representation of the current resolution. */

	    itoa(gr->XRes, res);

	    /* Calculate the difference, if any. */

	    switch (strlen(res)) {

	    case 3:
		diffhun = (res[0] - chun) * 0x01000000L;
		difften = (res[1] - cten) * 0x01000000L;
		diffone = (res[2] - cone) * 0x01000000L;
		break;

	    case 2:
		difften = (res[0] - cten) * 0x01000000L;
		diffone = (res[1] - cone) * 0x01000000L;
		break;

	    case 1:
		diffone = (res[0] - cone) * 0x01000000L;
		break;

	    }

	    /* Fix the object to display the current resolution. */

	    resobj[RES_ONE].ob_spec += diffone;
	    resobj[RES_TEN].ob_spec += difften;
	    resobj[RES_HUN].ob_spec += diffhun;

	    WinListAdd(WList, &reswin);
	    ObjWinOpen(resobj, &reswin, vw, st);
	}	
	break;

    case PAGESIZE:
	if (st->Event->KeyState) {
	    HandleHelp(SIZHELP, vw, st, pw, dest);
	}
	else {
	    WinListAdd(WList, &sizwin);
	    ObjWinOpen(sizobj, &sizwin, vw, st);
	}
	break;

    case ICON:
	if (st->Event->KeyState) {
	    HandleHelp(ICONHELP, vw, st, pw, dest);
	}
	else {
	  st->Event->SendMsg[0] = WM_ALLICONIFY;
	  appl_write(st->ApId, 16, st->Event->SendMsg);
	}
	break;

    case UNICON:
	if (st->Event->KeyState) {
	    HandleHelp(ICONHELP, vw, st, pw, dest);
	}
	else {
	    HandleUnIcon(vw, st);
	}
	break;

    case CYCLEF:
    case CYCLEB:
	if (st->Event->KeyState) {
	    HandleHelp(CYCLEHELP, vw, st, pw, dest);
	}
	else {
	    HandleCycle(st->Event->Message[4]);
	}
	break;

    case CLOSE:
	if (st->Event->KeyState) {
	    HandleHelp(CLOSEHELP, vw, st, pw, dest);
	}
	else {
	    st->Event->Message[3] = st->TopWin;
	    loop_end = HandleClose(st, pw, dest);
	}
	break;

    case COPY:
	if (st->Event->KeyState) {
	    HandleHelp(COPYHELP, vw, st, pw, dest);
	}
	else if (imagwin.opened) {
	    CopyImage(gr, vw, st);
	}
	else {
	    FormAlert(1, "[1][Ghostscript Error!|Nothing to Copy.][Continue]");
	}
	break;

    }

    return loop_end;

}

/* Handle help requests. */

void
HandleHelp(const char *file, VWRK *vw, PSTATE *st, stream_cursor_write *pw, byte **dest)
{
  char helpfile[MAXLEN];
	
  if (getenv("GS_HELPDIR") != NULL) {
    strcpy(gs_helpdir, getenv("GS_HELPDIR"));
    if (gs_helpdir[0] != '\0' &&
	gs_helpdir[strlen(gs_helpdir)-1] != '\\' &&
	gs_helpdir[strlen(gs_helpdir)-1] != '/')
      strcat(gs_helpdir, "/");
  }
  else {
    strcpy(helpfile, "GS_HELPDIR=");
    strcat(helpfile, gs_helpdir);
    putenv(gs_helpdir);
  }

  strcpy(helpfile, gs_helpdir);
  strcat(helpfile, file);

  if (st->PSHelp) {
    strcpy(st->Infile, file);
    strcat(st->Infile, ".ps");
    sprintf(st->Command, "(%s%s) runlibfile\n", helpfile, ".ps");
    SendCommand(st, pw, dest);
  }
  else {
    graf_mouse(BUSY_BEE, 0L);
    strcat(helpfile, ".hlp");
    HelpWinOpen(vw, st, helpfile);
    graf_mouse(ARROW, 0L);
  }
}

/* Open a file. */

int
HandleOpen(VWRK *vw, PSTATE *st, stream_cursor_write *pw, byte **dest)
{

  char *sptr, temp[MAXLEN]="";
  WINTEXT *text = conswin.obj;

  int loop_end=0;

  int prevline = (text->ln > 0) ? text->ln-1 : text->bh-1;

  /* If return was not pressed after a showpage,
   * then enter a return and send a message to
   * redraw the image window and run a new program
   * the next time through the event loop.
   */

  if ((sptr = strstr(TEXTLINE(text, prevline), ">>")) != NULL) {
    strcpy(st->Command, "\n");
    SendCommand(st, pw, dest);

    if (imagwin.opened && !imagwin.iconified) {
      st->Event->SendMsg[0] = WM_REDRAW;
      st->Event->SendMsg[3] = imagwin.handle;
      st->Event->SendMsg[4] = imagwin.canvas.g_x;
      st->Event->SendMsg[5] = imagwin.canvas.g_y;
      st->Event->SendMsg[6] = imagwin.canvas.g_w;
      st->Event->SendMsg[7] = imagwin.canvas.g_h;
      appl_write(st->ApId, 16, st->Event->SendMsg);
    }
	
    MenuWrite(st, -1, st->SelectPages ? PAGES : OPEN);

    graf_mouse(BUSY_BEE, 0L);
    
    return loop_end;
  }

  /* Make the current input file the default selection. */

  if (strlen(st->Infile)) {
    strcpy(st->FileSel, st->Infile);
  }
  else {
    strcpy(st->FileSel, "");
  }

  /* Get a file from the file selector. */

  if (GetFile(st, "*.PS")) {

    strcpy(st->InDir, st->DirSpec);
    strcpy(st->Infile, st->FileSel);

    strcpy(temp, st->DirSpec);
    strcat(temp, st->FileSel);

    if (st->SelectPages) {

      if (st->Doc->numpages > 1) {

	if (strcmp(st->Doc->name, temp) == 0) {

	  if (!pagewin.opened) {

	    if (!pagewin.iconified) {
	      WinListAdd(WList, &pagewin);
	    }

	    ObjWinOpen((OBJECT *)pagewin.obj, &pagewin, vw, st);
	  }

	  return 0;

	}

	DeletePageObj(st, &pagewin);
      }

      /* Send a message to filter pages from the file
       * after all pending events have been dispatched.
       */

      st->Event->SendMsg[0] = PGFILTER;
      appl_write(st->ApId, 16, st->Event->SendMsg);
    }
    else {

      if (st->Doc->numpages > 1) {
	DeletePageObj(st, &pagewin);
      }

      /* Execute the selected file. */
      sprintf(st->Command, "(%s) run\n", temp);
      SendCommand(st, pw, dest);
    }
    
  }

  return loop_end;
}

/* Display the previous page. */

int
HandlePrev(PSTATE *st)
{

    if (st->SelectPages) {
	int prevpage = st->Doc->currentpage - 1;

	if (prevpage < 1) {
	    return 0;
	}

	st->Event->SendMsg[0] = PGSELECT;
	st->Event->SendMsg[1] = prevpage;
	appl_write(st->ApId, 16, st->Event->SendMsg);
    }

    return 0;
}

/* Display the next page. */

int
HandleNext(PSTATE *st, stream_cursor_write *pw, byte **dest)
{
  if (st->SelectPages) {
    int nextpage = st->Doc->currentpage + 1;

    if (nextpage > st->Doc->numpages) {
      return 0;
    }

    st->Event->SendMsg[0] = PGSELECT;
    st->Event->SendMsg[1] = nextpage;
    appl_write(st->ApId, 16, st->Event->SendMsg);

  }
  else if (imagwin.opened || imagwin.iconified) {
    *++(*dest) = '\n';
    gemputc('\n');
    pw->ptr = *dest;
    
    graf_mouse(BUSY_BEE, 0L);

    return 1;
  }

  return 0;
}

/* Cycle the window positions. */

void
HandleCycle(int direction)
{
    WINLIST *wl = WList;
    int topwin, empty;

    wind_get(0, WF_TOP, &topwin, &empty,
	&empty, &empty);

    do {

	/* Find the current top window. */

	if (wl->Win->handle == topwin) {

	    /* Top the next open window in the appropriate direction. */

	    if (direction == CYCLEF) {
		while (!wl->Next->Win->opened) wl = wl->Next;
		wind_set(wl->Next->Win->handle, WF_TOP, 0, 0, 0, 0);
		break;
	    }
	    else if (direction == CYCLEB) {
		while (!wl->Prev->Win->opened) wl = wl->Prev;
		wind_set(wl->Prev->Win->handle, WF_TOP, 0, 0, 0, 0);
		break;
	    }
	}

	wl = wl->Next;
    }
    while (wl != WList);

}

/* Handle a click on the window arrows. */

void
HandleArrow(PSTATE *st)
{
    WINDOW *win = WinFindH(st->Event->Message[3]);
    if (win && win->arrow) (win->arrow)(st);
}

/* Handle horizontal slider movement. */

void
HorizontalSlider(PSTATE *st)
{
    WINDOW *win = WinFindH(st->Event->Message[3]);
    if (win && win->hslide) (win->hslide)(st);
}

/* Handle vertical slider movement. */

void
VerticalSlider(PSTATE *st)
{
    WINDOW *win = WinFindH(st->Event->Message[3]);
    if (win && win->vslide) (win->vslide)(st);

}

/* GS window has been topped. */

void
HandleOntop(VWRK *vw)
{
    if (vw->ColorBits > 1 && !vw->GSPalette && !vw->TrueColor) {

	/* Restore GS palette. */

	SetPalette(vw->Palette, vw);
	vw->GSPalette = 1;

    }
}

/* GW window has been untopped. */

void
HandleUntop(VWRK *vw)
{
    int topwin, empty;

    wind_get(0, WF_TOP, &topwin, &empty,
	&empty, &empty);

    /* If the top window is not in the window list, restore
     * the system palette.
     */

    if (vw->GSPalette) {

	if (WinFindH(topwin) != NULL)
	    return;

	SetPalette(vw->OldPalette, vw);
	vw->GSPalette = 0;

    }
}

/* GS window has been closed. */

#define CL_ERROR "[1][Note!|The designated window|cannot be closed.][Continue]"

int
HandleClose(PSTATE *st, stream_cursor_write *pw, byte **dest)
{
  char *sptr;
  int loop_end=0;
  int handle = st->Event->Message[3];

  WINDOW *win = WinFindH(handle);
  WINTEXT *text = conswin.obj;

  int prevline = (text->ln > 0) ? text->ln-1 : text->bh-1;

  if (win == NULL) {
    FormAlert(1, CL_ERROR);
    return 0;
  }

  if (!(win->gadgets & CLOSER)) {
    FormAlert(1, CL_ERROR);
    return 0;
  }

  WinClose(win);

  switch (win->type) {

  case BIT:
    /* If a return has not been entered after a
     * showpage, then enter one.
     */

    if (win == &imagwin &&
	(sptr = strstr(TEXTLINE(text,  prevline), ">>")) != NULL) {

      strcpy(st->Command, "\n");
      SendCommand(st, pw, dest);
    }

    break;
    
  case OBJ:
    /* If the closed window was an icon list, return to the icon. */

    if (win == &iconwin) {

      VWRK *vw = &VWork;
      OBJECT *obj = iconwin.obj;

      iconwin.gadgets &= ~CLOSER;
      obj = (OBJECT *)iconwin.obj = iconobj;

      IconListFree(obj);
      UpdateIconObj(obj, vw);

      ObjWinOpen(iconwin.obj, &iconwin, vw, st);

      return loop_end;

    }
  }

  WList = WinListDelete(WList, win);
  return loop_end;
}

/* Fuller window button has been clicked. */

void
HandleFull(PSTATE *st, VWRK *vw)
{

  int handle = st->Event->Message[3];

  WINDOW *win = WinFindH(handle);

  if (handle == iconwin.handle) {
    MenuWrite(st, -1, UNICON);
  }
  else if (st->Event->KeyState && st->IconManager) {

    st->Event->SendMsg[0] = WM_ICONIFY;
    st->Event->SendMsg[3] = win->handle;

    /* Send a message to iconify the window. */
    
    appl_write(st->ApId, 16, st->Event->SendMsg);

  }
  else {

    st->Event->SendMsg[0] = WM_SIZED;
    st->Event->SendMsg[3] = win->handle;

    wind_get(win->handle, WF_CURRXYWH,
	     &win->frame.g_x, &win->frame.g_y,
	     &win->frame.g_w, &win->frame.g_h);

    if (win->frame.g_x == vw->full.g_x &&
	win->frame.g_y == vw->full.g_y &&
	win->frame.g_w == MIN(win->mframe.g_w, vw->full.g_w) &&
	win->frame.g_h == MIN(win->mframe.g_h, vw->full.g_h)) {

      st->Event->SendMsg[4] = win->oframe.g_x;
      st->Event->SendMsg[5] = win->oframe.g_y;
      st->Event->SendMsg[6] = win->oframe.g_w;
      st->Event->SendMsg[7] = win->oframe.g_h;

    }
    else {
      win->oframe.g_x = win->frame.g_x;
      win->oframe.g_y = win->frame.g_y;
      win->oframe.g_w = win->frame.g_w;
      win->oframe.g_h = win->frame.g_h;
	
      st->Event->SendMsg[4] = vw->full.g_x;
      st->Event->SendMsg[5] = vw->full.g_y;
      st->Event->SendMsg[6] = MIN(win->mframe.g_w, vw->full.g_w);
      st->Event->SendMsg[7] = MIN(win->mframe.g_h, vw->full.g_h);
	
    }

    /* Send a message to handle the window resizing. */
      
    appl_write(st->ApId, 16, st->Event->SendMsg);
    
  }

}

/* Window has been moved. */

void
HandleMove(PSTATE *st)
{
    WINDOW *win = WinFindH(st->Event->Message[3]);
    (win->move)(st);
}

/* Window has been sized. */

void
HandleSize(PSTATE *st)
{
    WINDOW *win = WinFindH(st->Event->Message[3]);
    (win->size)(st);
}

/* One or more windows have been iconified by some means. */

void
HandleIcon(VWRK *vw, PSTATE *st)
{
  int handle = st->Event->Message[3];
  WINDOW *win = WinFindH(handle);

  if (iconwin.opened) {
    /* Close the icon window */

    WinClose(&iconwin);
    WList = WinListDelete(WList, &iconwin);
  }

  switch (st->Event->Message[0]) {

  case WM_ICONIFY:		/* close the iconified window */
    wind_close(handle);
    win->opened = 0;
    win->iconified = 1;

    if (win->type == OBJ) {
      OBJECT *obj = win->obj;
      obj[0].ob_flags ^= HIDETREE;
    }
    break;

  case WM_ALLICONIFY:		/* close all open windows */
    WinListClose(WList);
    break;
  }

  UpdateIconList(iconwin.obj, vw);
  WinListAdd(WList, &iconwin);
  ObjWinOpen(iconwin.obj, &iconwin, vw, st);

}

/* One or more windows have been uniconified. */

void
HandleUnIcon(VWRK *vw, PSTATE *st)
{
  if (iconwin.opened) {

    wind_get(iconwin.handle, WF_CURRXYWH,
	     &iconwin.frame.g_x, &iconwin.frame.g_y,
	     &iconwin.frame.g_w, &iconwin.frame.g_h);

    /* Close the icon window. */

    WinClose(&iconwin);
    WList = WinListDelete(WList, &iconwin);

    /* Open all windows in the window list. */

    WinListOpen(WList);
  }
}
						
/* Handle all button clicks not handled by the AES. */

int
HandleButton(VWRK *vw, PSTATE *st, GRAPHIC *gr, stream_cursor_write *pw, byte **dest)
{

    int ObjectIndex, loop_end=0;
    OBJECT *Object;

    /* Find where the mouse click took place. */		

    WINDOW *win = WinFindXY(st->Event->MouseX, st->Event->MouseY);

    if (win == NULL) {			/* Click was not in a window. */
	return 0;
    }
    else if (win->type != OBJ) {	/* Window contains no object. */
	return 0;
    }

    Object = (OBJECT *)win->obj;
    ObjectIndex = objc_find(Object, 0, 2,
	st->Event->MouseX, st->Event->MouseY);

    if (!(Object[ObjectIndex].ob_flags & SELECTABLE)) {
	return 0;			/* Object is not selectable */
    }

    if (Object[ObjectIndex].ob_state & DISABLED) {
	return 0;			/* Object is disabled */
    }

    ObjcChange(Object, ObjectIndex, 0, win,
	(Object[ObjectIndex].ob_state ^ SELECTED), 1);

    if (ObjectIndex >= 0) {
      loop_end = (win->button)(Object, ObjectIndex, win, vw, st, gr, pw, dest);
    }

    return loop_end;

}

/* Redraw all or part of a window. */

int
HandleRedraw(int flag, PSTATE *st)
{
    WINDOW *win = WinFindH(st->Event->Message[3]);

    if (win == NULL) return 0;	/* window is no longer open */

    wind_update(BEG_UPDATE);	/* lock the screen */
    (win->redraw)(flag, st);
    wind_update(END_UPDATE);	/* release screen */

    return 0;
}


/* Handle keyboard input. */

int
HandleKeybd(VWRK *vw, PSTATE *st, stream_cursor_write *pw, byte **dest)
{
    byte *limit = pw->limit - 1;  /* always leave room for \n */

    int loop_end=0;

    uint ch = (uint)( st->Event->Key & 255 );
    
    /* Check whether the key is a menu hotkey. If not, handle it
     * in the switch statement.
     */

    if (MenuSearch(st, menuobj) == 0) {

      switch (st->Event->Key) {

      case 0x4800:			/* up arrow */
	st->Event->KeyScroll = WA_UPLINE;
	break;

      case 0x4838:			/* shift up arrow */
	st->Event->KeyScroll = WA_UPPAGE;
	break;

      case 0x5000:			/* down arrow */
	st->Event->KeyScroll = WA_DNLINE;
	break;

      case 0x5032:			/* shift down arrow */
	st->Event->KeyScroll = WA_DNPAGE;
	break;

      case 0x4b00:			/* left arrow */
	st->Event->KeyScroll = WA_LFLINE;
	break;

      case 0x4b34:			/* shift left arrow */
	st->Event->KeyScroll = WA_LFPAGE;
	break;

      case 0x4d00:			/* right arrow */
	st->Event->KeyScroll = WA_RTLINE;
	break;

      case 0x4d36:			/* shift right arrow */
	st->Event->KeyScroll = WA_RTPAGE;
	break;

      case 0x4700:			/* clr/home */
	st->Event->KeyState = 1;
	st->Event->KeyScroll = WA_UPLINE;
	break;

      case 0x4737:			/* shift clr/home */
	st->Event->KeyState = 1;
	st->Event->KeyScroll = WA_DNLINE;
	break;

      case 0x6200:			/* help */
	HandleHelp(GENHELP, vw, st, pw, dest);
	break;

      case 0x1c0d:			/* return */
      case 0x720d:                      /* enter */

	/*
	 * Exit the event loop so the interpreter can
	 * read the input buffer.
	 */

	*++(*dest) = '\n';
	gemputc(ch);
	pw->ptr = *dest;

	graf_mouse(BUSY_BEE, 0L);
	loop_end = 1;
        break;

      default:		/* handle the input character. */

	switch (ch) {

	default:		/* put char in the input buffer. */

	  if ( *dest == limit ) {
	    ; /* MessageBeep(-1); */
	  }
	  else {
	    *++(*dest) = ch;
	    gemputc(ch);
	  }
	  break;

	case ERASELINE:
	  ClearLine(*dest - pw->ptr);
	  *dest = pw->ptr;
	  break;

	case ERASECHAR1:
	case ERASECHAR2:
	  if ( *dest > pw->ptr ) {
	    ClearLine(1);
	    (*dest)--;
	  }
	  break;

	}

      }

    }

    return loop_end;
}

/* Handle page filtering. */

#define FILT_ERR "[1][My simplistic routine|can not extract pages|\
from the selected file.|It may not conform to|Adobe's structure rules.]\
[Continue]"

void
HandlePageFilter(VWRK *vw, PSTATE *st, stream_cursor_write *pw,
	byte **dest)
{
  char input[MAXLEN];
  int num;

  sprintf(input, "%s%s", st->InDir, st->Infile);

  /* Filter the input file. */

  if ((num = PageFilter(input, st->Doc)) <= 1) {
    if (num <= 0) {
      FormAlert(1, FILT_ERR);
    }

    /* Execute the selected file. */
    sprintf(st->Command, "(%s) run\n", input);
    SendCommand(st, pw, dest);
  }
  else {
    /* Create and open the page dialog. */
    pagewin.obj = (void *)CreatePageObj(st->Doc->numpages);

    WinListAdd(WList, &pagewin);
    ObjWinOpen((OBJECT *)pagewin.obj, &pagewin, vw, st);

    /* Send a message to extract and execute a page
     * after all pending events have been dispatched.
     */

    st->Event->SendMsg[0] = PGSELECT;
    st->Event->SendMsg[1] = 0;
    appl_write(st->ApId, 16, st->Event->SendMsg);

  }

}

/* Display a selected page. */

void
HandlePageSelect(VWRK *vw, PSTATE *st, stream_cursor_write *pw,
	byte **dest, int page)
{
  char OutRoot[MAXLEN];

  st->Doc->currentpage = page;

  if (page == 0) {
    strcpy(OutRoot, "gs_pro");
  }
  else {
    if (page == 1) {
      menu_ienable(menuobj, PREV, 0);
      menu_ienable(menuobj, NEXT, 1);

    }
    else if (page == st->Doc->numpages) {
      menu_ienable(menuobj, PREV, 1);
      menu_ienable(menuobj, NEXT, 0);
    }
    else {
      menu_ienable(menuobj, PREV, 1);
      menu_ienable(menuobj, NEXT, 1);
    }

    strcpy(OutRoot, "gs_pg");
  }

  /* Extract the page and execute. */

  PageCopy(page, OutRoot, st->Doc);
  sprintf(st->Command, "(%s) run\n", OutRoot);
  SendCommand(st, pw, dest);

}

/* Clear a line on console. */

void
ClearLine(int cnt)
{
  int x, w;

  WINTEXT *text = conswin.obj;

  Cursor(OFF);
  State.Event->Message[3] = conswin.handle;

  if (cnt < 0) {
    *TEXTLINE(text, text->ln) = '\0';
    x = conswin.canvas.g_x;
    w = conswin.canvas.g_w;
    text->cx = align(text->edge + text->xoff);
    text->cn = 0;
  }
  else {
    text->cn = MAX(text->cn - cnt, 0);
    text->cx -= cnt * text->wc;
    TEXTLINE(text, text->ln)[text->cn] = '\0';
    x = MAX(text->cx, conswin.canvas.g_x);
    w = MIN(cnt * text->wc, conswin.canvas.g_w);
  }

  State.Event->Message[4] = x;
  State.Event->Message[5] = text->cy;
  State.Event->Message[6] = w;
  State.Event->Message[7] = text->hc;

  HandleRedraw(DIRT_RECT, &State);

  Cursor(ON);

}

/* Move to next line after printing a line to a window. */

void
NextLine(int oldcol)
{
  char outline[MAXLEN];

  int topwin, empty;

  GRECT rect;
  MFDB src, dest;
  WINTEXT *text = conswin.obj;

  src.fd_addr  = (long)NULL;
  dest.fd_addr = (long)NULL;

  wind_get(0, WF_TOP, &topwin, &empty, &empty, &empty);

  if (++text->ln >= text->bh) {
    text->ln = 0;
    text->scrolled = 1;
  }

  *TEXTLINE(text, text->ln) = '\0';

  if (text->scrolled) {
    if (++text->fl  >= text->bh) text->fl  = 0;
    if (++text->fdl >= text->bh) text->fdl = 0;
    if (++text->ll  >= text->bh) text->ll  = 0;
    if (++text->ldl >= text->bh) text->ldl = 0;

    if ((topwin == conswin.handle) && (text->ll == text->ldl)) {

      int offset = MAX(oldcol, text->fdc);
      int startx = MAX(text->cx, align(conswin.canvas.g_x + text->xoff));
      int count =  text->cols - oldcol + text->fdc;
      int prevline = (text->ln > 0) ? text->ln-1 : text->bh-1;

      graf_mouse(M_OFF, 0L );
      wind_update(BEG_UPDATE);	/* lock the screen */

      if ((count > 0) && (strlen(TEXTLINE(text, prevline)) > offset)) {
	*outline = '\0';
	strncat(outline, TEXTLINE(text, prevline)+offset, (unsigned)count);
	v_gtext(VWork.VdiHandle, startx, text->cy, outline);
      }

      State.pxy[0] = conswin.canvas.g_x;
      State.pxy[1] = conswin.canvas.g_y + text->hc;
      State.pxy[2] = conswin.canvas.g_x + conswin.canvas.g_w - 1;
      State.pxy[3] = conswin.canvas.g_y + conswin.canvas.g_h - 1;

      State.pxy[4] = conswin.canvas.g_x;
      State.pxy[5] = conswin.canvas.g_y;
      State.pxy[6] = conswin.canvas.g_x + conswin.canvas.g_w - 1;
      State.pxy[7] = conswin.canvas.g_y + conswin.canvas.g_h - text->hc - 1;

      vro_cpyfm(VWork.VdiHandle, 3, State.pxy, &src, &dest);

      rect.g_x = conswin.canvas.g_x;
      rect.g_y = text->cy;
      rect.g_w = conswin.canvas.g_x + conswin.canvas.g_w - rect.g_x;
      rect.g_h = text->hc;

      ClearWin(&rect);

      wind_update(END_UPDATE);	/* unlock the screen */
      graf_mouse(M_ON, 0L );

    }
    else {
      HandleRedraw(FULL_WIN, &State);
    }
  }
  else if (text->ln == (text->ldl + 1)) {

    text->top -= text->hc;
    text->fdl++;
    text->ldl++;

    if (topwin == conswin.handle) {

      int offset = MAX(oldcol, text->fdc);
      int startx = MAX(text->cx, align(conswin.canvas.g_x + text->xoff));
      int count =  text->cols - oldcol + text->fdc;
      int prevline = (text->ln > 0) ? text->ln-1 : text->bh-1;

      graf_mouse(M_OFF, 0L );
      wind_update(BEG_UPDATE);	/* lock the screen */

      if ((count > 0) && (strlen(TEXTLINE(text, prevline)) > offset)) {
	*outline = '\0';
	strncat(outline, TEXTLINE(text, prevline)+offset, (unsigned)count);
	v_gtext(VWork.VdiHandle, startx, text->cy, outline);
      }

      State.pxy[0] = conswin.canvas.g_x;
      State.pxy[1] = conswin.canvas.g_y + text->hc;
      State.pxy[2] = conswin.canvas.g_x + conswin.canvas.g_w - 1;
      State.pxy[3] = conswin.canvas.g_y + conswin.canvas.g_h - 1;

      State.pxy[4] = conswin.canvas.g_x;
      State.pxy[5] = conswin.canvas.g_y;
      State.pxy[6] = conswin.canvas.g_x + conswin.canvas.g_w - 1;
      State.pxy[7] = conswin.canvas.g_y + conswin.canvas.g_h - text->hc - 1;

      vro_cpyfm(VWork.VdiHandle, 3, State.pxy, &src, &dest);

      rect.g_x = conswin.canvas.g_x;
      rect.g_y = text->cy;
      rect.g_w = conswin.canvas.g_x + conswin.canvas.g_w - rect.g_x;
      rect.g_h = text->hc;

      ClearWin(&rect);

      wind_update(END_UPDATE);	/* unlock the screen */
      graf_mouse(M_ON, 0L );

    }
    else {
      HandleRedraw(FULL_WIN, &State);
    }

    UpdateScroll(conswin.handle);
  }
  else {
    State.Event->Message[4] = text->cx;
    State.Event->Message[5] = text->cy;
    State.Event->Message[6] = conswin.canvas.g_x + conswin.canvas.g_w
      - text->cx;
    State.Event->Message[7] = text->hc;
    HandleRedraw(DIRT_RECT, &State);

    text->cy += text->hc;
  }

  text->cx = align(text->edge + text->xoff);
  text->cn = 0;

}

/* Calculate the number of visible lines in the console window. */

int
VisibleLines(WINDOW *twin)
{
  WINTEXT *text = twin->obj;
  return ((twin->canvas.g_h - text->yoff) / text->hc);
}

/* Calculate the number of visible columns in the console window. */

int
VisibleCols(WINDOW *twin)
{
  WINTEXT *text = twin->obj;
  return ((twin->canvas.g_w - align(text->xoff)) / text->wc);
}

/* Turn the cursor on or off. */

void
Cursor(int state)
{
  int xy[4], txy[4];
  GRECT rect;
  WINTEXT *text = conswin.obj;

  if ((state == OFF && text->cstate == ON) ||
      (state == ON  && text->cstate == OFF)) {

    vsf_interior(VWork.VdiHandle, 1);	/* set fill mode to solid */
    vswr_mode(VWork.VdiHandle, 3);	/* set write mode to XOR */

    xy[0] = text->cx;
    xy[1] = text->cy;
    xy[2] = xy[0] + text->wc - 1;
    xy[3] = xy[1] + text->hc - 1;

    wind_get(conswin.handle, WF_FIRSTXYWH,
	     &rect.g_x, &rect.g_y,
	     &rect.g_w, &rect.g_h);

    while (rect.g_w && rect.g_h) {

      if (rc_intersect(&conswin.canvas, &rect) &&
	  rc_intersect(&VWork.full, &rect)) {

	grect_to_array(&rect, txy);
	vs_clip(VWork.VdiHandle, 1, txy);
	
	vr_recfl(VWork.VdiHandle, xy);	/* draw the cursor */
	
	vs_clip(VWork.VdiHandle, 0, txy);

      }

      wind_get(conswin.handle, WF_NEXTXYWH,
	       &rect.g_x, &rect.g_y,
	       &rect.g_w, &rect.g_h);

    }

    vs_clip(VWork.VdiHandle, 0, xy);
    vswr_mode(VWork.VdiHandle, 1);		/* restore replace mode */
    text->cstate = !text->cstate;		/* toggle cursor state  */

  }
}

/* Clear a window. */

void
ClearWin(GRECT *area)
{
  int xy[4];

  xy[0] = MAX(area->g_x, VWork.full.g_x);
  xy[1] = MAX(area->g_y, VWork.full.g_y);
  xy[2] = MIN(xy[0] + area->g_w - 1, VWork.full.g_x + VWork.full.g_w - 1);
  xy[3] = MIN(xy[1] + area->g_h - 1, VWork.full.g_y + VWork.full.g_h - 1);

  vsf_interior(VWork.VdiHandle, 0);	/* set fill mode to hollow */
  vr_recfl(VWork.VdiHandle, xy);		/* clear the window */
}

/*
 * The gem equivalent of putc(), which writes to a window instead of
 * to stdout.
 */

private void
gemputc(uint ch)
{
  char c[2];

  int topwin, empty;

  GRECT rect;
  MFDB src, dest;

  WINTEXT *text = conswin.obj;

  src.fd_addr  = (long)NULL;
  dest.fd_addr = (long)NULL;

  wind_get(0, WF_TOP, &topwin, &empty, &empty, &empty);

  Cursor(OFF);
  State.Event->Message[3] = conswin.handle;

  switch (ch) {

  default:

    c[0] = ch;
    c[1] = '\0';

    strcat(TEXTLINE(text, text->ln), c);

    if (topwin == conswin.handle) {
      if ((text->cx >= align(conswin.canvas.g_x + text->xoff)) &&
	  (text->cx < conswin.canvas.g_x+conswin.canvas.g_w-text->wc-1) &&
	  (text->cy >= conswin.canvas.g_y + text->yoff) &&
	  (text->cy < conswin.canvas.g_y+conswin.canvas.g_h-text->hc-1)) {

	graf_mouse(M_OFF, 0L);
	wind_update(BEG_UPDATE);	/* lock the screen */

	v_gtext(VWork.VdiHandle, text->cx, text->cy, c);

	wind_update(END_UPDATE);	/* unlock the screen */
	graf_mouse(M_ON, 0L);

      }
    }
    else {
      State.Event->Message[4] = text->cx;
      State.Event->Message[5] = text->cy;
      State.Event->Message[6] = text->wc;
      State.Event->Message[7] = text->hc;

      HandleRedraw(DIRT_RECT, &State);
    }

    if (++text->cn >= text->bw) goto newline;
    text->cx += text->wc;
    break;

  case '\r':
  case '\n':
  newline:

    TEXTLINE(text, text->ln)[text->cn] = '\0';

    if (++text->ln >= text->bh) {
      text->ln = 0;
      text->scrolled = 1;
    }

    *TEXTLINE(text, text->ln) = '\0';

    if (text->scrolled) {
      if (++text->fl >= text->bh) text->fl = 0;
      if (++text->fdl >= text->bh) text->fdl = 0;
      if (++text->ll >= text->bh) text->ll = 0;
      if (++text->ldl >= text->bh) text->ldl = 0;

      if ((topwin == conswin.handle) && (text->ll == text->ldl)) {

	State.pxy[0] = conswin.canvas.g_x;
	State.pxy[1] = conswin.canvas.g_y + text->hc;
	State.pxy[2] = conswin.canvas.g_x + conswin.canvas.g_w - 1;
	State.pxy[3] = conswin.canvas.g_y + conswin.canvas.g_h - 1;

	State.pxy[4] = conswin.canvas.g_x;
	State.pxy[5] = conswin.canvas.g_y;
	State.pxy[6] = conswin.canvas.g_x + conswin.canvas.g_w - 1;
	State.pxy[7] = conswin.canvas.g_y + conswin.canvas.g_h - text->hc - 1;

	graf_mouse(M_OFF, 0L);
	wind_update(BEG_UPDATE);	/* lock the screen */

	vro_cpyfm(VWork.VdiHandle, 3, State.pxy, &src, &dest);

	rect.g_x = State.Event->Message[4] = conswin.canvas.g_x;
	rect.g_y = State.Event->Message[5] = text->cy;
	rect.g_w = State.Event->Message[6] = conswin.canvas.g_x
	  + conswin.canvas.g_w - rect.g_x;
	rect.g_h = State.Event->Message[7] = text->hc;

	ClearWin(&rect);

	wind_update(END_UPDATE);	/* unlock the screen */
	graf_mouse(M_ON, 0L);

      }
      else {
	HandleRedraw(FULL_WIN, &State);
      }

    }
    else if (text->ln == (text->ldl + 1)) {
      text->top -= text->hc;
      text->fdl++;
      text->ldl++;

      if (topwin == conswin.handle) {

	State.pxy[0] = conswin.canvas.g_x;
	State.pxy[1] = conswin.canvas.g_y + text->hc;
	State.pxy[2] = conswin.canvas.g_x + conswin.canvas.g_w - 1;
	State.pxy[3] = conswin.canvas.g_y + conswin.canvas.g_h - 1;

	State.pxy[4] = conswin.canvas.g_x;
	State.pxy[5] = conswin.canvas.g_y;
	State.pxy[6] = conswin.canvas.g_x + conswin.canvas.g_w - 1;
	State.pxy[7] = conswin.canvas.g_y + conswin.canvas.g_h- text->hc - 1;

	graf_mouse(M_OFF, 0L);
	wind_update(BEG_UPDATE);	/* lock the screen */

	vro_cpyfm(VWork.VdiHandle, 3, State.pxy, &src, &dest);

	rect.g_x = align(conswin.canvas.g_x);
	rect.g_y = text->cy;
	rect.g_w = conswin.canvas.g_x + conswin.canvas.g_w - rect.g_x;
	rect.g_h = text->hc;

	ClearWin(&rect);

	wind_update(END_UPDATE);	/* unlock the screen */
	graf_mouse(M_ON, 0L);
	
      }
      else {
	HandleRedraw(FULL_WIN, &State);
      }

      UpdateScroll(conswin.handle);
      
    }
    else {
      text->cy += text->hc;
    }

    text->cx = align(text->edge + text->xoff);
    text->cn = 0;

    break;

  case '\b':
    TEXTLINE(text, text->ln)[--text->cn] = '\0';
    text->cx -= text->wc;
    break;

  }

  Cursor(ON);
}

/* The GEM equivalent of fprintf which prints to a window. */

int
gemprintf(FILE *stream, const char *format, ...)
{
  va_list args;

  char c, *lptr, *bptr;
  char buffer[1024], line[MAX_COLUMNS+1];
	
  WINTEXT *text = conswin.obj;

  int count;
  int oldcol = text->cn;

  va_start(args, format);

  lptr = line;
  bptr = buffer;

  if (gp_file_is_console(stream) && State.Windows) {

    /* Send console I/O to a window */
	
    if (text->buff == NULL) {
      return(0);
    }
	
    State.Event->Message[3] = conswin.handle;
    Cursor(OFF);

    count = vsprintf(buffer, format, args);

    for (; (c = *bptr) != '\0'; bptr++) {
      switch (c) {

      case '\r':
      case '\n':
      newline:
	*lptr = '\0';
	strcat(TEXTLINE(text, text->ln), line);
	NextLine(oldcol);

	oldcol = 0;
	lptr = line;
	break;

      default:
	*lptr++ = c;
	if (++text->cn >= text->bw) goto newline;
	
      }

    }

    if (lptr != line) {
      int dirtw;
      
      *lptr = '\0';
      strcat(TEXTLINE(text, text->ln), line);

      dirtw = strlen(line) * text->wc;
      State.Event->Message[4] = text->cx;
      State.Event->Message[5] = text->cy;
      State.Event->Message[6] = dirtw;
      State.Event->Message[7] = text->hc;

      HandleRedraw(DIRT_RECT, &State);

      text->cx += dirtw;

    }

    Cursor(ON);

  }
  else if (stream == NULL) {		/* output for the centronics port */
    OutFile = stream;
    count = vsprintf(buffer, format, args);
    l_start();
    lputs(buffer);
    l_stop();
  }
  else {				/* print to other streams normally */
    count = vfprintf(stream, format, args);
    con_flush();
  }

  va_end(args);
  return count;

}

/*
 * A version of fputc which works with Chris Strunk's code to print
 * directly to the centronics port.
 */

int
csputc(int c, FILE *stream)
{
  if (gp_file_is_console(stream) && State.Windows) {
    gemputc((unsigned)c);
  }
  else {
    OutFile = stream;
    l_start();
    lputc(c);
    l_stop();
  }

  return c;
}

/*
 * A version of fputs which works with Chris Strunk's code to print
 * directly to the centronics port.
 */

int
csputs(const char *s, FILE *stream)
{
  if (gp_file_is_console(stream) && State.Windows) {
    gemprintf(stream, s);
  }
  else {
    OutFile = stream;
    l_start();
    lputs(s);
    l_stop();
  }

  return 1;
}

/*
 * A version of fwrite which works with Chris Strunk's code to print
 * directly to the centronics port.
 */

size_t
cswrite(const void *ptr, size_t size, size_t nobj, FILE *stream)
{
  int count;

  if (stream == NULL) {
    OutFile = stream;
    l_start();
    for (count=0; count < size*nobj; count++) {
      lputc(*((char *)ptr)++);       /* send the data */
    }
    l_stop();
    return count;
  }
  else {
    int ret;
    ret = fwrite(ptr, size, nobj, stream);
    con_flush();
    return ret;
  }
		
}

/*
 * The remainder of this file contains printer output routines, most of
 * which were contributed by Chris Strunk (some of them were written by me).
 * I made some modifications to the following code (with permission), and
 * may have introduced errors not in the original code.
 * Tim Gallivan, 3/92.
 */

/*
 * This file is Copyright (C) 1990 Christoph Strunk.
 *
 * You are not allowed to copy or modify it.
 */

#define ATARI_TOS	(1)
#define MAKE_VOID	(void *)

int OutputIsAscii = 0;
#define RTX_Found State.MultiTOS
long *old_ssp = NULL;
FILE *OutFile = NULL;

#if PC_DOS || ATARI_TOS
#  define DIRECT	1	/* 1 = Direct Centronics port programming */
#endif

#if ATARI_TOS || ( PC_DOS && DIRECT )
static short prn_out ( int );
#endif

#if ATARI_TOS
private void call_yield(void);
private void *call_super(void *);
#if !DIRECT
private int prt_ready(int dev);
private void prt_out(int dev, int c);
#else
private void int_off __PROTO( ( void ) );
private void int_on __PROTO( ( void ) );
#endif
#endif

#if ATARI_TOS && LATTICE
#include <dos.h>
#endif

#ifdef __GNUC__
#include <osbind.h>
#endif

#if PC_DOS && DIRECT
#include <bios.h>
#endif

static int fatal_output_error = 0;


static void printer_not_ready(void)
{
    char str[80];
	
    l_stop();
    sprintf(str, "[1][Output error!|Printer not ready ?][Abort|Again]");
    if (FormAlert(2, str) == 1)
    {
        dputs ( "\n");
        dputs ("Output error --  Printer not ready." );
        fatal_output_error = 1;
        fatal ( "Output error" );
    }
    l_start();
}


/* output one character */

void
lputc(c)
int c;
{
    static short rc;
    static unsigned char global_count = 0; /* other processes every 256 chars */

    if ( fatal_output_error ) return;

    global_count += 1;

    c &= 255;            /* New in 2.9.44: avoid signed char problems */

#if ATARI_TOS || PC_DOS
    if ( ( c == '\n' ) && OutputIsAscii ) {
      lputc ( '\r' );    /* recursion */
    }
#endif

try_again:
    if ( OutFile ) {
       rc = ( fputc ( c, OutFile ) == EOF );
    } else {
#if ATARI_TOS || ( PC_DOS && DIRECT )
                   rc = prn_out ( c );
#else
                   rc = -1;
#endif
    }

    if ( rc ) {
        if ( OutFile ) {
            perror ( "\nlputc" );
            fprintf ( stderr, "\nOutput error -- %s ?\n",
                      "Disk full or printer not ready" );
            fclose ( OutFile );
            OutFile = NULL;
            l_stop();
            fatal_output_error = 1;
            fatal ( "Output error" );
        } else {
            printer_not_ready();
            goto try_again;
        }
    }

#if ATARI_TOS
    if ( RTX_Found && ! global_count ) call_yield(); /* allow other processes */
#endif
}

/* output a string */

void
lputs ( s )
const char *s;
{
    while ( *s ) lputc ( *s++ );
}

void
lflush()
{
   if ( OutFile ) fflush ( OutFile );
}

/* start/stop lputc device */

void
l_start()
{

#if ATARI_TOS && DIRECT
#if OLD
  volatile char *gpip = (char *) 0xFFFFFA01L;
#endif
  int cnt;

  if ( OutFile == NULL && old_ssp == NULL ) {
    old_ssp = (void *) call_super ( NULL );
try_again:
    cnt = 0;
#if OLD
    for ( cnt=0; ( *gpip & 1 ) && ( ++cnt <= 10 ); ) {
     printf("cnt = %d\n", cnt);
#endif
     (void)Ongibit(0x20);                                  /* set strobe bit */
#if OLD
    }
#endif
    if ( cnt > 10 ) {
        printer_not_ready();
        goto try_again;
    }
  }
#endif
#if ATARI_TOS && ! DIRECT
  int cnt;

  if ( OutFile == NULL && old_ssp == NULL ) {
    old_ssp = (void *) call_super ( NULL );

try_again:
    for ( cnt=0; ( ! prt_ready(0) ) && ( ++cnt <= 10 ); ) {
     (void)Ongibit(0x20);                                  /* set strobe bit */
    }
    if ( cnt > 10 ) {
        printer_not_ready();
        goto try_again;
    }
  }
#endif
#if PC_DOS && DIRECT
  if ( OutFile == NULL && ( biosprint ( 2, 0, BiosChannel ) & 0x29 ) ) {
    printer_not_ready();
    goto try_again;
  }
#endif
}

void l_stop()
{
    lflush();
#if ATARI_TOS
    if ( old_ssp != NULL ) {
        MAKE_VOID call_super ( old_ssp );
        old_ssp = NULL;
    }
#endif
}

#if ATARI_TOS && DIRECT


static short prn_out ( c )
int c;
{
  volatile unsigned long *hz_200;
  register unsigned long end_time;
  volatile char *gpip = (char *) 0xFFFFFA01L;
  register char g;

#if OLD
  unsigned char loop_count = 0;
#endif

  if ( old_ssp == NULL ) l_start();

  hz_200 = (unsigned long *) 0x04BAL;
#if OLD
  end_time = *hz_200 + 200;        /* check once per second */
#else
  end_time = *hz_200 + 2;          /* check 100 times per second */
#endif

  while ( *gpip & 1 ) {  /* wait */
#if OLD
    /* Printer 1 sec. or more not ready ? */
    if ( ( end_time < *hz_200 ) ||
         ( ( ( ++loop_count & 7 ) == 0 ) && ( BatchMode || RTX_Found ) ) ) {
        con_flush();         /* allow Control_C, other Processes etc. */
        end_time = *hz_200 + 2;          /* check 100 times per second */
    }
#else
    if ( end_time <= *hz_200 ) {
        con_flush();               /* allow Control_C, other Processes etc. */
        end_time = *hz_200 + 1;              /* check 200 times per second */
    }
#endif
  }

  int_off();                   /* disable interrupts */

  gpip = (char *) 0xFFFF8800L; /* load sound chip adress now */

  /* This next section of code was added by Tim Gallivan on 7/11/94
   * to fix a problem of this routine not working on some hardware.
   */

  *gpip = 7;		       /* select enable register */
  g = *gpip;                   /* get old value */
  g |= 0x80;                   /* set port B output bit */
  gpip[2] = g;

  /* End fix. */

  *gpip = 15;                  /* select port B */
  gpip[2] = (char) c;          /* write out char */
  *gpip = 14;                  /* select port A */
  g = *gpip;                   /* get old value */
#if OLD
  g &= 0xDF;                   /* clear strobe bit */
#else
  g &= ~0x20;                  /* clear strobe bit */
#endif
  gpip[2] = g;
  g |= 0x20;                   /* set strobe bit */
  g |= 0x20;                   /* short delay */
  gpip[2] = g;

  int_on();                    /* enable interrupts */

  return 0;
}
#endif

#if ATARI_TOS && ! DIRECT
static short prn_out ( c )
int c;
{
  volatile unsigned long *hz_200 = (unsigned long *) 0x04BAL;
  register unsigned long end_time;

  if ( old_ssp == NULL ) l_start();

  end_time = *hz_200 + 2;          /* check 200 times per second */

  while ( ! prt_ready(0) ) {  /* wait */
    if ( end_time <= *hz_200 ) {
        con_flush();               /* allow Control_C, other Processes etc. */
        end_time = *hz_200 + 1;              /* check 200 times per second */
    }
  }

  prt_out ( 0, c );

  return 0;
}
#endif

#if PC_DOS && DIRECT

static short prn_out ( c )
int c;
{
  while ( biosprint ( 0, c, BiosChannel ) & 0x29 ) {
    /* wait until Ok or Control-C pressed */
    con_flush();
  }

  return 0;
}

#endif

void
con_flush()
{
	int chin;

	chin = Crawio(0xFF);
	if ((chin & 0xFF) == 3) fatal("Keyboard Interrupt");
}

void
fatal(const char *message)
{
	fprintf(stderr, "%s\n", message);
	l_stop();
	exit(-1);
}

#undef puts

/* Restore the usual desktop background. */

void RestoreBG(VWRK *vw)
{
  puts("\003H\033f");
  v_hide_c(vw->VdiHandle);
  form_dial(FMD_FINISH, 0, 0, 8, 16, 0, 0, vw->XRes, vw->YRes);
  v_show_c(vw->VdiHandle, 0);
}

/*
#-----------------------------------------------------------------------------#
#:	Simple support routines for LATTICE C 5.04.01 and other C compilers
#
#:	Lattice C Assembler
#
#:	The parameter comes on the stack for functions starting with '_'.
#
#:	void *call_super ( void * );  supervisor mode on/off
#
#:	Original code by Chris Strunk. Modified for use with gcc in
#:	Ghostscript by Tim Gallivan, 3/92.
#-----------------------------------------------------------------------------#
*/

#if ATARI_TOS

private void *
call_super(void *new_ssp)
{
	register void *retvalue __asm__("d0");

	__asm__ volatile ("
	moveml   d2/a2-a3,sp@-
	movel    %1,sp@-
	movew    #0x20,sp@-
	movel    a7,a3     | save current stack pointer
	trap      #1
	movel    a3,a7     | make bad stack pointer good again !!!!!!!
	addql    #6,sp
	movel    d0,a0     | return value in both: d0 and a0
	moveml   sp@+,d2/a2-a3
	  "
	: "=r"(retvalue) 			/* outputs */
	: "d"(new_ssp)			/* inputs  */		\
	);
	return retvalue;
}

private void
call_yield(void)
{
	__asm__ volatile("
	moveml   d2/a2,sp@-
	movew    #0x00ff,sp@-   | MiNT Syield()  --  TOS illegal (rc -32)
	trap      #1
	addql    #2,sp
	moveml   sp@+,d2/a2
	  "
	);
}

#if !DIRECT

private int
prt_ready(int dev)
{
	register int retvalue __asm__("d0");
	
	__asm__("
	moveml	d2-d3/a2-a3/a5,a7@-
	subl	a5,a5
	movew	%1,a7@-
	movel	0x506,a0
	jsr	a0@
	addql	#2,a7
	moveml	a7@+,d2-d3/a2-a3/a5
	  "
	: "=r"(retvalue)			/* outputs */
	: "r"((short)(dev))			/* inputs  */
    );
    
    return retvalue;
}

private void
prt_out(int dev, int c)
{
	__asm__ volatile("
	moveml	d2-d3/a2-a3/a5,a7@-
	subl	a5,a5
	movew	%1,a7@-
	movew	%0,a7@-
	movel	0x50A,a0
	jsr	a0@
	addql	#4,a7
	moveml	a7@+,d2-d3/a2-a3/a5
    "
	: 			/* outputs */
	: "r"((short)(dev)), "r"((short)(c))			/* inputs  */
    );
}

#else

private void
int_off(void)
{
	__asm__("oriw  #0x0700,sr");
}

private void
int_on(void)
{
	__asm__("andiw #0xF8FF,sr");
}

#endif /* !DIRECT */
#endif /* ATARI_TOS */
