/*********************************************************************
							LEGOBITS.INC

Usage:  A Persistence of Vision 2.0 include file.
		#include "legobits.inc"
		This file provides a variety of basic LEGO Technic objects,
		plus the subsections needed to build up new ones of your own.
		Now you can have a 5x1 block, or purple LEGO, or whatever.

Author: Peter Middlestorb
		CIS:100065,3267
		May 1994

Status: Public domain. Fold, spindle or mutilate as you see fit.

Acknowledgements:

		LEGO is a registered trademark of LEGO Group.
		Much kudos to the POV-Ray Team for producing the excellent
		and freely available Persistence of Vision Raytracer.
*********************************************************************/

/*********************************************************************
							INFORMATION

All objects are based on real LEGO Technic blocks at 1 POV unit / mm.
Some dimensions have been idealised a bit. The "gap" has been built
into the objects. Thus to get proper gappy LEGO, you just use the
blocks with a displacement or pitch of 8 units in X and Z, and 9.6
units in Y (3.2 units for the thin slabs). If you make your own object
with a base of blocks or slabs then you may need to translate it by
Y=-0.2, otherwise it will not quite reach the floor due to the
built-in gap at the bottom. The block axle channel centre is Y=5.9.

Every non-trivial object, even the ones that are only used to build 
up more complex objects, has a bounding box. This seems to improve
rendering times.
*********************************************************************/

/*********************************************************************
a basic 2x1 block

	  +--------------b
	 /              /|
	+--------------+ |
	|              | +
	|              |/   Cell dimensions for positioning blocks:
	a--------------+    a=<0.0, 0.0, 0.0> b=<16.0, 9.6, 8.0)

*********************************************************************/

//========== former for 2x1 block ==========
#declare objUnionBlock2x1 = union {
	box {< 0.2, 0.2, 0.2>, <15.8, 9.4, 1.5>}            // front
	box {< 0.2, 0.2, 0.2>, < 1.5, 9.4, 7.8>}            // left
	box {<14.5, 0.2, 0.2>, <15.8, 9.4, 7.8>}            // right
	box {< 0.2, 0.2, 6.5>, <15.8, 9.4, 7.8>}            // back
	box {< 0.2, 8.1, 0.2>, <15.8, 9.4, 7.8>}            // top
	cylinder {< 8.0, 5.9, 0.2>, < 8.0,  5.9, 7.8>, 3.5} // axle channel (unbored)
	cylinder {< 8.0, 0.2, 4.0>, < 8.0,  3.0, 4.0>, 1.5} // underside inter-stud pip
	cylinder {< 4.0, 9.4, 4.0>, < 4.0, 11.4, 4.0>, 2.5} // top left stud (unbored)
	cylinder {<12.0, 9.4, 4.0>, <12.0, 11.4, 4.0>, 2.5} // top right stud (unbored)
	bounded_by {
		box {<0.0, 0.0, 0.0>, <16.0, 11.4, 8.0>}
	}
}

//========== 2x1 block ==========
#declare objBlock2x1 = difference {
	object {objUnionBlock2x1}
	cylinder {< 8.0, 5.9, 0.1>, < 8.0,  5.9, 7.9>, 2.4}  // axle channel bore
	cylinder {< 8.0, 5.9, 0.1>, < 8.0,  5.9, 0.8>, 3.0}  // front axle channel bezel
	cylinder {< 8.0, 5.9, 7.2>, < 8.0,  5.9, 7.9>, 3.0}  // back axle channel bezel
	cylinder {< 4.0, 9.4, 4.0>, < 4.0, 11.5, 4.0>, 1.7}  // top left stud bore
	cylinder {<12.0, 9.4, 4.0>, <12.0, 11.5, 4.0>, 1.7}  // top tight stud bore
	bounded_by {
		box {<0.0, 0.0, 0.0>, <16.0, 11.4, 8.0>}
	}
}

/*********************************************************************
block construction units (cut from the basic 2x1 block)
*********************************************************************/

//========== block left cap ==========
#declare objBlockLeftCapx1 = intersection {
	object {objBlock2x1}
	plane {x, 4}
	bounded_by {
		box {<0.0, 0.0, 0.0>, <4.0, 11.4, 8.0>}
	}
}

//========== block right cap ==========
#declare objBlockRightCapx1 = intersection {
	object {objBlock2x1
		translate <-12, 0, 0>
	}
	plane {-x, 0}
	bounded_by {
		box {<0.0, 0.0, 0.0>, <4.0, 11.4, 8.0>}
	}
}

//========== block body ==========
#declare objBlockBodyx1 = intersection {
	object {objBlock2x1
		translate <-4, 0, 0>
	}
	plane {x, 8}
	plane {-x, 0}
	bounded_by {
		box {<0.0, 0.0, 0.0>, <8.0, 11.4, 8.0>}
	}
}

/*********************************************************************
composite blocks made from the block construction units
*********************************************************************/

//========== 4x1 block ==========
#declare objBlock4x1 = union {
	object {objBlockLeftCapx1   translate < 0, 0, 0>}   // left cap, stud 1
	object {objBlockBodyx1      translate < 4, 0, 0>}   // studs 1/2
	object {objBlockBodyx1      translate <12, 0, 0>}   // studs 2/3
	object {objBlockBodyx1      translate <20, 0, 0>}   // studs 3/4
	object {objBlockRightCapx1  translate <28, 0, 0>}   // right cap, stud 4
	bounded_by {
		box {<0.0, 0.0, 0.0>, <32.0, 11.4, 8.0>}
	}
}

//========== 6x1 block ==========
#declare objBlock6x1 = union {
	object {objBlockLeftCapx1   translate < 0, 0, 0>}   // left cap, stud 1
	object {objBlockBodyx1      translate < 4, 0, 0>}   // studs 1/2
	object {objBlockBodyx1      translate <12, 0, 0>}   // studs 2/3
	object {objBlockBodyx1      translate <20, 0, 0>}   // studs 3/4
	object {objBlockBodyx1      translate <28, 0, 0>}   // studs 4/5
	object {objBlockBodyx1      translate <36, 0, 0>}   // studs 5/6
	object {objBlockRightCapx1  translate <44, 0, 0>}   // right cap, stud 6
	bounded_by {
		box {<0.0, 0.0, 0.0>, <48.0, 11.4, 8.0>}
	}
}

//========== 16x1 block ==========
#declare objBlock16x1 = union {
	object {objBlockLeftCapx1   translate < 0,  0, 0>}  // left cap, stud 1
	object {objBlockBodyx1      translate < 4,  0, 0>}  // studs 1/2
	object {objBlockBodyx1      translate < 12, 0, 0>}  // studs 2/3
	object {objBlockBodyx1      translate < 20, 0, 0>}  // studs 3/4
	object {objBlockBodyx1      translate < 28, 0, 0>}  // studs 4/5
	object {objBlockBodyx1      translate < 36, 0, 0>}  // studs 5/6
	object {objBlockBodyx1      translate < 44, 0, 0>}  // studs 6/7
	object {objBlockBodyx1      translate < 52, 0, 0>}  // studs 7/8
	object {objBlockBodyx1      translate < 60, 0, 0>}  // studs 8/9
	object {objBlockBodyx1      translate < 68, 0, 0>}  // studs 9/10
	object {objBlockBodyx1      translate < 76, 0, 0>}  // studs 10/11
	object {objBlockBodyx1      translate < 84, 0, 0>}  // studs 11/12
	object {objBlockBodyx1      translate < 92, 0, 0>}  // studs 12/13
	object {objBlockBodyx1      translate <100, 0, 0>}  // studs 13/14
	object {objBlockBodyx1      translate <108, 0, 0>}  // studs 14/15
	object {objBlockBodyx1      translate <116, 0, 0>}  // studs 15/16
	object {objBlockRightCapx1  translate <124, 0, 0>}  // right cap, stud 16
	bounded_by {
		box {<0.0, 0.0, 0.0>, <128.0, 11.4, 8.0>}
	}
}

/*********************************************************************
a basic 2x2 slab

	   +--------------b
	  /              /+
	 /              //
	+--------------+/   Cell dimensions for positioning slabs:
	a--------------+    a=<0.0, 0.0, 0.0> b=<16.0, 3.2, 16.0>

*********************************************************************/
//========== former for 2x2 slab ==========
#declare objUnionSlab2x2 = union {
	box {< 0.2, 0.2,  0.2>, <15.8, 3.0,  1.5>}              // front
	box {< 0.2, 0.2,  0.2>, < 1.5, 3.0, 15.8>}              // left
	box {<14.5, 0.2,  0.2>, <15.8, 3.0, 15.8>}              // right
	box {< 0.2, 0.2, 14.5>, <15.8, 3.0, 15.8>}              // back
	box {< 0.2, 1.7,  0.2>, <15.8, 3.0, 15.8>}              // top
	cylinder {< 8.0, 0.2,  8.0>, < 8.0, 3.0,  8.0>, 3.2}    // axle channel (unbored)
	cylinder {< 4.0, 3.0,  4.0>, < 4.0, 5.0,  4.0>, 2.5}    // top left front stud
	cylinder {<12.0, 3.0,  4.0>, <12.0, 5.0,  4.0>, 2.5}    // top right front stud
	cylinder {< 4.0, 3.0, 12.0>, < 4.0, 5.0, 12.0>, 2.5}    // top left back stud
	cylinder {<12.0, 3.0, 12.0>, <12.0, 5.0, 12.0>, 2.5}    // top right back stud
	bounded_by {
		box {<0.0, 0.0, 0.0>, <16.0, 5.0, 16.0>}
	}
}

//========== 2x2 slab ==========
#declare objSlab2x2 = difference {
	object {objUnionSlab2x2}
	cylinder {< 8.0, 0.1,  8.0>, < 8.0, 3.1,  8.0>, 2.4}   // axle channel bore
	cylinder {< 4.0, 1.6,  4.0>, < 4.0, 3.7,  4.0>, 1.3}   // top left front stud underpip
	cylinder {<12.0, 1.6,  4.0>, <12.0, 3.7,  4.0>, 1.3}   // top right front stud underpip
	cylinder {< 4.0, 1.6, 12.0>, < 4.0, 3.7, 12.0>, 1.3}   // top left back stud underpip
	cylinder {<12.0, 1.6, 12.0>, <12.0, 3.7, 12.0>, 1.3}   // top right back stud underpip
	bounded_by {
		box {<0.0, 0.0, 0.0>, <16.0, 5.0, 16.0>}
	}
}

/*********************************************************************
slab construction units (cut from the basic 2x2 slab)
*********************************************************************/

//========== slab left cap ==========
#declare objSlabLeftCapx2 = intersection {
	object {objSlab2x2}
	plane {x, 4}
	bounded_by {
		box {<0.0, 0.0, 0.0>, <4.0, 5.0, 16.0>}
	}
}

//========== slab right cap ==========
#declare objSlabRightCapx2 = intersection {
	object {objSlab2x2
		translate <-12, 0, 0>
	}
	plane {-x, 0}
	bounded_by {
		box {<0.0, 0.0, 0.0>, <4.0, 5.0, 16.0>}
	}
}

//========== slab body ==========
#declare objSlabBodyx2 = intersection {
	object {objSlab2x2
		translate <-4, 0, 0>
	}
	plane {x, 8}
	plane {-x, 0}
	bounded_by {
		box {<0.0, 0.0, 0.0>, <8.0, 5.0, 16.0>}
	}
}

/*********************************************************************
composite slabs made from the slab construction units
*********************************************************************/

//========== 4x2 slab ==========
#declare objSlab4x2 = union {
	object {objSlabLeftCapx2    translate < 0, 0, 0>}   // left cap, stud 1
	object {objSlabBodyx2       translate < 4, 0, 0>}   // studs 1/2
	object {objSlabBodyx2       translate <12, 0, 0>}   // studs 2/3
	object {objSlabBodyx2       translate <20, 0, 0>}   // studs 3/4
	object {objSlabRightCapx2   translate <28, 0, 0>}   // right cap, stud 4
	bounded_by {
		box {<0.0, 0.0, 0.0>, <32.0, 5.0, 16.0>}
	}
}

//========== 6x2 slab ==========
#declare objSlab6x2 = union {
	object {objSlabLeftCapx2    translate < 0, 0, 0>}   // left cap, stud 1
	object {objSlabBodyx2       translate < 4, 0, 0>}   // studs 1/2
	object {objSlabBodyx2       translate <12, 0, 0>}   // studs 2/3
	object {objSlabBodyx2       translate <20, 0, 0>}   // studs 3/4
	object {objSlabBodyx2       translate <28, 0, 0>}   // studs 4/5
	object {objSlabBodyx2       translate <36, 0, 0>}   // studs 5/6
	object {objSlabRightCapx2   translate <44, 0, 0>}   // right cap, stud 6
	bounded_by {
		box {<0.0, 0.0, 0.0>, <48.0, 5.0, 16.0>}
	}
}

//========== 8x2 slab ==========
#declare objSlab8x2 = union {
	object {objSlabLeftCapx2    translate < 0, 0, 0>}   // left cap, stud 1
	object {objSlabBodyx2       translate < 4, 0, 0>}   // studs 1/2
	object {objSlabBodyx2       translate <12, 0, 0>}   // studs 2/3
	object {objSlabBodyx2       translate <20, 0, 0>}   // studs 3/4
	object {objSlabBodyx2       translate <28, 0, 0>}   // studs 4/5
	object {objSlabBodyx2       translate <36, 0, 0>}   // studs 5/6
	object {objSlabBodyx2       translate <44, 0, 0>}   // studs 6/7
	object {objSlabBodyx2       translate <52, 0, 0>}   // studs 7/8
	object {objSlabRightCapx2   translate <60, 0, 0>}   // right cap, stud 8
	bounded_by {
		box {<0.0, 0.0, 0.0>, <64.0, 5.0, 16.0>}
	}
}

/*********************************************************************
wheel - built up in three stages (tyres are separate)

Cell dimensions: If you think of the wheel as facing you then the
centre of the axle channel at the front of the wheel is at <0, 0, 0>.
The axle axis is +Z. The radius is 8.5.

*********************************************************************/

//========== subtract the wheel centre ==========
#declare objDifferenceWheel = difference {
	cylinder {<0.0, 0.0,  0.0>, <0.0, 0.0, 7.2>, 8.5}
	cylinder {<0.0, 0.0, -0.1>, <0.0, 0.0, 7.3>, 6.0}
	bounded_by {
		box {<-8.5, -8.5, 0.0>, <8.5, 8.5, 7.2>}
	}
}

//========== put in the wheel hub and spokes ==========
#declare objUnionWheel = union {
	object {objDifferenceWheel}
	cylinder {<0.0, 0.0, 0.0>, <0.0, 0.0, 7.2>, 3.8}
	box {<-6.0, -0.9, 0.0>, <6.0, 0.9, 7.2>}            // spokes l-r
	bounded_by {
		box {<-8.5, -8.5, 0.0>, <8.5, 8.5, 7.2>}
	}
}

//========== subtract the axle slots ==========
#declare objWheel = difference {
	object {objUnionWheel}
	box {<-2.4, -0.9, -0.1>, <2.4, 0.9, 7.3>}            // short slot l-r
	box {<-0.9, -4.0, -0.1>, <0.9, 4.0, 7.3>}            // long slot u-d
	bounded_by {
		box {<-8.5, -8.5, 0.0>, <8.5, 8.5, 7.2>}
	}
}

/*********************************************************************
small tyre - built up in four stages

Cell dimensions: A small tyre has the same face centre as a wheel. It
has the same thickness (Z dimension), and a radius of 12.5.

*********************************************************************/

//========== small tread lump ==========
#declare objSmallTreadLump = intersection {
	box {<-0.9, 8.5, 0.0>, <0.9, 12.5, 3.6>}
	cylinder {<0.0, 0.0, 0.0>, <0.0, 0.0, 3.6>, 12.5}
}

//========== small tread ring ==========
#declare nSmallTreadAngle = 360 / 22
#declare objSmallTreadRing = union {
	object {objSmallTreadLump rotate <0, 0,  1 * nSmallTreadAngle>}
	object {objSmallTreadLump rotate <0, 0,  2 * nSmallTreadAngle>}
	object {objSmallTreadLump rotate <0, 0,  3 * nSmallTreadAngle>}
	object {objSmallTreadLump rotate <0, 0,  4 * nSmallTreadAngle>}
	object {objSmallTreadLump rotate <0, 0,  5 * nSmallTreadAngle>}
	object {objSmallTreadLump rotate <0, 0,  6 * nSmallTreadAngle>}
	object {objSmallTreadLump rotate <0, 0,  7 * nSmallTreadAngle>}
	object {objSmallTreadLump rotate <0, 0,  8 * nSmallTreadAngle>}
	object {objSmallTreadLump rotate <0, 0,  9 * nSmallTreadAngle>}
	object {objSmallTreadLump rotate <0, 0, 10 * nSmallTreadAngle>}
	object {objSmallTreadLump rotate <0, 0, 11 * nSmallTreadAngle>}
	object {objSmallTreadLump rotate <0, 0, 12 * nSmallTreadAngle>}
	object {objSmallTreadLump rotate <0, 0, 13 * nSmallTreadAngle>}
	object {objSmallTreadLump rotate <0, 0, 14 * nSmallTreadAngle>}
	object {objSmallTreadLump rotate <0, 0, 15 * nSmallTreadAngle>}
	object {objSmallTreadLump rotate <0, 0, 16 * nSmallTreadAngle>}
	object {objSmallTreadLump rotate <0, 0, 17 * nSmallTreadAngle>}
	object {objSmallTreadLump rotate <0, 0, 18 * nSmallTreadAngle>}
	object {objSmallTreadLump rotate <0, 0, 19 * nSmallTreadAngle>}
	object {objSmallTreadLump rotate <0, 0, 20 * nSmallTreadAngle>}
	object {objSmallTreadLump rotate <0, 0, 21 * nSmallTreadAngle>}
	object {objSmallTreadLump rotate <0, 0, 22 * nSmallTreadAngle>}
	bounded_by {
		box {<-12.5, -12.5, 0.0>, <12.5, 12.5, 3.6>}
	}
}

//========== small tyre body ==========
#declare objSmallTyreBody = difference {
	cylinder {<0.0, 0.0, 0.0>, <0.0, 0.0, 7.2>, 9.5}
	cylinder {<0.0, 0.0, -0.1>, <0.0, 0.0, 7.3>, 8.5}
	bounded_by {
		box {<-9.5, -9.5, 0.0>, <9.5, 9.5, 7.2>}
	}
}

//========== small tyre ==========
#declare objSmallTyre = union {
	object {objSmallTyreBody}
	object {objSmallTreadRing}
	object {objSmallTreadRing
		rotate <0, 0, 0.5 * nSmallTreadAngle>
		translate <0.0, 0.0, 3.6>
	}
	bounded_by {
		box {<-12.5, -12.5, 0.0>, <12.5, 12.5, 7.2>}
	}
}

/*********************************************************************
large tyre - built up in five stages

Cell dimensions: A large tyre is thicker than a wheel. It should be
translated by <0.0, 0.0, -3.6> when fitting to a wheel, as it
overhangs at the front. The radius is 15.0.

*********************************************************************/

//========== large tread lump ==========
#declare objLargeTreadLump = intersection {
	box {<-1.2, 12.0, 0.0>, <1.2, 15.0, 5.4>}
	cylinder {<0.0, 0.0, 0.0>, <0.0, 0.0, 5.4>, 15.0}
}

//========== large tread ring ==========
#declare nLargeTreadAngle = 360 / 20
#declare objLargeTreadRing = union {
	object {objLargeTreadLump rotate <0, 0,  1 * nLargeTreadAngle>}
	object {objLargeTreadLump rotate <0, 0,  2 * nLargeTreadAngle>}
	object {objLargeTreadLump rotate <0, 0,  3 * nLargeTreadAngle>}
	object {objLargeTreadLump rotate <0, 0,  4 * nLargeTreadAngle>}
	object {objLargeTreadLump rotate <0, 0,  5 * nLargeTreadAngle>}
	object {objLargeTreadLump rotate <0, 0,  6 * nLargeTreadAngle>}
	object {objLargeTreadLump rotate <0, 0,  7 * nLargeTreadAngle>}
	object {objLargeTreadLump rotate <0, 0,  8 * nLargeTreadAngle>}
	object {objLargeTreadLump rotate <0, 0,  9 * nLargeTreadAngle>}
	object {objLargeTreadLump rotate <0, 0, 10 * nLargeTreadAngle>}
	object {objLargeTreadLump rotate <0, 0, 11 * nLargeTreadAngle>}
	object {objLargeTreadLump rotate <0, 0, 12 * nLargeTreadAngle>}
	object {objLargeTreadLump rotate <0, 0, 13 * nLargeTreadAngle>}
	object {objLargeTreadLump rotate <0, 0, 14 * nLargeTreadAngle>}
	object {objLargeTreadLump rotate <0, 0, 15 * nLargeTreadAngle>}
	object {objLargeTreadLump rotate <0, 0, 16 * nLargeTreadAngle>}
	object {objLargeTreadLump rotate <0, 0, 17 * nLargeTreadAngle>}
	object {objLargeTreadLump rotate <0, 0, 18 * nLargeTreadAngle>}
	object {objLargeTreadLump rotate <0, 0, 19 * nLargeTreadAngle>}
	object {objLargeTreadLump rotate <0, 0, 20 * nLargeTreadAngle>}
	bounded_by {
		box {<-15.0, -15.0, 0.0>, <15.0, 15.0, 5.4>}
	}
}

//========== large tyre body ==========
#declare objLargeTyreBody = difference {
	cylinder {<0.0, 0.0, 0.0>, <0.0, 0.0, 10.8>, 13.0}
	cylinder {<0.0, 0.0, -0.1>, <0.0, 0.0, 10.9>, 8.5}
	cone {<0.0, 0.0, -0.1>, 10.0, <0.0, 0.0, 3.6>, 8.5}
	bounded_by {
		box {<-13.0, -13.0, 0.0>, <13.0, 13.0, 10.8>}
	}
}

//========== large tyre ==========
#declare objLargeTyre = union {
	object {objLargeTyreBody}
	object {objLargeTreadRing}
	object {objLargeTreadRing
		rotate <0, 0, 0.5 * nLargeTreadAngle>
		translate <0.0, 0.0, 5.4>
	}
	bounded_by {
		box {<-15.0, -15.0, 0.0>, <15.0, 15.0, 10.8>}
	}
}

/*********************************************************************
rods - for axles etc.

Cell dimensions: Rods are centred on the +X axis. Rods have an
effective radius of 2.4, to fit into a block or slab axle channel.

*********************************************************************/

//========== rod - 3 long ==========
#declare objRodx3 = union {
	box {<0.0, -2.4, -0.9>, <24.0, 2.4, 0.9>}   // x/y
	box {<0.0, -0.9, -2.4>, <24.0, 0.9, 2.4>}   // x/z
	bounded_by {
		box {<0.0, -2.4, -2.4>, <24.0, 2.4, 2.4>}
	}
}

//========== rod - 4 long ==========
#declare objRodx4 = union {
	box {<0.0, -2.4, -0.9>, <32.0, 2.4, 0.9>}   // x/y
	box {<0.0, -0.9, -2.4>, <32.0, 0.9, 2.4>}   // x/z
	bounded_by {
		box {<0.0, -2.4, -2.4>, <32.0, 2.4, 2.4>}
	}
}

//========== rod - 6 long ==========
#declare objRodx6 = union {
	box {<0.0, -2.4, -0.9>, <48.0, 2.4, 0.9>}   // x/y
	box {<0.0, -0.9, -2.4>, <48.0, 0.9, 2.4>}   // x/z
	bounded_by {
		box {<0.0, -2.4, -2.4>, <48.0, 2.4, 2.4>}
	}
}

/*********************************************************************
right angle rod joint and lock

Cell dimensions: A untranslated joint would normally join two rods
which were positioned thus:

object {objRod                          // X rod
	translate <4.0. 0.0. 0.0>
}
object {objRod                          // Z rod
	rotate <0, -90, 0>
	translate <0.0. 0.0. -4.0>
}

It may help to visualise a joint cell as the following box:

			 // <- Z rod
			//
		+------------b
	   /  //        /|
	 /   //        / |
	+-------------+  |
	|  //  =======|========== <- X rod
	|  O          | /
	|             |/   
	a-------------+    a=<-4.0, -4.0, -4.0> b=<12.0, 4.0, 4.0)

The joint lock goes as the front part of the Z rod channel, and should
be omitted if the Z rod is free to rotate in the joint.

*********************************************************************/

//========== joint core for z and x axles (x locked) ==========
#declare objUnionRod90Joint = union {
	cylinder {< 0.0, 0.0, 0.0>, < 0.0, 0.0, 3.8>, 3.8}  // z axle core
	cylinder {< 0.0, 0.0, 0.0>, < 5.4, 0.0, 0.0>, 3.8}  // z/x joiner
	cylinder {< 5.4, 0.0, 0.0>, <10.6, 0.0, 0.0>, 2.9}  // x axle core
	cylinder {<10.6, 0.0, 0.0>, <12.0, 0.0, 0.0>, 3.8}  // x axle flange
	bounded_by {
		box {<-3.8, -3.8, -3.8>, <12.0, 3.8, 4.0>}
	}
}

//========== subtract axle channels and stuff from joint core ==========
#declare objRod90Joint = difference {
	object {objUnionRod90Joint}
	cylinder {<0.0, 0.0, -0.1>, <0.0, 0.0, 3.9>, 2.4}   // z axle channel (free)
	box {<-0.1, -3.9, -3.9>, < 4.0, 3.9, 0.0>}          // z/x joiner cutout for part 2
	box {< 3.9, -2.4, -0.9>, <12.1, 2.4, 0.9>}          // x axle channel x/y axis
	box {< 3.9, -0.9, -2.4>, <12.1, 0.9, 2.4>}          // x axle channel x/z axis
	bounded_by {
		box {<-3.8, -3.8, -3.8>, <12.0, 3.8, 4.0>}
	}
}

//========== lock for z axle ==========
#declare objRod90Lock = difference {
	cylinder {<0.0, 0.0, -3.8>, <0.0, 0.0, 0.0>, 3.8}   // main part
	box {<-2.4, -0.9, -3.9>, <2.4, 0.9, 0.1>}           // z axle channel z/x axis
	box {<-0.9, -2.4, -3.9>, <0.9, 2.4, 0.1>}           // z axle channel z/y axis
	torus {3.8, 0.9                                     // outer lock channel
		rotate <90, 0, 0>
		translate <0.0, 0.0, -1.9>
	}
	bounded_by {
		box {<-3.8, -3.8, -4.0>, <3.8, 3.8, 0.0>}
	}
}

//========== joint for z and x axles with both axles locked ==========
#declare objRod90LockJoint = union {
	object {objRod90Joint}
	object {objRod90Lock}
	bounded_by {
		box {<-3.8, -3.8, -4.0>, <12.0, 3.8, 4.0>}
	}
}

/*********************************************************************
End of LEGOBITS.INC
*********************************************************************/
