/*
 *	This file contains the functions needed for processing outgoing
 *	TCP segments.
 *
 *	04/15/94, Kay Roemer.
 */

#include "atarierr.h"
#include "sockerr.h"
#include "kerbind.h"
#include "net.h"
#include "buf.h"
#include "tcp.h"
#include "ip.h"
#include "util.h"
#include "iov.h"

extern void	*memcpy (void *, const void *, unsigned long);

static void	tcbos_idle	(struct tcb *, short);
static void	tcbos_xmit	(struct tcb *, short);
static void	tcbos_retrans	(struct tcb *, short);
static void	tcbos_persist	(struct tcb *, short);

static BUF *	tcp_mkseg	(struct tcb *, long);
static long	tcp_sndseg	(struct tcb *, BUF *);
static void	tcp_retrans	(struct tcb *);
static void	tcp_probe	(struct tcb *);
static long	tcp_dropdata	(struct tcb *);
static long	tcp_sndhead	(struct tcb *);
static void	tcp_rtt		(struct tcb *);
static short	canretrans	(struct tcb *);
static void	wakeme		(long);

void (*tcb_ostate[]) (struct tcb *, short) = {
	tcbos_idle,
	tcbos_retrans,
	tcbos_persist,
	tcbos_xmit
};

/*
 * Output finite state machine.
 */

static void
tcbos_idle (tcb, event)
	struct tcb *tcb;
	short event;
{
	switch (event) {
	case TCBOE_SEND:
		if (SEQGE (tcb->snd_nxt, tcb->seq_write)) break;
		tcp_sndhead (tcb);
		tcb->ostate = canretrans (tcb) ? TCBOS_XMIT : TCBOS_PERSIST;
		tcb->persist_tmo =
		tcb->retrans_tmo = tcp_timeout (tcb);
		tcb->timer_evt = event_add (tcb->retrans_tmo,wakeme,(long)tcb);

DEBUG (("tcpout: port %d: IDLE -> %s", tcb->data->src.port,
	tcb->ostate == TCBOS_XMIT ? "XMIT" : "PERSIST"));

		break;
	}
}

static void
tcbos_xmit (tcb, event)
	struct tcb *tcb;
	short event;
{
	switch (event) {
	case TCBOE_SEND:
		if (SEQLT (tcb->snd_nxt, tcb->seq_write))
			tcp_sndhead (tcb);
		break;

	case TCBOE_WNDCLOSE:
		if (!canretrans (tcb)) {
			tcb->persist_tmo = tcb->retrans_tmo;
			tcb->ostate = TCBOS_PERSIST;

DEBUG (("tcpout: port %d: XMIT -> PERSIST", tcb->data->src.port));

		}
		break;

	case TCBOE_WNDOPEN:
		if (SEQLT (tcb->snd_nxt, tcb->seq_write))
			tcp_sndhead (tcb);

		if (!canretrans (tcb)) {
			tcb->persist_tmo = tcb->retrans_tmo;
			tcb->ostate = TCBOS_PERSIST;

DEBUG (("tcpout: port %d: XMIT -> PERSIST", tcb->data->src.port));

		}
		break;

	case TCBOE_ACKRCVD:
		tcp_rtt (tcb);
		tcp_dropdata (tcb);
		if (SEQGE (tcb->snd_una, tcb->seq_write)) {
			event_del (tcb->timer_evt);
			tcb->timer_evt = 0;
			tcb->ostate = TCBOS_IDLE;

DEBUG (("tcpout: port %d: XMIT -> IDLE", tcb->data->src.port));

			break;
		}
		tcb->retrans_tmo = tcp_timeout (tcb);
		tcb->nretrans = 0;
		event_reset (tcb->timer_evt, tcb->retrans_tmo);
		break;

	case TCBOE_TIMEOUT:

DEBUG (("tcpout: port %d: XMIT -> RETRANS", tcb->data->src.port));

		tcb->ostate = TCBOS_RETRANS;
		tcp_retrans (tcb);
		break;
	}
}

static void
tcbos_retrans (tcb, event)
	struct tcb *tcb;
	short event;
{
	switch (event) {
	case TCBOE_TIMEOUT:
		tcp_retrans (tcb);
		break;

	case TCBOE_WNDCLOSE:
		if (!canretrans (tcb)) {
			tcb->persist_tmo = tcb->retrans_tmo;
			tcb->ostate = TCBOS_PERSIST;

DEBUG (("tcpout: port %d: RETRANS -> PERSIST", tcb->data->src.port));

		}
		break;

	case TCBOE_ACKRCVD:
		tcp_dropdata (tcb);
		if (SEQGE (tcb->snd_una, tcb->seq_write)) {
			event_del (tcb->timer_evt);
			tcb->timer_evt = 0;
			tcb->ostate = TCBOS_IDLE;

DEBUG (("tcpout: port %d: RETRANS -> IDLE", tcb->data->src.port));

			break;
		}
		if (SEQLT (tcb->snd_nxt, tcb->seq_write))
			tcp_sndhead (tcb);


DEBUG (("tcpout: port %d: RETRANS -> XMIT", tcb->data->src.port));

		tcb->ostate = TCBOS_XMIT;
		tcb->retrans_tmo = tcp_timeout (tcb);
		tcb->nretrans = 0;
		event_reset (tcb->timer_evt, tcb->retrans_tmo);
		break;
	}
}

static void
tcbos_persist (tcb, event)
	struct tcb *tcb;
	short event;
{
	switch (event) {
	case TCBOE_TIMEOUT:
		tcp_probe (tcb);
		break;

	case TCBOE_ACKRCVD:
		tcp_dropdata (tcb);
		if (SEQGE (tcb->snd_una, tcb->seq_write)) {
			event_del (tcb->timer_evt);
			tcb->timer_evt = 0;
			tcb->ostate = TCBOS_IDLE;

DEBUG (("tcpout: port %d: PERSIST -> IDLE", tcb->data->src.port));

		}
		break;

	case TCBOE_WNDOPEN:
		if (SEQLT (tcb->snd_nxt, tcb->seq_write))
			tcp_sndhead (tcb);

		if (canretrans (tcb)) {

DEBUG (("tcpout: port %d: PERSIST -> XMIT", tcb->data->src.port));

			tcb->ostate = TCBOS_XMIT;
			tcb->retrans_tmo = tcp_timeout (tcb);
			event_reset (tcb->timer_evt, tcb->retrans_tmo);
		}
		break;
	}
}

/*
 * Helper routines for the output state machine.
 */

#define TH(b)		((struct tcp_dgram *)(b)->dstart)
#define SEQ1ST(b)	(TH(b)->seq)
#define DATLEN(b)	(TH(b)->urgptr)

/* Make a TCP segment of size 'size' an fill in some header fields. */
static BUF *
tcp_mkseg (tcb, size)
	struct tcb *tcb;
	long size;
{
	struct tcp_dgram *tcph;
	BUF *b;

	b = buf_alloc (TCP_RESERVE + size, TCP_RESERVE, BUF_NORMAL);
	if (b == 0) {
		DEBUG (("tcp_mkseg: no space for segment"));
		return 0;
	}
	tcph = (struct tcp_dgram *)b->dstart;
	tcph->srcport = tcb->data->src.port;
	tcph->dstport = tcb->data->dst.port;
	tcph->flags = 0;
	tcph->seq = tcb->snd_nxt;
	tcph->ack = tcb->rcv_nxt;
	tcph->hdrlen = TCP_MINLEN/4;
	tcph->window = 0;
	tcph->urgptr = 0;
	tcph->chksum = 0;
	b->dend += size;
	return b;
}

/* Send a copy of the segment in 'b' to the net. */
static long
tcp_sndseg (tcb, b)
	struct tcb *tcb;
	BUF *b;
{
	struct tcp_dgram *tcph;
	long len;
	BUF *nb;

	nb = buf_alloc (b->buflen, TCP_RESERVE, BUF_NORMAL);
	if (nb == 0) {
		DEBUG (("tcp_sndseg: no mem to send"));
		return ENSMEM;
	}
	len = (long)b->dend - (long)b->dstart;
	memcpy (nb->dstart, b->dstart, len);
	nb->dend += len;

	tcph = (struct tcp_dgram *)nb->dstart;
	tcph->ack = tcb->rcv_nxt;
	tcph->window = tcp_rcvwnd (tcb);
	tcph->chksum = 0;
	tcph->chksum = tcp_checksum (tcph, len,
		tcb->data->src.addr,
		tcb->data->dst.addr);
#if 0
	DEBUG (("tcp_sndseg::"));
	tcp_dump (nb);
#endif
	return ip_send (tcb->data->src.addr, tcb->data->dst.addr,
			nb, IPPROTO_TCP, 0);
}

/* Drop data from the segment retransmission queue. */
static long
tcp_dropdata (tcb)
	struct tcb *tcb;
{
	struct in_dataq *q = &tcb->data->snd;
	BUF *b, *nxtb;
	long una, oldlen;

	oldlen = q->curdatalen;
	una = tcb->snd_una;
	for (b = q->qfirst; b; b = nxtb) {
		if (SEQLE (SEQ1ST (b) + tcp_seglen (b, TH (b)), una)) {

DEBUG (("tcp_dropdata: port %d: <%ld,%ld>", TH(b)->srcport,
	SEQ1ST (b), tcp_seglen (b, TH (b))));

			if ((nxtb = b->next)) {
				nxtb->prev = 0;
				q->qfirst = nxtb;
				q->curdatalen -= DATLEN (b);
			} else {
				q->qfirst = 0;
				q->qlast = 0;
				q->curdatalen = 0;
			}
			buf_deref (b, BUF_NORMAL);
		} else	break;
	}
	if (tcb->data->sock && q->curdatalen < MIN (q->maxdatalen, oldlen)) {
		so_wakewsel (tcb->data->sock);
		wake (IO_Q, (long)tcb->data->sock);
		return 1;
	}
	return 0;
}

/* Send all unsent urgent segments and all non urgent segments that fall
 * into the receivers window. */
static long
tcp_sndhead (tcb)
	struct tcb *tcb;
{
	long wndnxt, seqnxt, r;
	short sent = 0;
	BUF *b;

	wndnxt = tcb->snd_wndack + tcb->snd_wnd;

/* Skip already sent segments */
	for (b = tcb->data->snd.qfirst; b; b = b->next) {
		seqnxt = SEQ1ST (b) + tcp_seglen (b, TH (b));
		if (SEQLT (tcb->snd_nxt, seqnxt)) break;
	}

/* Send the segments that fall into the window */
	for ( ; b; b = b->next) {
		seqnxt = SEQ1ST (b) + tcp_seglen (b, TH (b));

		/*
		 * Skip urgent segments, they will be sent by
		 * tcp_sndurg().
		 */
		if (TH(b)->flags & TCPF_URG) {
			tcb->snd_nxt = seqnxt;
			++sent;
			continue;
		}

		/*
		 * This segment does not fit into the window. Stop.
		 */
		if (SEQGT (seqnxt, wndnxt))
			break;

		/*
		 * Segment end falls into the window or is urgent.
		 * Send the datagram.
		 */
		r = tcp_sndseg (tcb, b);
		if (r < 0) break;

		tcb->snd_nxt = seqnxt;
		++sent;
	}
	return sent;
}

/* Probe the other TCP's window */
static void
tcp_probe (tcb)
	struct tcb *tcb;
{
	struct tcp_dgram *tcph;
	BUF *b;

DEBUG (("tcp_probe: port %d: sending probe", tcb->data->src.port));

	b = tcp_mkseg (tcb, TCP_MINLEN);
	if (b != 0) {
		tcph = (struct tcp_dgram *)b->dstart;
		tcph->flags = TCPF_ACK;
		tcph->seq = tcb->snd_una - 1;
		tcph->window = tcp_rcvwnd (tcb);
		tcph->chksum = 0;
		tcph->chksum = tcp_checksum (tcph, TCP_MINLEN,
			tcb->data->src.addr,
			tcb->data->dst.addr);

		ip_send (tcb->data->src.addr, tcb->data->dst.addr, b,
			IPPROTO_TCP, 0);
	} else	DEBUG (("do_probe: no memory to probe"));

	tcb->persist_tmo = MIN (TCP_MAXPROBE, tcb->persist_tmo << 1);
	tcb->timer_evt = event_add (tcb->persist_tmo, wakeme, (long)tcb);
}

/* Retransmit segments after timeout */
static void
tcp_retrans (tcb)
	struct tcb *tcb;
{
DEBUG (("tcp_retrans: port %d: sending %dst retransmit",
	tcb->data->src.port, tcb->nretrans+1));

	tcb->timer_evt = 0;
	if (++tcb->nretrans > TCP_MAXRETRY) {
		DEBUG (("do_retrans: max number of retransm. exceeded"));
		tcb_reset (tcb, ETIMEDOUT);
		tcb->state = TCBS_CLOSED;
		return;
	}
	if (tcb->data->snd.qfirst == 0) {
		DEBUG (("do_retrans: queue is empty ?!"));
		tcb->ostate = TCBOS_IDLE;
		return;
	}
	tcp_sndseg (tcb, tcb->data->snd.qfirst);

	tcb->retrans_tmo = MIN (TCP_MAXRETRANS, tcb->retrans_tmo << 1);
	tcb->timer_evt = event_add (tcb->retrans_tmo, wakeme, (long)tcb);
}

static void
wakeme (arg)
	long arg;
{
	TCB_OSTATE ((struct tcb *)arg, TCBOE_TIMEOUT);
}

static short
canretrans (tcb)
	struct tcb *tcb;
{
	long seqnxt;
	BUF *b;

	if (SEQGE (tcb->snd_una, tcb->snd_nxt))
		return 0;

	b = tcb->data->snd.qfirst;
	if (b == 0) return 0;

	seqnxt = SEQ1ST (b) + tcp_seglen (b, TH (b));
	return (SEQLE (seqnxt, tcb->snd_wndack + tcb->snd_wnd) &&
		SEQLE (seqnxt, tcb->snd_nxt));
}

/*
 * Exported functions for other modules.
 */

/* Generate and send TCP segments from the data in `iov' and/or with
 * the flags in `flags'. */
long
tcp_output (tcb, iov, niov, len, offset, flags)
	struct tcb *tcb;
	struct iovec *iov;
	long len, offset;
	short niov, flags;
{
	struct tcp_dgram *tcph;
	struct in_dataq *q = &tcb->data->snd;
	BUF *b, *first = 0;
	long r = 0;

/* Check if ACK only */
	if (len == 0 && (flags & (TCPF_SYN|TCPF_FIN)) == 0) {
		b = tcp_mkseg (tcb, TCP_MINLEN);
		if (b == 0) {
			DEBUG (("tcp_out: cannot send, memory low"));
			return ENSMEM;
		}
		tcph = TH (b);
		tcph->flags = TCPF_ACK | (flags & TCPF_PSH);
		tcph->window = tcp_rcvwnd (tcb);

		tcph->chksum = tcp_checksum (tcph, TCP_MINLEN,
			tcb->data->src.addr,
			tcb->data->dst.addr);
#if 0
		DEBUG (("tcp_output::"));
		tcp_dump (b);
#endif

		return ip_send (tcb->data->src.addr, tcb->data->dst.addr,
			b, IPPROTO_TCP, 0);
	}

	do {
		r = MIN (tcb->snd_mss, len);
		b = tcp_mkseg (tcb, r + TCP_MINLEN);
		if (b == 0) {
			DEBUG (("tcp_out: cannot send, memory low"));
			return ENSMEM;
		}
		tcph = TH (b);
		tcph->flags = flags & ~(TCPF_SYN|TCPF_FIN);
		tcph->seq = tcb->seq_write;
		if (r > 0) {
			if (!first) first = b;
			r = iov2buf_cpy (tcph->data, r, iov, niov, offset);
			offset += r;
			len -= r;
			tcph->flags |= TCPF_PSH;
		}
		if (len == 0 && flags & TCPF_FIN)
			tcph->flags |= TCPF_FIN;

		if (offset == r && flags & TCPF_SYN)
			tcph->flags |= TCPF_SYN;
		else	tcph->flags |= TCPF_ACK;

		tcph->urgptr = r; /* data length of segment */

DEBUG (("tcp_output: port %d: <%ld,%ld%s%s%s>",
	tcph->srcport, tcph->seq, r,
	tcph->flags & TCPF_URG ? ",URG" : "",
	tcph->flags & TCPF_SYN ? ",SYN" : "",
	tcph->flags & TCPF_FIN ? ",FIN" : ""));

		if (q->qlast == 0) {
			b->next = b->prev = 0;
			q->qlast = q->qfirst = b;
			q->curdatalen = r;
		} else {
			b->next = 0;
			b->prev = q->qlast;
			q->qlast->next = b;
			q->qlast = b;
			q->curdatalen += r;
		}
		tcb->seq_write += tcp_seglen (b, tcph);
	} while (len > 0);
	TCB_OSTATE (tcb, TCBOE_SEND);

	if (flags & TCPF_URG) while (first) {
		tcp_sndseg (tcb, first);
		first = first->next;
	}
	return 0;
}

/* Return the size of our window we should advertise to the remote TCP.
 * For now only return the current free buffer space, later we have
 * to take into account congestion control. */
long
tcp_rcvwnd (tcb)
	struct tcb *tcb;
{
	long space, minwnd;

	space = tcb->data->rcv.maxdatalen - tcb->data->rcv.curdatalen;
	if (space < tcb->rcv_mss || space*4 < tcb->data->rcv.maxdatalen)
		space = 0;

	if (tcb->state >= TCBS_SYNRCVD) {
		minwnd = tcb->rcv_wnd - tcb->rcv_nxt;
		if (space < minwnd)
			return minwnd;

		tcb->rcv_wnd = tcb->rcv_nxt + space;
	}
	return space;
}

/* Return the initial timeout to use for retransmission and persisting. */
long
tcp_timeout (tcb)
	struct tcb *tcb;
{
	long tmout = ((tcb->rtt >> 1) + tcb->rttdev) >> 1;
	return MAX (tmout, TCP_MINRETRANS);
}

/* update the round trip time mean and deviation. Note that tcb->rtt is scaled
 * by 8 and tcb->rttdev by 4. */
void
tcp_rtt (tcb)
	struct tcb *tcb;
{
	long err;

	err = tcb->retrans_tmo - event_delta (tcb->timer_evt);
	err -= tcb->rtt >> 3;
	tcb->rtt += err;
	if (err < 0) err = -err;
	tcb->rttdev += err - (tcb->rttdev >> 2);
}
