/*
 *	This file contains the functions needed for processing incoming
 *	TCP segments.
 *
 *	04/13/94, Kay Roemer.
 */

#include "kerbind.h"
#include "sockerr.h"
#include "signal.h"
#include "net.h"
#include "buf.h"
#include "tcp.h"
#include "ip.h"
#include "util.h"

extern void	*memcpy	(void *, const void *, unsigned long);

static void	tcbs_closed		(struct tcb *, BUF *);
static void	tcbs_listen		(struct tcb *, BUF *);
static void	tcbs_synsent		(struct tcb *, BUF *);
static void	tcbs_synrcvd		(struct tcb *, BUF *);
static void	tcbs_established	(struct tcb *, BUF *);
static void	tcbs_finwait1		(struct tcb *, BUF *);
static void	tcbs_finwait2		(struct tcb *, BUF *);
static void	tcbs_closewait		(struct tcb *, BUF *);
static void	tcbs_lastack		(struct tcb *, BUF *);
static void	tcbs_closing		(struct tcb *, BUF *);
static void	tcbs_timewait		(struct tcb *, BUF *);

static long	tcp_rcvdata		(struct tcb *, BUF *);
static long	tcp_addseg		(struct in_dataq *, BUF *);
static void	tcp_sndwnd		(struct tcb *, struct tcp_dgram *);
static long	tcp_ack			(struct tcb *, BUF *);
 
void (*tcb_state[]) (struct tcb *, BUF *) = {
	tcbs_closed,
	tcbs_listen,
	tcbs_synsent,
	tcbs_synrcvd,
	tcbs_established,
	tcbs_finwait1,
	tcbs_finwait2,
	tcbs_closewait,
	tcbs_lastack,
	tcbs_closing,
	tcbs_timewait
};

/*
 * Input finite state machine.
 */

static void
tcbs_closed (tcb, buf)
	struct tcb *tcb;
	BUF *buf;
{
	tcp_sndrst (buf);
	buf_deref (buf, BUF_NORMAL);
}

static void
tcbs_listen (tcb, buf)
	struct tcb *tcb;
	BUF *buf;
{
	struct tcp_dgram *tcph = (struct tcp_dgram *)IP_DATA (buf);
	struct socket *so;
	struct in_data *data;
	struct tcb *ntcb;
	long r;

	if ((tcph->flags & (TCPF_RST|TCPF_SYN)) != TCPF_SYN) {
		buf_deref (buf, BUF_NORMAL);
		return;
	}
	if (tcph->flags & TCPF_ACK) {
		tcp_sndrst (buf);
		buf_deref (buf, BUF_NORMAL);
		return;
	}
	ntcb = tcb_alloc ();
	data = in_data_create ();
	so = so_create ();
	if (!tcb || !data || !so) {
		DEBUG (("tcp_listen: cannot create new socket, memory low"));
		if (ntcb) tcb_free (ntcb);
		if (data) in_data_destroy (data, 0);
		if (so) kfree (so);
	}

/* Init the socket */
	so->state = SS_ISUNCONNECTED;
	so->data = data;

/* Init the inet data */
	data->protonum = tcb->data->protonum;
	data->proto = tcb->data->proto;
	data->sock = so;
	data->pcb = ntcb;
	data->src.addr = IP_DADDR (buf);
	data->src.port = tcph->dstport;
	data->dst.addr = IP_SADDR (buf);
	data->dst.port = tcph->srcport;
	data->linger = tcb->data->linger;
	data->flags = IN_ISBOUND|IN_ISCONNECTED;
	data->flags |= tcb->data->flags & (IN_KEEPALIVE|IN_OOBINLINE|
		IN_CHECKSUM|IN_DONTROUTE|IN_BROADCAST|IN_LINGER);

/* Init the TCB */
	ntcb->data = data;
	ntcb->flags |= TCBF_PASSIVE;
	ntcb->state = TCBS_SYNRCVD;
	ntcb->snd_isn =
	ntcb->snd_una =
	ntcb->snd_wndack =
	ntcb->snd_nxt =
	ntcb->snd_urg =
	ntcb->seq_write = tcp_isn ();

	ntcb->snd_wnd = 2; /* enough for SYN,FIN */

	ntcb->rcv_isn =
	ntcb->rcv_nxt =
	ntcb->rcv_wnd =
	ntcb->rcv_urg = tcph->seq;
	ntcb->seq_read =
	ntcb->seq_uread = tcph->seq + 1; /* SEQ of next byte to read() */

	in_data_put (data);

	r = so_connect (tcb->data->sock, so, tcb->data->backlog, 1, 0);
	if (r != EINPROGRESS) {
		DEBUG (("tcp_listen: cannot connect to server"));
		so_release (so);
		kfree (so);
		tcp_sndrst (buf);
		buf_deref (buf, BUF_NORMAL);
		return;
	}

	tcp_options (ntcb, tcph);
	tcp_rcvdata (ntcb, buf);
	tcp_output (ntcb, 0, 0, 0, 0, TCPF_SYN|TCPF_ACK);
}

static void
tcbs_synsent (tcb, buf)
	struct tcb *tcb;
	BUF *buf;
{
	struct tcp_dgram *tcph = (struct tcp_dgram *)IP_DATA (buf);

	if (tcph->flags & TCPF_ACK && (
	    SEQLE (tcph->ack, tcb->snd_isn) ||
	    SEQGT (tcph->ack, tcb->snd_nxt))) {
		tcp_sndrst (buf);
		buf_deref (buf, BUF_NORMAL);
		return;
	}
	if (tcph->flags & TCPF_RST) {
		if (tcph->flags & TCPF_ACK) {
			tcb_reset (tcb, ECONNRESET);
			tcb->state = TCBS_CLOSED;
			tcb->data->sock->state = SS_ISDISCONNECTING;
		}
		buf_deref (buf, BUF_NORMAL);
		return;
	}
	if (!(tcph->flags & TCPF_SYN)) {
		buf_deref (buf, BUF_NORMAL);
		return;
	}

	tcb->rcv_isn =
	tcb->rcv_nxt =
	tcb->rcv_wnd =
	tcb->rcv_urg = tcph->seq;
	tcb->seq_read =
	tcb->seq_uread = tcph->seq + 1; /* SEQ of next byte to read() */

	tcp_ack (tcb, buf);

	if (SEQGT (tcb->snd_una, tcb->snd_isn)) {
		/* SYN is acked */

DEBUG (("tcp port %d: SYNSENT -> ESTABLISHED", tcb->data->src.port));

		tcb->state = TCBS_ESTABLISHED;
		tcb->snd_wnd = tcph->window;
		tcb->snd_wndseq = tcph->seq;
		tcb->snd_wndack = tcph->ack;

		tcb->data->sock->state = SS_ISCONNECTED;
		so_wakewsel (tcb->data->sock);
		wake (IO_Q, (long)tcb->data->sock);
	} else {
		/* SYN not acked */

DEBUG (("tcp port %d: SYNSENT -> SYNRCVD", tcb->data->src.port));

		tcb->state = TCBS_SYNRCVD;
	}
	tcp_rcvdata (tcb, buf);
	tcp_output (tcb, 0, 0, 0, 0, TCPF_ACK);
	return;
}

static void
tcbs_synrcvd (tcb, buf)
	struct tcb *tcb;
	BUF *buf;
{
	struct tcp_dgram *tcph = (struct tcp_dgram *)IP_DATA (buf);

	if (tcph->flags & TCPF_RST) {
		if (tcb->flags & TCBF_PASSIVE) {
			struct socket *so = tcb->data->sock;
			so_release (so);
			kfree (so);
		} else {
			tcb_reset (tcb, ECONNREFUSED);
			tcb->state = TCBS_CLOSED;
			tcb->data->sock->state = SS_ISDISCONNECTING;
		}
		buf_deref (buf, BUF_NORMAL);
		return;
	}
	if (tcph->flags & TCPF_SYN) {
		tcp_sndrst (buf);
		if (tcb->flags & TCBF_PASSIVE) {
			struct socket *so = tcb->data->sock;
			so_release (so);
			kfree (so);
		} else {
			tcb_reset (tcb, ECONNRESET);
			tcb->state = TCBS_CLOSED;
			tcb->data->sock->state = SS_ISDISCONNECTING;
		}
		buf_deref (buf, BUF_NORMAL);
		return;
	}
	if (tcp_ack (tcb, buf)) {
		buf_deref (buf, BUF_NORMAL);
		return;
	}

DEBUG (("tcp port %d: SYNRCVD -> ESTABLISHED", tcb->data->src.port));

	tcb->state = TCBS_ESTABLISHED;
	tcb->snd_wnd = tcph->window;
	tcb->snd_wndseq = tcph->seq;
	tcb->snd_wndack = tcph->ack;

	tcp_rcvdata (tcb, buf);
	if (tcb->flags & TCBF_PASSIVE) {
		so_wakersel (tcb->data->sock->conn);
		wake (IO_Q, (long)tcb->data->sock->conn);
	} else {
		tcb->data->sock->state = SS_ISCONNECTED;
		so_wakewsel (tcb->data->sock);
		wake (IO_Q, (long)tcb->data->sock);
	}
	if (tcp_finished (tcb)) {

DEBUG (("tcp port %d: ESTABLISHED -> CLOSEWAIT", tcb->data->src.port));

		tcb->state = TCBS_CLOSEWAIT;
		tcb->data->sock->state = SS_ISDISCONNECTING;
	}
}

static void
tcbs_established (tcb, buf)
	struct tcb *tcb;
	BUF *buf;
{
	struct tcp_dgram *tcph = (struct tcp_dgram *)IP_DATA (buf);

	if (tcph->flags & TCPF_RST) {
		tcb_reset (tcb, ECONNRESET);
		tcb->state = TCBS_CLOSED;
		buf_deref (buf, BUF_NORMAL);
		return;
	}
	if (tcph->flags & TCPF_SYN) {
		tcp_sndrst (buf);
		tcb_reset (tcb, ECONNRESET);
		tcb->state = TCBS_CLOSED;
		buf_deref (buf, BUF_NORMAL);
		return;
	}
	if (tcp_ack (tcb, buf)) {
		buf_deref (buf, BUF_NORMAL);
		return;
	}
	tcp_sndwnd (tcb, tcph);
	tcp_rcvdata (tcb, buf);
	if (tcp_finished (tcb)) {

DEBUG (("tcp port %d: ESTABLISHED -> CLOSEWAIT", tcb->data->src.port));

		tcb->state = TCBS_CLOSEWAIT;
		so_wakersel (tcb->data->sock);
		wake (IO_Q, (long)tcb->data->sock);
	}
	return;
}

static void
tcbs_finwait1 (tcb, buf)
	struct tcb *tcb;
	BUF *buf;
{
	struct tcp_dgram *tcph = (struct tcp_dgram *)IP_DATA (buf);

	if (tcph->flags & (TCPF_RST|TCPF_SYN)) {
		tcp_sndrst (buf);
		tcb_reset (tcb, ECONNRESET);
		tcb->state = TCBS_CLOSED;
		buf_deref (buf, BUF_NORMAL);
		return;
	}
	if (tcp_ack (tcb, buf)) {
		buf_deref (buf, BUF_NORMAL);
		return;
	}
	tcp_sndwnd (tcb, tcph);
	tcp_rcvdata (tcb, buf);

	if (tcp_finished (tcb)) {
		if (SEQGT (tcb->seq_write, tcb->snd_una)) {
			/* FIN is not acked */

DEBUG (("tcp port %d: FINWAIT1 -> CLOSING", tcb->data->src.port));

			tcb->state = TCBS_CLOSING;
		} else {

DEBUG (("tcp port %d: FINWAIT1 -> TIMEWAIT", tcb->data->src.port));

			tcb->state = TCBS_TIMEWAIT;
			wake (IO_Q, (long)tcb->data->sock);
			tcb_wait (tcb);
		}
	} else if (SEQLE (tcb->seq_write, tcb->snd_una)) {
		/* FIN is acked */

DEBUG (("tcp port %d: FINWAIT1 -> FINWAIT2", tcb->data->src.port));

		tcb->state = TCBS_FINWAIT2;
		wake (IO_Q, (long)tcb->data->sock);
	}
}

static void
tcbs_finwait2 (tcb, buf)
	struct tcb *tcb;
	BUF *buf;
{
	struct tcp_dgram *tcph = (struct tcp_dgram *)IP_DATA (buf);

	if (tcph->flags & (TCPF_RST|TCPF_SYN)) {
		tcp_sndrst (buf);
		buf_deref (buf, BUF_NORMAL);
		tcb_delete ((long)tcb);
		return;
	}
	if (tcp_ack (tcb, buf)) {
		buf_deref (buf, BUF_NORMAL);
		return;
	}
	tcp_rcvdata (tcb, buf);

	if (tcp_finished (tcb)) {

DEBUG (("tcp port %d: FINWAIT2 -> TIMEWAIT", tcb->data->src.port));

		tcb->state = TCBS_TIMEWAIT;
		tcb_wait (tcb);
	}	
}

static void
tcbs_closewait (tcb, buf)
	struct tcb *tcb;
	BUF *buf;
{
	struct tcp_dgram *tcph = (struct tcp_dgram *)IP_DATA (buf);

	if (tcph->flags & (TCPF_RST|TCPF_SYN)) {
		tcp_sndrst (buf);
		tcb_reset (tcb, ECONNRESET);
		tcb->state = TCBS_CLOSED;
		buf_deref (buf, BUF_NORMAL);
		return;
	}
	tcp_ack (tcb, buf);
	tcp_sndwnd (tcb, tcph);
	buf_deref (buf, BUF_NORMAL);
}

static void
tcbs_lastack (tcb, buf)
	struct tcb *tcb;
	BUF *buf;
{
	struct tcp_dgram *tcph = (struct tcp_dgram *)IP_DATA (buf);

	if (tcph->flags & (TCPF_RST|TCPF_SYN)) {
		tcp_sndrst (buf);
		tcb_reset (tcb, ECONNRESET);
		tcb->state = TCBS_CLOSED;
		buf_deref (buf, BUF_NORMAL);
		return;
	}
	tcp_ack (tcb, buf);

	if (SEQLE (tcb->seq_write, tcb->snd_una)) {
		/* FIN is acked */

DEBUG (("tcp port %d: LASTACK -> CLOSED", tcb->data->src.port));

		tcb->state = TCBS_CLOSED;
		wake (IO_Q, (long)tcb->data->sock);
	}
}

static void
tcbs_closing (tcb, buf)
	struct tcb *tcb;
	BUF *buf;
{
	struct tcp_dgram *tcph = (struct tcp_dgram *)IP_DATA (buf);

	if (tcph->flags & (TCPF_RST|TCPF_SYN)) {
		tcp_sndrst (buf);
		tcb_reset (tcb, ECONNRESET);
		tcb->state = TCBS_CLOSED;
		buf_deref (buf, BUF_NORMAL);
		return;
	}
	tcp_ack (tcb, buf);

	if (SEQLE (tcb->seq_write, tcb->snd_una)) {
		/* FIN is acked */

DEBUG (("tcp port %d: CLOSING -> TIMEWAIT", tcb->data->src.port));

		tcb->state = TCBS_TIMEWAIT;
		wake (IO_Q, (long)tcb->data->sock);
		tcb_wait (tcb);
	}
}

/* Note that in TIMEWAIT state the tcb has no associated socket. */
static void
tcbs_timewait (tcb, buf)
	struct tcb *tcb;
	BUF *buf;
{
	struct tcp_dgram *tcph = (struct tcp_dgram *)IP_DATA (buf);

	if (tcph->flags & (TCPF_RST|TCPF_SYN)) {
		tcp_sndrst (buf);
		buf_deref (buf, BUF_NORMAL);
		tcb_delete ((long)tcb);
		return;
	}
	tcp_ack (tcb, buf);
	tcp_rcvdata (tcb, buf);
	tcb_wait (tcb);
}

/*
 * Helper routines for the finite state machine
 */

#define TH(b)		((struct tcp_dgram *)(b)->dstart)
#define SEQ1ST(b)	(TH(b)->seq)
#define SEQNXT(b)	(TH(b)->seq + TH(b)->urgptr)

static long
tcp_rcvdata (tcb, buf)
	struct tcb *tcb;
	BUF *buf;
{
	struct tcp_dgram *tcph;
	long nxt, datalen;

/* Strip the IP header */
	buf->dstart = IP_DATA (buf);
	tcph = (struct tcp_dgram *)buf->dstart;

/* Store the segments data length in urgptr */
	datalen = (long)buf->dend - (long)tcph - tcph->hdrlen*4;
	tcph->urgptr = datalen;
#if 0
	DEBUG (("tcp_rcvdata::"));
	tcp_dump (buf);
#endif
	if (tcph->flags & TCPF_SYN) {
		++tcb->rcv_nxt;
		++tcph->seq;
	}
	if (tcph->flags & TCPF_FIN) {
		tcb->flags |= TCBF_FIN;
		tcb->seq_fin = tcph->seq + datalen;
	}
	if (tcph->flags & TCPF_PSH) {
		tcb->flags |= TCBF_PSH;
		tcb->seq_psh = tcph->seq + datalen;
	}
	tcph->flags &= ~(TCPF_SYN|TCPF_PSH|TCPF_FIN);

/* Add the segment to the recv queue and wake things up */
	nxt = tcb->rcv_nxt;
	if (!tcp_addseg (&tcb->data->rcv, buf)) {
		BUF *b = buf;

/* Compute the next sequence number beyond the continuous sequence of
 * received bytes in `nxt'. */
		while (b && SEQLE (SEQ1ST (b), nxt)) {
			nxt = SEQNXT (b);
			b = b->next;
		}
		if (tcb->data->sock && SEQLT (tcb->seq_read, nxt)) {
			so_wakersel (tcb->data->sock);
			wake (IO_Q, (long)tcb->data->sock);
		}
	}
	if (tcb->flags & TCBF_FIN && SEQEQ (tcb->seq_fin, nxt)) {
		++nxt;
		/* Do FIN processing */
	}
	if (tcb->flags & TCBF_PSH && SEQLE (tcb->seq_psh, nxt)) {
		/* Do PUSH processing */
	}
	if (SEQLT (tcb->rcv_nxt, nxt)) {
		tcb->rcv_nxt = nxt;
		tcp_output (tcb, 0, 0, 0, 0, TCPF_ACK);
	}
	return 0;
}

/* Add the segment in `buf' to the queue `q'.
 * Note that q->curdatalen is the amount of sequence space in between the
 * two bytes with lowest and highest sequence number in the queue.
 * Also note that the `urgptr' field of the segment beeing insertred MUST
 * hold the segment length for urgent AND non urgent segments! */
static long
tcp_addseg (q, buf)
	struct in_dataq *q;
	BUF *buf;
{
	struct tcp_dgram *tcph = (struct tcp_dgram *)buf->dstart;
	long seq1st, seqnxt;
	BUF *b;

/* Check for zero length segments */
	if (tcph->urgptr == 0) {
		buf_deref (buf, BUF_NORMAL);
		return -1;
	}
	tcph->flags &= ~TCPF_FREEME;

	seq1st = tcph->seq;
	seqnxt = seq1st + tcph->urgptr;
	if (!q->qfirst) {
/* Queue was empty before. */
		buf->prev = buf->next = 0;
		q->qfirst = q->qlast = buf;
		q->curdatalen = tcph->urgptr;
		return 0;
	} else if (SEQLT (SEQNXT (q->qlast), seqnxt)) {
/* Insert at the end of the queue. */
		q->curdatalen += seqnxt - SEQNXT(q->qlast);
		buf->prev = q->qlast;
		buf->next = 0;
		q->qlast->next = buf;
		q->qlast = buf;
	} else if (SEQLT (seq1st, SEQ1ST (q->qfirst))) {
/* Insert at the front of the queue. */
		q->curdatalen += SEQ1ST(q->qfirst) - seq1st;
		buf->prev = 0;
		buf->next = q->qfirst;
		q->qfirst->prev = buf;
		q->qfirst = buf;
	} else {
/* Try to find a hole inbetween two segments which is (partially) filled
 * by the new segment. Search backwards for efficiency reasons. */
		for (b = q->qlast; b->prev; b = b->prev) {
			if (SEQLT (SEQNXT (b->prev), SEQ1ST (b)) &&
			    SEQLT (seq1st, SEQ1ST (b)) &&
			    SEQLT (SEQNXT (b->prev), seqnxt)) {
				buf->prev = b->prev;
				buf->next = b;
				buf->prev->next = buf;
				buf->next->prev = buf;
				break;
			}
		}
/* Now this segment must be a duplicate, free it. */
		if (!b->prev) {
			DEBUG (("tcp_addseg: duplicate segment"));
			buf_deref (buf, BUF_NORMAL);
			return -1;
		}
	}

/* Check for and free segments which are contained in the newly inserted
 * fragment. Note that we must search forward and backward here. */
	while ((b = buf->next) && SEQLE (SEQNXT (b), seqnxt)) {
		if (q->qlast == b) q->qlast = buf;
		buf->next = b->next;
		if (b->next) b->next->prev = buf;
		buf_deref (b, BUF_NORMAL);
	}
	while ((b = buf->prev) && SEQLE (seq1st, SEQ1ST (b))) {
		if (q->qfirst == b) q->qfirst = buf;
		buf->prev = b->prev;
		if (b->prev) b->prev->next = buf;
		buf_deref (b, BUF_NORMAL);
	}
	return 0;
}

/* Update the send window using the segment `tcph' */
static void
tcp_sndwnd (tcb, tcph)
	struct tcb *tcb;
	struct tcp_dgram *tcph;
{
	long owlast, wlast;

	if (!(tcph->flags & TCPF_ACK))
		return;

	if (SEQLT (tcph->seq, tcb->snd_wndseq) || (
	    SEQEQ (tcph->seq, tcb->snd_wndseq) &&
	    SEQLT (tcph->ack, tcb->snd_wndack)))
		return;

	owlast = tcb->snd_wndack + tcb->snd_wnd;
	wlast = tcph->ack + tcph->window;

	tcb->snd_wnd = tcph->window;
	tcb->snd_wndseq = tcph->seq;
	tcb->snd_wndack = tcph->ack;

	if (SEQLT (owlast, wlast) && tcb->snd_wnd > 0)
		TCB_OSTATE (tcb, TCBOE_WNDOPEN);
	else	TCB_OSTATE (tcb, TCBOE_WNDCLOSE);
}

/* Handle incoming acknowledgments */
static long
tcp_ack (tcb, buf)
	struct tcb *tcb;
	BUF *buf;
{
	struct tcp_dgram *tcph = (struct tcp_dgram *)IP_DATA (buf);
	long una;

	if (!(tcph->flags & TCPF_ACK))
		return -1;

	if (SEQLT (tcph->ack, tcb->snd_una)) {
		DEBUG (("tcp_ack: duplicate ack"));
		return 0;
	}
	if (SEQGT (tcph->ack, tcb->snd_nxt)) {
		DEBUG (("tcp_ack: ack for data not yet sent"));
		if (tcb->state == TCBS_SYNRCVD) tcp_sndrst (buf);
		else tcp_sndack (tcb, buf);
		return 0;
	}

	una = tcb->snd_una;
	tcb->snd_una = tcph->ack;
	if (SEQLT (una, tcb->snd_una)) {
		TCB_OSTATE (tcb, TCBOE_ACKRCVD);
	}
	return 0;
}

/*
 * Exported for other modules.
 */

long
tcp_rcvurg (tcb, buf)
	struct tcb *tcb;
	BUF *buf;
{
	struct tcp_dgram *utcph, *tcph;
	long urglen, datalen, hdrlen, nxt;
	BUF *ubuf;

	tcph = (struct tcp_dgram *)IP_DATA (buf);
	if (!(tcph->flags & TCPF_URG))
		return 0;

/* Ignore urgent data before connection is established */
	if (tcb->state < TCBS_SYNRCVD || tcb->state > TCBS_CLOSEWAIT ||
	    tcph->flags & TCPF_SYN) {
		tcph->flags &= ~TCPF_URG;
		return 0;
	}

	hdrlen = tcph->hdrlen*4;
	datalen = (long)buf->dend - (long)tcph - hdrlen;
	urglen = MIN (tcph->urgptr, datalen);
	if (datalen <= 0 || urglen <= 0) {
		tcph->flags &= ~TCPF_URG;
		return 0;
	}

/* Make a new segment containing the urgent data */
	ubuf = buf_alloc (hdrlen + urglen, 0, BUF_NORMAL);
	if (ubuf) {
		utcph = (struct tcp_dgram *)ubuf->dstart;
		memcpy (utcph, tcph, hdrlen + urglen);
		utcph->flags = TCPF_URG;
		utcph->urgptr = urglen;
		ubuf->dend += hdrlen + urglen;
#if 0
		DEBUG (("tcp_rcvurg::"));
		tcp_dump (buf);
#endif

/* Add the segment to the recv queue and wake things up.
 * BUG: here we should send a SIGURG to the sockets owner. But this would
 * require an extra process with euid = 0.
 * Also waking up processes selecting for exceptional conditions should
 * be done here. */
		if (!tcp_addseg (&tcb->data->rcv, ubuf)) {

/* Advance the rcv_nxt pointer and send an ack if necessary. */
			nxt = tcb->rcv_nxt;
			while (ubuf && SEQLE (SEQ1ST (ubuf), nxt)) {
				nxt = SEQNXT (ubuf);
				ubuf = ubuf->next;
			}
			if (SEQLT (tcb->rcv_nxt, nxt)) {
				tcb->rcv_nxt = nxt;
				tcp_output (tcb, 0, 0, 0, 0, TCPF_ACK);
			}
			if (tcb->data->sock) {
				so_wakersel (tcb->data->sock);
				wake (IO_Q, (long)tcb->data->sock);
				tcp_sendsig (tcb, SIGURG);
			}
		}
	} else	DEBUG (("tcp_rcvurg: no space for urgent data"));

/* Adjust the original segment */
	memcpy (TCP_DATA (tcph), TCP_DATA (tcph) + urglen, datalen - urglen);
	tcph->flags &= ~TCPF_URG;
	tcph->seq += urglen;
	buf->dend -= urglen;
	((struct ip_dgram *)buf->dstart)->length -= urglen;
	return 0;
}
