/*
 *	Simple IP implementation (missing fragmentation and option
 *	processing completely).
 *
 *	01/21/94, kay roemer.
 */

#include "kerbind.h"
#include "sockerr.h"
#include "atarierr.h"
#include "inet.h"
#include "in.h"
#include "ip.h"
#include "if.h"
#include "icmp.h"
#include "route.h"

#define SIN(x)	((struct sockaddr_in *)x)

struct in_ip_ops *allipprotos = 0;

/* Return the local IP address which is 'nearest' to `dstaddr'. Note
 * that one machine can have several IP addresses on several networks,
 * so this routine must know `dstaddr'.
 */

unsigned long
ip_local_addr (dstaddr)
	unsigned long dstaddr;
{
	struct route *rt;
	struct ifaddr *ifa;

	if (dstaddr == INADDR_ANY)
		return INADDR_LOOPBACK;
		
	rt = route_get (dstaddr);
	if (!rt) return INADDR_LOOPBACK;

	ifa = if_af2ifaddr (rt->nif, AF_INET);
	route_deref (rt);
	if (!ifa) return INADDR_LOOPBACK;

	return SIN (&ifa->addr)->sin_addr.s_addr;
}

short
ip_is_brdcst_addr (addr)
	unsigned long addr;
{
	struct netif *nif;
	struct ifaddr *ifa;
	
	if (addr == INADDR_ANY || addr == INADDR_BROADCAST)
		return 1;

	for (nif = allinterfaces; nif; nif = nif->next) {
		ifa = if_af2ifaddr (nif, AF_INET);
		if (!ifa) continue;
		
		if (addr == ifa->net ||
		    addr == ifa->subnet ||
		    addr == ifa->net_broadaddr)
			return 1;

		if (nif->flags & IFF_BROADCAST &&
		    addr == SIN (&ifa->ifu.broadaddr)->sin_addr.s_addr)
			return 1;
	}
	return 0;
}
 
short
ip_is_local_addr (addr)
	unsigned long addr;
{
	struct netif *nif;
	struct ifaddr *ifa;

	if (addr == INADDR_LOOPBACK)
		return 1;

	for (nif = allinterfaces; nif; nif = nif->next) {
		ifa = if_af2ifaddr (nif, AF_INET);
		if (ifa && addr == SIN (&ifa->addr)->sin_addr.s_addr)
			return 1;
	}
	return 0;
}

short
ip_chk_addr (addr, nif)
	unsigned long addr;
	struct netif *nif;
{
	struct ifaddr *ifa;

	if (addr == INADDR_ANY || addr == INADDR_BROADCAST)
		return IPADDR_BRDCST;

	if ((addr & 0x7f000000ul) == 0x7f000000ul)
		return IPADDR_LOCAL;

	ifa = if_af2ifaddr (nif, AF_INET);
	if (ifa) {
		if (addr == SIN (&ifa->addr)->sin_addr.s_addr)
			return IPADDR_LOCAL;

		if (addr == ifa->net ||
		    addr == ifa->subnet ||
		    addr == ifa->net_broadaddr)
			return IPADDR_BRDCST;

		if (nif->flags & IFF_BROADCAST &&
		    addr == SIN (&ifa->ifu.broadaddr)->sin_addr.s_addr)
			return IPADDR_BRDCST;
	}
	if (!IN_CLASSA (addr) && !IN_CLASSB (addr) && !IN_CLASSC (addr))
		return IPADDR_BADCLASS;

	return IPADDR_NONE;
}

/* Check whether foreign IP address `foreign' matches local IP address
 * `local'. Note that `foreign' might be a broadcast address that matches
 * our network and thus our machines address.
 * This routine requires the subnet portion to be a multiple of 8 bits to
 * recognice subnet broadcasts.
 */

short
ip_same_addr (local, foreign)
	unsigned long local, foreign;
{
	unsigned long mask = 0xffffffff;

	if (local == foreign) return 1;
	do {
		mask <<= 8;
		if ((local & mask) == (foreign & mask) &&
		    ((foreign & ~mask) == 0 || (foreign & ~mask) == ~mask))
			return 1;
	} while (mask);
	return 0;
}

unsigned long
ip_dst_addr (addr)
	unsigned long addr;
{
	struct netif *nif;
	struct ifaddr *ifa;
	
	if (addr == INADDR_ANY) {
		/* Connect to primary interface */
		nif = if_primary ();
		if (nif) {
			ifa = if_af2ifaddr (nif, AF_INET);
			if (ifa) return SIN (&ifa->addr)->sin_addr.s_addr;
		}
		DEBUG (("ip_dst_addr: no primary if with inet address"));
		return INADDR_LOOPBACK; /* sould never happen */
	} else if (addr == INADDR_BROADCAST) {
		/* Broadcast on local net */
		nif = if_primary ();
		if (nif && nif->flags & IFF_BROADCAST) {
			ifa = if_af2ifaddr (nif, AF_INET);
			if (ifa) return SIN (&ifa->ifu.broadaddr)->sin_addr.s_addr;
		}
		return INADDR_BROADCAST;
	}
	return addr;
}	

unsigned long
ip_netmask (addr)
	unsigned long addr;
{
	struct netif *nif;
	struct ifaddr *ifa;
	unsigned long netmask;

	if (IN_CLASSA (addr))
		netmask = IN_CLASSA_NET;
	else if (IN_CLASSB (addr))
		netmask = IN_CLASSB_NET;
	else if (IN_CLASSC (addr))
		netmask = IN_CLASSC_NET;
	else return 0;

	for (nif = allinterfaces; nif; nif = nif->next) {
		ifa = if_af2ifaddr (nif, AF_INET);
		if (ifa && ifa->net == (addr & netmask)) {
			return ifa->subnetmask;
		}
	}
	return netmask;
}

void
ip_register (proto)
	struct in_ip_ops *proto;
{
	proto->next = allipprotos;
	allipprotos = proto;
}

BUF *
ip_brdcst_copy (buf, nif, rt, addrtype)
	BUF *buf;
	struct netif *nif;
	struct route *rt;
	short addrtype;
{
	BUF *buf2;
	long offset;
	extern void *memcpy (void *, const void *, unsigned long);
	
	if (nif == rt->nif || addrtype != IPADDR_BRDCST) return 0;
	DEBUG (("ip_broadcast: doing directed broadcast copying"));
	
	offset = (long)buf->dstart - (long)buf + sizeof (BUF);
	buf2 = buf_alloc (buf->buflen, offset, BUF_NORMAL);
	if (!buf2) {
		DEBUG (("ip_broadcast: out of mem"));
		return 0;
	}
	offset = (long)buf->dend - (long)buf->dstart;
	memcpy (buf2->dstart, buf->dstart, offset);
	buf2->dend = buf2->dstart + offset;
	return buf2;
}

/* Forward IP datagram in `buf' to its next destination.
 * `local' -- 1 if locally generated datagram, 0 otherwise.
 * `nif'   -- network interface the datagram came from.
 */
void
ip_forward (buf, nif, local)
	BUF *buf;
	struct netif *nif;
	short local;
{
	struct ip_dgram *iph = (struct ip_dgram *)buf->dstart;
	struct route *rt;
	short addrtype;

/* Validate incoming datagram */
	if (!local) {
		short pktlen = (long)buf->dend - (long)buf->dstart;
		
		if (pktlen < IP_MINLEN || pktlen != iph->length) {
			DEBUG (("ip_forward: invalid packet length"));
			buf_deref (buf, BUF_NORMAL);
			return;
		}
		if (chksum (iph, iph->hdrlen * sizeof (short))) {
			DEBUG (("ip_forward: bad checksum"));
			buf_deref (buf, BUF_NORMAL);
			return;
		}
		if (iph->version != IP_VERSION) {
			DEBUG (("ip_forward: %d: unsupp. IP version",
				iph->version));
			buf_deref (buf, BUF_NORMAL);
			return;
		}
		if (iph->fragoff & (IP_MF|IP_FRAGOFF)) {
			DEBUG (("ip_forward: fragmentation not supported"));
			icmp_send (ICMPT_DSTUR, ICMPC_PROTOUR, iph->saddr,
				buf, 0);
			return;
		}
	}

/* Check if time to live exeeded */
	if (--iph->ttl <= 0) {
		DEBUG (("ip_forward: ttl exeeded"));
		icmp_send (ICMPT_TIMEX, ICMPC_TTLEX, iph->saddr, buf, 0);
		return;
	}

/* Route datagram to next interface */
	rt = route_get (iph->daddr);
	if (!rt) {
		DEBUG (("ip_forward: not route to dst %lx", iph->daddr));
		icmp_send (ICMPT_DSTUR, ICMPC_NETUR, iph->saddr, buf, 0);
		return;
	}

	addrtype = ip_chk_addr (iph->daddr, rt->nif);
	if (addrtype == IPADDR_BADCLASS) {
		DEBUG (("ip_forward: Dst addr not in class A/B/C"));
		buf_deref (buf, BUF_NORMAL);
		route_deref (rt);
		return;
	}

/* Check if the datagram is destined to this interface. If so send
 * the datagram to the local software. */
	if (!local && (addrtype == IPADDR_LOCAL || addrtype == IPADDR_BRDCST)) {
		struct in_ip_ops *p;
		BUF *buf2;

/* GATEWAY: If datagram is broadcast to a net we are directly connected to,
 * we must send the datagram to the local software AND forward it to the
 * network. buf2 will hold a copy if copying is necessary or zero if not. */
		buf2 = ip_brdcst_copy (buf, nif, rt, addrtype);

		for (p = allipprotos; p; p = p->next) {
			if (p->proto == iph->proto) {
				(*p->input) (nif, buf, iph->saddr, iph->daddr);
				break;
			}
		}
		if (!p) {
			DEBUG (("ip_forward: %d: Warning: No such proto",
				iph->proto));
			icmp_send (ICMPT_DSTUR, ICMPC_PROTOUR, iph->saddr,
				buf, 0);
		}
		buf = buf2;
		if (!buf) {
			route_deref (rt);
			return;
		}
	}

/* Fill in the source of the datagram (if not already done at higher level) */
	if (local && iph->saddr == INADDR_ANY) {
		struct ifaddr *ifa = if_af2ifaddr (rt->nif, AF_INET);

		if (!ifa) {
			DEBUG (("if_forward: chosen net if has no inet addr"));
			buf_deref (buf, BUF_NORMAL);
			route_deref (rt);
			return;
		}
		iph->saddr = SIN (&ifa->addr)->sin_addr.s_addr;
	}

/* Recompute checksum */
	iph->chksum = 0;
	iph->chksum = chksum (iph, iph->hdrlen * sizeof (short));

	if (!local && rt->nif == nif) {
		DEBUG (("ip_forward: sending redirect"));
		/* should send ICMP redirect here */
	}
	if (rt->nif->mtu < iph->length) {
		DEBUG (("ip_forward: packet size greater than MTU"));
		buf_deref (buf, BUF_NORMAL);
		route_deref (rt);
		return;
	}
	if_send (rt->nif, buf, rt->flags & RTF_GATEWAY ? rt->gway : iph->daddr);
	route_deref (rt);
}

long
ip_send (saddr, daddr, buf, proto, flags)
	unsigned long saddr, daddr;
	BUF *buf;
	short proto, flags;
{
	BUF *nbuf;
	struct ip_dgram *iph;
	static short dgram_id = 0;

	daddr = ip_dst_addr (daddr);
	if (!(flags & IP_BROADCAST) && ip_is_brdcst_addr (daddr)) {
		DEBUG (("ip_send: broadcasts not allowed"));
		buf_deref (buf, BUF_NORMAL);
		return EACCDN;
	}

/* Allocate and fill in IP header */
	nbuf = buf_reserve (buf, sizeof (*iph), BUF_RESERVE_START);
	if (!nbuf) {
		DEBUG (("ip_send: no space for IP header"));
		buf_deref (buf, BUF_NORMAL);
		return ENSMEM;
	}
	nbuf->dstart -= sizeof (*iph);
	iph = (struct ip_dgram *)nbuf->dstart;
	iph->version = IP_VERSION;
	iph->hdrlen  = sizeof (*iph) / sizeof (long);
	iph->tos     = 0;
	iph->length  = (short)((long)nbuf->dend - (long)nbuf->dstart);
	iph->id      = dgram_id++;
	iph->fragoff = 0;
	iph->ttl     = 30;
	iph->proto   = proto;
	iph->saddr   = saddr;
	iph->daddr   = daddr;
	iph->chksum  = 0;

	ip_forward (nbuf, (struct netif *)0, 1);
	return 0;
}

void
ip_input (void)
{
	struct netif *nif;
	BUF *buf;
	short todo, comeagain = 0;

	for (nif = allinterfaces; nif; nif = nif->next) {
		if ((nif->flags & (IFF_UP|IFF_RUNNING)) != (IFF_UP|IFF_RUNNING))
			continue;
		for (todo = IP_PACKETS_PER_IF; todo; --todo) {
			buf = if_dequeue (&nif->rcv);
			if (!buf) break;
			ip_forward (buf, nif, 0);
		}
		if (!todo) comeagain = 1;
	}
	if (comeagain) {
		/* Come again at next context switch, since we did
		 * not check all interfaces, there might be packets
		 * waiting for us. */
		if_input (nif, (BUF *)0, 100, PKTYPE_IP);
	}
}	
