/*
 * Driver for Dlink's DE600 pocket lan adaptor connected to the cartridge
 * port via the TUW hardware (adapter hardware description can be found
 * at fortec.tuwien.ac.at).
 *
 * This driver REQUIRES you to connect the interrupt wire to pin 11 (BUSY)
 * of your centronics port, ie it does not poll for incomming packets during
 * the VBL interrupt like the tuw-tcp driver.
 *
 * This driver has been developed and tested on a 16MHz ST. If you run in
 * on substantially faster machines a few wait()'s may be needed here and
 * there. Please contact me in case of success or failure...
 *
 * On a 16MHz ST the driver reaches the following throughput:
 * ftp send:  ~88 k/sec (ncftp 1.7.5 -> wu-ftpd 2.0)
 * ftp recv:  ~85 k/sec (ncftp 1.7.5 <- wu-ftpd 2.0)
 * tcp send:  ~90 k/sec (tcpcl -> tcpsv)
 * tcp recv:  ~90 k/sec (tcpsv <- tcpcl)
 * nfs read:  ~50 k/sec
 * nfs write: ~45 k/sec
 *
 * A note to NFS. When you mount a filesystem you must tell 'mount' to
 * use a maximum receive block size of MTU-300 bytes! Otherwise the
 * UDP datagrams must be fragmented and the lots of fragments will flood
 * the DE600's receive buffers wich leads to *very* low performance, ie
 * ALWAYS mount like this:
 *	mount -o rsize=1200 remote:/dir /nfs/dir
 *
 * 01/15/94, Kay Roemer.
 */

#include <string.h>
#include "config.h"
#include "netinfo.h"
#include "kerbind.h"
#include "atarierr.h"
#include "sockios.h"
#include "sockerr.h"
#include "buf.h"
#include "if.h"
#include "ifeth.h"
#include "lancemem.h"
#include "pktqueue.h"
#include "util.h"

#define offsetof(s,e) ((long)(&((s *)0)->e))

/*
 * 200 Hz timer
 */
#define HZ200 (*(volatile long *)0x4baL)

#define MIN_LEN 60

#define TIMEOUTCNT 10000
#define TIMEOUT	1000

static PKTBUF *pkt_ring[RECVBUFFS];
static PKTPOOL *p_free;

void (*ihandler) (void) = 0;
static int pkt_recv;
static volatile int pkt_sent = 0;
static volatile TMD *tmd_sent;
static u_short freecnt;

static struct netif if_lance;

static long	lance_open	(struct netif *);
static long	lance_close	(struct netif *);
static long	lance_output	(struct netif *, BUF *, char *, short, short);
static long	lance_ioctl	(struct netif *, short, long);

static long	lance_probe	(struct netif *);
static long	lance_reset	(struct netif *);
static void	lance_install_ints (void);
static void	lance_int	(void);

static long
lance_open (nif)
	struct netif *nif;
{
	lance_reset (nif);
	return 0;
}

static long
lance_close (nif)
	struct netif *nif;
{
	return 0;
}

static long
lance_output (nif, buf, hwaddr, hwlen, pktype)
	struct netif *nif;
	BUF *buf;
	char *hwaddr;
	short hwlen, pktype;
{
	long ret;
	short len;
	BUF *nbuf;
	volatile PKTBUF *send;
	register volatile int *rdp;
	register volatile TMD *tmd;
	register long timeout;

	nbuf = eth_build_hdr (buf, nif, hwaddr, pktype);
	if (nbuf == 0) {
		++nif->out_errors;
		return ENSMEM;
	}
	send = p_getpkt (1, p_free);
	if (!send) {
		buf_deref (nbuf, BUF_NORMAL);
		++nif->out_errors;
		return ENSMEM;		
	}
DEBUG (("LANCE: send = %lx", send));
	len = nbuf->dend - nbuf->dstart;
	if (len < MIN_LEN)
		len = MIN_LEN;
#ifdef PAMs_INTERN
	len = (len+1) & ~1;
#endif

DEBUG (("LANCE: len = %d", len));

	memcpy (send, nbuf->dstart, len);
	buf_deref (nbuf, BUF_NORMAL);

DEBUG (("LANCE: nach memcpy/deref"));

	if(!send || (send < (PKTBUF *)PPKT) || (send > (PKTBUF *)PPOOL)) {
		++nif->out_errors;
		ret = EINTRN;
		goto bad;
	}
DEBUG (("LANCE:      A"));
	rdp = RCP_RDP;
DEBUG (("LANCE:      B"));
	pkt_sent = 0;
	*(rdp+1) = CSR0;
DEBUG (("LANCE:      C"));
	if(!(*rdp & CSR0_TXON)) {
DEBUG (("LANCE:      D"));
		++nif->out_errors;
		ret = EINTRN;
		goto bad;
	}
DEBUG (("LANCE: 1"));

	tmd = (TMD *)tmd_sent;
	if((tmd->tmd1 & OWN) == OWN_CHIP) {
		++nif->out_errors;
		ret = EINTRN;
		goto bad;
	}
	tmd->ladr = (u_short)(send);		/* point to packet to send */
	tmd->tmd2 = -len;
	tmd->tmd3 = 0;
	tmd->tmd1 = (STP | ENP | OWN_CHIP);	/* start/end of packet */
DEBUG (("LANCE: 2"));
	timeout = HZ200 + 3*TIMEOUT;
	while(!pkt_sent && !tmd->tmd3 && HZ200 - timeout < 0)
		__asm__ volatile ("andiw #0xf8ff, sr;");   /* wait till packet sent */
DEBUG (("LANCE: 3"));
	if(!pkt_sent && !tmd->tmd3) {
		++nif->out_errors;
		ret = ETIMEDOUT;
		goto bad;
	}
	if(tmd->tmd3 || pkt_sent < 0) {
		++nif->collisions;
		ret = ETIMEDOUT;
		goto bad;
	}
	ret = 0;
bad:
DEBUG (("LANCE: 4"));
	p_putpkt (p_free, send);
DEBUG (("LANCE: leaving output"));
	return ret;	
}

static long
lance_ioctl (nif, cmd, arg)
	struct netif *nif;
	short cmd;
	long arg;
{
	switch (cmd) {
	case SIOCSIFNETMASK:
	case SIOCSIFFLAGS:
	case SIOCSIFADDR:
		return 0;

	case SIOCSIFMTU:
		/*
		 * Limit MTU to 1500 bytes. MintNet has alraedy set nif->mtu
		 * to the new value, we only limit it here.
		 */
		if (nif->mtu > ETH_MAX_DLEN)
			nif->mtu = ETH_MAX_DLEN;
		return 0;
	}
	return EINVFN;
}

static long
lance_pkt_init (void)
{
	short i;
	
	p_free = p_init (MAXPKT, PPOOL, PPKT);
	freecnt = MAXPKT;
	for (i = 0; i < RECVBUFFS; ++i) {
		pkt_ring[i] = p_getpkt (1, p_free);
		if (!pkt_ring[i])
			return ENSMEM;
		--freecnt;
	}
	return 0;
}

long
driver_init (void)
{
	static char message[100];

	strcpy (if_lance.name, "en");
	if_lance.unit = if_getfreeunit ("en");
	if_lance.metric = 0;
	if_lance.flags = IFF_BROADCAST;
	if_lance.mtu = 1500;
	if_lance.timer = 0;
	if_lance.hwtype = HWTYPE_ETH;
	if_lance.hwlocal.len =
	if_lance.hwbrcst.len = ETH_ALEN;

	if_lance.rcv.maxqlen = IF_MAXQ;
	if_lance.snd.maxqlen = IF_MAXQ;

	if_lance.open = lance_open;
	if_lance.close = lance_close;
	if_lance.output = lance_output;
	if_lance.ioctl = lance_ioctl;

	if_lance.timeout = 0;

	if_lance.data = 0;

	/*
	 * Tell upper layers the max. number of packets we are able to
	 * receive in fast succession.
	 */
	if_lance.maxpackets = 0;

	lance_pkt_init ();
	if (lance_probe (&if_lance) < 0) {
		c_conws ("lance: driver not installed.\r\n");
		return 1;
	}
	lance_install_ints ();

	if_register (&if_lance);

	sprintf (message, "LANCE driver v%s (en%d) (%x:%x:%x:%x:%x:%x)\r\n",
		"0.0",
		if_lance.unit,
		if_lance.hwlocal.addr[0],
		if_lance.hwlocal.addr[1],
		if_lance.hwlocal.addr[2],
		if_lance.hwlocal.addr[3],
		if_lance.hwlocal.addr[4],
		if_lance.hwlocal.addr[5]);
	c_conws (message);
	return 0;
}

static long
lance_reset (nif)
	struct netif *nif;
{
	return 0;
}

static long
lance_probe (nif)
	struct netif *nif;
{
	register volatile LNCMEM *mem;
	register volatile TMD	*md;
	register u_long i;
	register volatile int *rdp;
#ifdef PAMs_INTERN
	char volatile *ram;
	char volatile *eeprom;
	u_short volatile *eemem;
#endif
	char my_haddr[6];

#ifdef PAMs_INTERN
	eeprom = (char *)LANCE_EEPROM;
	*(volatile char *)eeprom;
#endif	
	rdp = RCP_RDP;
	*(rdp+1) = CSR0;
	*rdp = CSR0_STOP;
		/* init data structures */
		
#ifdef PAMs_INTERN
	eemem = (u_short *)RCP_MEMBOT;
	ram = (char *)LANCE_MEM;
	eeprom = (char *)LANCE_EEPROM;
	i = *eeprom;
	for (i=0 ; i<6 ; i++)
		nif->hwlocal.addr[i] = my_haddr[i] =
			((eemem[i<<1] & 0xf) << 4) | (eemem[(i<<1)+1] & 0xf); 
	i = *ram;
#endif
	mem = (LNCMEM *)RCP_MEMBOT;
	mem->init.mode = 0;		/* mode word */
	mem->init.haddr[5] = my_haddr[4];
	mem->init.haddr[4] = my_haddr[5];
	mem->init.haddr[3] = my_haddr[2];
	mem->init.haddr[2] = my_haddr[3];
	mem->init.haddr[1] = my_haddr[0];
	mem->init.haddr[0] = my_haddr[1];
    	
	mem->init.laf[0] = 0;	/* logical adress filter */
	mem->init.laf[1] = 0;	/* logical adress filter */

	mem->init.rdrp.drp_lo = (u_short)offsetof(LNCMEM,rmd[0]);
/*	((u_long)&(PRMD[0]));	*/ /* receive ring descr. pointer */
	mem->init.rdrp.drp_hi = 0;
	mem->init.rdrp.len = RECVRLEN;
	
	mem->init.tdrp.drp_lo = (u_short)offsetof(LNCMEM,tmd);
/*	((u_long)&PTMD);	*/	/* transmit ring descr. pointer */
	mem->init.tdrp.drp_hi = 0;
	mem->init.tdrp.len = XMITRLEN;

	md = &mem->tmd;
	md->ladr = 0;							/* dont know adress of packet to send */
	md->tmd1 = OWN_HOST;
	md->tmd2 = 0;							/* zero pkt size */
	md->tmd3 = 0;
	for(i=0; i< RECVBUFFS; i++)
	{
		(RMD *)md = mem->rmd+i;
		((RMD *)md)->ladr = (u_short)((char *)pkt_ring[i]);
		((RMD *)md)->rmd1 = OWN_CHIP;
		((RMD *)md)->rmd2 = -MAXPKTLEN;			/* 2's complement (maximum size) */
		((RMD *)md)->mcnt = 0;
	}
	tmd_sent = &mem->tmd;
	pkt_recv = 0;
	*(rdp+1) = CSR3;
#ifndef PAMs_INTERN
	*rdp = CSR3_BSWP;
#else
	*rdp = CSR3_BSWP | CSR3_ACON;
#endif
	*(rdp+1) = CSR2;
	*rdp = 0;
	*(rdp+1) = CSR1;
	*rdp = 0;
	*(rdp+1) = CSR0;
	*rdp = CSR0_STRT | CSR0_INIT;
	i = TIMEOUTCNT * 100;
	do
	{
		if(*rdp & CSR0_IDON) break;
	}while(--i > 0);
	if(i<=0 || (*rdp & CSR0_ERR))
	{
		*rdp = CSR0_STOP;
		ihandler = NULL;
		return(-1);
	}
	ihandler = lance_int;
	*rdp = CSR0_IDON;		/* clear idon-bit */
	*rdp = CSR0_INEA;		/* enable interrupts */
	return(0);
}

/*
 * Busy interrupt routine
 */
void
lance_int (void)
{
	register int 		csr0;
	register int		type;

	register volatile int	*rdp;
	register volatile TMD	*md;
	register volatile PKTBUF *pkt;

	struct netif *nif = &if_lance;
	BUF *buf;
	int pktlen;
	
	rdp = RCP_RDP;		/* register data port */
	*(rdp+1) = CSR0;
	csr0 = *rdp;
	if(!(csr0 & CSR0_INTR))
		return;
	
	if(csr0 & CSR0_IDON)
		*(rdp) = CSR0_IDON;
	else if(csr0 & CSR0_RINT)
	{
	  do
	  {
		(RMD *)md  = &PRMD[pkt_recv];
		*rdp = CSR0_RINT;
		if(!(((RMD *)md)->rmd1 & ERR))	/* packet ok */
		{
/*			pkt = pkt_ring[pkt_recv]; */
			pkt = (PKTBUF *)((long)RCP_MEMBOT+(((RMD *)md)->ladr));
			type = pkt->et_type;
			++nif->in_packets;

			pktlen = ((RMD *)md)->mcnt;
			buf = buf_alloc (pktlen+100, 50, BUF_ATOMIC);
			if (buf == 0)
				++nif->in_errors;
			else {
				buf->dend += pktlen;
				memcpy (buf->dstart, pkt, pktlen);
				if (!if_input (nif, buf, 0, eth_remove_hdr (buf)))
					++nif->in_packets;
				else
				++nif->in_errors;
			}
		}
		else
		{
		  	if(((RMD *)md)->rmd1 & CRC)
		  		++nif->in_errors;
			lance_reset (nif);
			return;
		}
		((RMD *)md)->mcnt = 0;			/* free packet */
		((RMD *)md)->rmd1 = OWN_CHIP;	/* give packet back to lance */
		pkt_recv++;
		if(pkt_recv >= RECVBUFFS)
			pkt_recv = 0;
	  } while((PRMD[pkt_recv].rmd1 & OWN) == OWN_HOST);
	}
	else if(csr0 & CSR0_TINT)
	{
DEBUG (("LANCE: int: 1"));
		md = (TMD *)tmd_sent;
		if((md->tmd1 & OWN) == OWN_HOST)  /* packet given back to host */
		{
DEBUG (("LANCE: int: 2"));
			if((md->tmd1) & ERR)
			{
				pkt_sent = -1;
				++nif->out_errors;
			}
			else
			{
				pkt_sent = 1;
				++nif->out_packets;
				if(md->tmd1 & (MORE | ONE | DEF))
					++nif->collisions;
			}
		}
		else
			++nif->out_errors;
		*rdp = CSR0_TINT;	/* clear interrupt bit */
	}
	else if(csr0 & CSR0_ERR)
	{
		++nif->in_errors;
		*(rdp+1) = 0;
		*rdp = CSR0_CERR | CSR0_MERR | CSR0_BABL | CSR0_MISS;
	}
	*(rdp+1) = CSR0;
	*rdp = CSR0_INEA;
}

extern void lance_hbi_int (void);
extern void lance_v5_int (void);

/*
 * Install interrupt handler
 */
static void
lance_install_ints (void)
{
	long old_v5, old_hbl;
	short sr;

	sr = spl7 ();
	old_v5 = s_etexec ((short)LANCEIVEC, (long)lance_v5_int);
#ifdef PAMs_INTERN
	*(char *)LANCEIVECREG = LANCEIVEC;
	old_hbl = s_etexec ((short)HBI, (long)lance_hbi_int);
#endif
	spl (sr);
}
