/*
 *	slattach(8) utility for MintNet (w) 1994, Kay Roemer.
 *
 *	Options:
 *
 *	-h			Don't hang the line up when exiting.
 *
 *	-e			Exit after setting up the connection.
 *
 *	-d			Don't add default route.
 *
 *	-t <termial line>	Specify terminal device to use. If this
 *				option is missing use stdin instead.
 *
 *	-r <remote host>	Specify the name of the remote host.
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <fcntl.h>
#include <signal.h>
#include <sys/types.h>
#include <sys/time.h>
#include <sys/ioctl.h>
#include <support.h>
#include <netdb.h>

#define _PATH_IFLINK	"/etc/iflink"
#define _PATH_IFCONFIG	"/etc/ifconfig"
#define _PATH_ROUTE	"/etc/route"

static int ttyfd;
static char command[100], hostname[64], ifname[16];
static int exit_opt = 0, hangup_opt = 1, defrt_opt = 1;

void
set_raw_tty (fd)
	int fd;
{
	int r;
	struct sgttyb sg;

	r = gtty (fd, &sg);
	if (r < 0) {
		perror ("gtty");
		exit (1);
	}
	sg.sg_flags |= RAW;
	sg.sg_flags &= ~(CBREAK|ECHO|CRMOD|XKEY);
	r = stty (fd, &sg);
	if (r < 0) {
		perror ("stty");
		exit (1);
	}
}

void
set_canon_tty (fd)
	int fd;
{
	int r;
	struct sgttyb sg;

	r = gtty (fd, &sg);
	if (r < 0) {
		perror ("gtty");
		exit (1);
	}
	sg.sg_flags |= CBREAK|ECHO|XKEY|CRMOD;
	sg.sg_flags &= ~RAW;
	r = stty (fd, &sg);
	if (r < 0) {
		perror ("stty");
		exit (1);
	}
}

static void
hangup (void)
{
	struct timeval tm = { 0, 600000L };
	struct sgttyb sg;
	char ispeed, ospeed;

	if (ioctl (ttyfd, TIOCGETP, &sg) < 0) {
		perror ("cannot drop DTR: ioctl (TIOCGETP):");
		return;
	}
	ispeed = sg.sg_ispeed;
	ospeed = sg.sg_ospeed;
	sg.sg_ispeed = B0;
	sg.sg_ospeed = B0;
	if (ioctl (ttyfd, TIOCSETN, &sg) < 0) {
		perror ("cannot drop DTR: ioctl (TIOCSETN):");
		return;
	}
	select (0, 0, 0, 0, &tm);
	sg.sg_ispeed = ispeed;
	sg.sg_ospeed = ospeed;
	if (ioctl (ttyfd, TIOCSETN, &sg) < 0) {
		perror ("cannot drop DTR: ioctl (TIOCSETN):");
		return;
	}
}

void
sig_handler (sig)
	int sig;
{
	set_canon_tty (ttyfd);

	sprintf (command, "%s %s down\n", _PATH_IFCONFIG, ifname);
	if (system (command) != 0) {
		fprintf (stderr, "cannot shut down interface %s\n", ifname);
		exit (1);
	}

	if (hangup_opt)
		hangup ();

	signal (SIGINT, SIG_DFL);
	raise (SIGINT);
}

void
usage (void)
{
	printf ("usage: slattach [-h] [-e] [-d] [-t <tty device>] -r <remote host>\n");
	exit (1);
}

int
main (argc, argv)
	int argc;
	char *argv[];
{
	char *tty = NULL, *rhost = NULL;
	extern char *optarg;
	FILE *fp;
	int c;

	strcpy (ifname, "sl");
	while ((c = getopt (argc, argv, "dhet:r:p")) != EOF) switch (c) {
	case 't':
		tty = optarg;
		break;

	case 'r':
		rhost = optarg;
		break;

	case 'e':
		exit_opt = 1;
		break;

	case 'h':
		hangup_opt = 0;
		break;

	case 'd':
		defrt_opt = 0;
		break;		

	case 'p':
		strcpy (ifname, "ppp");
		break;

	case '?':
		usage ();
		break;
	}
	if (tty == NULL)
		tty = ttyname (0);

	if (rhost == NULL || tty == NULL)
		usage ();

	ttyfd = open (tty, O_RDONLY);
	if (ttyfd < 0) {
		perror ("open");
		exit (1);
	}

	signal (SIGQUIT, sig_handler);
	signal (SIGINT, sig_handler);
	signal (SIGTERM, sig_handler);
	signal (SIGHUP, sig_handler);

	if (gethostname (hostname, sizeof (hostname)) != 0) {
		perror ("gethostname");
		exit (1);
	}

	sprintf (command, "%s -i %s -d %s", _PATH_IFLINK, ifname, tty);
	fp = popen (command, "rt");
	if (fp == 0) {
		fprintf (stderr, "cannot link tty to interface\n");
		exit (1);
	}
	if (fgets (ifname, sizeof (ifname), fp) == NULL) {
		fprintf (stderr, "cannot get interface name");
		exit (1);
	}
	pclose (fp);
	ifname[strlen (ifname) - 1] = '\0';

	sprintf (command, "%s %s addr %s dstaddr %s up\n", _PATH_IFCONFIG,
		ifname, hostname, rhost);
	if (system (command) != 0) {
		fprintf (stderr, "cannot activate interface %s\n", ifname);
		exit (1);
	}

	sprintf (command, "%s add %s %s\n", _PATH_ROUTE, rhost, ifname);
	if (system (command) != 0) {
		fprintf (stderr, "cannot install routing entry\n");
		exit (1);
	}
	if (defrt_opt) {
		sprintf (command, "%s add default %s gw %s\n", _PATH_ROUTE,
			ifname, rhost);
		if (system (command) != 0) {
			fprintf (stderr, "cannot install default route\n");
			exit (1);
		}
	}

	set_raw_tty (ttyfd);
	if (!exit_opt) for (;;)
		pause ();
	return 0;
}
