/*
 *	This file provides a timer with 100 ms granularity. We must
 *	have our own routines for this, because the Mint builtin
 *	timer does not provide for an argument to the timeout function.
 *
 *	03/22/94, Kay Roemer.
 */

#include "config.h"
#include "kerbind.h"
#include "file.h"
#include "timer.h"
#include "util.h"

static void	update_head	(void);
static void	check_events	(long);
static void	event_insert	(struct event *, long);
static short	event_remove	(struct event *);

static struct event	*allevents = 0;
static struct timeout	*nexttimeout = 0;

static void
update_head (void)
{
#define HZ200	(*(volatile long *)0x4baL)
	static long last = 0;
	static short init = 1;
	long diff, curr = HZ200;

	if (init || !allevents || curr - last < 0) {
		last = curr;
		init = 0;
	} else {
		diff = (curr - last)/(EVTGRAN/5);
		allevents->delta -= diff;
		last += diff*(EVTGRAN/5);
	}
}

static void
check_events (proc)
	long proc;
{
	register struct event *ep;
	register void (*func) (long);
	register long arg;
	register void *sp;

	sp = setstack (stack + sizeof (stack));

	update_head ();
	while ((ep = allevents) && ep->delta <= 0) {
		arg = ep->arg;
		func = ep->func;
		allevents = ep->next;
		if (allevents) allevents->delta += ep->delta;
		kfree (ep);
		(*func) (arg);
		update_head ();
	}
	if (ep) {
		nexttimeout = addroottimeout(ep->delta*EVTGRAN,check_events,0);
		if (!nexttimeout) FATAL (("timer: out of kernel memory"));
	} else	nexttimeout = 0;
	setstack (sp);
}

static void
event_insert (ep, delta)
	struct event *ep;
	long delta;
{
	struct event **prev, *curr;

	update_head ();
	prev = &allevents;
	curr = allevents;
	for (; curr; prev = &curr->next, curr = curr->next) {
		if (curr->delta <= delta) delta -= curr->delta;
		else {
			curr->delta -= delta;
			break;
		}
	}
	ep->delta = delta;
	ep->next = curr;
	*prev = ep;
	if (allevents == ep) {
		if (nexttimeout) cancelroottimeout (nexttimeout);
		nexttimeout = addroottimeout (delta*EVTGRAN, check_events, 0);
		if (!nexttimeout) FATAL (("timer: out of kernel memory"));
	}
}

static short
event_remove (ep)
	struct event *ep;
{
	struct event **prev, *curr;

	prev = &allevents;
	curr = allevents;
	for (; curr; prev = &curr->next, curr = curr->next) {
		if (ep == curr) {
			*prev = curr->next;
			if (curr->next) curr->next->delta += curr->delta;
			return 1;
		}
	}
	return 0;
}

struct event *
event_add (delta, func, arg)
	long delta, arg;
	void (*func) (long);
{
	struct event *ep;

	if (delta <= 0) {
		(*func) (arg);
		return 0;
	}
	ep = kmalloc (sizeof (struct event));
	if (!ep) FATAL (("event_add: no memory for timeout struct"));

	ep->func = func;
	ep->arg = arg;
	event_insert (ep, delta);

	return ep;
}

void
event_del (ep)
	struct event *ep;
{
	if (event_remove (ep)) kfree (ep);
}

/* return the time in ms until the event `ep' happens */
long
event_delta (ep)
	struct event *ep;
{
	struct event *cur;
	long ticks = 0;
	
	update_head ();
	for (cur = allevents; cur; cur = cur->next) {
		ticks += cur->delta;
		if (cur == ep) return ticks;
	}
	DEBUG (("event_delta: event not found"));
	return 0;
}

struct event *
event_reset (ep, delta)
	struct event *ep;
	long delta;
{
	if (event_remove (ep)) {
		event_insert (ep, delta);
		return ep;
	} else	FATAL (("event_reset: event not found"));
	return 0;
}
