/*** config.c ***/

#include <stdio.h>
#include "windows.h"
#include "config.h"
#include "expect.h"
#include "sioerror.h"
#include "pcl4w.h"
#include "term.h"

extern int  OnLineFlag;
extern HWND hMainWnd;
extern char XferState;

static struct
 {int Port;
  int BaudRate;
  int WordLength;
  int Parity;
  int StopBits;
 } Config = {COM1,Baud9600,WordLength8,NoParity,OneStopBit};

static long BaudRateList[10] =
    {300L,600L,1200L,2400L,4800L,9600L,19200L,38400L,57600L,115200L};
static char StopBitList[2] = {'1','2'};
static char ParityList[4]  = {'N','O','?','E'};
static char WordLengthList[4] = {0,0,'7','8'};

static int PortMsgID[4] = {MSG_COM1,MSG_COM2,MSG_COM3,MSG_COM4};
static int BaudRateMsgID[10] = {0,0,MSG_1200,MSG_2400,
                                MSG_4800,MSG_9600,MSG_19200,MSG_38400,
                                MSG_57600,MSG_115200};
static int StopBitsMsgID[2] = {MSG_1_SB,MSG_2_SB};
static int ParityMsgID[4] = {MSG_NONE,MSG_ODD,0,MSG_EVEN};
static int WordLengthMsgID[4] = {0,0,MSG_7_DB,MSG_8_DB};

static char *LineText[2] = {"OFF", "ON"};

static void SetPSL(int,int,int);
static void CheckTheMenu(int);
static void UncheckTheMenu(int);
static void CheckAll(void);
static void UncheckAll(void);
static int  UpdateTheChecks(int,int);

static LastPortID = MSG_COM1;
static LastParityID = MSG_NONE;
static LastStopBitsID = MSG_1_SB;
static LastWordLengthID = MSG_8_DB;
static LastBaudRateID = MSG_9600;

void InitConfig()
{CheckAll();
 SetTitle();
}

void LoadConfig()
{int hFile;
 hFile = _lopen("term.cfg",OF_READ | OF_SHARE_DENY_WRITE);
 if(hFile<0) ErrorMessage("TERM.CFG not found");
 else
   {/* uncheck menu selections */
    UncheckAll();
    /* read in new configuration */
    _lread(hFile,&Config,sizeof(Config));
    _lclose(hFile);
    SetTitle();
    /* set current menu selections */
    LastPortID = PortMsgID[Config.Port];
    LastParityID = ParityMsgID[Config.Parity];
    LastStopBitsID = StopBitsMsgID[Config.StopBits];
    LastWordLengthID = WordLengthMsgID[Config.WordLength];
    LastBaudRateID = BaudRateMsgID[Config.BaudRate];
    /* check current menu selections */
    CheckAll();
   }
} /* end LoadConfig */

void SaveConfig()
{int hFile;
 hFile = _lcreat("term.cfg",0);
 _lwrite(hFile,&Config,sizeof(Config));
 _lclose(hFile);
} /* end LoadConfig */

void SetParity(int Parity)
{
 SetPSL(Parity,Config.StopBits,Config.WordLength);
 LastParityID = UpdateTheChecks(LastParityID,ParityMsgID[Parity]);
}

void SetStopBits(int StopBits)
{
 SetPSL(Config.Parity,StopBits,Config.WordLength);
 LastStopBitsID = UpdateTheChecks(LastStopBitsID,StopBitsMsgID[StopBits]);
}

void SetWordLength(int WordLength)
{
 SetPSL(Config.Parity,Config.StopBits,WordLength);
 LastWordLengthID = UpdateTheChecks(LastWordLengthID,WordLengthMsgID[WordLength]);
}

void SetPSL(int Parity,int StopBits,int WordLength)
{int RetCode;
 if(!ExpectNoXfer()) return;
 Config.Parity = Parity;
 Config.StopBits = StopBits;
 Config.WordLength = WordLength;
 if(OnLineFlag)
   {RetCode = SioParms(Config.Port,Config.Parity,
                       Config.StopBits,Config.WordLength);
    if(RetCode<0) SioError(RetCode,"SioParms");
   }
 SetTitle();
 /* end SetPSL */
}

void SetBaud(int BaudRate)
{int RetCode;
 if(!ExpectNoXfer()) return;
 Config.BaudRate = BaudRate;
 if(OnLineFlag)
   {RetCode = SioBaud(Config.Port,Config.BaudRate);
    if(RetCode<0) SioError(RetCode,"SioBaud");
   }
 LastBaudRateID = UpdateTheChecks(LastBaudRateID,BaudRateMsgID[BaudRate]);
 SetTitle();
} /* end SetBaud */

void SetPort(int Port)
{
 if(!ExpectNoXfer()) return;
 if(!ExpectOffLine()) return;
 Config.Port = Port;
 LastPortID = UpdateTheChecks(LastPortID,PortMsgID[Port]);
 SetTitle();
} /* end SetPort */

int GetBaud(void)
{return (Config.BaudRate);
}

int GetParity(void)
{return (Config.Parity);
}

int GetStopBits(void)
{return (Config.StopBits);
}

int GetWordLength(void)
{return(Config.WordLength);
}

int GetPort(void)
{return (Config.Port);
}

void SetTitle()
{char Text[80];
 sprintf(Text,"TERM: COM%d @ %ld baud %c%c%c %sLINE %c",
    Config.Port+1,
    BaudRateList[Config.BaudRate],
    WordLengthList[Config.WordLength],
    ParityList[Config.Parity],
    StopBitList[Config.StopBits],
    LineText[OnLineFlag&1],
    XferState);
 SetWindowText(hMainWnd,Text);
} /* end SetTitle */

void CheckTheMenu(int MenuID)
{HMENU hMenu;
 if(MenuID)
   {hMenu = GetMenu(hMainWnd);
    CheckMenuItem(hMenu,MenuID,MF_BYCOMMAND | MF_CHECKED);
   }
} /* end CheckTheMenu */

void UncheckTheMenu(int MenuID)
{HMENU hMenu;
 if(MenuID)
   {hMenu = GetMenu(hMainWnd);
    CheckMenuItem(hMenu,MenuID,MF_BYCOMMAND | MF_UNCHECKED);
   }
} /* end CheckTheMenu */

void UncheckAll(void)
{UncheckTheMenu(LastPortID);
 UncheckTheMenu(LastParityID);
 UncheckTheMenu(LastStopBitsID);
 UncheckTheMenu(LastWordLengthID);
 UncheckTheMenu(LastBaudRateID);
} /* end UncheckAll */

void CheckAll(void)
{CheckTheMenu(LastPortID);
 CheckTheMenu(LastParityID);
 CheckTheMenu(LastStopBitsID);
 CheckTheMenu(LastWordLengthID);
 CheckTheMenu(LastBaudRateID);
} /* end CheckAll */

int UpdateTheChecks(int OldMsgID,int NewMsgID)
{UncheckTheMenu(OldMsgID);
 CheckTheMenu(NewMsgID);
 return NewMsgID;
} /* end UpdateTheChecks */
