#include <ctype.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <io.h>

#include "1adini.h"

extern FILE *log;

int get_ini_value(char *fname, char *area, char *indent, char *value, int max_value)
{
  ini_inforec ini_info;

  if(!open_ini_file(fname, &ini_info))
    return -1;    /* Wasn't able to retrieve or create fname.ini */

  read_ini_string(&ini_info, area, indent, value, max_value);

  if(ini_info.changes_made)
    write_ini_file(&ini_info);   // value had to be added, so write it to disk

  free_ini_info(&ini_info);
  return 1;       /* Successful read */
}

int put_ini_value(char *fname, char *area, char *indent, char *value)
{
  ini_inforec ini_info;

  if(!open_ini_file(fname, &ini_info))
    return -1;    /* Wasn't able to retrieve or create fname.ini */

  set_ini_string(&ini_info, area, indent, value);
  write_ini_file(&ini_info);   // set_ini_string only sets the memory, it must
                               // be written to disk

  free_ini_info(&ini_info);
  return 1;       /* Successful read */
}

int get_ini_number(char *fname, char *area, char *indent, int default_value)
{
  ini_inforec ini_info;
  long x;

  if(!open_ini_file(fname, &ini_info))
    return -1;    /* Wasn't able to retrieve or create fname.ini */

  x = read_ini_number(&ini_info, area, indent, default_value);

  if(ini_info.changes_made)
    write_ini_file(&ini_info);   // value had to be added, so write it to disk

  free_ini_info(&ini_info);
  return x;       /* Successful read */
}

int put_ini_number(char *fname, char *area, char *indent, long value)
{
  ini_inforec ini_info;

  if(!open_ini_file(fname, &ini_info))
    return -1;    /* Wasn't able to retrieve or create fname.ini */

  set_ini_number(&ini_info, area, indent, value);
  write_ini_file(&ini_info);   // set_ini_string only sets the memory, it must
                               // be written to disk

  free_ini_info(&ini_info);
  return 1;       /* Successful read */
}


int get_ini_boolean(char *fname, char *area, char *indent, char *value)
{
  char s[51];

  strcpy(s, value);
  get_ini_value(fname, area, indent, s, 50);

  strupr(s);

  if(strcmp(s, "YES")==0 || strcmp(s, "TRUE")==0 || strcmp(s, "1")==0)
    return 1;

  if(strcmp(s, "NO")==0 || strcmp(s, "FALSE")==0 || strcmp(s, "0")==0)
    return 0;

  return -1;
}

int read_ini_boolean(ini_inforec *ini_info, char *area, char *indent, char *value)
{
  char s[51];

  strcpy(s, value);
  read_ini_string(ini_info, area, indent, s, 50);

  strupr(s);

  if(strcmp(s, "YES")==0 || strcmp(s, "TRUE")==0 || strcmp(s, "1")==0)
    return 1;

  if(strcmp(s, "NO")==0 || strcmp(s, "FALSE")==0 || strcmp(s, "0")==0)
    return 0;

  return -1;
}


/* ------------------------------------------------------------------------ */
/* read a string in area, if area doesn't exist, create it...  if indent    */
/* doesn't exist, create it                                                 */

int read_ini_string(ini_inforec *ini_info, char *area, char *indent, char *value, int max_value)
{
  return(ll_read_ini_string(ini_info, area, indent, value, max_value, 1, 0));
}

/* ------------------------------------------------------------------------ */
/* Look for string in area, returns 0 if not there, does NOT create it      */
int look_ini_string(ini_inforec *ini_info, char *area, char *indent, char *value, int max_value)
{
  return(ll_read_ini_string(ini_info, area, indent, value, max_value, 0, 0));
}

/* ------------------------------------------------------------------------ */
/* For duplicatate indents, searchs for the "which'th" indent and returns   */
/* that value, unexistant ones are not created                              */
int look_ini_string_dup(ini_inforec *ini_info, char *area, char *indent, char *value, int max_value, int which)
{
  return(ll_read_ini_string(ini_info, area, indent, value, max_value, 0, which));
}



/* ------------------------------------------------------------------------ */
/* just like read_ini string, creates missing parts if needed (including    */
/* ini file                                                                 */
long read_ini_number(ini_inforec *ini_info, char *area, char *indent, long value)
{
  return(ll_read_ini_number(ini_info, area, indent, value, 1, 0));
}

long look_ini_number(ini_inforec *ini_info, char *area, char *indent, long value)
{
  return(ll_read_ini_number(ini_info, area, indent, value, 0, 0));
}

long look_ini_number_dup(ini_inforec *ini_info, char *area, char *indent, long value, int which)
{
  return(ll_read_ini_number(ini_info, area, indent, value, 0, which));
}



void set_ini_number(ini_inforec *ini_info, char *area, char *indent, long value)
{
  set_ini_number_dup(ini_info, area, indent, value, 0);
}

void set_ini_string(ini_inforec *ini_info, char *area, char *indent, char *value)
{
  set_ini_string_dup(ini_info, area, indent, value, 0);
}






void free_ini_info(ini_inforec *ini_info)
{
  if(ini_info->ini && ini_info->open)
    free(ini_info->ini);

  ini_info->open = 0;
  return;
}



int write_ini_file(ini_inforec *ini_info)
{
  FILE *fp;

  if(!ini_info->changes_made)
    return 2;

  fp = fopen(ini_info->fname, "w+t");
  if(!fp)
    return 0;               /*  Couldn't create ini file */

  fseek(fp, 0, SEEK_SET);
  fwrite((void *)ini_info->ini, 1, ini_info->fsize, fp);

  fclose(fp);
  return 1;
}

char *open_ini_file(char *fname, ini_inforec *ini_info)
{
  FILE *fp;

  memset((void *)ini_info, 0, sizeof(ini_inforec));
  strcpy(ini_info->fname, fname);

  fp = fopen(fname, "rt" );

  if(!fp)                        /* if it doesn't exist, create it */
  {
    fp = fopen(fname, "w+t");
    if(!fp)
      return NULL;               /*  Couldn't create ini file */
  }

  ini_info->fsize=filesize(fp);

  if(ini_info->fsize < 60000L)   /* Make 60,000 our file size */
    ini_info->ini = (char *) malloc(ini_info->fsize + EXTRA_INI_SPACE);

  if(!ini_info->ini)
  {
    fclose(fp);
    return NULL;
  }

  fseek(fp, 0, SEEK_SET);
  ini_info->fsize = fread((void *)ini_info->ini, 1, ini_info->fsize, fp);
  ini_info->ini[ini_info->fsize]=0;

  ini_info->allocated=ini_info->fsize + EXTRA_INI_SPACE;
  ini_info->open = 1;                // turn closed flag off

  fclose(fp);
  return(ini_info->ini);
}

char * increase_ini_allocation(ini_inforec *ini_info)
{
  ini_info->allocated += EXTRA_INI_SPACE;

  if(ini_info->allocated > 0xffff)
    return NULL;

  ini_info->ini = (char *)realloc(ini_info->ini, ini_info->allocated);

  return(ini_info->ini);
}








char * get_ini_line(char *buff, int max_len, long *pos, char *ini)
{
  int c, buff_pos=0;
  char *temp;

  if(!ini[*pos])     // If at end of buffer, return NULL
    return NULL;

  while(1)
  {
    c = ini[*pos];

    if(c == 0 || c == '\n')
    {
      buff[buff_pos] = 0;
      if(c)    // Only increment if we are on a newline
        ++*pos;

      temp=strchr(buff, ';');
      if(temp)
        temp[0]=0;      // Null out all after ; (comment)

      return(buff);
    }

    buff[buff_pos] = c;

    ++buff_pos;
    ++*pos;

    if(buff_pos >= max_len)
    {
      buff[buff_pos] = 0;

      temp=strchr(buff, ';');
      if(temp)
        temp[0]=0;      // Null out all after ; (comment)

      return(buff);
    }
  }
}



void break_up_ini(char *buff, char *w1, char *w2)
{
  int pos=0, spos;
  int c;

  strip_string(buff);

  while(1)
  {
    c = buff[pos];

    if(c == '=' || c == 0)
    {
      buff[pos]=0;
      if(c)
        ++pos;
      break;
    }
    ++pos;
  }
  strcpy(w1, buff);

  spos=pos;

  while(1)
  {
    c = buff[pos];

    if(c == 0)
    {
      buff[pos] = 0;
      break;
    }

    ++pos;
  }

  strcpy(w2, buff+spos);

  strip_string(w1);
  strip_string(w2);
}


long find_ini_area(char *ini, char *area)
{
  char buff[255], *tmp;
  long pos=0;

  if(!area[0])       /* If we don't specify an area, start at top */
    return 0;

  while(1)
  {
    tmp = get_ini_line(buff, 255, &pos, ini);

    if(!tmp)
      return(pos);

    strip_string(buff);

    if(buff[0] != '[')
      continue;

    tmp=strchr(buff, ']');

    if(!tmp)
      continue;

    tmp[0]=0;

    if(strcmpi(buff+1, area) == 0)
      return(pos);

  }
}

long count_ini_areas(ini_inforec *ini, char *area)
{
  char buff[255], *tmp;
  long pos=0;
  int x = strlen(area), matches=0;


  while(1)
  {
    tmp = get_ini_line(buff, 255, &pos, ini->ini);

    if(!tmp)
      return(matches);

    strip_string(buff);

    if(buff[0] != '[')
      continue;

    tmp=strchr(buff, ']');

    if(!tmp)
      return(pos);

    tmp[0]=0;

    if(x)
    {
      if(strncmpi(buff+1, area, x) == 0)
        ++matches;
    }
    else
      ++matches;
  }
}



/* ------------------------------------------------------------------------- */
/*  read_ini_area                                                            */
/*                                                                           */
/*  Memory allocated by this function must be 'free_2d' to release the memory*/
/*  that it allocates.                                                       */
/*                                                                           */
/*  ie                                                                       */
/*                                                                           */
/*  char **tmp;                                                              */
/*  tmp = read_ini_areas(&ini_info, "USER"));                                */
/*  ...                                                                      */
/*  free_2d(tmp);                                                            */
/*                                                                           */
/*  area is a sorta a wildcard, if it is specified, it will read in all areas*/
/*  that match the area specified, up to the size of the area specified, so  */
/*  if you want to count 'USER 1', 'USER 2', 'USER 3', etc..., you would     */
/*  users = read_ini_areas(&ini, "USER ");                                   */
/* ------------------------------------------------------------------------- */

#define MAX_INI_AREA_SIZE 50
char ** read_ini_areas(ini_inforec *ini, char *area)
{
  char buff[255], *tmp;
  char **areas;
  long pos=0;
  int x = strlen(area), matches=0;
  int amount;


  amount = count_ini_areas(ini, area) + 1;
  areas = alloc_2d(amount, MAX_INI_AREA_SIZE + 1, sizeof(char));


  while(1)
  {
    tmp = get_ini_line(buff, 255, &pos, ini->ini);

    if(!tmp)
    {
      areas[matches]=NULL;
      return(areas);
    }

    strip_string(buff);

    if(buff[0] != '[')
      continue;

    tmp=strchr(buff, ']');

    if(!tmp)
    {
      areas[matches]=NULL;
      return(areas);
    }

    tmp[0]=0;

    if(x)
    {
      if(strncmpi(buff+1, area, x) == 0)
      {
        strncpy(areas[matches], buff+1, MAX_INI_AREA_SIZE);
        areas[matches][MAX_INI_AREA_SIZE] = 0;
        ++matches;
      }
    }
    else
    {
      strncpy(areas[matches], buff+1, MAX_INI_AREA_SIZE);
      areas[matches][MAX_INI_AREA_SIZE] = 0;
      ++matches;
    }
  }
}

long area_exist(ini_inforec *ini, char *area)
{
  char buff[255], *tmp;
  long pos=0;

  if(!area[0])     /* If no area was specified, of course it was found */
    return 1;

  while(1)
  {
    tmp = get_ini_line(buff, 255, &pos, ini->ini);

    if(!tmp)
      return 0;                      /* DID NOT find area */

    strip_string(buff);

    if(buff[0] != '[')
      continue;

    tmp=strchr(buff, ']');

    if(!tmp)
      continue;

    tmp[0]=0;

    if(strcmpi(buff+1, area) == 0)
      return 1;                     /* Found area */

  }
}



int count_ini_indents(ini_inforec *ini_info, char *area, char *indent)
{
  char *tmp, buff[255], w1[128], w2[128];
  long area_start, pos;
  int amount = 0, size = strlen(indent);


  area_start = find_ini_area(ini_info->ini, area);


  if(area_start >= ini_info->fsize)
    return 0;                        /* No matches, the area doesn't exist */


  pos = area_start;
  while(1)
  {
    tmp = get_ini_line(buff, 255, &pos, ini_info->ini);

    if(!tmp)          // No more lines
      return amount;

    strip_string(tmp);

    if(buff[0]=='[')  // Star of a new area
      return amount;


    break_up_ini(buff, w1, w2);
    if(strncmpi(w1, indent, size) == 0)    /* Found a match */
      ++amount;
  }
}


/* ------------------------------------------------------------------------- */
/*  read_ini_indents                                                         */
/*                                                                           */
/*  Memory allocated by this function must be 'free_2d' to release the memory*/
/*  that it allocates.                                                       */
/*                                                                           */
/*  ie                                                                       */
/*                                                                           */
/*  char **tmp;                                                              */
/*  tmp = read_ini_areas(&ini_info, "USER", "LOAD"));                        */
/*  ...                                                                      */
/*  free_2d(tmp);                                                            */
/*                                                                           */
/*  Indent is sort of wildcard, 'Load' will find any thing with load at the  */
/*  start of it, 'Load 1', 'Load 2' etc...                                   */
/* ------------------------------------------------------------------------- */

#define MAX_INI_INDENT_SIZE 80
char ** read_ini_indents(ini_inforec *ini_info, char *area, char *indent)
{
  char buff[255], *tmp;
  char **indents, w1[128], w2[128];
  long pos=0;
  int size = strlen(indent), matches=0, amount;
  unsigned area_start;


  amount = count_ini_indents(ini_info, area, indent) + 1;
  indents = alloc_2d(amount, MAX_INI_INDENT_SIZE + 1, sizeof(char));

  area_start = find_ini_area(ini_info->ini, area);


  if(area_start >= ini_info->fsize)
    return 0;                        /* No matches, the area doesn't exist */


  pos = area_start;
  while(1)
  {
    tmp = get_ini_line(buff, 255, &pos, ini_info->ini);

    if(!tmp)          // No more lines
    {
      indents[matches]=NULL;
      return(indents);
    }
    strip_string(tmp);

    if(buff[0]=='[')  // Star of a new area
    {
      indents[matches]=NULL;
      return(indents);
    }

    break_up_ini(buff, w1, w2);
    if(strncmpi(w1, indent, size) == 0)    /* Found a match */
    {
      strncpy(indents[matches], w2, MAX_INI_AREA_SIZE);
      indents[matches][MAX_INI_AREA_SIZE] = 0;
      ++matches;
    }
  }

}



long ll_read_ini_number(ini_inforec *ini_info, char *area, char *indent, long value, int should_update, int which)
{
  char *tmp, buff[255], w1[128], w2[128];
  long area_start, pos;
  int amount, this_which = 0;


  area_start = find_ini_area(ini_info->ini, area);


  if(area_start >= ini_info->fsize)
  {

    if(should_update)
    {
      if(area[0])
      {
        amount = sprintf(buff, "\n[%s]\n", area);

        if(ini_info->fsize + amount >= ini_info->allocated)
          if(!increase_ini_allocation(ini_info))
            return value;

        strcat(ini_info->ini, buff);
        ini_info->fsize += amount;
      }

      amount = sprintf(buff, "%s = %ld\n", indent, value);

      if(ini_info->fsize + amount >= ini_info->allocated)
        if(!increase_ini_allocation(ini_info))
          return value;

      strcat(ini_info->ini, buff);
      ini_info->fsize += amount;


      ini_info->changes_made = 1;
    }
    return(value);
  }



  pos = area_start;
  while(1)
  {
    tmp = get_ini_line(buff, 255, &pos, ini_info->ini);

    if(!tmp)          // No more lines
    {
      if(should_update)
        set_ini_number(ini_info, area, indent, value);
      return value;
    }

    strip_string(tmp);


    if(buff[0]=='[')  // Star of a new area
    {
      if(should_update)
        set_ini_number(ini_info, area, indent, value);
      return value;
    }

    break_up_ini(buff, w1, w2);

    if(strcmpi(w1, indent) == 0)
    {
      if(this_which == which)
        return(atol(w2));
      else
        ++this_which;
    }
  }
}

      


int ll_read_ini_string(ini_inforec *ini_info, char *area, char *indent, char *value, int max_value, int should_update, int which)
{
  char *tmp, buff[255], w1[128], w2[128];
  long area_start, pos;
  int amount, this_which = 0;


  area_start = find_ini_area(ini_info->ini, area);


  if(area_start >= ini_info->fsize)
  {

    if(should_update)
    {
      if(area[0])
      {
        amount = sprintf(buff, "\n[%s]\n", area);

        if(ini_info->fsize + amount >= ini_info->allocated)
          if(!increase_ini_allocation(ini_info))
            return 0;

        strcat(ini_info->ini, buff);
        ini_info->fsize += amount;
      }

      amount = sprintf(buff, "%s = %s\n", indent, value);

      if(ini_info->fsize + amount >= ini_info->allocated)
        if(!increase_ini_allocation(ini_info))
          return 0;

      strcat(ini_info->ini, buff);
      ini_info->fsize += amount;

      ini_info->changes_made = 1;
      return 1;           /* Value was added, area was created */
    }
    else
      return 0;
  }



  pos = area_start;
  while(1)
  {
    tmp = get_ini_line(buff, 255, &pos, ini_info->ini);

    if(!tmp)          // No more lines
    {
      if(should_update)
      {
        set_ini_string(ini_info, area, indent, value);
        return 2;
      }
      else
        return 0;
    }

    strip_string(tmp);

    if(buff[0]=='[')  // Star of a new area
    {
      if(should_update)
      {
        set_ini_string(ini_info, area, indent, value);
        return 2;
      }
      else
        return 0;
    }

    break_up_ini(buff, w1, w2);


    /* Check to see if indents match */
    if(strcmpi(w1, indent) == 0)
    {
      // which is for duplicate indents, only return it if this is the one
      // the user specified
      if(this_which == which)
      {
        if(w2[0])
        {
          strncpy(value, w2, max_value);
          value[max_value]=0;
        }
        return 3;
      }
      else
      {
        // otherwise, increment our 'this_which'
        ++this_which;
      }
    }
  }
}


void set_ini_string_dup(ini_inforec *ini_info, char *area, char *indent, char *value, int which)
{
  char buff[255], *tmp, w1[128], w2[128];
  long area_start, pos, save_pos;
  int amount, this_which=0;


  area_start = find_ini_area(ini_info->ini, area);

  if(area_start >= ini_info->fsize)
  {

    if(area[0])
    {
      amount = sprintf(buff, "\n[%s]\n", area);

      if(ini_info->fsize + amount >= ini_info->allocated)
        if(!increase_ini_allocation(ini_info))
          return;

      strcat(ini_info->ini, buff);
      ini_info->fsize+= amount;
    }

    amount = sprintf(buff, "%s = %s\n", indent, value);

    if(ini_info->fsize + amount >= ini_info->allocated)
      if(!increase_ini_allocation(ini_info))
        return;

    strcat(ini_info->ini, buff);
    ini_info->fsize+= amount;

    ini_info->changes_made = 1;
    return;
  }


  pos = area_start;
  while(1)
  {
    save_pos = pos;
    tmp = get_ini_line(buff, 255, &pos, ini_info->ini);

    if(!tmp)
      break;

    strip_string(buff);

    if(buff[0]=='[')  // Star of a new area, so break
    {
      save_pos = area_start;    // Put value right after [area], comment out this line to put at end
      break;
    }
    break_up_ini(buff, w1, w2);

    if(strcmpi(w1, indent) == 0)
    {
      if(this_which == which)
      {
        if(!w2[0])
          strcpy(w2, value);

        memmove(ini_info->ini + save_pos, ini_info->ini + pos, ini_info->fsize - pos+1);
        ini_info->fsize = ini_info->fsize - (pos - save_pos);
        break;
      }
      else
        ++this_which;
    }
  }

  amount = sprintf(buff, "%s = %s\n", indent, value);

  if(ini_info->fsize + amount >= ini_info->allocated)
    if(!increase_ini_allocation(ini_info))
      return;

  memmove(ini_info->ini+amount+save_pos, ini_info->ini+save_pos,
                               ini_info->fsize+amount-save_pos+1);
  strncpy(ini_info->ini+save_pos, buff, amount);
  ini_info->fsize += amount;

  ini_info->changes_made = 1;
  return;         // Value was updated
}


void set_ini_number_dup(ini_inforec *ini_info, char *area, char *indent, long value, int which)
{
  char buff[255], *tmp, w1[128], w2[128];
  long area_start, pos, save_pos;
  int amount, this_which = 0;


  area_start = find_ini_area(ini_info->ini, area);


  if(area_start >= ini_info->fsize)
  {

    if(area[0])
    {
      amount = sprintf(buff, "\n[%s]\n", area);

      if(ini_info->fsize + amount >= ini_info->allocated)
        if(!increase_ini_allocation(ini_info))
          return;

      strcat(ini_info->ini, buff);
      ini_info->fsize+= amount;
    }

    amount = sprintf(buff, "%s = %ld\n", indent, value);

    if(ini_info->fsize + amount >= ini_info->allocated)
      if(!increase_ini_allocation(ini_info))
        return;

    strcat(ini_info->ini, buff);
    ini_info->fsize+= amount;

    ini_info->changes_made = 1;
    return;
  }


  pos = area_start;
  while(1)
  {
    save_pos = pos;
    tmp = get_ini_line(buff, 255, &pos, ini_info->ini);

    if(!tmp)
      break;

    strip_string(buff);

    if(buff[0]=='[')  // Star of a new area, so break
    {
      save_pos = area_start;    // Put value right after [area], comment out this line to put at end
      break;
    }
    break_up_ini(buff, w1, w2);

    if(strcmpi(w1, indent) == 0)
    {
      if(this_which == which)
      {
        memmove(ini_info->ini + save_pos, ini_info->ini + pos, ini_info->fsize - pos+1);
        ini_info->fsize = ini_info->fsize - (pos - save_pos);
        break;
      }
      else
        ++this_which;
    }
  }

  amount = sprintf(buff, "%s = %ld\n", indent, value);

  if(ini_info->fsize + amount >= ini_info->allocated)
    if(!increase_ini_allocation(ini_info))
      return;

  memmove(ini_info->ini+amount+save_pos, ini_info->ini+save_pos,
                               ini_info->fsize+amount-save_pos+1);
  strncpy(ini_info->ini+save_pos, buff, amount);
  ini_info->fsize += amount;

  ini_info->changes_made = 1;
  return;
}


char **alloc_2d(int row, int col, unsigned size)
{
   int i;
   char **prow, *pdata;

   pdata = (char *) calloc(row * col, size);
   if (pdata == (char *) NULL)
   {
      fprintf(log, "No heap space for data\n");
      printf("Memory Error in alloc2d!\n");
      return(NULL);
   }
   prow  = (char **) malloc(row * sizeof (char *));

   if (prow == (char **) NULL)
   {
      fprintf(log, "No heap space for row pointers\n");
      printf("Memory Error (2) in alloc2d!\n");
      free(pdata);
      return(NULL);
   }

   for (i = 0; i < row; i++)
   {
     prow[i] = pdata;             /* store pointers to rows */
     pdata += size * col;         /* move to next row */
   }
   return prow;                   /* pointer to 2D array */
}

void free_2d(char **pa)
{
   free(*pa);                    /* free the data */
   free(pa);                     /* free pointer to row pointers */
}

long filesize(FILE *stream)
{
  long curpos, length;
  
  curpos=ftell(stream);
  
  fseek(stream, 0L, SEEK_END);
  length=ftell(stream);
  
  fseek(stream, curpos, SEEK_SET);
  
  return length;
}

char * strip_string(char *string)
{
  int x=0, y;
  while(isspace(string[x]) && string[x])
    ++x;
    
  y=strlen(string);
  memmove(string, string+x, y-x+1);
  string[y-x+1]=0;
  
  y=strlen(string);
  --y;
  
  while(isspace(string[y]) && y >= 0)
    --y;
    
  string[y+1]=0;
  
  
  return(string);
}

